/**
 * \file       dia_SystemSettingsModule.cpp
 *
 * \brief      Abstract base class for all kinds of modules need to be triggered by a system setting operation
 *
 * \details    Abstract base class for all kinds of modules need to be triggered by a system setting operation
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSystemSettings
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#include "dia_SystemSettingsModule.h"

#ifndef __INCLUDED_DIA_HASH_CALCULATOR__
#include <common/framework/utils/dia_HashCalculator.h>
#endif

static const dia_SystemSettingsModuleTypeInfo arSystemSettingsModuleTypeInfo[] = {
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_HMI,                   DIA_C_STR_SYSTEM_SETTING_TYPE_HMI,                    DIA_C_UID_SYSTEM_SETTING_TYPE_HMI },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_PRODUCTION,            DIA_C_STR_SYSTEM_SETTING_TYPE_PRODUCTION,             DIA_C_UID_SYSTEM_SETTING_TYPE_PRODUCTION },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_ALIGNMENT,             DIA_C_STR_SYSTEM_SETTING_TYPE_ALIGNMENT,              DIA_C_UID_SYSTEM_SETTING_TYPE_ALIGNMENT },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_CALIBRATION,           DIA_C_STR_SYSTEM_SETTING_TYPE_CALIBRATION,            DIA_C_UID_SYSTEM_SETTING_TYPE_CALIBRATION },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_CUSTOMER,              DIA_C_STR_SYSTEM_SETTING_TYPE_CUSTOMER,               DIA_C_UID_SYSTEM_SETTING_TYPE_CUSTOMER },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_USRMGMT_RESET_ALL,     DIA_C_STR_SYSTEM_SETTING_TYPE_USRMGMT_RESET_ALL,      DIA_C_UID_SYSTEM_SETTING_TYPE_USRMGMT_RESET_ALL },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_USRMGMT_RESET_USER,    DIA_C_STR_SYSTEM_SETTING_TYPE_USRMGMT_RESET_USER,     DIA_C_UID_SYSTEM_SETTING_TYPE_USRMGMT_RESET_USER },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_USRMGMT_RESET_BANK,    DIA_C_STR_SYSTEM_SETTING_TYPE_USRMGMT_RESET_BANK,     DIA_C_UID_SYSTEM_SETTING_TYPE_USRMGMT_RESET_BANK },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_USRMGMT_SWITCH_USER,   DIA_C_STR_SYSTEM_SETTING_TYPE_USRMGMT_SWITCH_USER,    DIA_C_UID_SYSTEM_SETTING_TYPE_USRMGMT_SWITCH_USER },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_USRMGMT_SWITCH_BANK,   DIA_C_STR_SYSTEM_SETTING_TYPE_USRMGMT_SWITCH_BANK,    DIA_C_UID_SYSTEM_SETTING_TYPE_USRMGMT_SWITCH_BANK },
      { DIA_C_U32_SYSTEM_SETTING_TYPE_MASK_USRMGMT_SAVE_BANK,     DIA_C_STR_SYSTEM_SETTING_TYPE_USRMGMT_SAVE_BANK,      DIA_C_UID_SYSTEM_SETTING_TYPE_USRMGMT_SAVE_BANK }
};

//-----------------------------------------------------------------------------

dia_SystemSettingsModule::dia_SystemSettingsModule ( tCString name )
   : mName(name), mTypeMask(0),isTypeMaskUsed(false)
{
   mUID = dia_getHashCodeFromString(mName);
}

//-----------------------------------------------------------------------------

dia_SystemSettingsModule::dia_SystemSettingsModule ( tCString name, tU32 mask )
   : mName(name), mTypeMask(mask),isTypeMaskUsed(true)
{
   mUID = dia_getHashCodeFromString(mName);
   dia_SystemSettingsModule::evaluateTypeMask();
}

//-----------------------------------------------------------------------------

dia_SystemSettingsModule::~dia_SystemSettingsModule ( void )
{
   mName = 0;
}

//-----------------------------------------------------------------------------

tCString
dia_SystemSettingsModule::getName ( void ) const
{
   return mName;
}

//-----------------------------------------------------------------------------

dia_UID
dia_SystemSettingsModule::getUID ( void ) const
{
   return mUID;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsModule::setTypeMask ( tU32 mask )
{
   mTypeMask = mask;
   evaluateTypeMask();
   isTypeMaskUsed = true;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsModule::evaluateTypeMask ( void  )
{
   dia_tclFnctTrace trc("dia_SystemSettingsModule::evaluateTypeMask()");

   mTypeLookupRep.clear();
   for ( tU16 i=0; i<(sizeof(arSystemSettingsModuleTypeInfo)/sizeof(dia_SystemSettingsModuleTypeInfo)); i++ )
   {
      if ( mTypeMask & arSystemSettingsModuleTypeInfo[i].mMaskValue )
      {
         DIA_TR_INF("Module \"%s\" supports system setting type \"%s\" (UID = 0x%08x)",mName,arSystemSettingsModuleTypeInfo[i].mTypeName,arSystemSettingsModuleTypeInfo[i].mUID);
         mTypeLookupRep[arSystemSettingsModuleTypeInfo[i].mUID] = arSystemSettingsModuleTypeInfo[i];
      }
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsModule::loadSystemSettingsModuleTypeInfo(const dia_SystemSettingsModuleTypeInfo typeInfoData[], tU16 numOfTypeInfos)
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsModule::loadSystemSettingTypeInformation()");
   tDiaResult retCode = DIA_FAILED;

   if (typeInfoData && (numOfTypeInfos > 0)) {
      for (tU16 i = 0; i < numOfTypeInfos; ++i) {
         dia_UID key = typeInfoData[i].mUID;
         if (!key) {
            key = dia_getHashCodeFromString(typeInfoData[i].mTypeName);
         }

         mTypeLookupRep[key] = typeInfoData[i];
         mTypeLookupRep[key].mUID = key;

         DIA_TR_INF("Added System Setting Type \"%s\" (UID=0x%08x)", mTypeLookupRep[key].mTypeName, mTypeLookupRep[key].mUID);
      }
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

bool
dia_SystemSettingsModule::isSystemSettingTypeSupported ( dia_UID typeUID )
{
   if ( !isTypeMaskUsed ) return true;

   std::map<dia_UID,dia_SystemSettingsModuleTypeInfo>::iterator iter = mTypeLookupRep.find(typeUID);
   if ( iter == mTypeLookupRep.end() )
   {
      DIA_TR_INF("System setting type (UID = 0x%08x) is not supported by module \"%s\"",typeUID,mName);
      return false;
   }

   DIA_TR_INF("System setting type \"%s\" (UID = 0x%08x) is supported by module \"%s\"",iter->second.mTypeName,typeUID,mName);
   return true;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsModule::getSystemSettingTypeName ( dia_UID typeUID, std::string& name )
{
   std::map<dia_UID,dia_SystemSettingsModuleTypeInfo>::iterator iter = mTypeLookupRep.find(typeUID);
   if ( iter == mTypeLookupRep.end() )
   {
      DIA_TR_INF("System setting type (UID = 0x%08x) not found in module \"%s\"",typeUID,mName);
      return DIA_FAILED;
   }

   name = std::string(iter->second.mTypeName);
   return DIA_SUCCESS;
}
