/*!
 * \file       dia_SystemSettingsManager.h
 *
 * \brief      Entry point to the system settings framework. Responsible for rolling out system
 *             wide system settings operations.
 *
 * \details    Class dia_SystemSettingsManager is the central entry point to the system settings
 *             framework of the diagnosis core. It is responsible for rolling out system wide
 *             system setting operations in a synchronized way. Therefore every system setting
 *             operation consists of three levels.
 *
 *             1st level - prepare:
 *             On receiving the prepare message from the diagnosis application every component
 *             should reject all messages from other applications that will have an impact on the
 *             component state. The component has to confirm that it is prepared for the system
 *             setting operation and that is waiting for the next level.
 *
 *             2nd level - execute:
 *             On receiving the execute message from the diagnosis application every component
 *             shall carry out all tasks required for the given system setting operation. The
 *             component has to inform the diagnosis about success or failure of the execute step.
 *
 *             3rd level - finalize:
 *             On receiving the finalize message from the diagnosis application every component
 *             is informed that the system setting operation is complete and that the component
 *             can continue with its normal operation mode.
 *
 *             Example sequence
 *
 *             \startuml
 *               actor Tester
 *               participant Diag #4090FF
 *               participant Comp1
 *               participant Comp2
 *               Tester -> Diag : setSystemToDeliveryState()
 *               Diag -> Comp1 : prepareSystemSet(DeliveryState)
 *               Diag -> Comp2 : prepareSystemSet(DeliveryState)
 *               Comp2 --> Diag : prepareSystemSetResponse(OK)
 *               Diag -> Diag : timeout()
 *               Diag -> Comp1 : prepareSystemSet(DeliveryState)
 *               Comp1 --> Diag : prepareSystemSetResponse(OK)
 *               Diag -> Comp1 : executeSystemSet(DeliveryState)
 *               Diag -> Comp2 : executeSystemSet(DeliveryState)
 *               Comp1 --> Diag : executeSystemSetResponse(OK)
 *               Comp2 --> Diag : executeSystemSetResponse(OK)
 *               Diag -> Comp1 : finalizeSystemSet(DeliveryState)
 *               Diag -> Comp2 : finalizeSystemSet(DeliveryState)
 *               Diag -> Tester : setSystemToDeliveryStateResponse(OK)
 *               opt finalize response
 *                 Comp1 --> Diag : finalizeSystemSetResponse(OK)
 *               end
 *             \enduml
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSystemSettings
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_MANAGER__
#define __INCLUDED_DIA_SYSTEM_SETTINGS_MANAGER__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SYSTEMSETTINGS__
#include <common/interfaces/dia_ISystemSettings.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEMSETTINGSFSM__
#include <common/framework/fsm/generated/dia_SystemSettingsFSM.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_TIMER_LISTENER__
#include <common/interfaces/dia_ITimerListener.h>
#endif

#ifndef __INCLUDED_DIA_QUEUE__
#include <common/framework/application/dia_Queue.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_SYSSET__
#include <common/framework/sysset/dia_common_sysset.h>
#endif

#ifndef __INCLUDED_DIA_FUNCTOR__
#include <common/framework/application/dia_Functor.h>
#endif

// forward declarations
class dia_SystemSettingsModule;
class dia_SystemSetPlugin;
class dia_SystemSettingsConfiguration;
/*!
 * \class   dia_SystemSettingsManager
 *
 * \brief   Foundation class that acts as entrypoint to the system settings subframework
 *
 *          This foundation class acts as entrypoint to the system settings subframework.
 *          It acts as manager (manager-plugin design pattern) that maintains a set of
 *          plugins (dia_SystemSettingsModule and dia_SystemSettingsPlugin) and implements
 *          the system settings state machine to control the process of a system wide
 *          system settings operation (setSystemToDeliveryState, language and unit changes,
 *          user profile switching,...)
 *
 * \see     dia_ISystemSettings, dia_SystemSettingsFSM::FsmBehaviour, dia_ITimerListener
 *
 * \ingroup diaCoreSystemSettings
 *
 */
class dia_SystemSettingsManager
   : public dia_ISystemSettings,
     public dia_SystemSettingsFSM::FsmBehaviour,
     public dia_ITimerListener
{
   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_SystemSettingsManager);
   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_SystemSettingsManager);

   enum dia_enProcessingStatus
   {
      DIA_EN_SYSSET_PROCSTATUS_UNKNOWN = 0,
      DIA_EN_SYSSET_PROCSTATUS_PROCESSING_PLUGINS,
      DIA_EN_SYSSET_PROCSTATUS_PROCESSING_MODULES,
      DIA_EN_SYSSET_PROCSTATUS_COMPLETE,
      DIA_EN_SYSSET_PROCSTATUS_COUNT
   };

   struct dia_SystemSettingStatus
   {
      dia_SystemSettingStatus ( void ) : mpModule(0), mRequestSent(false), mACKReceived(false), mIsACKRequired(true), mErrorCode(DIA_E_NOERROR) {}
      dia_SystemSettingStatus ( dia_SystemSettingsModule& module ) : mpModule(&module), mRequestSent(false), mACKReceived(false), mIsACKRequired(true), mErrorCode(DIA_E_NOERROR) {}
      dia_SystemSettingStatus ( const dia_SystemSettingStatus& obj ) : mpModule(obj.mpModule), mRequestSent(obj.mRequestSent), mACKReceived(obj.mACKReceived), mIsACKRequired(obj.mIsACKRequired), mErrorCode(obj.mErrorCode) {}
      dia_SystemSettingStatus& operator = ( const dia_SystemSettingStatus& obj ) {
         if ( this != &obj ) {
            mpModule       = obj.mpModule;
            mRequestSent   = obj.mRequestSent;
            mACKReceived   = obj.mACKReceived;
            mIsACKRequired = obj.mIsACKRequired;
            mErrorCode     = obj.mErrorCode;
         }
         return *this;
      }

      dia_SystemSettingsModule* mpModule;
      bool  mRequestSent;
      bool  mACKReceived;
      bool  mIsACKRequired;
      tDiaResult mErrorCode;
   };
   


   struct dia_SystemSettingRequest
   {
      DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_SystemSettingRequest);

   public:
      //! constructor
      dia_SystemSettingRequest ( dia_UID typeID, const dia_SystemSettingExtendedData& extData, dia_ISystemSettingsListener& requester, tCookieType cookie )
         : mType(typeID), mExtData(extData), mRequester(requester), mCookie(cookie)
      {}
      //! constructor
      dia_SystemSettingRequest ( dia_UID typeID, const dia_SystemSettingExtendedData& extData, std::vector<dia_UID> modules, dia_ISystemSettingsListener& requester, tCookieType cookie )
         : mType(typeID), mExtData(extData), mModules(modules), mRequester(requester), mCookie(cookie)
      {}
      //! constructor
      dia_SystemSettingRequest ( dia_UID typeID, dia_ISystemSettingsListener& requester, tCookieType cookie )
         : mType(typeID), mRequester(requester), mCookie(cookie)
      {}
      //! destructor
      virtual ~dia_SystemSettingRequest ( void )
      {}

      //! requested type
      dia_UID mType;
      //! extended data for the requested system setting type
      dia_SystemSettingExtendedData mExtData;
      //! selection of modules
      std::vector<dia_UID> mModules;
      //! service ID to be used for executing the request
      dia_ISystemSettingsListener& mRequester;
      //! data ID to be used for executing the request
      tCookieType mCookie;
   };

   struct dia_RemoteControlRequest
   {
      DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_RemoteControlRequest);

   public:
      //! constructor
      dia_RemoteControlRequest ( dia_enSystemSettingMode mode, dia_ISystemSettingsListener& requester, tCookieType cookie )
         : mMode(mode), mRequester(requester), mCookie(cookie)
      {}
      //! destructor
      virtual ~dia_RemoteControlRequest ( void )
      {}

      //! request mode
      dia_enSystemSettingMode mMode;
      //! service ID to be used for executing the request
      dia_ISystemSettingsListener& mRequester;
      //! data ID to be used for executing the request
      tCookieType mCookie;
   };

   struct dia_RemoteSystemSettingRequest
   {
      DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_RemoteSystemSettingRequest);

   public:
      //! constructor
      dia_RemoteSystemSettingRequest ( dia_UID typeID, dia_enSystemSettingLevel level, const dia_SystemSettingExtendedData& extData, dia_ISystemSettingsListener& requester, tCookieType cookie )
         : mType(typeID), mLevel(level), mExtData(extData), mRequester(requester), mCookie(cookie)
      {}
      //! constructor
      dia_RemoteSystemSettingRequest ( dia_UID typeID, dia_enSystemSettingLevel level, dia_ISystemSettingsListener& requester, tCookieType cookie )
         : mType(typeID), mLevel(level), mRequester(requester), mCookie(cookie)
      {}
      //! destructor
      virtual ~dia_RemoteSystemSettingRequest ( void )
      {}

      //! requested type of system setting
      dia_UID mType;
      //! requested level (prepare, execute or finalize)
      dia_enSystemSettingLevel mLevel;
      //! extended data for the requested system setting type
      dia_SystemSettingExtendedData mExtData;
      //! service ID to be used for executing the request
      dia_ISystemSettingsListener& mRequester;
      //! data ID to be used for executing the request
      tCookieType mCookie;
   };


   class SystemSettingsFirstError {
   public:
      enum enLevelPhase
         {
            enUNKNOWN = 0,
            enPLUGINS,
            enMODULES,
            enCOUNT
         };

      SystemSettingsFirstError() {
         reset();
      }

      void reset(tU16 retryCounter=0);
      void addError(char const *itemName, dia_enSystemSettingLevel level, tDiaResult errorCode, char const *errorText);
      void addTimeout(dia_enSystemSettingLevel level, uint32_t numTimeout);

      bool bHasError() const {
         return mErrorCode!=DIA_SUCCESS;
      }
      void setLevelPhase(enLevelPhase levelPhase) {
         mLevelPhase=levelPhase;
      }

      void toString(std::string &res) const;
      void setProperty() const;

   private:
      char const *levelPhaseToString() const;

      char const *levelToString() const;
      uint32_t mNumFailed;
      uint32_t mNumTimeout;
      std::string mItemName;
      dia_enSystemSettingLevel mLevel;
      enLevelPhase mLevelPhase;
      tDiaResult mErrorCode;
      uint32_t mRetryCounter;
      std::string mErrorText;
   }; 

public:
   //! set system setting framework to slave or master mode. in slave mode an external master will trigger the level requests
   virtual tDiaResult setSystemSettingMode ( dia_enSystemSettingMode mode, dia_ISystemSettingsListener& requester, tCookieType cookie );
   //! retrieve the current mode of the system setting framework. can be either in master or slave mode
   virtual dia_enSystemSettingMode getSystemSettingMode ( void ) { return mMode; }

   //! perform the requested type of system setting in master mode. that means all three levels (prepare, execute, finallize) will be carried out by the framework itself
   virtual tDiaResult processSystemSetting ( dia_UID typeID, dia_ISystemSettingsListener& pRequester, tCookieType cookie );
   //! perform the requested type of system setting in master mode. that means all three levels (prepare, execute, finallize) will be carried out by the framework itself
   virtual tDiaResult processSystemSetting ( dia_UID typeID, const dia_SystemSettingExtendedData& extData, dia_ISystemSettingsListener& pRequester, tCookieType cookie );
   //! perform the requested type of system setting in master mode. that means all three levels (prepare, execute, finallize) will be carried out by the framework itself
   virtual tDiaResult processSystemSetting ( dia_UID typeID, const std::vector<dia_UID>& modules, dia_ISystemSettingsListener& pRequester, tCookieType cookie );
   //! perform the requested type of system setting in master mode. that means all three levels (prepare, execute, finallize) will be carried out by the framework itself
   virtual tDiaResult processSystemSetting ( dia_UID typeID, const dia_SystemSettingExtendedData& extData, const std::vector<dia_UID>& modules, dia_ISystemSettingsListener& pRequester, tCookieType cookie );

   //! perform the requested type of system setting in slave mode. that means the level to be carried out (prepare, execute, finallize) is explicitly requested by the remote master
   virtual tDiaResult processSystemSetting ( dia_UID typeID, dia_enSystemSettingLevel level, dia_ISystemSettingsListener& pRequester, tCookieType cookie );
   //! perform the requested type of system setting in slave mode. that means the level to be carried out (prepare, execute, finallize) is explicitly requested by the remote master
   virtual tDiaResult processSystemSetting ( dia_UID typeID, dia_enSystemSettingLevel level, const dia_SystemSettingExtendedData& extData, dia_ISystemSettingsListener& pRequester, tCookieType cookie );

   //! install a signal handler
   virtual tDiaResult addSystemSettingsModule ( dia_SystemSettingsModule& module );
   //! install a signal handler
   virtual tDiaResult addSystemSettingsPlugin ( dia_SystemSettingsPlugin& plugin );

   //! retrieve the list of system setting modules
   const std::set<dia_SystemSettingsModule*>* getModules ( void ) const { return &mModuleRep; }
   //! retrieve the list of system setting modules
   const std::list<dia_SystemSettingsPlugin*>* getPlugins ( void ) const { return &mPluginRep; }

   //! return the configuration object
//   /*const*/ dia_SystemSettingsConfiguration& getConfiguration ( void ) { return (*mpConfig); }

   //! query for a certain plugin based on a UID
   virtual tDiaResult queryPlugin ( dia_UID uid, dia_SystemSettingsPlugin** ppPlugin );
   //! query for a certain plugin based on a name (UID will be calculated internally)
   virtual tDiaResult queryPlugin ( tCString name, dia_SystemSettingsPlugin** ppPlugin );

   //! overloaded method from class dia_ITimerListener
   void vOnTimerElapsed ( dia_TimerID id );

   //! overloaded method from class dia_ITimerListener
   virtual void vOnTimerElapsedLocal ( dia_TimerID id );


   void onSystemSettingsModuleUpdate ( dia_SystemSettingsModule& module, dia_UID sysSetTypeUID, dia_enSystemSettingLevel level, tDiaResult errCode );

   virtual void onSystemSettingsModuleUpdateLocal ( dia_SystemSettingsModule& module, dia_UID sysSetTypeUID, dia_enSystemSettingLevel level, tDiaResult errCode );


   virtual void onProcessEvents ( void );

   //! assign a new configuration to the system setting manager
   tDiaResult setConfiguration ( dia_SystemSettingsConfiguration& config );
   //! assign a new configuration to the system setting manager
   /*const*/ dia_SystemSettingsConfiguration* getConfiguration ( void ) const { return mpConfig; }

   virtual tU16 getRetryCounter() const;

protected:
    //! default class constructor
   dia_SystemSettingsManager ( void );
    //! class destructor
   virtual ~dia_SystemSettingsManager ( void );

   //! setup method for the application
   virtual tDiaResult setup ( void );
   //! shutdown method for the application
   virtual tDiaResult tearDown ( void );

   //! perform actions on plugins for mActiveLevel
   virtual tDiaResult execPlugins ( dia_UID sysSetTypeUID );
   //! perform the system set operation
   virtual tDiaResult runToCompletionPlugins ( dia_UID sysSetTypeUID );

   char const *levelToString() {
      switch (mActiveLevel) {
         case DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE: return "PREPARE";
         case DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE: return "EXECUTE";
         case DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE: return "FINALIZE";
         case DIA_EN_SYSTEM_SETTING_LEVEL_COUNT:
         case DIA_EN_SYSTEM_SETTING_LEVEL_UNKNOWN:
         default:
            return "UNKNOWN";
      }
   }


   // FSM actions
   virtual void vFsmEvaluateResponse ( void* pArg );
   virtual void vFsmExecutionCompleted ( void* pArg );
   virtual void vFsmFinalizationCompleted ( void* pArg );
   virtual void vFsmFinalizeSlaveMode ( void* pArg );
   virtual void vFsmFinalizeSlaveProcessing ( void* pArg );
   virtual void vFsmFinalizeSystemSetting ( void* pArg );
   virtual void vFsmHandleError ( void* pArg );
   virtual void vFsmInitializeExecution ( void* pArg );
   virtual void vFsmInitializeFinalization ( void* pArg );
   virtual void vFsmInitializeMasterSystemSetting ( void* pArg );
   virtual void vFsmInitializePreparation ( void* pArg );
   virtual void vFsmInitializeProcessing ( void* pArg );
   virtual void vFsmInitializeRetry ( void* pArg );
   virtual void vFsmInitializeSlaveMode ( void* pArg );
   virtual void vFsmInitializeSlaveProcessing ( void* pArg );
   virtual void vFsmInitializeSlaveSystemSetting ( void* pArg );
   //! \see  dia_SystemSettingsFSM::FsmBehaviour::vFsmInitializeSystemSetting(void* pArg)
   virtual void vFsmInitializeSystemSetting ( void* pArg );
   virtual void vFsmPreparationCompleted ( void* pArg );
   virtual void vFsmProcessPlugins ( void* pArg );
   virtual void vFsmRejectRemoteSystemSettingRequest ( void* pArg );
   virtual void vFsmRejectSystemSettingRequest ( void* pArg );
   virtual void vFsmSendNegResponseToMaster ( void* pArg );
   virtual void vFsmSendPosResponseToMaster ( void* pArg );
   virtual void vFsmSendRemoteControlResponseToMaster ( void* pArg );
   virtual void vFsmSendResponse ( void* pArg );
   virtual void vFsmSendSystemSettingsRequests ( void* pArg );
   virtual void vFsmSetProcessingTimeout ( void* pArg );
   virtual void vFsmSetRetryTimeout ( void* pArg );
   virtual void vFsmStartProcessingTimer ( void* pArg );
   virtual void vFsmStopProcessingTimer ( void* pArg );

   // FSM guards
   virtual bool bFsmIsCompleteOK ( void* pArg );
   virtual bool bFsmIsNotificationRequired ( void* pArg );
   virtual bool bFsmIsRetryCheckOK ( void* pArg );
   virtual bool bFsmIsRetryCounterExceeded ( void* pArg );

   virtual void preProcess ( void );
   virtual void postProcess ( void );

private:
   virtual void postEvent ( dia_FunctorVoid *functor );

protected:
   //! pointer to the configuration object
   dia_SystemSettingsConfiguration* mpConfig; //lint -sem(dia_SystemSettingsManager::tearDown,cleanup)
   //! generated state machine
   dia_SystemSettingsFSM::Fsm* mpFSM; //lint -sem(dia_SystemSettingsManager::tearDown,cleanup)
   //! message queue used for communication with the server engine
   dia_Queue<dia_FunctorVoid>* mpMsgQueue; //lint -sem(dia_SystemSettingsManager::tearDown,cleanup)
   //! timer used when rolling out a system setting
   dia_Timer mTimer;
   //! processing timeout value
   tU32 mProcessingTimeout;
   //! retry counter
   tU16 mRetryCounter;
   //!
   tDiaResult mErrorCode;

   //! system setting mode. Could be master or slave mode. In slave mode the system setting operation is controlled by a remote master
   dia_enSystemSettingMode mMode;
   //! active request
   dia_SystemSettingRequest* mpActiveRequest; //lint -sem(dia_SystemSettingsManager::tearDown,cleanup)
   //! active request
   dia_RemoteSystemSettingRequest* mpActiveRemoteRequest; //lint -sem(dia_SystemSettingsManager::tearDown,cleanup)
   //! active remote control request
// dia_RemoteControlRequest* mpActiveRemoteControlRequest; //lint -sem(dia_SystemSettingsManager::tearDown,cleanup)
   //! ID of active system setting type
   dia_UID mActiveTypeID;
   //! active level
   dia_enSystemSettingLevel mActiveLevel;
   //! processing status for the existing levels
   dia_enProcessingStatus mProcessingStatus[DIA_EN_SYSTEM_SETTING_LEVEL_COUNT];
   //! processing status for the existing levels
   tDiaResult mPluginResults[DIA_EN_SYSTEM_SETTING_LEVEL_COUNT];
   //! processing status for the existing levels
   tDiaResult mModuleResults[DIA_EN_SYSTEM_SETTING_LEVEL_COUNT];
   //! requested type of system setting
   dia_UID mRequestedTypeID;
   //! requested area to which the system setting operation shall be rolled out
   dia_UID mRequestedAreaID;
   //!
   dia_enSystemSettingLevel mRequestedLevel;

   bool mIsSetupDone;

   //! repository of system setting modules
   std::set<dia_SystemSettingsModule*>  mModuleRep;
   //! repository of diagnostic specific plugins controlled by this manager class
   std::list<dia_SystemSettingsPlugin*> mPluginRep;

   std::map<dia_SystemSettingsModule*,dia_SystemSettingStatus> mStatusRep;

   SystemSettingsFirstError mSystemSettingsFirstError;
};

//-----------------------------------------------------------------------------

dia_SystemSettingsManager* getInstanceOfSystemSettingsManager ( void );
void releaseInstanceOfSystemSettingsManager ( void );

#endif /* __INCLUDED_DIA_SYSTEM_SETTINGS_MANAGER__ */
