/**
 * \file       dia_SystemSettingsManager.cpp
 *
 * \brief      Entry point to the system settings framework. Responsible for rolling out system
 *             wide system setting operations.
 *
 * \details    Class dia_SystemSettingsManager is the central entry point to the system settings
 *             framework of the diagnosis core. It is responsible for rolling out system wide
 *             system setting operations in a synchronized way. Therefore every system setting
 *             operation consists of three levels.
 *
 *             1st level - prepare:
 *             On receiving the prepare message from the diagnosis application every component
 *             should reject all messages from other applications that will have an impact on the
 *             component state. The component has to confirm that it is prepared for the system
 *             setting operation and that is waiting for the next level.
 *
 *             2nd level - execute:
 *             On receiving the execute message from the diagnosis application every component
 *             shall carry out all tasks required for the given system setting operation. The
 *             component has to inform the diagnosis about success or failure of the execute step.
 *
 *             3rd level - finalize:
 *             On receiving the finalize message from the diagnosis application every component
 *             is informed that the system setting operation is complete and that the component
 *             can continue with its normal operation mode.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSystemSettings
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#ifndef __INCLUDED_DIA_MAIN__
#include "../../depricated/dia_main.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include <common/framework/config/dia_ConfigManager.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_MODULE__
#include <common/framework/sysset/dia_SystemSettingsModule.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_PLUGIN__
#include <common/framework/sysset/dia_SystemSettingsPlugin.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_MANAGER__
#include <common/framework/sysset/dia_SystemSettingsManager.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_CONFIGURATION__
#include <common/framework/sysset/dia_SystemSettingsConfiguration.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SYSTEMSETTINGS_LISTENER__
#include <common/interfaces/dia_ISystemSettingsListener.h>
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include <common/framework/application/dia_Application.h>
#endif

#ifndef __INCLUDED_DIA_FUNCTOR__
#include <common/framework/application/dia_Functor.h>
#endif


#ifndef __INCLUDED_DIA_HASH_CALCULATOR__
#include <common/framework/utils/dia_HashCalculator.h>
#endif

#include <sstream>

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_SystemSettingsManager)


#ifndef __DIA_UNIT_TESTING__

dia_SystemSettingsManager*
getInstanceOfSystemSettingsManager ( void )
{
   return dia_SystemSettingsManager::getInstance();
}

void
releaseInstanceOfSystemSettingsManager ( void )
{
   dia_SystemSettingsManager::deleteInstance();
}

#endif

using namespace dia;

static const tU32 DIA_C_U32_DEFAULT_PROCESSING_TIMEOUT = 500;

static const tCString arSystemSettingLevelMapping[DIA_EN_SYSTEM_SETTING_LEVEL_COUNT] = {
   "DIA_EN_SYSTEM_SETTING_LEVEL_UNKNOWN",
   "DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE",
   "DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE",
   "DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE"
};

#define SYSSET_QUEUE_SIZE       100

static tCString strSystemSettingQueueName  = "DIA_SYSSETQ";

//-----------------------------------------------------------------------------

dia_SystemSettingsManager::dia_SystemSettingsManager ( void )
   : mpConfig(0),
     mpFSM(0),
     mpMsgQueue(0),
     mProcessingTimeout(0),
     mRetryCounter(0),
     mErrorCode(DIA_SUCCESS),
     mMode(DIA_EN_SYSTEM_SETTING_MODE_MASTER),
     mpActiveRequest(0),
     mpActiveRemoteRequest(0),
     mActiveTypeID(0),
     mActiveLevel(DIA_EN_SYSTEM_SETTING_LEVEL_UNKNOWN),
     mRequestedTypeID(0),
     mRequestedAreaID(0),
     mRequestedLevel(DIA_EN_SYSTEM_SETTING_LEVEL_UNKNOWN),
     mIsSetupDone(false)
{
   ScopeTrace trc("dia_SystemSettingsManager::dia_SystemSettingsManager");

   mTimer.s32Create();

   for ( tU16 i=0; i<DIA_EN_SYSTEM_SETTING_LEVEL_COUNT; i++ )
   {
      mProcessingStatus[i] = DIA_EN_SYSSET_PROCSTATUS_UNKNOWN;
      mPluginResults[i]    = DIA_SUCCESS;
      mModuleResults[i]    = DIA_SUCCESS;
   }
}

//-----------------------------------------------------------------------------

dia_SystemSettingsManager::~dia_SystemSettingsManager ( void )
{
   _BP_TRY_BEGIN
   {
      (void) dia_SystemSettingsManager::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SystemSettingsManager::~dia_SystemSettingsManager !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::setup ( void )
{
   ScopeTrace trc("dia_SystemSettingsManager::setup");

   // we setup the engine object only once
   if ( mIsSetupDone )
   {
      DIA_TR_INF("### SYSTEM SET MANAGER WAS ALREADY SET UP. RETURNING... ###");
      return DIA_SUCCESS;
   }

   mIsSetupDone = true;

   // create the state machine object
   if ( dia_SystemSettingsFSM::Fsm::createFSM (&mpFSM,this) != true )
   {
      DIA_TR_INF("### FAILED TO CREATE SYSTEM SETTING MANAGER STATE MACHINE ###");
      return DIA_FAILED;
   }

   //
   // create message queue
   //
   mpMsgQueue = new dia_Queue<dia_FunctorVoid>(strSystemSettingQueueName,SYSSET_QUEUE_SIZE);
   if ( !mpMsgQueue || (mpMsgQueue->open() != DIA_SUCCESS) ) return DIA_FAILED;

   mpConfig = new dia_SystemSettingsConfiguration("DefaultSystemSettingsConfiguration");
   if ( !mpConfig )
   {
      DIA_TR_ERR("##### FAILED TO CREATE DEFAULT SYSTEM SETTING CONFIGURATION OBJECT #####");
      return DIA_E_MEMALLOC_FAILED;
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::tearDown ( void )
{
   ScopeTrace trc("dia_SystemSettingsManager::tearDown");

   DIA_IMPL_SET_REPOSITORY_TEAR_DOWN(dia_SystemSettingsModule,mModuleRep);
   DIA_IMPL_LIST_REPOSITORY_TEAR_DOWN(dia_SystemSettingsPlugin,mPluginRep);

   mTimer.s32Delete();
   mTimer.removeTimerListener(this);

   if ( mpMsgQueue )
   {
      mpMsgQueue->close();
      delete mpMsgQueue;
      mpMsgQueue = 0;
   }

   delete mpFSM;                  mpFSM = 0;
   delete mpActiveRequest;        mpActiveRequest = 0;
   delete mpActiveRemoteRequest;  mpActiveRemoteRequest = 0;
   delete mpConfig;               mpConfig = 0;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::setConfiguration ( dia_SystemSettingsConfiguration& config )
{
   delete mpConfig;
   mpConfig = config.clone();
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tU16 dia_SystemSettingsManager::getRetryCounter() const {
   return mRetryCounter;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::addSystemSettingsModule ( dia_SystemSettingsModule& module )
{
   ScopeTrace trc("dia_SystemSettingsManager::addSystemSettingsModule");

   mModuleRep.insert(&module);
   DIA_TR_INF("dia_SystemSettingsManager::addSystemSetModule \"%s\"",module.getName());
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::addSystemSettingsPlugin ( dia_SystemSettingsPlugin& plugin )
{
   ScopeTrace trc("dia_SystemSettingsManager::addSystemSettingsPlugin");

   tDiaResult retCode = DIA_FAILED;

   bool found = false;

   std::list<dia_SystemSettingsPlugin*>::iterator iter = mPluginRep.begin();
   for ( ; iter != mPluginRep.end(); ++iter )
   {
      if ( (*iter)->getUID() == plugin.getUID() )
      {
         found = true;
         break;
      }
   }

   if ( !found )
   {
      mPluginRep.push_back(&plugin);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::execPlugins ( dia_UID sysSetTypeUID) {
   dia_ScopeTraceVarg oTrace("dia_SystemSettingsManager::execPlugins() %s", levelToString());
   tDiaResult retCode = DIA_SUCCESS;
   std::list<dia_SystemSettingsPlugin*>::iterator iter = mPluginRep.begin();
   mSystemSettingsFirstError.setLevelPhase(SystemSettingsFirstError::enPLUGINS);
   for ( ; iter != mPluginRep.end() ; ++iter )
   {
      dia_SystemSettingsPlugin *curPlugin=*iter;
      if ( !curPlugin->isSystemSettingTypeSupported(sysSetTypeUID)) {
         continue;
      } 
      tDiaResult errCode=DIA_FAILED;
      switch (mActiveLevel) {
         case DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE:
            errCode=curPlugin->prepare(sysSetTypeUID);
            break;
         case DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE:
            errCode=curPlugin->execute(sysSetTypeUID);
            break;
         case DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE:
            errCode=curPlugin->finalize(sysSetTypeUID);
            break;
         default:
            break;
      }
      if (errCode != DIA_SUCCESS)
      {
         DIA_TR_ERR("##### %s FAILED FOR PLUGIN \"%s\" (MODE = %d)", levelToString(), curPlugin->getName(), sysSetTypeUID);
         //EM_TRACE Entry
//		 DIA_TR_ERRMEM("##### PREPARATION FAILED FOR PLUGIN \"%s\" (MODE = %d)", curPlugin->getName(), sysSetTypeUID);
         mSystemSettingsFirstError.addError(curPlugin->getName(), mActiveLevel, errCode, "execute plugin failed");
         retCode = DIA_FAILED;
      }
   }
   mSystemSettingsFirstError.setLevelPhase(SystemSettingsFirstError::enUNKNOWN);

   return retCode;

}



//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::runToCompletionPlugins ( dia_UID sysSetTypeUID )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::runToCompletionPlugins()");

   tDiaResult retCode = DIA_SUCCESS;

   std::list<dia_SystemSettingsPlugin*>::iterator iter = mPluginRep.begin();

   for ( ; iter != mPluginRep.end(); ++iter )
   {
      if ( (*iter)->runToCompletion(sysSetTypeUID) != DIA_SUCCESS )
      {
         DIA_TR_ERR("##### RUN TO COMPLETION FAILED FOR PLUGIN \"%s\" (MODE = %d)", (*iter)->getName(), sysSetTypeUID);
         //EM_TRACE Entry
//         DIA_TR_ERRMEM("##### RUN TO COMPLETION FAILED FOR PLUGIN \"%s\" (MODE = %d)", (*iter)->getName(), sysSetTypeUID);
         retCode = DIA_FAILED;
      }
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::queryPlugin ( dia_UID uid, dia_SystemSettingsPlugin** ppPlugin )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::queryPlugin(dia_UID,dia_SystemSettingsPlugin**)");

   tDiaResult retCode = DIA_FAILED;

   if ( ppPlugin )
   {
      *ppPlugin = 0;

      std::list<dia_SystemSettingsPlugin*>::iterator iter = mPluginRep.begin();

      for ( ; iter != mPluginRep.end(); ++iter )
      {
         dia_SystemSettingsPlugin* pPlugin = (*iter);
         if ( pPlugin && (pPlugin->getUID() == uid) )
         {
            *ppPlugin = pPlugin;
            retCode = DIA_SUCCESS;
            break;
         }
      }
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::queryPlugin ( tCString name, dia_SystemSettingsPlugin** ppPlugin )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::queryPlugin(tCString,dia_SystemSettingsPlugin**)");

   return queryPlugin(dia_getHashCodeFromString(name),ppPlugin);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::setSystemSettingMode ( dia_enSystemSettingMode mode, dia_ISystemSettingsListener& requester, tCookieType cookie )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::setSystemSettingMode(mode)");
   /* todo hpe2hi: 
      check if we are in the wrong thread.
      if so, we have send needed information to the worker-thread.
      we must not use any variabled of the class the might be changed by other threads
   */
   // mode has to be DIA_EN_SYSTEM_SETTING_MODE_SLAVE or DIA_EN_SYSTEM_SETTING_MODE_MASTER

   if ( mode == mMode )
   {
      DIA_TR_INF("##### Requested mode already active. No action performed #####");
      requester.onSetSystemSettingMode(mode,DIA_SUCCESS,cookie);
      return DIA_SUCCESS;
   }

   // here we know that we have to change the mode
   dia_RemoteControlRequest* pRemoteControlRequest = new dia_RemoteControlRequest(mode,requester,cookie);

   if ( pRemoteControlRequest && mpFSM )
   {
      DIA_TR_INF("pRemoteControlRequest = %p",pRemoteControlRequest);

      if ( mode == DIA_EN_SYSTEM_SETTING_MODE_SLAVE )
      {
         DIA_TR_INF("##### Switching to SLAVE mode #####");
         mpFSM->acceptEvent(dia_SystemSettingsFSM::evSlaveMode,pRemoteControlRequest); //lint !e429: custodial pointer is freed by FSM after message has been processed or in destructor
      }
      else
      {
         DIA_TR_INF("##### Switching to MASTER mode #####");
         mpFSM->acceptEvent(dia_SystemSettingsFSM::evReturnControl,pRemoteControlRequest); //lint !e429: custodial pointer is freed by FSM after message has been processed or in destructor
      }  //lint !e429: custodial pointer is freed by FSM after message has been processed or in destructor

      return DIA_SUCCESS;  //lint !e429: custodial pointer is freed by FSM after message has been processed or in destructor
   } //lint !e429: custodial pointer is freed by FSM after message has been processed or in destructor

   // make lint happy
   pRemoteControlRequest = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pRemoteControlRequest'. --> lifetime is controlled by system settings state machine

   return DIA_FAILED; //lint !e438 Warning: last value assigned to variable 'pRemoteControlRequest' not used
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::processSystemSetting ( dia_UID typeID, const dia_SystemSettingExtendedData& extData, dia_ISystemSettingsListener& requester, tCookieType cookie )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::processSystemSetting(dia_UID,dia_SystemSettingExtendedData&,dia_ISystemSettingsListener&,tCookieType)");

   dia_SystemSettingsConfiguration* pConfig = getConfiguration();
   if ( (!pConfig) || (!(pConfig->isSystemSettingTypeSupported(typeID))) )
   {
      DIA_TR_ERR("##### UNABLE TO ROLL OUT SYSTEM SETTING REQUEST --- INVALID TYPE  UID (0x%08x) #####",typeID);
      //EM_TRACE Entry
//      DIA_TR_ERRMEM("##### UNABLE TO ROLL OUT SYSTEM SETTING REQUEST --- INVALID TYPE  UID (0x%08x) #####",typeID);
      return DIA_E_SYSSET_INVALID_TYPE;
   }

   dia_SystemSettingRequest* pActiveRequest  = new dia_SystemSettingRequest(typeID,extData,requester,cookie);

   if ( pActiveRequest && mpFSM ) mpFSM->acceptEvent(dia_SystemSettingsFSM::evSystemSettingRequest,pActiveRequest); //lint !e429: custodial pointer is freed by FSM after message has been processed or in destructor

   // make lint happy
   pActiveRequest = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pActiveRequest'. --> lifetime is controlled by system settings state machine

   return DIA_SUCCESS; //lint !e438 Warning: last value assigned to variable 'pActiveRequest' not used
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::processSystemSetting ( dia_UID typeID, dia_ISystemSettingsListener& requester, tCookieType cookie )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::processSystemSetting(dia_UID,dia_ISystemSettingsListener&,tCookieType)");
   dia_SystemSettingExtendedData extData;
   return processSystemSetting(typeID,extData,requester,cookie);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::processSystemSetting ( dia_UID typeID, const dia_SystemSettingExtendedData& extData, const std::vector<dia_UID>& modules, dia_ISystemSettingsListener& requester, tCookieType cookie )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::processSystemSetting(dia_UID,dia_SystemSettingExtendedData&,vector<dia_UID>&,dia_ISystemSettingsListener&,tCookieType)");

   dia_SystemSettingsConfiguration* pConfig = getConfiguration();
   if ( (!pConfig) || (!(pConfig->isSystemSettingTypeSupported(typeID))) )
   {
      DIA_TR_ERR("##### UNABLE TO ROLL OUT SYSTEM SETTING REQUEST --- INVALID TYPE  UID (0x%08x) #####",typeID);
      //EM_TRACE Entry
//      DIA_TR_ERRMEM("##### UNABLE TO ROLL OUT SYSTEM SETTING REQUEST --- INVALID TYPE  UID (0x%08x) #####",typeID);
      return DIA_E_SYSSET_INVALID_TYPE;
   }

   dia_SystemSettingRequest* pRequest = new dia_SystemSettingRequest(typeID,extData,modules,requester,cookie); //lint !e429: custodial pointer is freed by FSM after message has been processed or in destructor

   if ( pRequest && mpFSM ) mpFSM->acceptEvent(dia_SystemSettingsFSM::evSystemSettingRequest,pRequest);

   // make lint happy
   pRequest = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pActiveRequest'. --> lifetime is controlled by system settings state machine

   return DIA_SUCCESS; //lint !e438 Warning: last value assigned to variable 'pActiveRequest' not used
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::processSystemSetting ( dia_UID typeID, const std::vector<dia_UID>& modules, dia_ISystemSettingsListener& requester, tCookieType cookie )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::processSystemSetting(dia_UID,vector<dia_UID>&,dia_ISystemSettingsListener&,tCookieType)");
   dia_SystemSettingExtendedData extData;
   return processSystemSetting(typeID,extData,modules,requester,cookie);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::processSystemSetting ( dia_UID typeID, dia_enSystemSettingLevel level, const dia_SystemSettingExtendedData& extData, dia_ISystemSettingsListener& requester, tCookieType cookie )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::processSystemSetting(dia_UID,dia_enSystemSettingLevel,extData)");

   dia_SystemSettingsConfiguration* pConfig = getConfiguration();
   if ( (!pConfig) || (!(pConfig->isSystemSettingTypeSupported(typeID))) )
   {
      DIA_TR_ERR("##### UNABLE TO ROLL OUT REMOTE SYSTEM SETTING REQUEST --- INVALID TYPE  UID (0x%08x) #####",typeID);
      //EM_TRACE Entry
//      DIA_TR_ERRMEM("##### UNABLE TO ROLL OUT REMOTE SYSTEM SETTING REQUEST --- INVALID TYPE  UID (0x%08x) #####",typeID);
      return DIA_E_SYSSET_INVALID_TYPE;
   }

   dia_RemoteSystemSettingRequest* pRemoteRequest  = new dia_RemoteSystemSettingRequest(typeID,level,extData,requester,cookie); //lint !e429: custodial pointer is freed by FSM after message has been processed or in destructor

   if ( pRemoteRequest && mpFSM ) {
      mpFSM->acceptEvent(dia_SystemSettingsFSM::evRemoteSystemSettingRequest,pRemoteRequest);
   }

   // make lint happy
   pRemoteRequest = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pActiveRequest'. --> lifetime is controlled by system settings state machine

   return DIA_SUCCESS; //lint !e438 Warning: last value assigned to variable 'pActiveRequest' not used
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsManager::processSystemSetting ( dia_UID typeID, dia_enSystemSettingLevel level, dia_ISystemSettingsListener& requester, tCookieType cookie )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::processSystemSetting(dia_UID,dia_enSystemSettingLevel)");
   dia_SystemSettingExtendedData extData;
   return processSystemSetting(typeID,level,extData,requester,cookie);
}

//-------------------------------------------------------------------------

void
dia_SystemSettingsManager::vOnTimerElapsed ( dia_TimerID id )
{
   DIA_TR_INF("### dia_SystemSettingsManager::vOnTimerElapsed ###");
   // hpe2hi: we should have a timer-class that work on our own message-queue
   // we are posting a state machine event to our queue and trigger the event thread for switching the thread context

   //hpe2i: forward to worker-thread
   postEvent(new dia_FunctorVoidImpl<dia_SystemSettingsManager, dia_TimerID>
             (this, &dia_SystemSettingsManager::vOnTimerElapsedLocal, id));
}

void
dia_SystemSettingsManager::vOnTimerElapsedLocal ( dia_TimerID /* id */ )
{
   DIA_TR_INF("### dia_SystemSettingsManager::vOnTimerElapsed ###");
   mSystemSettingsFirstError.addTimeout(mActiveLevel, (uint32_t)mStatusRep.size());
   mpFSM->acceptEvent(dia_SystemSettingsFSM::evProcessingTimeout,0);
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::postEvent ( dia_FunctorVoid *functor )
{
ScopeTrace oTrace("dia_SystemSettingsManager::postEvent(dia_SystemSettingsFSM::FsmEvent,void*)");

   if ( functor && mpMsgQueue )
   {
      mpMsgQueue->addElement(functor);
      DIA_TR_INF("Added event to system setting event queue");
   }

   // we are posting the event to the application to switch the thread context
   getInstanceOfApplication()->postMessage(DIA_EVENT_SYSSET_FSM_EVENTS);
} //lint !e438 Warning: last value assigned to variables 'pEvent' not used

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::onProcessEvents ( void )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::processEvents()");

   if ( (!mpMsgQueue) || (!mpFSM) || (mpMsgQueue->getSize() == 0) ) return;

   dia_FunctorVoid* functor = nullptr;

   while ( mpMsgQueue->getSize() )
   {
      if ( mpMsgQueue->getElement(&functor) != DIA_SUCCESS )
      {
         DIA_TR_ERR("##### FAILED TO READ ELEMENT FROM QUEUE #####");
         break;
      }

      if (functor) {
         functor->execute();
         delete functor;
      }
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmRejectRemoteSystemSettingRequest ( void* pArg )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmRejectRemoteSystemSettingRequest()");

   if ( !pArg ) return;

   dia_RemoteSystemSettingRequest* pActiveRequest = (dia_RemoteSystemSettingRequest*) pArg;

   pActiveRequest->mRequester.onSystemSettingProcessed(pActiveRequest->mType,pActiveRequest->mLevel,pActiveRequest->mExtData,DIA_E_SYSSET_REQUEST_REJECTED_INVALID_MODE,pActiveRequest->mCookie);
   delete pActiveRequest;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmRejectSystemSettingRequest ( void* pArg )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmRejectSystemSettingRequest()");

   if ( !pArg ) return;

   dia_SystemSettingRequest* pActiveRequest = (dia_SystemSettingRequest*) pArg;

   pActiveRequest->mRequester.onSystemSettingProcessed(pActiveRequest->mType,pActiveRequest->mExtData,DIA_E_SYSSET_REQUEST_REJECTED_INVALID_MODE,pActiveRequest->mCookie);
   delete pActiveRequest;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmSendNegResponseToMaster ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmSendNegResponseToMaster()");

   if ( mpActiveRemoteRequest )
   {
      if ( mErrorCode == DIA_SUCCESS )
      {
         DIA_TR_INF("##### GENERATING DEFAULT ERROR CODE AS NO ERROR CODE WAS ASSIGNED #####");
         mErrorCode = DIA_FAILED;
      }

      mpActiveRemoteRequest->mRequester.onSystemSettingProcessed(mActiveTypeID,mActiveLevel,mpActiveRemoteRequest->mExtData,mErrorCode,mpActiveRemoteRequest->mCookie);

      delete mpActiveRemoteRequest;
      mpActiveRemoteRequest = 0;
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmSendPosResponseToMaster ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmSendPosResponseToMaster()");

   if ( mpActiveRemoteRequest )
   {
      mpActiveRemoteRequest->mRequester.onSystemSettingProcessed(mActiveTypeID,mActiveLevel,mpActiveRemoteRequest->mExtData,DIA_SUCCESS,mpActiveRemoteRequest->mCookie);
      delete mpActiveRemoteRequest;
      mpActiveRemoteRequest = 0;
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmSendResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmSendResponse()");

   if ( mpActiveRequest )
   {
      tDiaResult retCode = DIA_SUCCESS;

      for ( tU16 i=0; i<DIA_EN_SYSTEM_SETTING_LEVEL_COUNT; i++ )
      {
         if ( ( mPluginResults[i] != DIA_SUCCESS ) || ( mModuleResults[i] != DIA_SUCCESS ) )
         {
            retCode = DIA_FAILED;
            break;
         }
      }

      if ( mpActiveRequest->mModules.size() )
      {
         DIA_TR_INF("##### SYSTEM SETTING DONE. SENDING RESPONSE (typeUID = 0x%08x,retCode=0x%08x) #####", mActiveTypeID,retCode);
         mpActiveRequest->mRequester.onSystemSettingProcessed(mActiveTypeID,mpActiveRequest->mExtData,mpActiveRequest->mModules,retCode,mpActiveRequest->mCookie);
      }
      else
      {
         DIA_TR_INF("##### SYSTEM SETTING DONE. SENDING RESPONSE (typeUID = 0x%08x,retCode=0x%08x) #####", mActiveTypeID,retCode);
         mpActiveRequest->mRequester.onSystemSettingProcessed(mActiveTypeID,mpActiveRequest->mExtData,retCode,mpActiveRequest->mCookie);
      }
      delete mpActiveRequest;
      mpActiveRequest = 0;
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmSendRemoteControlResponseToMaster ( void* pArg )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmSendRemoteControlResponseToMaster()");

   if ( !pArg )
   {
      DIA_TR_ERR("##### dia_SystemSettingsManager::vFsmSendRemoteControlResponseToMaster(): PARAMETER EXPECTED, BUT NOT PASSED #####");
      return;
   }

   dia_RemoteControlRequest* pRemoteControlRequest = (dia_RemoteControlRequest*) pArg;
   DIA_TR_INF("pRemoteControlRequest = %p",pRemoteControlRequest);
   pRemoteControlRequest->mRequester.onSetSystemSettingMode(pRemoteControlRequest->mMode,DIA_SUCCESS,pRemoteControlRequest->mCookie);
   delete pRemoteControlRequest;
}

/**************************************************************************************************
 *
 * Basic initialization of the state machine when the idle state is entered.
 * Resets all member variables to their default values.
 *
 * This method is part of the state machines behaviour interface generated by
 * SMC.
 *
 * \param[in]  pArg  parameter passed to the state machines accept() method
 *
 * \return n/a
 *
 *************************************************************************************************/

void
dia_SystemSettingsManager::vFsmInitializeSystemSetting ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializeSystemSetting()");

   mMode            = DIA_EN_SYSTEM_SETTING_MODE_MASTER;
   mActiveLevel     = DIA_EN_SYSTEM_SETTING_LEVEL_UNKNOWN;
   mRequestedTypeID = 0;
   mRequestedAreaID = 0;
   mRequestedLevel  = DIA_EN_SYSTEM_SETTING_LEVEL_UNKNOWN;
   mRetryCounter    = 0;
   mErrorCode       = DIA_SUCCESS;

   delete mpActiveRequest;        mpActiveRequest = 0;
   delete mpActiveRemoteRequest;  mpActiveRemoteRequest = 0;
}

/******************************************************************************
 *
 * Initializes the master mode. In master mode all levels of system setting
 * operations (prepare, execute and finalize) are rolled out under control of
 * dia_SystemSettingsManager.
 *
 * This method is part of the state machines behaviour interface generated by
 * SMC.
 *
 * \param[in]  pArg  parameter passed to the state machines accept() method
 *
 * \return n/a
 *
 *****************************************************************************/

void
dia_SystemSettingsManager::vFsmInitializeMasterSystemSetting ( void* pArg )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializeMasterSystemSetting()");

   if ( !pArg ) return;

   mSystemSettingsFirstError.reset();
   mMode           = DIA_EN_SYSTEM_SETTING_MODE_MASTER;
   mpActiveRequest = (dia_SystemSettingRequest*) pArg;
   mActiveTypeID   = mpActiveRequest->mType;
   mActiveLevel    = DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE; // we will start with preparation

}

/******************************************************************************
 *
 * Initializes the slave mode. In slave mode all levels of system setting
 * operations (prepare, execute and finalize) are rolled out under control of
 * a remote master.
 *
 * This method is part of the state machines behaviour interface generated by
 * SMC.
 *
 * \param[in]  pArg  parameter passed to the state machines accept() method
 *
 * \return n/a
 *
 *****************************************************************************/

void
dia_SystemSettingsManager::vFsmInitializeSlaveMode ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializeSlaveMode()");

   DIA_TR_INF("##### ENTERED SYSTEM SETTINGS SLAVE MODE #####");
   mMode = DIA_EN_SYSTEM_SETTING_MODE_SLAVE;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmFinalizeSlaveMode ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmFinalizeSlaveMode()");
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmInitializePreparation ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializePreparation()");

   mActiveLevel = DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE;
   mModuleResults[mActiveLevel] = DIA_FAILED;

   // initialize status map
   preProcess();
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmPreparationCompleted ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmPreparationCompleted()");

   mProcessingStatus[DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE] = DIA_EN_SYSSET_PROCSTATUS_COMPLETE;
   mModuleResults[DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE]    = DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmProcessPlugins ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmProcessPlugins()");

   tDiaResult retCode = DIA_FAILED;

   mProcessingStatus[mActiveLevel] = DIA_EN_SYSSET_PROCSTATUS_PROCESSING_PLUGINS;

   switch ( mActiveLevel )
   {
   case DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE:
   case DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE:
   case DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE:
      retCode = execPlugins(mActiveTypeID);
      break;
   default:
      mProcessingStatus[mActiveLevel] = DIA_EN_SYSSET_PROCSTATUS_UNKNOWN;
      break;
   }

   if ( retCode == DIA_SUCCESS )
   {
      mPluginResults[mActiveLevel] = DIA_SUCCESS;

      if ( mpConfig )
      {
         DIA_TR_INF("dia_SystemSettingsManager::vFsmProcessPlugins(): Successfully processed internal plugins (type = \"%s\") !!",mpConfig->getSystemSettingTypeName(mActiveTypeID));
      }
      // this method should only be called by the fsm, so why do we need postEvent???
      mpFSM->acceptEvent(dia_SystemSettingsFSM::evPluginsProcessed,0);
      //      postEvent(dia_SystemSettingsFSM::evPluginsProcessed,0);
   }
   else
   {
      mPluginResults[mActiveLevel] = retCode;

      if ( mpConfig )
      {
         DIA_TR_ERR("dia_SystemSettingsManager::vFsmProcessPlugins(): Failed to process internal plugins (type = \"%s\", errCode = 0x%08x) !!",mpConfig->getSystemSettingTypeName(mActiveTypeID),retCode);
         //EM_TRACE Entry
//         DIA_TR_ERRMEM("dia_SystemSettingsManager::vFsmProcessPlugins(): Failed to process internal plugins (errCode = 0x%08x, type = %s) !!",retCode, mpConfig->getSystemSettingTypeName(mActiveTypeID));
      }

      // TBD: depending on a strategy we break at this point or we continue
      if ( ( mActiveLevel == DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE ) || ( mActiveLevel == DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE ) )
      {
         mpFSM->acceptEvent(dia_SystemSettingsFSM::evFailed,0);
         //         postEvent(dia_SystemSettingsFSM::evFailed,0);
      }
      else
      {
         mpFSM->acceptEvent(dia_SystemSettingsFSM::evPluginsProcessed,0);
         //         postEvent(dia_SystemSettingsFSM::evPluginsProcessed,0);
      }
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmInitializeExecution ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializeExecution()");

   mActiveLevel = DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE;
   mModuleResults[mActiveLevel] = DIA_FAILED;

   // initialize status map
   preProcess();
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmExecutionCompleted ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmExecutionCompleted()");

   mProcessingStatus[DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE] = DIA_EN_SYSSET_PROCSTATUS_COMPLETE;
   mModuleResults[DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE]    = DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmInitializeFinalization ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializeFinalization()");

   mActiveLevel = DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE;
   mModuleResults[mActiveLevel] = DIA_FAILED;

   // initialize status map
   preProcess();
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmFinalizationCompleted ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmFinalizationCompleted()");

   mProcessingStatus[DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE] = DIA_EN_SYSSET_PROCSTATUS_COMPLETE;
   mModuleResults[DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE]    = DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmFinalizeSystemSetting ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmFinalizeSystemSetting()");
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmInitializeProcessing ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializeProcessing()");

   mProcessingStatus[mActiveLevel] = DIA_EN_SYSSET_PROCSTATUS_PROCESSING_MODULES;

   // reset the retry counter
   mRetryCounter = 0;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmFinalizeSlaveProcessing ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmFinalizeSlaveProcessing()");
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmInitializeRetry ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializeRetry()");

   // we are going to start a retry, so increment the retry counter
   ++mRetryCounter;
   mSystemSettingsFirstError.reset(mRetryCounter);


}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmInitializeSlaveProcessing ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializeSlaveProcessing()");
   mSystemSettingsFirstError.reset();
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmInitializeSlaveSystemSetting ( void* pArg )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmInitializeSlaveSystemSetting()");

   if ( !pArg ) return;

   mpActiveRemoteRequest = (dia_RemoteSystemSettingRequest*) pArg;
   mActiveTypeID         = mpActiveRemoteRequest->mType;
   mActiveLevel          = mpActiveRemoteRequest->mLevel;

   mModuleResults[mActiveLevel] = DIA_FAILED;

   mStatusRep.clear();

   std::set<dia_SystemSettingsModule*>::iterator moduleIter = mModuleRep.begin();
   for ( ; moduleIter != mModuleRep.end(); ++moduleIter )
   {
      dia_SystemSettingsModule* pModule = *moduleIter;
      if ( pModule->isSystemSettingTypeSupported(mpActiveRemoteRequest->mType) )
      {
         dia_SystemSettingStatus moduleStatus(*pModule);
         DIA_TR_INF("dia_SystemSettingsManager::vFsmInitializeSlaveSystemSetting - initialized status for system setting module \"%s\" (UID = 0x%08x)",pModule->getName(),pModule->getUID());
         if ( mActiveLevel == DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE )
         {
            moduleStatus.mIsACKRequired = false;
         }
         mStatusRep[pModule] =  moduleStatus;
      }
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmSendSystemSettingsRequests ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmSendSystemSettingsRequests()");

   DIA_TR_INF("System Setting - Sending requests to %zu modules (level =\"%s\")", mStatusRep.size(), arSystemSettingLevelMapping[mActiveLevel]);

   mSystemSettingsFirstError.setLevelPhase(SystemSettingsFirstError::enMODULES);

   if ( mStatusRep.size() )
   {
      std::map<dia_SystemSettingsModule*,dia_SystemSettingStatus>::iterator iter = mStatusRep.begin();
      for ( ; iter != mStatusRep.end(); ++iter )
      {
         dia_SystemSettingsModule *curModule=iter->first;
         dia_SystemSettingStatus &curStatus=iter->second;
         if( curModule && !curStatus.mACKReceived )
         {
            DIA_TR_INF("Sending request to module \"%s\"", curModule->getName());

            dia_SystemSettingExtendedData extData;
            if ( mpActiveRemoteRequest )
            {
               extData = mpActiveRemoteRequest->mExtData;
            }

            tDiaResult retCode = DIA_FAILED;

            switch ( mActiveLevel )
            {
            case DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE:
               retCode = curModule->prepareSystemSetting(mActiveTypeID, extData);
               break;

            case DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE:
               retCode = curModule->executeSystemSetting(mActiveTypeID, extData);
               break;

            case DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE:
               retCode = curModule->finalizeSystemSetting(mActiveTypeID, extData);
               break;

            default:
               continue;
            };



            if ( retCode == DIA_SUCCESS )
            {
               curStatus.mRequestSent = true;
               if ( curStatus.mIsACKRequired == false )
               {
                  // simulate response as we don't need for a response from the external component
                  onSystemSettingsModuleUpdate(*curModule,mActiveTypeID,mActiveLevel,DIA_SUCCESS);
               }
            }
            else
            {
               DIA_TR_ERR("##### FAILED TO SEND SYSTEM SETTING REQUEST TO MODULE \"%s\"", curModule->getName());
               //EM_TRACE Entry
               DIA_TR_ERRMEM("##### FAILED TO SEND SYSTEM SETTING REQUEST TO MODULE \"%s\"", curModule->getName());
               mSystemSettingsFirstError.addError(curModule->getName(), mActiveLevel, retCode, "call module failed");
            }
         }
      }
   }
   else
   {
         mpFSM->acceptEvent(dia_SystemSettingsFSM::evProcessingSucceeded,0);
         //      postEvent(dia_SystemSettingsFSM::evProcessingSucceeded,0);
   }

}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmSetProcessingTimeout ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmSetProcessingTimeout()");

   // retrieve the processing timeout from the configuration object, returns 0 in case of error
   mProcessingTimeout = ( mpConfig ) ? mpConfig->getTimeoutValue(mActiveLevel,0) : 0;
   if ( !mProcessingTimeout )
   {
      mProcessingTimeout = DIA_C_U32_DEFAULT_PROCESSING_TIMEOUT;
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmSetRetryTimeout ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmSetRetryTimeout()");

   // retrieve the processing timeout from the configuration object, returns 0 in case of error
   mProcessingTimeout = ( mpConfig ) ? mpConfig->getTimeoutValue(mActiveLevel,mRetryCounter) : 0;
   if ( !mProcessingTimeout )
   {
      mProcessingTimeout = DIA_C_U32_DEFAULT_PROCESSING_TIMEOUT;
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmStartProcessingTimer ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmStartProcessingTimer()");

   mTimer.addTimerListener(this);
   mTimer.s32SetTime(0,0);
   mTimer.s32SetTime(mProcessingTimeout, 0);
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmStopProcessingTimer ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmStopProcessingTimer()");

   mTimer.s32SetTime(0,0);
   mTimer.removeTimerListener(this);
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmEvaluateResponse ( void* pArg )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmEvaluateResponse()");

   if( !pArg ) return;
   std::map<dia_SystemSettingsModule*,dia_SystemSettingStatus>::iterator statusIter = mStatusRep.find( (dia_SystemSettingsModule*) pArg );
   if ( statusIter != mStatusRep.end() )
   {
      if ( statusIter->second.mIsACKRequired )
      {
         if ( statusIter->second.mACKReceived ) mStatusRep.erase(statusIter);
      }
      else
      {
         mStatusRep.erase(statusIter);
      }
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::vFsmHandleError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::vFsmHandleError()");
   mErrorCode = DIA_FAILED;
}


//-----------------------------------------------------------------------------

bool
dia_SystemSettingsManager::bFsmIsNotificationRequired ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::bFsmIsNotificationRequired()");

   // check if we have sent preparation requests, if so we need to send finalization requests as well
   return ( mProcessingStatus[DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE] > DIA_EN_SYSSET_PROCSTATUS_PROCESSING_PLUGINS ) ? true : false;
}

//-----------------------------------------------------------------------------

bool
dia_SystemSettingsManager::bFsmIsCompleteOK ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::bFsmIsCompleteOK()");

   bool retCode = ( mStatusRep.size() ) ? false : true;
   if ( retCode )
   {
      DIA_TR_INF("##### SYSTEM SETTINGS LEVEL COMPLETED (typeUID = 0x%08x, level=%s) #####", mActiveTypeID, arSystemSettingLevelMapping[mActiveLevel]);
   }

   return retCode;
}

//-----------------------------------------------------------------------------

bool
dia_SystemSettingsManager::bFsmIsRetryCheckOK ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::bFsmIsRetryCheckOK()");
   return true;
}

//-----------------------------------------------------------------------------

bool
dia_SystemSettingsManager::bFsmIsRetryCounterExceeded ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::bFsmIsRetryCounterExceeded()");
   if ( !mpConfig ) return true;
   DIA_TR_INF("##### RETRY COUNTER(\"%s\" = %d, MAX RETRIES = %d #####",arSystemSettingLevelMapping[mActiveLevel],mRetryCounter,mpConfig->getMaxRetries(mActiveLevel));
   return ( mRetryCounter < mpConfig->getMaxRetries(mActiveLevel) ) ? false : true;
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::preProcess ( void )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::preProcess()");

   mStatusRep.clear();

   if ( mpActiveRequest )
   {
      if ( mpActiveRequest->mModules.size() )
      {
         std::vector<dia_UID>::iterator uidIter = mpActiveRequest->mModules.begin();
         for ( ; uidIter != mpActiveRequest->mModules.end(); ++uidIter )
         {
            std::set<dia_SystemSettingsModule*>::iterator moduleIter = mModuleRep.begin();
            for ( ; moduleIter != mModuleRep.end(); ++moduleIter )
            {
               dia_SystemSettingsModule* pModule = *moduleIter;
               if ( *uidIter != pModule->getUID() ) continue;
               if ( pModule->isSystemSettingTypeSupported(mpActiveRequest->mType) )
               {
                  dia_SystemSettingStatus moduleStatus(*pModule);
                  DIA_TR_INF("dia_SystemSettingsManager::preProcess - initialized status for system setting module \"%s\" (UID = 0x%08x)",pModule->getName(),pModule->getUID());
                  if ( mActiveLevel == DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE )
                  {
                     moduleStatus.mIsACKRequired = false;
                  }
                  mStatusRep[pModule] =  moduleStatus;
               }
            }
         }
      }
      else
      {
         std::set<dia_SystemSettingsModule*>::iterator moduleIter = mModuleRep.begin();
         for ( ; moduleIter != mModuleRep.end(); ++moduleIter )
         {
            dia_SystemSettingsModule* pModule = *moduleIter;
            if ( pModule->isSystemSettingTypeSupported(mpActiveRequest->mType) )
            {
               dia_SystemSettingStatus moduleStatus(*pModule);
               DIA_TR_INF("dia_SystemSettingsManager::preProcess - initialized status for system setting module \"%s\" (UID = 0x%08x)",pModule->getName(),pModule->getUID());
               if ( mActiveLevel == DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE )
               {
                  moduleStatus.mIsACKRequired = false;
               }
               mStatusRep[pModule] =  moduleStatus;
            }
         }
      }
   }
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::postProcess ( void )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::postProcess()");
}

//-----------------------------------------------------------------------------

void
dia_SystemSettingsManager::onSystemSettingsModuleUpdate ( dia_SystemSettingsModule& module, dia_UID sysSetTypeUID, dia_enSystemSettingLevel level, tDiaResult errCode ) {
   //hpe2i: forward to worker-thread
   postEvent(new dia_FunctorVoidImpl<dia_SystemSettingsManager, dia_SystemSettingsModule&, dia_UID, dia_enSystemSettingLevel, tDiaResult>
             (this, &dia_SystemSettingsManager::onSystemSettingsModuleUpdateLocal,
              module, sysSetTypeUID, level, errCode));

   dia_FunctorRetImpl<tDiaResult, dia_SystemSettingsManager,dia_UID, dia_SystemSettingsPlugin**>
   (this, &dia_SystemSettingsManager::queryPlugin, 0, 0);
}


void
dia_SystemSettingsManager::onSystemSettingsModuleUpdateLocal ( dia_SystemSettingsModule& module, dia_UID /*sysSetTypeUID*/, dia_enSystemSettingLevel level, tDiaResult errCode )
{
   ScopeTrace oTrace("dia_SystemSettingsManager::onSystemSettingsModuleUpdate()");

   if ( mActiveLevel == level )
   {
      std::map<dia_SystemSettingsModule*,dia_SystemSettingStatus>::iterator statusIter = mStatusRep.find(&module);
      if ( statusIter != mStatusRep.end() )
      {
         DIA_TR_INF("##### UPDATING SYSTEM SETTING STATUS FOR MODULE \"%s\" #####", module.getName());
         DIA_TR_INF("##### (level=%s,retCode=0x%08x) #####", arSystemSettingLevelMapping[level],errCode);
         if ( errCode == DIA_SUCCESS )
         {
            DIA_TR_INF("##### ACK RECEIVED #####");
            statusIter->second.mACKReceived = true;
            statusIter->second.mErrorCode   = errCode;
         }
         else
         {
            DIA_TR_ERR("##### ACK NOT RECEIVED #####");
            //EM_TRACE Entry
            DIA_TR_ERRMEM("##### ACK NOT RECEIVED FROM MODULE \"%s\" (level=%s,retCode=0x%08x) #####", module.getName(), arSystemSettingLevelMapping[level],errCode);
            mSystemSettingsFirstError.addError(module.getName(), level, errCode, "module failed");
         }

         mpFSM->acceptEvent(dia_SystemSettingsFSM::evResponseReceived, (void*) &module);
         //         postEvent(dia_SystemSettingsFSM::evResponseReceived, (void*) &module);
      }
      else
      {
         DIA_TR_ERR("##### NO STATUS ENTRY AVAILABLE FOR SYSTEM SETTING MODULE \"%s\"#####", module.getName() );
         DIA_TR_ERR("##### (level=%s,retCode=0x%08x)  #####", arSystemSettingLevelMapping[level], errCode);
         //EM_TRACE Entry
         DIA_TR_ERRMEM("##### NO STATUS ENTRY AVAILABLE FOR SYSTEM SETTING MODULE \"%s\" (level=%s,retCode=0x%08x) #####", module.getName(), arSystemSettingLevelMapping[level],errCode);
      }
   }
}

void dia_SystemSettingsManager::SystemSettingsFirstError::toString(std::string &res) const {
   if (mNumFailed==0 && mErrorCode==DIA_SUCCESS) {
      res = "no error";
      return;
   }
   std::stringstream ss;
   ss<< mErrorText << "\n";
   ss<< "nFail=" << mNumFailed << "\n";
   ss<< "nTo=" << mNumTimeout - mNumFailed << "\n";
   ss<< "level=" << levelToString() << "\n";
   ss<< "phase=" <<levelPhaseToString() << "\n";
   ss<< "item=" << mItemName << "\n";
   char errBuf[9];
   snprintf(errBuf, sizeof(errBuf), "%08X", (uint16_t)mErrorCode);
   ss << "err=" << errBuf << "\n";
   ss << "retries=" << mRetryCounter << "\n";
   res= ss.str();
}

void dia_SystemSettingsManager::SystemSettingsFirstError::reset(tU16 retryCounter) {
   mNumFailed=0;
   mNumTimeout=0;
   mItemName="";
   mLevel=DIA_EN_SYSTEM_SETTING_LEVEL_UNKNOWN;
   mLevelPhase=enUNKNOWN;
   mErrorCode=DIA_SUCCESS;
   mRetryCounter=retryCounter;
}

void dia_SystemSettingsManager::SystemSettingsFirstError::addError(char const *itemName, dia_enSystemSettingLevel level, tDiaResult errorCode, char const *errorText) {
   DIA_TR_INF("SystemSettingsFirstError::addError() itemName=%s level=%u errorCode=%u", 
              itemName, (uint32_t)level, errorCode);

   ++mNumFailed;
   if (!bHasError()) {
      mItemName=itemName;
      mLevel=level;
      mErrorCode=errorCode;
      mErrorText=errorText;
   }
   setProperty();
}

void dia_SystemSettingsManager::SystemSettingsFirstError::addTimeout(dia_enSystemSettingLevel level, uint32_t numTimeout) {
   DIA_TR_INF("SystemSettingsFirstError::addTimeout() level=%u numTimeout=%u", (uint32_t)level, numTimeout);

   if (!mNumTimeout) {
      mNumTimeout=numTimeout;
   }
   
   if (!bHasError()) {
      mErrorText="Timeout";
      mErrorCode=DIA_E_TIMEOUT;
      mLevel=level;
   }
   setProperty();
}

char const *dia_SystemSettingsManager::SystemSettingsFirstError::levelPhaseToString() const {
   switch (mLevelPhase) {
      case enPLUGINS: return "plugin";
      case enMODULES: return "module";
      case enUNKNOWN:
      case enCOUNT:
      default: 
         return "unknown";
   }
}

char const *dia_SystemSettingsManager::SystemSettingsFirstError::levelToString() const {
   switch (mLevel) {
      case DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE: return "prepare";
      case DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE: return "execute";
      case DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE: return "finalize";
      case DIA_EN_SYSTEM_SETTING_LEVEL_UNKNOWN:
      case DIA_EN_SYSTEM_SETTING_LEVEL_COUNT:
      default: 
         return "unknown";
   }
}

void dia_SystemSettingsManager::SystemSettingsFirstError::setProperty() const {
   std::string resString;
   toString(resString);
   DIA_TR_INF("##### CONFIG_LAST_DIAG_ERROR #####\n'%s'", resString.c_str());
   
   (void) dia_setProperty(DIA_PROP_CM_CONFIG_LAST_DIAG_ERROR, (uint8_t*)resString.c_str(), (uint16_t)resString.length());

}
