#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include "common/framework/application/dia_Application.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER__
#include "common/framework/application/dia_CommandController.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND__
#include "common/framework/application/dia_Command.h"
#endif

#ifndef __INCLUDED_DIA_PREDICATE_SERVICE_STATE_CCA__
#include "common/framework/platform/cca/dia_PredicateServiceStateCCA.h"
#endif

#ifndef __INCLUDED_DIA_PREDICATE_SERVICE_REGISTRATION_CCA__
#include "common/framework/platform/cca/dia_PredicateServiceRegistrationCCA.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SYSTEM_ADAPTERS__
#include "common/framework/sysadapters/dia_common_system_adapters.h"
#endif

#define MIDW_FI_S_IMPORT_INTERFACE_FI_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "midw_fi_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#ifndef __INCLUDED_DIA_MAIN__
#include "common/depricated/dia_main.h"
#endif

#define DIA_C_CONDVAR_SIGNAL_SERVICE_AVAILABLE     DIA_C_CONDVAR_EVENT_01
#define DIA_C_CONDVAR_SIGNAL_REGID_AVAILABLE       DIA_C_CONDVAR_EVENT_02
#define DIA_C_CONDVAR_RETRIES                      5
#define DIA_C_CONDVAR_TIMEOUT                      2000
#define DIA_C_CONDVAR_NAME_EXTENSION               "_CV"
#define DIA_C_LOCK_NAME_EXTENSION                  "_LK"

/*#define C_U16_CMDCOUNT_SETGETMSG    0*/
#define C_U16_CMDCOUNT_UPREGMSG     1
#define C_U16_CMDCOUNT_RELUPREGMSG  2
/*#define C_U16_CMDCOUNT_METHODSTART  3*/

using namespace dia;

//-----------------------------------------------------------------------------

dia_SystemAdapterServicePlugin::dia_SystemAdapterServicePlugin (
      tCString name,    // name used to create name for lock and condvar object (in future!!)
      tU16 srvID,           // service of the client we are using
      tU16 srvMajVersion,   // major version number of the service
      tU16 srvMinVersion,   // minor version number of the service
      dia_SystemAdapter& adapter
   )
   : mIsRegistrationInProgress(false),
     mBlockingMode(false),
     mpAdapter(&adapter),
     mSyncObj(std::string(std::string(name) + DIA_C_LOCK_NAME_EXTENSION).c_str()),
     mCondVar(std::string(std::string(name) + DIA_C_CONDVAR_NAME_EXTENSION).c_str()),
     mCondVarRetries(DIA_C_CONDVAR_RETRIES),
     mConditionTimeout(DIA_C_CONDVAR_TIMEOUT),
     mConditionMask(DIA_C_CONDVAR_SIGNAL_SERVICE_AVAILABLE),
     _u16RegID(AMT_C_U16_REGID_INVALID),
     _bFidReg(FALSE),
     _u16ServiceId(0),
     _u16SourceSubID(0),
     mServiceID(srvID),
     mServiceMajorVersion(srvMajVersion),
     mServiceMinorVersion(srvMinVersion)
{
   mClientAppID = getClientID();
}

//----------------------------------------------------------------------------------------

dia_SystemAdapterServicePlugin::~dia_SystemAdapterServicePlugin ( void )
{
   _BP_TRY_BEGIN
   {
      mpAdapter = OSAL_NULL;
      mFunctionMapping.clear();
      mFeatureRep.clear();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SystemAdapterServicePlugin::~dia_SystemAdapterServicePlugin !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

tU16
dia_SystemAdapterServicePlugin::getClientID ( void ) const
{
   return mpAdapter->getClientID();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemAdapterServicePlugin::onServiceRegistered ( tU16 regID, tU16 srcSubID )
{
   // store registration id, u16ServiceId, u16SourceSubID
   _u16RegID = regID;
   _u16SourceSubID = srcSubID;

   // To avoid multiple reg
   mIsRegistrationInProgress = false;

   // signal condition variable
   if ( _u16RegID == AMT_C_U16_REGID_INVALID )
   {
      mCondVar.signal(DIA_C_CONDVAR_SIGNAL_REGID_AVAILABLE,FALSE);
   }
   else
   {
      mCondVar.signal(DIA_C_CONDVAR_SIGNAL_REGID_AVAILABLE,TRUE);
   }

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_SystemAdapterServicePlugin::registerService ( void )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::registerService");

   tDiaResult retCode = DIA_SUCCESS;

   // Check that registration ID is invalid, because we don't want to register twice
   if ( (_u16RegID == AMT_C_U16_REGID_INVALID) && (!mIsRegistrationInProgress) )
   {
      // reset the connection
      resetConnection();


      // OK try to register for the service
      tBool bRetVal = diagnostics_tclApp::getInstance()->bRegisterAsync (
            mServiceID,
            mServiceMajorVersion,
            mServiceMinorVersion,
            0x0,
            mClientAppID
      );

      // Check wether the registration request was OK
      if ( bRetVal == TRUE )
      {
         DIA_TR_SM("RegisterForService: appID=0x%04x, srvID=0x%04x succeeded", mClientAppID, mServiceID);
         mIsRegistrationInProgress = true;
      }
      else
      {
         DIA_TR_ERR("RegisterForService: appID=0x%04x, srvID=0x%04x FAILED", mClientAppID, mServiceID);
         retCode = DIA_FAILED;
      }
   }
   else
   {
      DIA_TR_ERR("registerService: appID=0x%04x, srvID=0x%04x _u16RegID=0x%04X mIsRegistrationInProgress=%s", mClientAppID, mServiceID, _u16RegID, ((mIsRegistrationInProgress) ? "true" : "false"));
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_SystemAdapterServicePlugin::unregisterService ( void )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::unregisterService");

   tDiaResult retCode = DIA_SUCCESS;

   // registration valid?
   if ( _u16RegID != AMT_C_U16_REGID_INVALID )
   {
      // unregister for Service
      diagnostics_tclApp::getInstance()->vUnregisterService(mServiceID);

      // Set that the function ID's are unregistered
      _bFidReg = FALSE;

      // Destroy local copy of registration Id
      _u16RegID = AMT_C_U16_REGID_INVALID;
      mCondVar.signal(DIA_C_CONDVAR_SIGNAL_REGID_AVAILABLE,FALSE);

      // reset the connection
      resetConnection();
   }
   else
   {
      // Oopse an error may be we are already unregistered
#if 0
      // Don't return FAILED, this would prevent the un-registration of all other dia_SystemAdapterServicePlugins !
      // See: dia_SystemAdapter::vFsmUnregisterServices()
      retCode = DIA_FAILED;
#endif

      mIsRegistrationInProgress = false;

      DIA_TR_ERR("Error - No unregistration triggered AppID=0x%04x, SrvID=0x%04x", mClientAppID, mServiceID);
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

void
dia_SystemAdapterServicePlugin::onServiceStateChanged ( tU8 u8ServiceState, tU16 u16SubId )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::onServiceStateChanged");

   // analyze the state of the server
   switch ( u8ServiceState )
   {
   case AMT_C_U8_SVCSTATE_AVAILABLE:
      {
         // signal condition variable
         mCondVar.signal(DIA_C_CONDVAR_SIGNAL_SERVICE_AVAILABLE,TRUE);

         // the service is now available, register only once, for each FID
         if ( !_bFidReg )
         {
            _bFidReg = bOnServiceStateAvailable(mServiceID,u16SubId);
         }
         else
         {
            DIA_TR_INF("We are already registering for this service. Don't do again");
         }
      }
      break;

   case AMT_C_U8_SVCSTATE_NOT_AVAILABLE:
      {
         // signal condition variable
         mCondVar.signal(DIA_C_CONDVAR_SIGNAL_SERVICE_AVAILABLE,FALSE);

         // the service is not available anymore
         vOnServiceStateNotAvailable(mServiceID,u16SubId);
      }
      break;

   default:
      DIA_TR_ERR("Error - Unknown service state");
      break;
   }
}

//----------------------------------------------------------------------------------------

tBool
dia_SystemAdapterServicePlugin::bOnServiceStateAvailable ( tU16 /*u16SrvID*/, tU16 /*u16SubId*/ )
{
   tBool retCode = TRUE;

   for ( std::list<dia_SystemAdapterFeature*>::iterator it = mFeatureRep.begin(); it != mFeatureRep.end(); ++it )
   {
      dia_SystemAdapterFeature* pFeature = (*it);
      // if ( (!pFeature) || (pFeature->startMonitoring() != DIA_SUCCESS) )
      if ( pFeature )
      {
         // retCode = FALSE;
#ifndef __DIA_UNIT_TESTING__
         DIA_TR_INF("Forwarding startMonitoring function call for '%s'", pFeature->mName);

         getInstanceOfApplication()->postMessage (
               OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                     OSAL_NEW dia_FunctorNoArgs< dia_SystemAdapterFeature >(pFeature, &dia_SystemAdapterFeature::startMonitoring)
               )
         );
#endif
      }
      else
      {
         retCode = FALSE;
      }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

void
dia_SystemAdapterServicePlugin::vOnServiceStateNotAvailable ( tU16 /*u16SrvID*/, tU16 /*u16SubId*/ )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::vOnServiceStateNotAvailable");
}

//----------------------------------------------------------------------------------------

void
dia_SystemAdapterServicePlugin::resetConnection ( void )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::resetConnection");
   mIsRegistrationInProgress = false;

   for ( std::list<dia_SystemAdapterFeature*>::iterator it = mFeatureRep.begin(); it != mFeatureRep.end(); ++it )
   {
      dia_SystemAdapterFeature* pFeature = (*it);
      if ( pFeature )
      {
         pFeature->reset();
      }
   }
}

//----------------------------------------------------------------------------------------

tBool
dia_SystemAdapterServicePlugin::bTransmitMessage ( amt_tclServiceData* poMessage )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::bTransmitMessage");

   tBool retCode = FALSE;

   diagnostics_tclApp* pApp = diagnostics_tclApp::getInstance();

   if ( !poMessage || !pApp )
   {
      DIA_TR_ERR("dia_SystemAdapterServicePlugin::bTransmitMessage FAILED (poMessage = %p, pApp = %p) !!!", poMessage, pApp);
      return FALSE;
   }

   if ( getRegistrationID() == AMT_C_U16_REGID_INVALID )
   {
      DIA_TR_ERR("dia_SystemAdapterServicePlugin::bTransmitMessage _u16RegID is AMT_C_U16_REGID_INVALID.");
      return FALSE;
   }

   if ( !mBlockingMode )
   {
      for ( tU16 i=0; i<mCondVarRetries; i++ )
      {
         DIA_TR_INF("Trying to send message AppId=0x%04x SrvId=0x%04x", mClientAppID, mServiceID);

         tDiaResult result = mCondVar.wait(mConditionMask,mConditionTimeout);
         if ( result == DIA_SUCCESS )
         {
            if ( pApp->enPostMessage(poMessage) == AIL_EN_N_NO_ERROR )
            {
               retCode = TRUE;
            }
            else
            {
               DIA_TR_ERR("dia_SystemAdapterServicePlugin::bTransmitMessage enRetCode is different than AIL_EN_N_NO_ERROR.");
            }
            break;
         }
         else if ( result == DIA_E_CONDVAR_BROADCAST )
         {
            DIA_TR_ERR("dia_SystemAdapterServicePlugin::bTransmitMessage: condition variable released via broadcast signal.");
            break;
         }
      }
   }

   DIA_TR_INF("dia_SystemAdapterServicePlugin::bTransmitMessage %s", (retCode == TRUE) ? "SUCCEEDED" : "FAILED");

   return retCode;
}

void
dia_SystemAdapterServicePlugin::vInitServiceData ( fi_tclVisitorMessage& oMsg, tU16 u16FuncId, tU8 u8OpCode, tU16 u16CmdCounter )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::vInitServiceData");

   oMsg.vInitServiceData ( DIA_USED_APP_ID,
                           mClientAppID,
                           AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                           0,
                           _u16RegID,
                           u16CmdCounter,
                           mServiceID,
                           u16FuncId,
                           u8OpCode );
}

void
dia_SystemAdapterServicePlugin::vOnErrorMessage ( amt_tclServiceData* poMessage ) const
{
    amt_tclServiceDataError oErrorMsg( poMessage );
    tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();

    switch ( u16ErrorCode )
    {
        case AMT_C_U16_ERROR_FUNCTION_BUSY: //The function is available, but is busy (there is no queueing for this function)
        {
            DIA_TR_INF("--- In CASE: AMT_C_U16_ERROR_FUNCTION_BUSY");
            break;
        }

        case AMT_C_U16_ERROR_INTERNAL_FAILURE: //The execution of the function failed
        {
            DIA_TR_INF("--- In CASE: AMT_C_U16_ERROR_INTERNAL_FAILURE");
            break;
        }

        case AMT_C_U16_ERROR_UNKNOWN_REG_ID:
        {
            DIA_TR_INF("--- In CASE: AMT_C_U16_ERROR_UNKNOWN_REG_ID");
            break;
        }

        case AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE:
        {
            DIA_TR_INF("--- In CASE: AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE");
            // Todo: Service availability changed
            break;
        }

        default:
        {
            DIA_TR_INF("--- In CASE: default");
        }
    }
}

//-----------------------------------------------------------------------------

bool
dia_SystemAdapterServicePlugin::isRegistrationInProgress ( void ) const
{
   return mIsRegistrationInProgress;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemAdapterServicePlugin::addFeature ( dia_SystemAdapterFeature& feature )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::addFeature");

   std::list<tU16> supportedFunctions;
   feature.getFunctionIDs(supportedFunctions);
   for ( std::list<tU16>::iterator funcIter = supportedFunctions.begin(); funcIter != supportedFunctions.end(); ++funcIter )
   {
      mFunctionMapping[*funcIter] = &feature;
   }

   std::list<dia_SystemAdapterFeature*>::iterator objIter = std::find(mFeatureRep.begin(),mFeatureRep.end(),&feature);
   if ( objIter == mFeatureRep.end() )
   {
      mFeatureRep.push_back(&feature);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemAdapterServicePlugin::dispatchMessage ( amt_tclBaseMessage *pMessage )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::vDispatchMessage");

   tDiaResult retCode = DIA_FAILED;

   // check if the received message is a AMT Service data message
   if ( pMessage->u8GetType() == AMT_C_U8_CCAMSGTYPE_SVCDATA )
   {
      // cast the received message to a AMT Service data message so we can
      // access the standard members
      amt_tclServiceData oServiceData(pMessage);

      tU16 u16FunctionID = oServiceData.u16GetFunctionID();

      std::map<tU16,dia_SystemAdapterFeature*>::iterator iter =  mFunctionMapping.find(u16FunctionID);
      if ( iter != mFunctionMapping.end() )
      {
         dia_SystemAdapterFeature* pFeature = iter->second;
         if ( pFeature )
         {
            retCode = pFeature->dispatchMessage(pMessage);
         }
      }
   }

   return retCode;
}

tBool
dia_SystemAdapterServicePlugin::bRegisterForFID ( tU16 u16FID )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::bRegisterForFID");

   tBool bRetVal = FALSE;

   diagnostics_tclApp* pApp = diagnostics_tclApp::getInstance();

   // Check that the registration Id is valid
   if ( pApp && ( _u16RegID != AMT_C_U16_REGID_INVALID ) )
   {
      // create message with upreg - request
      gm_tclEmptyMessage oUpRegMessage (
            DIA_USED_APP_ID,        // AppID of this application
                mClientAppID,   // AppID of the Server
                _u16RegID,      // RegId for the service (see bRegisterForService )
                0,              // always 0
                mServiceID,     // SID of the service
                u16FID,         // FID to register for
                AMT_C_U8_CCAMSG_OPCODE_UPREG
                );

      oUpRegMessage.vSetCmdCounter(C_U16_CMDCOUNT_UPREGMSG);

      if ( pApp->enPostMessage(&oUpRegMessage) == AIL_EN_N_NO_ERROR )
      {
         // message send success - set the return value to TRUE
         bRetVal = TRUE;
         DIA_TR_INF("FID registration successful AppId=0x%04x SrvId=0x%04x, FctId=0x%04x", mClientAppID, mServiceID,u16FID);
      }
      else
      {
         DIA_TR_INF("FID registration failed AppId=0x%04x SrvId=0x%04x, FctId=0x%04x", mClientAppID, mServiceID,u16FID);
      }
   }
   else
   {
      DIA_TR_INF("FID registration not sent AppId=0x%04x SrvId=0x%04x, FctId=0x%04x", mClientAppID, mServiceID,u16FID);
   }

   return bRetVal;
}

//----------------------------------------------------------------------------------------

tBool
dia_SystemAdapterServicePlugin::bUnregisterForFID ( tU16 u16FID )
{
   ScopeTrace oTrace("dia_SystemAdapterServicePlugin::bUnregisterForFID");

   tBool bRetVal = FALSE;

   diagnostics_tclApp* pApp = diagnostics_tclApp::getInstance();

   // registration invalid?
   if ( pApp && ( _u16RegID != AMT_C_U16_REGID_INVALID ) )
   {
      /* create message with relupreg - request */
      gm_tclEmptyMessage oRelupRegMessage (
            DIA_USED_APP_ID,        // AppID of this application
            mClientAppID,   // AppID of the Server
            _u16RegID,      // RegId for the service (see bRegisterForService )
            0,              // always 0
            mServiceID,     // SID of the service
            u16FID,         // FID to register for
            AMT_C_U8_CCAMSG_OPCODE_RELUPREG
      );

      oRelupRegMessage.vSetCmdCounter(C_U16_CMDCOUNT_RELUPREGMSG);

      if ( pApp->enPostMessage(&oRelupRegMessage) == AIL_EN_N_NO_ERROR )
      {
         // message send success - set the return value to TRUE
         bRetVal = TRUE;
         DIA_TR_INF("FID unregistration successful AppId=0x%04x SrvId=0x%04x, FctId=0x%04x", mClientAppID, mServiceID,u16FID);
      }
      else
      {
         DIA_TR_INF("FID unregistration failed AppId=0x%04x SrvId=0x%04x, FctId=0x%04x", mClientAppID, mServiceID,u16FID);
      }
   }
   else
   {
      DIA_TR_INF("FID unregistration not sent AppId=0x%04x SrvId=0x%04x, FctId=0x%04x", mClientAppID, mServiceID,u16FID);
   }

   return bRetVal;
}

//----------------------------------------------------------------------------------------

tU16
dia_SystemAdapterServicePlugin::getRegistrationID ( void )
{
   tU16 regID = AMT_C_U16_REGID_INVALID;

   if ( !mBlockingMode )
   {
      for ( tU16 i=0; i<mCondVarRetries; i++ )
      {
         tDiaResult retCode = mCondVar.wait(DIA_C_CONDVAR_SIGNAL_REGID_AVAILABLE,mConditionTimeout);
         if ( retCode == DIA_SUCCESS )
         {
            regID = _u16RegID;
            break;
         }
         else if ( retCode == DIA_E_CONDVAR_BROADCAST )
         {
            // during start-up AMT_C_U16_REGID_INVALID will be returned if we have not already received the regID
            DIA_TR_ERR("dia_SystemAdapterServicePlugin::getRegistrationID: condition variable released via broadcast signal.");
            regID = _u16RegID;
            break;
         }
         else
         {
            DIA_TR_ERR("#### RETRY: WAITING FOR REGID ... (app=0x%04x, srv=0x%04x) ####", mClientAppID, mServiceID);
         }
      }
   }
   else
   {
      // during start-up AMT_C_U16_REGID_INVALID will be returned if we have not already received the regID
      DIA_TR_ERR("UNABLE TO GET REGISTRATION ID (BLOCKING MODE). RETURNING CURRENT VALUE.");
      regID = _u16RegID;
   }

   return regID;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_SystemAdapterServicePlugin::executeServiceFunctor(std::unique_ptr<dia_Functor> pFunctor)
{
   std::unique_ptr<dia::Command> pCmd(new dia::Command);

   if (!pFunctor || !pCmd)
   {
      return DIA_E_INVALID_POINTER;
   }

   tDiaResult retCode = pCmd->addFunctor(pFunctor.release());
   if (retCode != DIA_SUCCESS)
   {
      return retCode;
   }

   std::list<std::unique_ptr<dia_Predicate>> predicateList;
   predicateList.push_back(std::unique_ptr<dia_Predicate>(new dia_PredicateServiceRegistrationCCA(mServiceID, mClientAppID)));
   predicateList.push_back(std::unique_ptr<dia_Predicate>(new dia_PredicateServiceStateCCA(mServiceID, mClientAppID, TRUE)));

   for (auto& predicate: predicateList)
   {
      retCode = pCmd->addPredicate(predicate.release());
      if (retCode != DIA_SUCCESS)
      {
         return retCode;
      }
   }

   return getInstanceOfCommandController()->addCommand(pCmd.release());
}

