/*!
 * \file       dia_SystemAdapterFacade.h
 *
 * \brief      Definition of the facade for the system adapter package.
 *
 * \details    Definition of the facade for the system adapter package. The facade acts as
 *             the single single entrypoint to the set of available system adapters
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSysAdapters
 *
 * \copyright  (c) 2008-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#define __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACES__
#include "common/interfaces/dia_interfaces.h"
#endif

#ifndef __INCLUDED_DIA_LOCK__
#include <common/framework/application/dia_Lock.h>
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

/**
 *
 *  \class dia_SystemAdapterFacade
 *
 *  \brief single entrypoint to the set of available system adapters
 *
 *  The dia_SystemAdapterFacade class acts as single entrypoint to the system adapter
 *  package. It hides the internal design and the implementation of system adapters
 *  to external clients. Instead it exposes a set of interfaces and clients can query for
 *  pointers to interfaces. If an interface implementation is available in the current
 *  project a valid interface pointer will be returned. Otherwise an error code and a NULL
 *  pointer is returned.
 *
 *  \ingroup diagFramework
 */

class dia_SystemAdapterFacade
{
   friend class dia_UTSystemAdapterFacade;

public:
    static dia_SystemAdapterFacade* createInstance();
    static dia_SystemAdapterFacade* getInstance();
    static void deleteInstance();
    //! setup method for SA Facade
    virtual tDiaResult setup ( void );
    //! shutdown method for SA Facade
    virtual tDiaResult tearDown ( void );
    //! retrieve an interface pointer via the facade
    template<class _T> tDiaResult queryInterface ( _T** pInterface );
    //! uncouple the interface from the provided implementation
    template<class _T> tDiaResult unsetInterface ( _T* pInterface );
    //! couple the given interface pointer with the given IID
    template<class _T> tDiaResult setInterface ( _T* pInterface );

    //! retrieve an interface listener pointer via the facade
    template<class _T> tDiaResult queryListener ( _T** pListener );
    //! uncouple the interface from the provided implementation
    template<class _T> tDiaResult unsetInterfaceListener ( _T* pListener );
    //! couple the given interface pointer with the given IID
    template<class _T> tDiaResult setInterfaceListener ( _T* pListener );

    //! install a notifier for the given interface
    template<class _L, class _N> tDiaResult setInterfaceNotifier ( _L* pInterface, _N*  pNotifier );
    template<class _L, class _N> tDiaResult setInterfaceNotifier ( _N* pNotifier );

protected:
    //! singleton instance
    static dia_SystemAdapterFacade* mpInstance;
    //! default constructor
    dia_SystemAdapterFacade ( void );
    //! destructor
    virtual ~dia_SystemAdapterFacade ( void );
    //! query the repository of interface pointers
    virtual void* queryInterface ( dia::IID iid );
    //! try to create a SystemAdapter for an interface
    virtual void* createInterface ( dia::IID iid );

    //! synchronization object used to synchronize concurrent access from multiple threads
    mutable dia::Lock mSyncObj;

    //! repository of interface pointers
    std::map<dia::IID,void*> mInterfaceRep;
    //! repository of interface pointers
    std::map<dia::IID,void*> mListenerRep;
    //! repository of notifier pointers
    std::map<dia::IID,dia::NotifierBase*> mNotifierRep;
};

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
queryInterface ( _T** pInterface )
{
    return getInstanceOfSystemAdapterFacade()->queryInterface<_T>(pInterface);
}

template<class _T> tDiaResult
querySysAdapterInterface ( _T** pInterface )
{
    return getInstanceOfSystemAdapterFacade()->queryInterface<_T>(pInterface);
}

template<class _T> tDiaResult
setSysAdapterInterface ( _T* pInterface )
{
    return getInstanceOfSystemAdapterFacade()->setInterface<_T>(pInterface);
}

template<class _T> tDiaResult
unsetSysAdapterInterface ( _T* pInterface )
{
    return getInstanceOfSystemAdapterFacade()->unsetInterface<_T>(pInterface);
}

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
queryListener ( _T** pListener )
{
    return getInstanceOfSystemAdapterFacade()->queryListener<_T>(pListener);
}

template<class _T> tDiaResult
querySysAdapterListener ( _T** pListener )
{
    return getInstanceOfSystemAdapterFacade()->queryListener<_T>(pListener);
}

template<class _T> tDiaResult
setSysAdapterListener ( _T* pListener )
{
    return getInstanceOfSystemAdapterFacade()->setInterfaceListener<_T>(pListener);
}

template<class _T> tDiaResult
unsetSysAdapterListener ( _T* pListener )
{
    return getInstanceOfSystemAdapterFacade()->unsetInterfaceListener<_T>(pListener);
}

template<class _T> tDiaResult
setInterfaceListener ( _T* pListener )
{
    return getInstanceOfSystemAdapterFacade()->setInterfaceListener<_T>(pListener);
}

template<class _T> tDiaResult
unsetInterfaceListener ( _T* pListener )
{
    return getInstanceOfSystemAdapterFacade()->unsetInterfaceListener<_T>(pListener);
}

//----------------------------------------------------------------------------------------

template<class _L, class _N>
tDiaResult
setSysAdapterInterfaceNotifier ( _L* /*pInterface*/, _N* pNotifier )
{
    return getInstanceOfSystemAdapterFacade()->setInterfaceNotifier<_L,_N>(pNotifier, pNotifier);
}
//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
dia_SystemAdapterFacade::queryInterface ( _T** pInterface )
{
   if ( !pInterface ) return DIA_E_INVALID_POINTER;

   dia::IID iid = _T::getUID();
   void* p = queryInterface(iid);

   if (p == nullptr)
   {
      p = createInterface(iid);
   }

   return (*pInterface = static_cast<_T*>(p)) == nullptr? DIA_E_NOT_FOUND: DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
dia_SystemAdapterFacade::setInterface ( _T* pInterface )
{
   dia::LockScope lock(mSyncObj);

   if ( !pInterface ) return DIA_E_INVALID_POINTER;
   if ( pInterface->getUID() != _T::getUID() ) return DIA_E_INTERFACE_ID_MISMATCH;

   mInterfaceRep[_T::getUID()] = (void*) pInterface;

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
dia_SystemAdapterFacade::unsetInterface ( _T* pInterface )
{
   dia::LockScope lock(mSyncObj);

   if ( !pInterface ) return DIA_E_INVALID_POINTER;
   if ( pInterface->getUID() != _T::getUID() ) return DIA_E_INTERFACE_ID_MISMATCH;

   std::map<dia::IID,void*>::iterator iter = mInterfaceRep.find(_T::getUID());
   if ( iter != mInterfaceRep.end() )
   {
      if( ((void*) pInterface) == iter->second )
      {
         mInterfaceRep.erase(iter);
      }
   }

    return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
dia_SystemAdapterFacade::queryListener ( _T** pListener )
{
   dia::LockScope lock(mSyncObj);

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   std::map<dia::IID,void*>::iterator iter = mListenerRep.find(_T::getUID());
   if ( iter != mListenerRep.end() )
   {
      *pListener = static_cast<_T*>(mListenerRep[_T::getUID()]);
      return DIA_SUCCESS;
   }

   *pListener = 0;
   return DIA_E_NOT_FOUND;
}

//----------------------------------------------------------------------------------------

template<class _T>
tDiaResult
dia_SystemAdapterFacade::setInterfaceListener(_T* pListener)
{
    dia::LockScope lock(mSyncObj);

    if (!pListener)
        return DIA_E_INVALID_POINTER;
    if (pListener->getUID() != _T::getUID())
        return DIA_E_INTERFACE_ID_MISMATCH;

    tDiaResult retCode = DIA_SUCCESS;

    std::map<dia::IID, dia::NotifierBase*>::iterator iter = mNotifierRep.find(_T::getUID());
    if (iter != mNotifierRep.end())
    {
        dia::NotifierBase* pNotifier = iter->second;
        retCode                      = pNotifier->attachListener((void*)pListener);
    }
    else
    {
        dia::NotifierBase* pNotifier = new dia::NotifierBase;
        if (pNotifier)
        {
            retCode                    = pNotifier->attachListener((void*)pListener);
            mNotifierRep[_T::getUID()] = pNotifier;
        }
        else
        {
            retCode = DIA_E_OUT_OF_MEMORY;
        }
    }

    return retCode;
}

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
dia_SystemAdapterFacade::unsetInterfaceListener ( _T* pListener )
{
   dia::LockScope lock(mSyncObj);

   if ( !pListener ) return DIA_E_INVALID_POINTER;
   if ( pListener->getUID() != _T::getUID() ) return DIA_E_INTERFACE_ID_MISMATCH;

   tDiaResult retCode = DIA_SUCCESS;

   std::map<dia::IID,dia::NotifierBase*>::iterator iter = mNotifierRep.find(_T::getUID());
   if ( iter != mNotifierRep.end() )
   {
      dia::NotifierBase* pNotifier = iter->second;
      retCode = pNotifier->detachListener((void*) pListener);
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

template<class _L, class _N>
tDiaResult
dia_SystemAdapterFacade::setInterfaceNotifier ( _L* /*pInterface*/, _N* pNotifier )
{
   dia::LockScope lock(mSyncObj);

   if ( !pNotifier ) return DIA_E_INVALID_POINTER;
   if ( pNotifier->getUID() != _L::getUID() ) return DIA_E_INTERFACE_ID_MISMATCH;

   dia::NotifierBase* pExistingNotifier = 0;
   std::map<dia::IID,dia::NotifierBase*>::iterator iter = mNotifierRep.find(_L::getUID());
   if ( iter != mNotifierRep.end() )
   {
      pExistingNotifier = iter->second;
   }

   if ( pExistingNotifier )
   {
      pNotifier->mergeListeners(*pExistingNotifier);
      DIA_DELETE pExistingNotifier;
   }

   mListenerRep[_L::getUID()] = (void*) ((_L*) pNotifier);
   mNotifierRep[_L::getUID()] = pNotifier;

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

template<class _L, class _N>
tDiaResult
dia_SystemAdapterFacade::setInterfaceNotifier ( _N* pNotifier )
{
   dia::LockScope lock(mSyncObj);

   if ( !pNotifier ) return DIA_E_INVALID_POINTER;
   if ( pNotifier->getUID() != _L::getUID() ) return DIA_E_INTERFACE_ID_MISMATCH;

   dia::NotifierBase* pExistingNotifier = 0;
   std::map<dia::IID,dia::NotifierBase*>::iterator iter = mNotifierRep.find(_L::getUID());
   if ( iter != mNotifierRep.end() )
   {
      pExistingNotifier = iter->second;
   }

   if ( pExistingNotifier )
   {
      pNotifier->mergeListeners(*pExistingNotifier);
      DIA_DELETE pExistingNotifier;
   }

   mListenerRep[_L::getUID()] = (void*) ((_L*) pNotifier);
   mNotifierRep[_L::getUID()] = pNotifier;

   return DIA_SUCCESS;
}

#endif
