/*!
 * \file       dia_SystemAdapterFacade.cpp
 *
 * \brief      Definition of the facade for the system adapter package.
 *
 * \details    Definition of the facade for the system adapter package. The facade acts as
 *             the single single entrypoint to the set of available system adapters
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSysAdapters
 *
 * \copyright  (c) 2008-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include <string.h>
#include <future>

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SYSTEM_ADAPTER_FACTORY_LISTENER__
#include "common/interfaces/dia_ISystemAdapterFactoryListener.h"
#endif

using namespace dia;

dia_SystemAdapterFacade* dia_SystemAdapterFacade::mpInstance = nullptr;

dia_SystemAdapterFacade*
getInstanceOfSystemAdapterFacade ( void )
{
   return dia_SystemAdapterFacade::getInstance();
}

void releaseInstanceOfSystemAdapterFacade( void )
{
   return dia_SystemAdapterFacade::deleteInstance();
}

dia_SystemAdapterFacade*
createInstanceOfSystemAdapterFacade ( void )
{
   return dia_SystemAdapterFacade::createInstance();
}

//----------------------------------------------------------------------------------------

dia_SystemAdapterFacade::dia_SystemAdapterFacade ( void )
   : mSyncObj("dia_SystemAdapterFacade_LK")
{
   DIA_TR_INF("dia_SystemAdapterFacade::dia_SystemAdapterFacade");
}

//----------------------------------------------------------------------------------------

dia_SystemAdapterFacade::~dia_SystemAdapterFacade ( void )
{
   DIA_TR_INF("dia_SystemAdapterFacade::~dia_SystemAdapterFacade");
}

//----------------------------------------------------------------------------------------

dia_SystemAdapterFacade*
dia_SystemAdapterFacade::createInstance ( void )
{
   if ( nullptr == mpInstance )
   {
      mpInstance = new dia_SystemAdapterFacade;
      if ( mpInstance )
      {
         if ( mpInstance->setup() != DIA_SUCCESS )
         {
            delete mpInstance;
            mpInstance = nullptr;
         }
      }
   }
   return mpInstance;
}

//----------------------------------------------------------------------------------------

dia_SystemAdapterFacade*
dia_SystemAdapterFacade::getInstance ( void )
{
   return mpInstance;
}

//----------------------------------------------------------------------------------------

void
dia_SystemAdapterFacade::deleteInstance ( void )
{
   if ( mpInstance )
   {
      (void) mpInstance->tearDown();
      delete mpInstance;
      mpInstance = nullptr;
   }
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_SystemAdapterFacade::setup ( void )
{
   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_SystemAdapterFacade::tearDown ( void )
{
   ScopeTrace oTrace("dia_SystemAdapterFacade::tearDown");

   LockScope lock(mSyncObj);

   mInterfaceRep.clear();
   mListenerRep.clear();
   DIA_IMPL_MAP_REPOSITORY_TEAR_DOWN(dia::IID,dia::NotifierBase,mNotifierRep);

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

void*
dia_SystemAdapterFacade::queryInterface ( dia::IID iid )
{
   LockScope lock(mSyncObj);

   auto iter = mInterfaceRep.find(iid);
   if ( iter != mInterfaceRep.end() )
   {
      return iter->second;
   }

   return nullptr;
}

//----------------------------------------------------------------------------------------

void*
dia_SystemAdapterFacade::createInterface ( dia::IID iid )
{
   dia_tclFnctTrace trc("dia_SystemAdapterFacade::createInterface");

   DIA_TR_INF("dia_SystemAdapterFacade::createInterface: iid = 0x%x", iid);

   // Prevent an endless recursion
   if (iid == dia_ISystemAdapterFactory::getUID())
   {
      DIA_TR_ERR("The SystemAdapter for dia_ISystemAdapterFactory cannot be created from itself!");
      return nullptr;
   }

   dia_ISystemAdapterFactory* pSystemAdapterFactory = nullptr;
   if ((DIA_SUCCESS != queryInterface<dia_ISystemAdapterFactory>(&pSystemAdapterFactory)) || !pSystemAdapterFactory)
   {
      DIA_TR_ERR("dia_ISystemAdapterFactory with IID=0x%x not available!", dia_ISystemAdapterFactory::getUID());
      return nullptr;
   }

   class Promise : public std::promise<void*>,
         private dia_ISystemAdapterFactoryListenerImpl
   {
   public:
      typedef std::future<void*> Future;
      Promise(dia_SystemAdapterFacade* pFacade) : _pFacade(pFacade)
      {
         (void) _pFacade->setInterfaceListener<dia_ISystemAdapterFactoryListener>(this);
      }
      virtual ~Promise() {}
   private:
      virtual void onCreateSystemAdapter (dia::IID iid, tDiaResult result )
      {
         DIA_TR_INF("onCreateSystemAdapter: iid = 0x%x, result = 0x%x", iid, result);

         (void) _pFacade->unsetInterfaceListener<dia_ISystemAdapterFactoryListener>(this);
         set_value( (result == DIA_SUCCESS)? _pFacade->queryInterface(iid): nullptr);
      }
      Promise ( void );
      dia_SystemAdapterFacade* _pFacade;
   };

   Promise promise(this);
   Promise::Future future = promise.get_future();

   if (DIA_SUCCESS != pSystemAdapterFactory->createSystemAdapter(iid))
   {
      DIA_TR_ERR("dia_ISystemAdapterFactory::createSystemAdapter failed for IID=0x%x", iid);
      return nullptr;
   }

   switch (future.wait_for(std::chrono::seconds(10)))
   {
   case std::future_status::ready:
      DIA_TR_INF("future: ready");
      return future.get();

   case std::future_status::timeout:
      DIA_TR_ERR("future: timeout");
      break;

   case std::future_status::deferred:
      DIA_TR_ERR("future: deferred");
      break;

   default:
      DIA_TR_ERR("future: unknown");
      break;
   }

   return nullptr;
}

