/*!
 * \file       dia_SystemAdapter.cpp
 *
 * \brief      System adapter class
 *
 * \details    System adapter class that is assembled with Service plugins and features
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSysAdapters
 *
 * \copyright  (c) 2011-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_SERVICE_PLUGIN__
#include "common/framework/sysadapters/dia_SystemAdapterServicePlugin.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_CCA__
#include "project/framework/sysadapters/dia_SAFeatureCCA.h"
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include "common/framework/application/dia_AppController.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER__
#include "dia_SystemAdapter.h"
#endif

using namespace dia;

//---------------------------------------------------------------------------------------------------------------------

dia_SystemAdapter*
dia_SystemAdapter::create ( tCString name, tU16 clientAppID )
{
   dia_SystemAdapter* pInstance = DIA_NEW dia_SystemAdapter(name,clientAppID);
   if ( pInstance )
   {
      if ( pInstance->setup() != DIA_SUCCESS )
      {
         OSAL_DELETE pInstance;
         pInstance = 0;
      }
   }

   return pInstance;
}

//---------------------------------------------------------------------------------------------------------------------

dia_SystemAdapter::dia_SystemAdapter (
      tCString name,
      tU16 clientAppID        // appID of the client application
   )
   : mName(name),
     mSyncObj(name),
     mAppID(DIA_C_U16_DIAGNOSTICS_APP_ID),
     mClientAppID(clientAppID),
     mApplicationStateRequested(DIA_C_U32_APP_STATE_UNKNOWN),
     mpFSM(0),
     mIsApplicationStateChangeInProgress(false),
     mIsRegistrationInProgress(false)
{
   ScopeTrace oTrace( "dia_SystemAdapter::dia_SystemAdapter" );
}

//---------------------------------------------------------------------------------------------------------------------

dia_SystemAdapter::~dia_SystemAdapter ( tVoid )
{
    _BP_TRY_BEGIN
    {
       getInstanceOfAppController()->removeAppStateListener(this);
        mName = 0;
        DIA_DELETE mpFSM;
        mpFSM = 0;
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_SystemAdapter::~dia_SystemAdapter !!!");
        DIA_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
dia_SystemAdapter::setup ( void )
{
   ScopeTrace oTrace("dia_SystemAdapter::setup()");

   // create the state machine object
   if ( !(dia_SystemAdapterFSM::Fsm::createFSM(&mpFSM,this)) )
   {
      DIA_TR_INF( "### STATE MACHINE INITIALIZATION FAILED ###");
      return DIA_FAILED;
   }

   getInstanceOfAppController()->addAppStateListener(this);

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vOnUnknownMessage ( amt_tclBaseMessage* poMessage )
{
   ScopeTrace oTrace("dia_SystemAdapter::vOnUnknownMessage");

   if ( poMessage && poMessage->bIsValid() )
   {
      (void) poMessage->bDelete();
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vOnErrorMessage ( amt_tclServiceData* poMessage ) const
{
   amt_tclServiceDataError oErrorMsg( poMessage );
   tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();

   switch ( u16ErrorCode )
   {
   case AMT_C_U16_ERROR_FUNCTION_BUSY: //The function is available, but is busy (there is no queueing for this function)
      DIA_TR_INF("--- In CASE: AMT_C_U16_ERROR_FUNCTION_BUSY");
      break;

   case AMT_C_U16_ERROR_INTERNAL_FAILURE: //The execution of the function failed
      DIA_TR_INF("--- In CASE: AMT_C_U16_ERROR_INTERNAL_FAILURE");
      break;

   case AMT_C_U16_ERROR_UNKNOWN_REG_ID:
      DIA_TR_INF("--- In CASE: AMT_C_U16_ERROR_UNKNOWN_REG_ID");
      break;

   case AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE:
      DIA_TR_INF("--- In CASE: AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE");
      // Todo: Service availability changed
      break;

   default:
      DIA_TR_INF("--- In CASE: default");
      break;
    }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vDispatchMessage ( amt_tclBaseMessage* poMessage )
{
   ScopeTrace oTrace("dia_SystemAdapter::vDispatchMessage");

   if ( poMessage )
   {
      acceptEvent(dia_SystemAdapterFSM::evMessage,poMessage);
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::acceptEvent ( dia_SystemAdapterFSM::FsmEvent event, void* pArg )
{
   DIA_TR_INF("Accept Event Id=%d ('%s')", event, dia_SystemAdapterFSM::getEventName(event));

   if ( mpFSM )
   {
      LockScope oLock(mSyncObj);

       DIA_TR_INF("### State Before: %s", mpFSM->getStateName());
       mpFSM->acceptEvent(event,pArg);
       DIA_TR_INF("### State After : %s", mpFSM->getStateName());
   }
   else
   {
       DIA_TR_INF( "### Event Not Accepted By System Adapter '%s'. No FSM !!!", this->mName);
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::onApplicationStateChangePrepare ( dia::ApplicationStateID /*currentState*/, dia::ApplicationStateID /*nextState*/ )
{
   ScopeTrace oTrace("dia_SystemAdapter::onApplicationStateChangePrepare");

   // be aware: this method is not executed in the context of the diagnostics worker thread to prevent
   // that an application state change can be delayed by other messages that need to be processed first

   mIsApplicationStateChangeInProgress = true;

   std::map<tU16,dia_SystemAdapterServicePlugin*>::iterator iter = mPluginRep.begin();
   for ( ; iter != mPluginRep.end(); ++iter )
   {
      dia_SystemAdapterServicePlugin* pPlugin = iter->second;
      if ( pPlugin )
      {
         // 1st step: activate blocking mode at plugin to avoid that we enter the waiting loop on the plugin's condition variable
         pPlugin->setBlockingMode(true);
         // 2nd step: if the plugin is currently waiting at the condition variable we unblock it via a call to method broadcast()
         pPlugin->onStopSending();
      }
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::onApplicationStateChange ( dia::ApplicationStateID currentState, dia::ApplicationStateID nextState )
{
   ScopeTrace oTrace("dia_SystemAdapter::onApplicationStateChange");

   DIA_TR_INF("--- STATE TO LEAVE 0x%02x, STATE TO ENTER 0x%02x",currentState,nextState);

   // Take a copy of the state we wish to enter
   mApplicationStateRequested = nextState;

   // Select which OLD state we are leaving
   switch ( nextState )
   {
   case DIA_C_U32_APP_STATE_NORMAL:
      {
         mIsRegistrationInProgress = true;
         acceptEvent(dia_SystemAdapterFSM::evNormal,0);
      }
      break;

   case DIA_C_U32_APP_STATE_PAUSE:
      acceptEvent(dia_SystemAdapterFSM::evPause,0);
      break;

   case DIA_C_U32_APP_STATE_OFF:
      acceptEvent(dia_SystemAdapterFSM::evOff,0);
      break;

   default:
      break;
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::onApplicationStateChangeFinalize ( dia::ApplicationStateID /*appStateID*/ )
{
   ScopeTrace oTrace("dia_SystemAdapter::onApplicationStateChangeFinalize");
   mIsApplicationStateChangeInProgress = false;

   std::map<tU16,dia_SystemAdapterServicePlugin*>::iterator iter = mPluginRep.begin();
   for ( ; iter != mPluginRep.end(); ++iter )
   {
      dia_SystemAdapterServicePlugin* pPlugin = iter->second;
      if ( pPlugin )
      {
         pPlugin->onResumeSending();
         pPlugin->setBlockingMode(false);
      }
   }

}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vLogRegistrationID ( tU16 u16RegID, tU16 u16AppID, tU16 u16SrvID, tU16 u16SrcSubID )
{
   ScopeTrace oTrace("dia_SystemAdapter::vLogRegistrationID");

   // This method is called in the context of the CCA body thread. The thread context should not be switched to the
   // worker thread, as the worker thread might be blocked on a system adapters condition variable waiting for a
   // registration ID

   if (mClientAppID == AMT_C_U16_APPID_INVALID)
   {
      mClientAppID = u16AppID;
   }

   if ( u16AppID != mClientAppID )
   {
      return;
   }

   if ( u16RegID == AMT_C_U16_REGID_INVALID )
   {
      DIA_TR_ERR("##### ERR: RECEIVED INVALID REGISTRATION ID");
      return;
   }

   DIA_TR_INF("--- dia_SystemAdapter::vLogRegistrationID: AppID=0x%0x SrvID=0x%0x) !!",u16AppID,u16SrvID );

   std::map<tU16,dia_SystemAdapterServicePlugin*>::iterator iter = mPluginRep.find(u16SrvID);
   if ( iter != mPluginRep.end() )
   {
      dia_SystemAdapterServicePlugin* pPlugin = iter->second;
      if ( pPlugin && ( pPlugin->onServiceRegistered(u16RegID,u16SrcSubID) != DIA_SUCCESS ) )
      {
         DIA_TR_INF("dia_SystemAdapter::vLogRegistrationID: pPlugin->onServiceRegistered FAILED !!!");
      }
   }

   acceptEvent(dia_SystemAdapterFSM::evRegistrationID,0);
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vFsmNormalModeReached ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemAdapter::vFsmNormalModeReached");
   getInstanceOfAppController()->onApplicationStateChanged(DIA_C_U32_APP_STATE_NORMAL,this);
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vFsmPauseModeReached ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemAdapter::vFsmPauseModeReached");
   getInstanceOfAppController()->onApplicationStateChanged(DIA_C_U32_APP_STATE_PAUSE,this);
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vFsmOffModeReached ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemAdapter::vFsmOffModeReached");
   getInstanceOfAppController()->onApplicationStateChanged(DIA_C_U32_APP_STATE_OFF,this);
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vFsmEvaluateRegistrationIDs ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemAdapter::vFsmEvaluateRegistrationIDs");

   if ( mIsRegistrationInProgress )
   {
      mIsRegistrationInProgress = bFsmAreNoRegistrationsPending(0);
      if ( !mIsRegistrationInProgress )
      {
         DIA_TR_INF("All registrations from SystemAdapter %s sent ... " ,mName);
      }
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vOnServiceState (
      tU16 u16ServiceId,
      tU16 u16ServerId,
      tU16 /*u16RegisterId*/,
      tU8  u8ServiceState,
      tU16 u16SubId
      )
{
   ScopeTrace oTrace("dia_SystemAdapter::vOnServiceState");

   DIA_TR_INF("--- dia_SystemAdapter::vOnServiceState: SrvID=0x%04x,AppID=0x%04x,State=0x%02x) !!",u16ServiceId,u16ServerId,u8ServiceState);

   std::map<tU16,dia_SystemAdapterServicePlugin*>::iterator iter = mPluginRep.find(u16ServiceId);
   if ( iter != mPluginRep.end() )
   {
      dia_SystemAdapterServicePlugin* pPlugin = iter->second;
      if ( pPlugin )
      {
         if ( u8ServiceState == AMT_C_U8_SVCSTATE_REG_INVALID )
         {
            pPlugin->registerService();
            pPlugin->resetConnection();
         }
         else
         {
            pPlugin->onServiceStateChanged(u8ServiceState,u16SubId);
         }
      }
    }
    else
    {
       DIA_TR_ERR("RECEIVED SERVICE STATE FOR UNKOWN SERVICE ID");
    }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vFsmProcessMessage ( void* pArg )
{
   ScopeTrace oTrace("dia_SystemAdapter::vFsmProcessMessage");

   if ( !pArg ) return;

   amt_tclBaseMessage* poMessage = (amt_tclBaseMessage*) pArg;

   // check if the received message is a AMT Service data message
   if ( poMessage->u8GetType() != AMT_C_U8_CCAMSGTYPE_SVCDATA ) return;

   // cast the received message to a AMT Service data message so we can access the standard members
   amt_tclServiceData oServiceData(poMessage);

   tU16 u16ServiceID = oServiceData.u16GetServiceID();

   std::map<tU16, dia_SystemAdapterServicePlugin*>::iterator iter = mPluginRep.find(u16ServiceID);
   if (iter != mPluginRep.end())
   {
      dia_SystemAdapterServicePlugin* pPlugin = iter->second;
      if ( pPlugin && (pPlugin->dispatchMessage(poMessage) != DIA_SUCCESS) )
      {
         DIA_TR_ERR("### DISPATCHING TO SERVICE PLUGIN HAS FAILED !!! ###");
      }
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vFsmUnknownMessage ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SystemAdapter::vFsmUnknownMessage");
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vFsmRegisterServices ( void* /*args*/ )
{
   ScopeTrace oTrace("dia_SystemAdapter::vFsmRegisterServices()");

   tU16 failedCounter = 0;

   // forward the state change request to all system adapters found in the repository
   std::map<tU16, dia_SystemAdapterServicePlugin*>::iterator iter = mPluginRep.begin();
   for ( ; iter != mPluginRep.end(); iter++ )
   {
      dia_SystemAdapterServicePlugin* pPlugin = iter->second;

      if ( pPlugin && ( pPlugin->registerService() == DIA_FAILED ) )
      {
         failedCounter++;
      }
   }

   if ( failedCounter )
   {
      DIA_TR_ERR("REGISTRATION FAILED FOR SOME APPLICATIONS");
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SystemAdapter::vFsmUnregisterServices ( void* /*args*/ )
{
   ScopeTrace oTrace("dia_SystemAdapter::vFsmUnregisterServices()");

   tU16 failedCounter = 0;

   // forward the state change request to all system adapters found in the repository
   std::map<tU16, dia_SystemAdapterServicePlugin*>::iterator iter = mPluginRep.begin();
   for ( ; iter != mPluginRep.end(); iter++ )
   {
      dia_SystemAdapterServicePlugin* pPlugin = iter->second;

      if ( pPlugin && ( pPlugin->unregisterService() == DIA_FAILED ) )
      {
         failedCounter++;
      }
   }

   if ( failedCounter )
   {
      DIA_TR_ERR("DE-REGISTRATION FAILED FOR SOME APPLICATIONS");
   }
}

//---------------------------------------------------------------------------------------------------------------------

bool
dia_SystemAdapter::bFsmAreNoRegistrationsPending ( void* /*args*/ )
{
   tU16 countPlugins = 0;

   // check all plugins if a registration process is currently running
   std::map<tU16,dia_SystemAdapterServicePlugin*>::iterator iter = mPluginRep.begin();
   for ( ; iter != mPluginRep.end(); iter++ )
   {
      dia_SystemAdapterServicePlugin* pPlugin = iter->second;
      if ( pPlugin && pPlugin->isRegistrationInProgress() )
      {
         countPlugins++;
      }
   }

   return ( countPlugins == mPluginRep.size() ) ? true : false;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
dia_SystemAdapter::addServicePlugin ( dia_SystemAdapterServicePlugin& pPlugin )
{
   ScopeTrace oTrace("dia_SystemAdapter::addServicePlugin");

   tDiaResult retCode = DIA_FAILED;

   tU16 key = pPlugin.getServiceID();
   DIA_TR_INF("--- dia_SystemAdapter::addServicePlugin: AppID=0x%0x, ClientAppID=0x%0x, SrvID=0x%0x) !!" ,mAppID,mClientAppID,key );
   std::map<tU16,dia_SystemAdapterServicePlugin*>::iterator iter = mPluginRep.find(key);
   if ( iter == mPluginRep.end() )
   {
      mPluginRep[key] = &pPlugin;

      dia_SAFeatureCCA* pCCA = dia_SAFeatureCCA::getInstance();
      DIA_TR_INF("SystemAdapter created AppId=0x%0x SrvId=0x%0x",mClientAppID,pPlugin.getServiceID());
      if ( pCCA->setInfo(mClientAppID,pPlugin.getServiceID(), this) != DIA_SUCCESS )
      {
         DIA_TR_ERR("Error setting SystemAdapterInfo AppId=0x%0x SrvId=0x%0x",mClientAppID,pPlugin.getServiceID());
      }
      retCode = DIA_SUCCESS;
   }
   else
   {
      retCode = DIA_E_KEY_ALREADY_EXISTS;
   }

   return retCode;
}
