/*!
 * \file       dia_SAFeatureWifiBLConflictManagement.cpp
 *
 * \brief      header file for dia_SAFeatureWifiBLConflictManagement
 *
 * \details    ASF component for Wifi BL ConflictManagement
 *
 * \component  Diagnosis
 *
 * \ingroup    diaASFComponents
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include "dia_SAFeatureWifiBLConflictManagement.h"

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_SERVICE_PLUGIN_ASF__
#include <common/framework/platform/asf/dia_SystemAdapterServicePluginASF.h>
#endif

const std::string dia_SAFeatureWifiBLConflictManagement::sAPMode = "AP";
const std::string dia_SAFeatureWifiBLConflictManagement::sStationMode = "STA";

dia_SAFeatureWifiBLConflictManagement::dia_SAFeatureWifiBLConflictManagement (dia_SystemAdapterServicePluginASF<ConflictmanagementProxy>& pSrvPlugin )
   : dia_SystemAdapterFeatureASF<ConflictmanagementProxy>(pSrvPlugin),
     mActReg(0)
{

}

//-----------------------------------------------------------------------------

dia_SAFeatureWifiBLConflictManagement::~dia_SAFeatureWifiBLConflictManagement()
{
   // TODO Auto-generated destructor stub
}

//-----------------------------------------------------------------------------


tDiaResult
dia_SAFeatureWifiBLConflictManagement::startMonitoring()
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::startMonitoring");

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------


tDiaResult
dia_SAFeatureWifiBLConflictManagement::stopMonitoring()
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::stopMonitoring");

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureWifiBLConflictManagement::prepareSetup(const ::std::string& mode)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::prepareSetup(string)");

   tDiaResult retCode = DIA_FAILED;

   ::asf::dbus::DBusVariant apConfigVariant;
   ::std::map < ::std::string, ::asf::dbus::DBusVariant > setupsMap;

   DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::prepareSetup mode='%s'", mode.c_str());

   if (!sAPMode.compare(mode))
   {
      DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::prepareSetup AP.");
      if (DIA_SUCCESS!=insert_map_element_dbus(setupsMap, "Mode", DBUS_TYPE_STRING, &mode))
      {
         DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::prepareSetup insert_map_element_dbus failed.");
         return DIA_FAILED;
      }

      if (DIA_SUCCESS!=insertAPConfigElements_dbus(apConfigVariant))
      {
         DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::prepareSetup insertAPConfigElements_dbus failed.");
         return DIA_FAILED;
      }
   }
   else if (!sStationMode.compare(mode))
   {
      DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::prepareSetup STA.");
      if (DIA_SUCCESS!=insert_map_element_dbus(setupsMap, "Mode", DBUS_TYPE_STRING, &mode))
      {
         DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::prepareSetup insert_map_element_dbus failed.");
         return DIA_FAILED;
      }
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::prepareSetup Wrong input param mode='%s'.", mode.c_str());
      return DIA_FAILED;
   }

   if ((mpSrvPlugin) && (mpSrvPlugin->getProxy()))
   {
      act_t act = mpSrvPlugin->getProxy()->sendPrepareSetupRequest(*this, setupsMap);
      if (0!=act)
      {
         retCode = DIA_SUCCESS;
      }
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::prepareSetup. mpSrvPlugin or mpSrvPlugin->getProxy is NULL.");
   }

   DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::prepareSetup. returned %s", (DIA_SUCCESS==retCode? "SUCCESS": "FAILED"));

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureWifiBLConflictManagement::regActiveSetups()
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::regActiveSetups");

   tDiaResult retCode = DIA_FAILED;

   if ((mpSrvPlugin) && (mpSrvPlugin->getProxy()))
   {
      mActReg = mpSrvPlugin->getProxy()->sendActiveWiFiSetupsRegister(*this);
      if (0!=mActReg)
      {
         retCode = DIA_SUCCESS;
      }
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::regActiveSetups. mpSrvPlugin or mpSrvPlugin->getProxy is NULL.");
   }

   DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::regActiveSetups returned %s with mActReg=0x%08lX.", (DIA_SUCCESS==retCode? "SUCCESS": "FAILED"), mActReg);

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureWifiBLConflictManagement::deregActiveSetups()
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::deregActiveSetups()");

   tDiaResult retCode = DIA_FAILED;

   if ((mpSrvPlugin) && (mpSrvPlugin->getProxy()))
   {
      if (0!=mActReg)
      {
         DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::DeregActiveSetups mActReg=0x%08lX.", mActReg);
         bool result = mpSrvPlugin->getProxy()->sendActiveWiFiSetupsDeregister(mActReg);
         if (true==result)
         {
            retCode = DIA_SUCCESS;
            mActReg = 0;
         }
         else
         {
            DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::DeregActiveSetups. Deregister return false, mActReg=0x%08lX.", mActReg);
         }
      }
      else
      {
         DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::DeregActiveSetups mActReg is 0x%08lX.", mActReg);
      }
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::DeregActiveSetups. mpSrvPlugin or mpSrvPlugin->getProxy is NULL.");
   }

   DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::DeregActiveSetups. returned %s", (DIA_SUCCESS==retCode? "SUCCESS": "FAILED"));

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureWifiBLConflictManagement::GetActiveSetups( )
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::GetActiveSetups");

   tDiaResult retCode = DIA_FAILED;

   if ((mpSrvPlugin) && (mpSrvPlugin->getProxy()))
   {
      act_t token = mpSrvPlugin->getProxy()->sendActiveWiFiSetupsGet(*this);
      if (0!=token)
      {
         retCode = DIA_SUCCESS;
      }
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::GetActiveSetups. mpSrvPlugin or mpSrvPlugin->getProxy is NULL.");
   }

   DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::GetActiveSetups. returned %s", (DIA_SUCCESS==retCode? "SUCCESS": "FAILED"));

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureWifiBLConflictManagement::deActivateSetup(const ::std::string& mode)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::deActivateSetup(string)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::deActivateSetup mode='%s'", mode.c_str());

   if ((mpSrvPlugin) && (mpSrvPlugin->getProxy()))
   {
      act_t act = mpSrvPlugin->getProxy()->sendDeActivateSetupRequest(*this, mode);
      if (0!=act)
      {
         retCode = DIA_SUCCESS;
      }
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureWifiBLConflictManagement::deActivateSetup. mpSrvPlugin or mpSrvPlugin->getProxy is NULL.");
   }

   DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::deActivateSetup. returned %s", (DIA_SUCCESS==retCode? "SUCCESS": "FAILED"));

   return retCode;
}

//=============================================================================
//=============================================================================

void
dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupError(const ::boost::shared_ptr< ConflictmanagementProxy >& /*proxy*/, const ::boost::shared_ptr< DeActivateSetupError >& error)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupError(const ::boost::shared_ptr< ConflictmanagementProxy >&, const ::boost::shared_ptr< DeActivateSetupError >&)");

   DIA_TR_ERR( "dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupError. ErrorName     = \"%s\"", error->getName().c_str() );
   DIA_TR_ERR( "dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupError. ErrorMessage  = \"%s\"", error->getMessage().c_str() );

   tDiaResult retCode = DIA_FAILED;
   getInstanceOfApplication()->postMessage (
         new dia_tclDiagSession::tclEventIntMsgRxGeneric (
               new dia_FunctorOneArgNoReturnValue<dia_SAFeatureWifiBLConflictManagement,tDiaResult>(this,&dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupResponse,retCode)
         )
   );
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupResponse(const ::boost::shared_ptr< ConflictmanagementProxy >& /*proxy*/, const ::boost::shared_ptr< DeActivateSetupResponse >& /*response*/)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupResponse(const ::boost::shared_ptr< ConflictmanagementProxy >&, const ::boost::shared_ptr< DeActivateSetupResponse >&)");

   tDiaResult retCode = DIA_SUCCESS;
   getInstanceOfApplication()->postMessage (
         new dia_tclDiagSession::tclEventIntMsgRxGeneric (
               new dia_FunctorOneArgNoReturnValue<dia_SAFeatureWifiBLConflictManagement,tDiaResult>(this,&dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupResponse,retCode)
         )
   );
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupResponse(tDiaResult response)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupResponse(tDiaResult)");

   DIA_TR_INF( "dia_SAFeatureWifiBLConflictManagement::onDeActivateSetupResponse response=0x%08X.", response );

   dia_IWifiSetupsListener* pListener = 0;
   if ((querySysAdapterListener<dia_IWifiSetupsListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->onDeActivateSetup(response);
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiBLConflictManagement::onPrepareSetupError(const ::boost::shared_ptr< ConflictmanagementProxy >& /*proxy*/, const ::boost::shared_ptr< PrepareSetupError >& error)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::onPrepareSetupError");

   DIA_TR_ERR( "dia_SAFeatureWifiBLConflictManagement::onPrepareSetupError. ErrorName     = \"%s\"", error->getName().c_str() );
   DIA_TR_ERR( "dia_SAFeatureWifiBLConflictManagement::onPrepareSetupError. ErrorMessage  = \"%s\"", error->getMessage().c_str() );

   tDiaResult retCode = DIA_FAILED;
   getInstanceOfApplication()->postMessage (
         new dia_tclDiagSession::tclEventIntMsgRxGeneric (
               new dia_FunctorOneArgNoReturnValue<dia_SAFeatureWifiBLConflictManagement,tDiaResult>(this,&dia_SAFeatureWifiBLConflictManagement::onPrepareSetupResponse,retCode)
         )
   );
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiBLConflictManagement::onPrepareSetupResponse(const ::boost::shared_ptr< ConflictmanagementProxy >& /*proxy*/, const ::boost::shared_ptr< PrepareSetupResponse >& /*response*/)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::onPrepareSetupResponse(const ::boost::shared_ptr< ConflictmanagementProxy >&, const ::boost::shared_ptr< PrepareSetupResponse >&)");

   /*
    * In General for the any PrepareSetup Request you could expect response / Error. If it is response then it means no Error. If it gives Error then you would receive the below errors...
    * org.bosch.wbl.Error.InvalidArguments
    * org.bosch.wbl.Error.PermissionDenied
    * org.bosch.wbl.Error.NotSupported
    * org.bosch.wbl.Error.Rejected
    * org.bosch.wbl.Error.InProgress
    * org.bosch.wbl.Error.Cancelled
    */

   tDiaResult retCode = DIA_SUCCESS;
   getInstanceOfApplication()->postMessage (
         new dia_tclDiagSession::tclEventIntMsgRxGeneric (
               new dia_FunctorOneArgNoReturnValue<dia_SAFeatureWifiBLConflictManagement,tDiaResult>(this,&dia_SAFeatureWifiBLConflictManagement::onPrepareSetupResponse,retCode)
         )
   );
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiBLConflictManagement::onPrepareSetupResponse(tDiaResult response)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::onPrepareSetupResponse(tDiaResult)");

   DIA_TR_INF( "dia_SAFeatureWifiBLConflictManagement::onPrepareSetupResponse response=0x%08X.", response );

   dia_IWifiSetupsListener* pListener = nullptr;
   if ((querySysAdapterListener<dia_IWifiSetupsListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->onPrepareSetup(response);
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiBLConflictManagement::onActiveWiFiSetupsError(const ::boost::shared_ptr< ConflictmanagementProxy >& /*proxy*/, const ::boost::shared_ptr< ActiveWiFiSetupsError >& error)
{
    dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::onActiveWiFiSetupsError");

    DIA_TR_ERR( "dia_SAFeatureWifiBLConflictManagement::onActiveWiFiSetupsError. ErrorName     = \"%s\"", error->getName().c_str() );
    DIA_TR_ERR( "dia_SAFeatureWifiBLConflictManagement::onActiveWiFiSetupsError. ErrorMessage  = \"%s\"", error->getMessage().c_str() );
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiBLConflictManagement::onActiveWiFiSetupsUpdate(const ::boost::shared_ptr< ConflictmanagementProxy >& /*proxy*/, const ::boost::shared_ptr< ActiveWiFiSetupsUpdate >& update)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::onActiveWiFiSetupsUpdate");

   asf::dbus::DBusVariant Variant;
   DBusMessageIter* iter = nullptr;
   void* vpVal = nullptr;
   std::vector<dia_stWifiSetup> diaWifiSetups;
   std::vector< ActiveWiFiSetupsStruct > wifiSetups = update->getActiveWiFiSetups();

   for (std::vector< ActiveWiFiSetupsStruct >::const_iterator it = wifiSetups.begin(); it != wifiSetups.end(); ++it)
   {
      std::string objPath = it->getElem1();

      DIA_TR_INF("objPath = '%s'", objPath.c_str());

      std::map< ::std::string, ::asf::dbus::DBusVariant > Var = it->getElem2();

      Variant = Var.find("PowerState")->second;
      iter = Variant.getReadIterator();
      vpVal = getDbusIterValue(iter);
      std::string PowerState = reinterpret_cast<char*>(vpVal);
      //Variant.clear();

      DIA_TR_INF("Power State: '%s'", PowerState.c_str());

      Variant = Var.find("CurrentOperatingChannel")->second;
      iter = Variant.getReadIterator();
      vpVal = getDbusIterValue(iter);
      tU32 u32Channel = *(reinterpret_cast<tU32*>(vpVal));
      //Variant.clear();

      DIA_TR_INF("Current Operating Channel: %d", u32Channel);

      Variant = Var.find("Frequency")->second;
      iter = Variant.getReadIterator();
      vpVal = getDbusIterValue(iter);
      std::string Frequency = reinterpret_cast<char*>(vpVal);
      dia_enWifiFrequency eFreq = DIA_ENUM_WIFI_FREQ_UNKNOWN;
      if (!Frequency.compare("5 GHz"))
      {
         eFreq = DIA_ENUM_WIFI_FREQ_5GHZ;
      }
      else if (!Frequency.compare("2.4 GHz"))
      {
         eFreq = DIA_ENUM_WIFI_FREQ_2_4GHZ;
      }
      //Variant.clear();

      DIA_TR_INF("Frequency: '%s'", Frequency.c_str());

      if (objPath.compare("/org/bosch/wbl/wlan0/wifisetup/STA") == 0)
      {
         dia_stWifiSetup diaWifiSetup;
         diaWifiSetup.mMode = DIA_ENUM_WIFI_MODE_STA;
         diaWifiSetup.mPoweredOn = (PowerState.compare("WIFI_STATE_POWERED_ON") == 0)? true: false;
         diaWifiSetup.mChannel = u32Channel;
         diaWifiSetup.mFrequency = eFreq;

         diaWifiSetups.push_back(diaWifiSetup);
      } //STA
      else if (objPath.compare("/org/bosch/wbl/wlan1/wifisetup/AP") == 0)
      {
         dia_stWifiSetup diaWifiSetup;
         diaWifiSetup.mMode = DIA_ENUM_WIFI_MODE_AP;
         diaWifiSetup.mPoweredOn = (PowerState.compare("WIFI_STATE_POWERED_ON") == 0)? true: false;
         diaWifiSetup.mChannel = u32Channel;
         diaWifiSetup.mFrequency = eFreq;

         Variant = Var.find("APConfig")->second;
         DBusMessageIter* iter1 = Variant.getReadIterator();

         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter1, &subiter);

         while (dbus_message_iter_get_arg_type(&subiter) != DBUS_TYPE_INVALID)
         {
            std::string strAPConfigKey;
            DBusMessageIter sub2Iter;
            dbus_message_iter_recurse(&subiter, &sub2Iter);

            const char* value = nullptr;
            dbus_message_iter_get_basic(&sub2Iter, &value);
            strAPConfigKey.assign(value);

            DIA_TR_INF("strAPConfigKey: %s", strAPConfigKey.c_str());

            while (dbus_message_iter_next(&sub2Iter))
            {
               if (strAPConfigKey.compare("AssociatedStations") == 0)
               {
                  std::vector<std::string> deviceList;
                  getConnectedDeviceList(&sub2Iter, deviceList, 0);

                  diaWifiSetup.mConnectedDevices = (tU32)(deviceList.size());

               } //if (strAPConfigKey.compare("AssociatedStations") == 0)
               else if (strAPConfigKey.compare("SSID") == 0)
               {
                  DBusMessageIter sub3Iter;
                  dbus_message_iter_recurse(&sub2Iter, &sub3Iter);
                  while (dbus_message_iter_get_arg_type(&sub3Iter) != DBUS_TYPE_INVALID)
                  {
                     ::DBusMessageIter subIter;
                     ::std::string subSignature;

                     dbus_message_iter_recurse(&sub3Iter, &subIter);
                     char* sigPtr = dbus_message_iter_get_signature(&subIter);
                     subSignature.assign(sigPtr);
                     dbus_free(sigPtr);
                     if (0 == subSignature.compare("y")) //byte
                     {
                        while (dbus_message_iter_get_arg_type(&subIter) != DBUS_TYPE_INVALID)
                        {
                           unsigned char value = 0;
                           dbus_message_iter_get_basic(&subIter, &value);
                           diaWifiSetup.mSSID.push_back(value);
                           dbus_message_iter_next(&subIter);
                        }
                     }
                     diaWifiSetup.mSSID.push_back(0);

                     DIA_TR_INF("Hotspot SSID: '%s'", diaWifiSetup.mSSID.data());

                     dbus_message_iter_next(&sub3Iter);
                  } //while (dbus_message_iter_get_arg_type(&sub3Iter) != DBUS_TYPE_INVALID)
               } //else if (strAPConfigKey.compare("SSID") == 0)
            } //while (dbus_message_iter_next(&sub2Iter))

            dbus_message_iter_next(&subiter);
         } //while (dbus_message_iter_get_arg_type(&subiter) != DBUS_TYPE_INVALID)

         diaWifiSetups.push_back(diaWifiSetup);
      } //AP
   } //for (std::vector< ActiveWiFiSetupsStruct >::const_iterator it = wifiSetups.begin(); it != wifiSetups.end(); it++)

   getInstanceOfApplication()->postMessage(
       OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
            OSAL_NEW dia_FunctorOneTemplateArgNoReturnValue<dia_SAFeatureWifiBLConflictManagement,dia_stWifiSetup,std::vector>(this, &dia_SAFeatureWifiBLConflictManagement::onActiveWiFiSetupsUpdate, diaWifiSetups)
       )
   );
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiBLConflictManagement::onActiveWiFiSetupsUpdate( const std::vector<dia_stWifiSetup>& setupList )
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::onActiveWiFiSetupsUpdate");

   dia_IWifiSetupsListener* pListener = 0;
   if ((querySysAdapterListener<dia_IWifiSetupsListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->OnActiveSetups(setupList);
   }
}

//-----------------------------------------------------------------------------

void*
dia_SAFeatureWifiBLConflictManagement::getDbusIterValue(DBusMessageIter* iter)
{
   void* pVal = NULL;
   int type = dbus_message_iter_get_arg_type(iter);

   DIA_TR_INF("dia_SAFeatureWifiBLConflictManagement::getDbusIterValue: type = %d", type);

   if (type == DBUS_TYPE_INVALID)
   {
      return NULL;
   }
   switch (type)
   {
      case DBUS_TYPE_STRING:
      case DBUS_TYPE_SIGNATURE:
      case DBUS_TYPE_OBJECT_PATH:
      {
         char* val = nullptr;
         dbus_message_iter_get_basic(iter, &val);
         if (val)
         {
            char* cpVal = (char*) malloc(strlen(val) + 1);
            if (cpVal)
            {
               strcpy(cpVal, val);
               return cpVal;
            }
         }
         break;
      }
      case DBUS_TYPE_INT16:
      case DBUS_TYPE_UINT16:
      {
         dbus_uint16_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_INT32:
      case DBUS_TYPE_UINT32:
      {
         dbus_uint32_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_INT64:
      case DBUS_TYPE_UINT64:
      {
         dbus_uint64_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_DOUBLE:
      {
         double val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_BYTE:
      {
         unsigned char val;
         dbus_message_iter_get_basic(iter, &val);
         int iVal = static_cast< int >(val);
         pVal = malloc(sizeof(int));
         if (pVal != NULL)
         {
            memcpy(pVal, &iVal, sizeof(val));
         }
         break;
      }

      case DBUS_TYPE_BOOLEAN:
      {
         dbus_bool_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = malloc(sizeof(bool));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(bool));
         }
         return pVal;
         break;
      }
      case DBUS_TYPE_VARIANT:
      {
         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter, &subiter);
         void* vpVal = getDbusIterValue(&subiter);
         return vpVal;
      }
      case DBUS_TYPE_DICT_ENTRY:
      {
         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter, &subiter);
         getDbusIterValue(&subiter);
         dbus_message_iter_next(&subiter);
         void* vpVal = getDbusIterValue(&subiter);
         return vpVal;
      }
      case DBUS_TYPE_ARRAY:
      case DBUS_TYPE_STRUCT:
      {
         int current_type;
         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter, &subiter);
         void* vpVal = NULL;
         while ((current_type = dbus_message_iter_get_arg_type(&subiter))
                != DBUS_TYPE_INVALID)
         {
            vpVal = getDbusIterValue(&subiter);
            dbus_message_iter_next(&subiter);
            if (dbus_message_iter_get_arg_type(&subiter) != DBUS_TYPE_INVALID)
            {
               //TO DO:
            }
         }
         return vpVal;
         break;
      }

      default:
      {
         return pVal;
      }
   }
   return pVal;
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureWifiBLConflictManagement::getConnectedDeviceList(DBusMessageIter* iter, std::vector< std::string >& deviceList, int check)
{
   DBusMessageIter subiter, sub2Iter, sub3Iter;

   dbus_message_iter_recurse(iter, &subiter);
   dbus_message_iter_recurse(&subiter, &sub2Iter);

   while (dbus_message_iter_get_arg_type(&sub2Iter) != DBUS_TYPE_INVALID)
   {
      dbus_message_iter_recurse(&sub2Iter, &sub3Iter);

      if (check == 0)
      {
         while (dbus_message_iter_next(&sub3Iter))
         {
            getConnectedDeviceList(&sub3Iter, deviceList, 1);
         }
      }
      else
      {
         const char* value = NULL;
         dbus_message_iter_get_basic(&sub3Iter, &value);
         std::string strKey;
         strKey.assign(value);

         DIA_TR_INF("getConnectedDeviceList: strKey = '%s'", strKey.c_str());

         if (strKey.compare("HostName") == 0)
         {
            char* sigPtr = dbus_message_iter_get_signature(&sub3Iter);
            std::string Signature;
            Signature.assign(sigPtr);
            dbus_free(sigPtr);

            if (Signature.compare("s") == 0)
            {
               while (dbus_message_iter_get_arg_type(&sub3Iter) != DBUS_TYPE_INVALID)
               {
                  void* vpVal = getDbusIterValue(&sub3Iter);
                  char* HostName = reinterpret_cast<char*>(vpVal);
                  std::string strValue;
                  strValue.assign(HostName);

                  DIA_TR_INF("getConnectedDeviceList: strValue = '%s'", strValue.c_str());

                  if (strValue.compare("HostName") != 0)
                  {
                     deviceList.push_back(strValue);
                  }
                  dbus_message_iter_next(&sub3Iter);
               }
            }
         }
      }
      dbus_message_iter_next(&sub2Iter);
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureWifiBLConflictManagement::insert_map_element_dbus(::std::map< ::std::string, ::asf::dbus::DBusVariant >& map, const ::std::string& key, int type, const void* val)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::insert_map_element_dbus");

   ::asf::dbus::DBusVariant variant;
   DBusMessageIter* iterVariant = variant.getWriteIterator();
   if (!dbus_message_iter_append_basic(iterVariant, type, val))
   {
      DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::insert_map_element_dbus: append failed for type %d.", type);
      return DIA_FAILED;
   }
   map.insert(std::pair< ::std::string, ::asf::dbus::DBusVariant >(key, variant));
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureWifiBLConflictManagement::insertAPConfigElements_dbus(::asf::dbus::DBusVariant& apConfigVariant)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::insertAPConfigElements_dbus");

   DBusMessageIter* iterAPConfigVariant = nullptr;
   DBusMessageIter iterApElement;

   iterAPConfigVariant = apConfigVariant.getWriteIterator();

   if (!dbus_message_iter_open_container(iterAPConfigVariant, DBUS_TYPE_ARRAY, DBUS_DICT_ENTRY_BEGIN_CHAR_AS_STRING
                                          DBUS_TYPE_STRING_AS_STRING DBUS_TYPE_VARIANT_AS_STRING
                                          DBUS_DICT_ENTRY_END_CHAR_AS_STRING, &iterApElement))
   {
      DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::insertAPConfigElements_dbus: open container failed.");
      return DIA_FAILED;
   }

   dbus_uint32_t StationsToBeReserved = 5;
   if (DIA_SUCCESS!=dict_append_entry_dbus(&iterApElement, "StationsToBeReserved", DBUS_TYPE_UINT32, &StationsToBeReserved))
   {
      DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::insertAPConfigElements_dbus: insertAPConfigElements_dbus failed.");
      return DIA_FAILED;
   }

   std::string type = "Normal";
   dict_append_entry_dbus(&iterApElement, "Type", DBUS_TYPE_STRING, &type);

   if (!dbus_message_iter_close_container(iterAPConfigVariant, &iterApElement))
   {
      DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::insertAPConfigElements_dbus: close container failed.");
      return DIA_FAILED;
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureWifiBLConflictManagement::dict_append_entry_dbus(DBusMessageIter* dict, const char* key, int type, void* val)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::dict_append_entry_dbus");

   DBusMessageIter entry;
   if (type == DBUS_TYPE_STRING)
   {
      const char* str = *((const char**) val);
      if (str == NULL)
      {
         DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::dict_append_entry_dbus: str null.");
         return DIA_FAILED;
      }
   }

   if (!dbus_message_iter_open_container(dict, DBUS_TYPE_DICT_ENTRY, NULL, &entry))    { DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::dict_append_entry_dbus: open container failed.");        return DIA_FAILED; }
   if (!dbus_message_iter_append_basic(&entry, DBUS_TYPE_STRING, &key))                { DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::dict_append_entry_dbus: append basic failed.");          return DIA_FAILED; }
   if (DIA_SUCCESS!=append_variant_dbus(&entry, type, val))                            { DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::dict_append_entry_dbus: append_variant_dbus failed.");   return DIA_FAILED; }
   if (!dbus_message_iter_close_container(dict, &entry))                               { DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::dict_append_entry_dbus: close container failed.");       return DIA_FAILED; }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureWifiBLConflictManagement::append_variant_dbus(DBusMessageIter* iter, int type, void* val)
{
   dia_tclFnctTrace trc("dia_SAFeatureWifiBLConflictManagement::append_variant_dbus");

   DBusMessageIter value;
   char sig[2] = { char(type), '\0'};

   if (!dbus_message_iter_open_container(iter, DBUS_TYPE_VARIANT, sig, &value))  { DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::append_variant_dbus: open container failed.");     return DIA_FAILED; }
   if (!dbus_message_iter_append_basic(&value, type, val))                       { DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::append_variant_dbus: append basic failed.");       return DIA_FAILED; }
   if (!dbus_message_iter_close_container(iter, &value))                         { DIA_TR_ERR("### dia_SAFeatureWifiBLConflictManagement::append_variant_dbus: close container failed.");    return DIA_FAILED; }

   return DIA_SUCCESS;
}
