/************************************************************************
* FILE:           dia_SAFeatureErrorLog.cpp
* PROJECT:        Mazda
* SW-COMPONENT:   fc_diagnosis
* AUTHOR:         Andreas Evers
* DESCRIPTION:    System Adapter for vd_diaglog
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*************************************************************************
* October 2014 - refactoring of file according existing midw-FI
*************************************************************************/
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLOGFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLOGFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLOGFI_SERVICEINFO
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE

#ifndef DIA_SAFEATUREERRORLOG_H_
#include "dia_SAFeatureErrorLog.h"
#endif

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#ifndef __INCLUDED_DIA_ERROR_INFO_INTERFACE__
#include <common/framework/errmgmt/dia_ErrorInfoInterface.h>
#endif

#ifndef __INCLUDED_DIA_TEST_CONTROLLER__
#include "common/framework/test/dia_TestController.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGNOSTICS_SYSADAPT
#include "trcGenProj/Header/dia_SAFeatureErrorLog.cpp.trc.h"
#endif

#define BP_LEN_REPORT_ITC_STATUS    0x03   // data length to report ITC status

//#undef MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION_AVAILABLE
#define MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION_AVAILABLE

/*************************************************************************
*
* FUNCTION: MESSAGE MAP and ON_MESSAGE
*
* DESCRIPTION: the function will be called when a message with the corresponding FID arrives
*
* PARAMETER of ON_MESSAGE: FKTID and handle function called by server
*
* RETURNVALUE: none
*
**************************************************************************/

BEGIN_MSG_MAP(dia_SAFeatureErrorLog, dia_SystemAdapterFeature)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_CLEARDTC,                   vHandleClearResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_CONTROL,                    vHandleControlResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETDTCBYSTATUSMASK, vHandleReadDTCbyStatusMaskResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETEXTENDEDDATABYDTC,       vHandleGetExtendedDataByDTCResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETLASTCONFIRMEDDTC,        vHandleReadLastConfirmedDTCResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETNUMBEROFDTCBYSTATUSMASK, vHandleReadNumberOfDTCbyStatusMaskResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETSTATUSBYDTC, vHandleReadStatusByDTCResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETSTATUSBYITC, vHandleReadStatusByITCResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETSUPPORTEDDTC, vHandleReadSupportedDTCResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT,             vHandleTestResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT,vHandleSendNextTestResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTIDENTIFICATION,vHandleSnapShotIdResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTRECORDBYDTCNUMBER,vHandleSnapShotDataResult)
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST,vHandleActiveDTCListResult)
#ifdef MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION_AVAILABLE //To be prepared when it is available
   ON_MESSAGE(MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION,vHandleGetItcInformationResult)
#endif
END_MSG_MAP()

//-----------------------------------------------------------------------------

const midw_fi_tcl_e8_TestResult::tenType dia_SAFeatureErrorLog::mTestResultMapping[DIA_EN_ERRLOG_TESTRESULT_COUNT] =
{
   midw_fi_tcl_e8_TestResult::FI_EN_FAILED,
   midw_fi_tcl_e8_TestResult::FI_EN_PASSED,
   midw_fi_tcl_e8_TestResult::FI_EN_FAILED
};

//-----------------------------------------------------------------------------

dia_SAFeatureErrorLog::dia_SAFeatureErrorLog  (
      tCString name,      // feature name
      dia_SystemAdapterServicePlugin& pSrvPlugin
   )
   : dia_SystemAdapterFeature(name,pSrvPlugin)
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::dia_SAFeatureErrorLog");
}
//-----------------------------------------------------------------------------

dia_SAFeatureErrorLog::~dia_SAFeatureErrorLog()
{
}

//-----------------------------------------------------------------------------

tDiaResult dia_SAFeatureErrorLog::startMonitoring(tVoid)
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::startMonitoring()");
   tDiaResult retCode = DIA_SUCCESS;

   if (_bFidReg == FALSE)
   {
      if (registerFunction(MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT) != DIA_SUCCESS)
      {
         retCode = DIA_FAILED;
         DIA_TR_ERR("##### REGISTRATION TO DIAGLOG PROPERTY MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT FAILED #####");
      }
      if (this->config.isActiveDTCListSupported() && (registerFunction(MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST) != DIA_SUCCESS))
      {
         DIA_TR_ERR("##### REGISTRATION TO DIAGLOG PROPERTY MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST FAILED #####");
         retCode = DIA_FAILED;
      }
      if(retCode == DIA_SUCCESS)
      {
         _bFidReg = TRUE;
      }

   }
   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::u32TestDTCStart ( const tU8* au8TestResultData )
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::u32TestDTCStart");
   tDiaResult u32RetVal = DIA_FAILED;

   if((au8TestResultData[0] < BP_LEN_REPORT_ITC_STATUS) || ((au8TestResultData[0] % BP_LEN_REPORT_ITC_STATUS) != 0))
   {
      DIA_TR_ERR("dia_SAFeatureErrorLog::u32TestDTCStart => ERROR: data length not correct");

      return u32RetVal;
   }

   // the data buffer is not empty and can be composed to a TestResultList
   DIA_TR_INF("dia_SAFeatureErrorLog::u32TestDTCStart => test result data hat length = %d", au8TestResultData[0]);

   midw_diaglogfi_tclMsgSaveTestResultMethodStart oTestData;

   // add test result
   for ( int i=0; i < au8TestResultData[0]; i+=BP_LEN_REPORT_ITC_STATUS )
   {
      midw_fi_tcl_TestResult oTestResult;

      //fill this oTestResult with ITC from data
      oTestResult.TroubleCode  = au8TestResultData[i+2];
      oTestResult.TroubleCode |= au8TestResultData[i+1] << 8;

      //fill this oTestResult with result type from data
      oTestResult.Result.enType = (midw_fi_tcl_e8_TestResult::tenType) au8TestResultData[i+3];

      //add the oTestResult into TestResultList
      oTestData.TestResultList.TestResultList.push_back(oTestResult);
   }

   //put the oTestData into MethodStart message
   fi_tclVisitorMessage oTestDTCStartMsg(oTestData);

   //construct start message
   mpSrvPlugin->vInitServiceData(oTestDTCStartMsg, MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   DIA_TR_INF("dia_SAFeatureErrorLog::u32TestDTCStart => post the Message to DiagLog server");

   if (mpSrvPlugin->bTransmitMessage(&oTestDTCStartMsg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::clearDTC ( tU8 u8Group, tU32 u32GroupOfDtc )
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::clearDTC");
   tDiaResult u32RetVal = DIA_FAILED;

   DIA_TR_INF("dia_SAFeatureErrorLog::clearDTC => u8Group = %d, u32GroupOfDtc = 0x%x", u8Group, u32GroupOfDtc);

   midw_diaglogfi_tclMsgclearDTCMethodStart oRequestData;

   oRequestData.Group = u8Group;
   oRequestData.DTCMask = u32GroupOfDtc;

   //put the data in message
   fi_tclVisitorMessage oClearDTCStartMsg(oRequestData);

   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(oClearDTCStartMsg,
         MIDW_DIAGLOGFI_C_U16_CLEARDTC,
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   DIA_TR_INF("dia_SAFeatureErrorLog::clearDTC => post the message to DiagLog server");

   if (mpSrvPlugin->bTransmitMessage(&oClearDTCStartMsg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::getStatusByDTC ( tU8 u8Group, tU32 u32DTCMask )
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::getStatusByDTC");
   tDiaResult u32RetVal = DIA_FAILED;

   DIA_TR_INF("dia_SAFeatureErrorLog::getStatusByDTC => u8Group = %d, u32DTCMask = 0x%x", u8Group, u32DTCMask);

   midw_diaglogfi_tclMsggetStatusByDTCMethodStart oRequestData;

   oRequestData.Group = u8Group;
   oRequestData.DTCMask = u32DTCMask;

   //put the data in message
   fi_tclVisitorMessage oReadDTCStartMsg(oRequestData);

   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(oReadDTCStartMsg,
         MIDW_DIAGLOGFI_C_U16_GETSTATUSBYDTC,
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   DIA_TR_INF("dia_SAFeatureErrorLog::getStatusByDTC => post the message to DiagLog server");

   if (mpSrvPlugin->bTransmitMessage(&oReadDTCStartMsg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::getStatusByITC ( tU8 u8Group, tU32 u32ITCMask )
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::getStatusByITC");
   tDiaResult u32RetVal = DIA_FAILED;

   DIA_TR_INF("dia_SAFeatureErrorLog::getStatusByITC => u8Group = %d, u32ITCMask = 0x%x", u8Group, u32ITCMask);

   midw_diaglogfi_tclMsggetStatusByITCMethodStart oRequestData;

   oRequestData.Group = u8Group;
   oRequestData.ITCMask = u32ITCMask;

   //put the data in message
   fi_tclVisitorMessage oReadDTCStartMsg(oRequestData);

   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(oReadDTCStartMsg,
         MIDW_DIAGLOGFI_C_U16_GETSTATUSBYITC,
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   DIA_TR_INF("dia_SAFeatureErrorLog::getStatusByITC => post the message to DiagLog server");

   if (mpSrvPlugin->bTransmitMessage(&oReadDTCStartMsg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::getDTCbyStatusMask ( tU8 u8Group, tU8 u8Mask )
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::getDTCbyStatusMask");
   tDiaResult u32RetVal = DIA_FAILED;

   DIA_TR_INF("dia_SAFeatureErrorLog::getDTCbyStatusMask => u8Group = %d, u8Mask = 0x%x", u8Group, u8Mask);

   midw_diaglogfi_tclMsggetDTCbyStatusMaskMethodStart oRequestData;

   oRequestData.Group = u8Group;
   oRequestData.StatusMask = u8Mask;

   //put the data in message
   fi_tclVisitorMessage oReadDTCStartMsg(oRequestData);

   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(oReadDTCStartMsg,
         MIDW_DIAGLOGFI_C_U16_GETDTCBYSTATUSMASK,
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   DIA_TR_INF("dia_SAFeatureErrorLog::getDTCbyStatusMask => post the message to DiagLog server");

   if (mpSrvPlugin->bTransmitMessage(&oReadDTCStartMsg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::getNumberOfDTCbyStatusMask ( tU8 u8Group, tU8 u8Mask )
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::getNumberOfDTCbyStatusMask");
   tDiaResult u32RetVal = DIA_FAILED;

   DIA_TR_INF("dia_SAFeatureErrorLog::getNumberOfDTCbyStatusMask => u8Group = %d, u8Mask = 0x%x", u8Group, u8Mask);

   midw_diaglogfi_tclMsggetNumberOfDTCByStatusMaskMethodStart oRequestData;

   oRequestData.Group = u8Group;
   oRequestData.StatusMask = u8Mask;

   //put the data in message
   fi_tclVisitorMessage oReadDTCStartMsg(oRequestData);

   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(oReadDTCStartMsg,
         MIDW_DIAGLOGFI_C_U16_GETNUMBEROFDTCBYSTATUSMASK,
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   DIA_TR_INF("dia_SAFeatureErrorLog::getNumberOfDTCbyStatusMask => post the message to DiagLog server");

   if (mpSrvPlugin->bTransmitMessage(&oReadDTCStartMsg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::getSupportedDTC ( tU8 u8Group )
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::getSupportedDTC");
   tDiaResult u32RetVal = DIA_FAILED;

   DIA_TR_INF("dia_SAFeatureErrorLog::getSupportedDTC => u8Group = %d", u8Group);

   midw_diaglogfi_tclMsggetSupportedDTCMethodStart oRequestData;

   oRequestData.Group = u8Group;

   //put the data in message
   fi_tclVisitorMessage oReadDTCStartMsg(oRequestData);

   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(oReadDTCStartMsg,
         MIDW_DIAGLOGFI_C_U16_GETSUPPORTEDDTC,
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   DIA_TR_INF("dia_SAFeatureErrorLog::getSupportedDTC => post the message to DiagLog server");

   if (mpSrvPlugin->bTransmitMessage(&oReadDTCStartMsg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::getLastConfirmedDTC ( tU8 u8Group )
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::getLastConfirmedDTC");

   tDiaResult u32RetVal = DIA_FAILED;
   DIA_TR_INF("dia_SAFeatureErrorLog::getLastConfirmedDTC => u8Group = %d", u8Group);

   midw_diaglogfi_tclMsggetLastConfirmedDTCMethodStart oRequestData;

   oRequestData.Group = u8Group;

   //put the data in message
   fi_tclVisitorMessage oReadDTCStartMsg(oRequestData);

   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(oReadDTCStartMsg,
         MIDW_DIAGLOGFI_C_U16_GETLASTCONFIRMEDDTC,
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   DIA_TR_INF("dia_SAFeatureErrorLog::getLastConfirmedDTC => post the message to DiagLog server");

   if (mpSrvPlugin->bTransmitMessage(&oReadDTCStartMsg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::getExtendedData ( tU8 group, tU32 dtc, tU8 recordNumber )
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::getExtendedData");
   tDiaResult u32RetVal = DIA_FAILED;

   midw_diaglogfi_tclMsggetExtendedDataByDTCMethodStart oMsgData;
   oMsgData.Group = group;
   oMsgData.DTC = dtc;
   oMsgData.DTCExtDataRecordNumber = recordNumber;
   //put the data in message
   fi_tclVisitorMessage msg(oMsgData);
   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(msg, MIDW_DIAGLOGFI_C_U16_GETEXTENDEDDATABYDTC, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);
   DIA_TR_INF("dia_SAFeatureErrorLog::getExtendedData => post the message to DiagLog server");
   if (mpSrvPlugin->bTransmitMessage(&msg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }
   return u32RetVal;
}
//-----------------------------------------------------------------------------

tDiaResult
#ifdef MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION_AVAILABLE //To be prepared when it is available
dia_SAFeatureErrorLog::getItcInformation ( tU8 u8Group, const std::vector<tU16>& ItcList )
#else //Lint :)
dia_SAFeatureErrorLog::getItcInformation ( tU8 u8Group, const std::vector<tU16>&  )
#endif
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::getItcInformation");
   tDiaResult u32RetVal = DIA_FAILED;

   DIA_TR_INF("dia_SAFeatureErrorLog::getItcInformation => u8Group = %d", u8Group);

#ifdef MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION_AVAILABLE //To be prepared when it is available
   midw_diaglogfi_tclMsggetItcInformationMethodStart oRequestData;

   oRequestData.Group = u8Group;

   if (ItcList.size() != 0)
   {
      std::vector<tU16>::const_iterator iter = ItcList.begin();
      for ( ; iter != ItcList.end(); iter++ )
      {
         oRequestData.ITCList.push_back(*iter);
      }
   }

   //put the data in message
   fi_tclVisitorMessage oReadDTCStartMsg(oRequestData);

   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(oReadDTCStartMsg,
         MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION,
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   DIA_TR_INF("dia_SAFeatureErrorLog::getItcInformation => post the message to DiagLog server");

   if (mpSrvPlugin->bTransmitMessage(&oReadDTCStartMsg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }
#endif

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::getSnapShotId ( tU8 group )
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::getSnapShotId");
   tDiaResult u32RetVal = DIA_FAILED;

   midw_diaglogfi_tclMsgreportDTCSnapshotIdentificationMethodStart oMsgData;
   oMsgData.Group = group;
   //put the data in message
   fi_tclVisitorMessage msg(oMsgData);
   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(msg, MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTIDENTIFICATION, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);
   DIA_TR_INF("dia_SAFeatureErrorLog::getSnapShotId => post the message to DiagLog server");
   if (mpSrvPlugin->bTransmitMessage(&msg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }
   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::getSnapShotData ( tU8 group, tU32 dtc, tU8 recordNumber )
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::getSnapShotData");
   tDiaResult u32RetVal = DIA_FAILED;

   midw_diaglogfi_tclMsgreportDTCSnapshotRecordByDTCNumberMethodStart oMsgData;
   oMsgData.Group = group;
   oMsgData.DTC = dtc;
   oMsgData.DTCSnapshotRecordNumber = recordNumber;
   //put the data in message
   fi_tclVisitorMessage msg(oMsgData);
   //construct MethodStart message
   mpSrvPlugin->vInitServiceData(msg, MIDW_DIAGLOGFI_C_U16_REPORTDTCSNAPSHOTRECORDBYDTCNUMBER, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);
   DIA_TR_INF("dia_SAFeatureErrorLog::getSnapShotData => post the message to DiagLog server");
   if (mpSrvPlugin->bTransmitMessage(&msg) == TRUE)
   {
      u32RetVal = DIA_SUCCESS;
   }
   return u32RetVal;
}


/******************************************************************************
* FUNCTION:    controlDTCSetting
*
* DESCRIPTION: send CCA MethodStart message to trigger DTC Setting
*
* PARAMETER:   u8Mode - ControlMode to forward
*
* RETURNVALUE: TRUE  - post message to DiagLog server no error
*              FALSE - post message to DiagLog server failed
*
******************************************************************************/
tDiaResult
dia_SAFeatureErrorLog::controlDTCSetting(const dia_tenControlType eMode)
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::controlDTCSetting");
   tDiaResult u32RetVal = DIA_FAILED;

   //fill data with the setting
   midw_diaglogfi_tclMsgControlMethodStart oData;

   switch(eMode)
   {
      case DIA_EN_ERRLOG_CONTROL_DTC_SETTING_ON:
      {
         oData.ControlParameter.enType = midw_fi_tcl_e8_ControlParameter::FI_EN_ENABLELOGGING;
         break;
      }

      case DIA_EN_ERRLOG_CONTROL_DTC_SETTING_ON_PERSISTENT:
      {
         oData.ControlParameter.enType = midw_fi_tcl_e8_ControlParameter::FI_EN_ENABLELOGGINGPERSISTENT;
         break;
      }

      case DIA_EN_ERRLOG_CONTROL_DTC_SETTING_OFF:
      {
         oData.ControlParameter.enType = midw_fi_tcl_e8_ControlParameter::FI_EN_DISABLELOGGING;
         break;
      }

      case DIA_EN_ERRLOG_CONTROL_DTC_SETTING_OFF_PERSISTENT:
      {
         oData.ControlParameter.enType = midw_fi_tcl_e8_ControlParameter::FI_EN_DISABLELOGGINPERSISTENT;
         break;
      }

      default:
      {
         DIA_ASSERT_ALWAYS();
         return u32RetVal;
      }
   }// switch(u8Mode)

   //put the data in message
   fi_tclVisitorMessage oControlDTCStartMsg(oData);
   mpSrvPlugin->vInitServiceData(oControlDTCStartMsg, MIDW_DIAGLOGFI_C_U16_CONTROL, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   u32RetVal = (mpSrvPlugin->bTransmitMessage(&oControlDTCStartMsg))? DIA_SUCCESS: DIA_FAILED;

   return u32RetVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::reportTestResults( const std::map<tU16,dia_enTestResult>& mapTestResults )
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::reportTestResults");
   tDiaResult retCode = DIA_FAILED;

   midw_diaglogfi_tclMsgSaveTestResultMethodStart oTestData;

   if ( ( mapTestResults.size() != 0 ))
   {
      std::map<tU16,dia_enTestResult>::const_iterator iter = mapTestResults.begin();
      for ( ; iter != mapTestResults.end(); iter++ )
      {
         midw_fi_tcl_TestResult oTestResult;

         oTestResult.TroubleCode   = iter->first;
         oTestResult.Result.enType = mTestResultMapping[iter->second];
         oTestData.TestResultList.TestResultList.push_back(oTestResult);
      }

      fi_tclVisitorMessage oTestDTCStartMsg(oTestData);
      mpSrvPlugin->vInitServiceData(oTestDTCStartMsg, MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

      if(TRUE == mpSrvPlugin->bTransmitMessage(&oTestDTCStartMsg))
      {
         retCode = DIA_SUCCESS;
      }
   }
   else
   {
      DIA_TR_INF( "data length not correct");
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::reportTestResult ( tU16 u16Itc, dia_enTestResult status )
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::reportTestResult");
   tDiaResult retCode = DIA_FAILED;

   midw_diaglogfi_tclMsgSaveTestResultMethodStart oTestData;
   midw_fi_tcl_TestResult oTestResult;

   oTestResult.TroubleCode   = u16Itc;
   oTestResult.Result.enType = mTestResultMapping[status];
   oTestData.TestResultList.TestResultList.push_back(oTestResult);

   fi_tclVisitorMessage oTestDTCStartMsg(oTestData);
   mpSrvPlugin->vInitServiceData(oTestDTCStartMsg, MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   if(TRUE == mpSrvPlugin->bTransmitMessage(&oTestDTCStartMsg))
   {
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

/*************************************************************************
*  FUNCTION:    vHandleTestResult
*
*   DESCRIPTION: react on messages with FKTID = MIDW_DIAGLOGFI_C_U16_SAVETESTRESULT
*
*   PARAMETER:   message to analyse
*
*   RETURNVALUE: NONE
*
**************************************************************************/

void
dia_SAFeatureErrorLog::vHandleTestResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleTestResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
          DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

          // Extract the op code of the message called
          tU8 u8OpCode = poMessage->u8GetOpCode();

          // Switch on the opCode received
          switch (u8OpCode)
          {
              case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleTestResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
                  fi_tclVisitorMessage oResultMsg(poMessage);
                  midw_diaglogfi_tclMsgSaveTestResultMethodResult oResultData;
                  tS32 s32Result = oResultMsg.s32GetData(oResultData);
                  if ( s32Result == OSAL_OK )
                  {
                     if(oResultData.Status.enType != midw_fi_tcl_e8_MethodResultReply::FI_EN_REJECTED)
                     {
                        pListener->vOnDiaglogWriteTestResult();
                     }
                     else
                     {
                        DIA_TR_INF("dia_SAFeatureErrorLog::vHandleTestResult => oResultData.Status.enType => FI_EN_REJECTED");
                        pListener->vOnDiaglogWriteTestError(EN_ERROR_FATAL_INTERNAL);
                     }
                  }
                  else
                  {
                     DIA_TR_INF("dia_SAFeatureErrorLog::vHandleTestResult => oResultMsg.s32GetData FAILED");
                     pListener->vOnDiaglogWriteTestError(EN_ERROR_FATAL_INTERNAL);
                  }
                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleTestResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");
                  pListener->vOnDiaglogWriteTestError(EN_ERROR_FATAL_INTERNAL);
                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleTestResult => ERROR: unsuported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureErrorLog::vHandleClearResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleClearResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
          DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

          // Extract the op code of the message called
          tU8 u8OpCode = poMessage->u8GetOpCode();

          // Switch on the opCode received
          switch (u8OpCode)
          {
              case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleClearResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
                  fi_tclVisitorMessage oResultMsg(poMessage);
                  midw_diaglogfi_tclMsgclearDTCMethodResult oCompResp;
                  tS32 s32Result = oResultMsg.s32GetData(oCompResp);

                  if ( s32Result == OSAL_OK )
                  {
                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        pListener->vOnClearResult();
                     }
                     else
                     {
                          pListener->vOnClearError(EN_ERROR_FATAL_INTERNAL);
                     }
                  }
                  else
                  {
                     DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleClearResult => ERROR: oResultMsg.s32GetData FAILED");

                     //STC2HI: Error handling TODO
                  }
                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleClearResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                   amt_tclServiceDataError oErrorMsg(poMessage);
                   tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
                   switch (u16ErrorCode)
                   {
                     case AMT_C_U16_ERROR_PARAMETER_OUT_OF_RANGE:
                     {
                        DIA_TR_INF( "dia_SAFeatureErrorLog::vHandleClearResult => AMT_C_U8_CCAMSG_OPCODE_ERROR -> AMT_C_U16_ERROR_PARAMETER_OUT_OF_RANGE");
                        pListener->vOnClearError(EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE);
                        break;
                     }
                     default:
                     {
                        DIA_TR_INF( "dia_SAFeatureErrorLog::vHandleClearResult => AMT_C_U8_CCAMSG_OPCODE_ERROR -> DEFAULT");
                        pListener->vOnClearError(EN_ERROR_FATAL_INTERNAL);
                        break;
                     }
                   }// switch (u16ErrorCode)

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleClearResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureErrorLog::vHandleReadDTCbyStatusMaskResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleReadDTCbyStatusMaskResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
          DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

          // Extract the op code of the message called
          tU8 u8OpCode = poMessage->u8GetOpCode();

          // Switch on the opCode received
          switch (u8OpCode)
          {
              case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadDTCbyStatusMaskResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
                  fi_tclVisitorMessage oResultMsg(poMessage);
                  midw_diaglogfi_tclMsggetDTCbyStatusMaskMethodResult oCompResp;
                  tS32 s32Result = oResultMsg.s32GetData(oCompResp);

                  if ( s32Result == OSAL_OK )
                  {
                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        std::list<tdia_IErrorLogResult> ErrorLogList;

                        for (tU32 i = 0; i < oCompResp.ReadResultList.DTCList.size(); i++)
                        {
                           tdia_IErrorLogResult dtc;

                           dtc.u8Status = oCompResp.ReadResultList.DTCList[i].StatusOfDtc;
                           dtc.u32DTC = oCompResp.ReadResultList.DTCList[i].DTC;

                           ErrorLogList.push_back(dtc);
                        }

                        pListener->vOnReadDTCbyStatusMaskResult(oCompResp.StatusAvailabilityMask,ErrorLogList);
                     }
                     else
                     {
                        //STC2HI: Error handling TODO
                     }
                  }
                  else
                  {
                     DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadDTCbyStatusMaskResult => ERROR: oResultMsg.s32GetData FAILED");

                     //STC2HI: Error handling TODO
                  }
                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadDTCbyStatusMaskResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                  //STC2HI: Error handling TODO

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadDTCbyStatusMaskResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureErrorLog::vHandleReadStatusByDTCResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleReadStatusByDTCResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
          DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

          // Extract the op code of the message called
          tU8 u8OpCode = poMessage->u8GetOpCode();

          // Switch on the opCode received
          switch (u8OpCode)
          {
              case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadStatusByDTCResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
                  fi_tclVisitorMessage oResultMsg(poMessage);
                  midw_diaglogfi_tclMsggetStatusByDTCMethodResult oCompResp;
                  tS32 s32Result = oResultMsg.s32GetData(oCompResp);

                  if ( s32Result == OSAL_OK )
                  {
                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        std::list<tdia_IErrorLogResult> ErrorLogList;

                        for (tU32 i = 0; i < oCompResp.ReadResultList.DTCList.size(); i++)
                        {
                           tdia_IErrorLogResult dtc;

                           dtc.u8Status = oCompResp.ReadResultList.DTCList[i].StatusOfDtc;
                           dtc.u32DTC = oCompResp.ReadResultList.DTCList[i].DTC;

                           ErrorLogList.push_back(dtc);
                        }

                        pListener->vOnReadStatusByDTCResult(oCompResp.StatusAvailabilityMask,ErrorLogList);
                     }
                     else
                     {
                        //STC2HI: Error handling TODO
                     }
                  }
                  else
                  {
                     DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadStatusByDTCResult => ERROR: oResultMsg.s32GetData FAILED");

                     //STC2HI: Error handling TODO
                  }
                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadStatusByDTCResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                  //STC2HI: Error handling TODO

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadStatusByDTCResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureErrorLog::vHandleReadStatusByITCResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleReadStatusByITCResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
          DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

          // Extract the op code of the message called
          tU8 u8OpCode = poMessage->u8GetOpCode();

          // Switch on the opCode received
          switch (u8OpCode)
          {
              case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadStatusByITCResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
                  fi_tclVisitorMessage oResultMsg(poMessage);
                  midw_diaglogfi_tclMsggetStatusByITCMethodResult oCompResp;
                  tS32 s32Result = oResultMsg.s32GetData(oCompResp);

                  if ( s32Result == OSAL_OK )
                  {
                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        std::list<tdia_IErrorLogResult> ErrorLogList;

                        for (tU32 i = 0; i < oCompResp.ReadResultList.DTCList.size(); i++)
                        {
                           tdia_IErrorLogResult dtc;

                           dtc.u8Status = oCompResp.ReadResultList.DTCList[i].StatusOfDtc;
                           dtc.u32DTC = oCompResp.ReadResultList.DTCList[i].DTC;

                           ErrorLogList.push_back(dtc);
                        }

                        pListener->vOnReadStatusByITCResult(ErrorLogList);
                     }
                     else
                     {
                        //STC2HI: Error handling TODO
                     }
                  }
                  else
                  {
                     DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadStatusByITCResult => ERROR: oResultMsg.s32GetData FAILED");

                     //STC2HI: Error handling TODO
                  }
                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadStatusByITCResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                  //STC2HI: Error handling TODO

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadStatusByITCResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureErrorLog::vHandleReadNumberOfDTCbyStatusMaskResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleReadNumberOfDTCbyStatusMaskResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
          DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

          // Extract the op code of the message called
          tU8 u8OpCode = poMessage->u8GetOpCode();

          // Switch on the opCode received
          switch (u8OpCode)
          {
              case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadNumberOfDTCbyStatusMaskResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
                  fi_tclVisitorMessage oResultMsg(poMessage);
                  midw_diaglogfi_tclMsggetNumberOfDTCByStatusMaskMethodResult oCompResp;
                  tS32 s32Result = oResultMsg.s32GetData(oCompResp);

                  if ( s32Result == OSAL_OK )
                  {
                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        pListener->vOnReadNumberOfDTCByStatusMaskResult(oCompResp.StatusAvailabilityMask, oCompResp.NumberOfDTC);
                     }
                     else
                     {
                        //STC2HI: Error handling TODO
                     }
                  }
                  else
                  {
                     DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadNumberOfDTCbyStatusMaskResult => ERROR: oResultMsg.s32GetData FAILED");

                     //STC2HI: Error handling TODO
                  }
                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadNumberOfDTCbyStatusMaskResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                  //STC2HI: Error handling TODO

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadNumberOfDTCbyStatusMaskResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureErrorLog::vHandleReadSupportedDTCResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleReadSupportedDTCResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
          DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

          // Extract the op code of the message called
          tU8 u8OpCode = poMessage->u8GetOpCode();

          // Switch on the opCode received
          switch (u8OpCode)
          {
              case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadSupportedDTCResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
                  fi_tclVisitorMessage oResultMsg(poMessage);
                  midw_diaglogfi_tclMsggetSupportedDTCMethodResult oCompResp;
                  tS32 s32Result = oResultMsg.s32GetData(oCompResp);

                  if ( s32Result == OSAL_OK )
                  {
                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        std::list<tdia_IErrorLogResult> ErrorLogList;

                        for (tU32 i = 0; i < oCompResp.ReadResultList.DTCList.size(); i++)
                        {
                           tdia_IErrorLogResult dtc;

                           dtc.u8Status = oCompResp.ReadResultList.DTCList[i].StatusOfDtc;
                           dtc.u32DTC = oCompResp.ReadResultList.DTCList[i].DTC;

                           ErrorLogList.push_back(dtc);
                        }

                        pListener->vOnReadSupportedDTCResult(oCompResp.StatusAvailabilityMask, ErrorLogList);
                     }
                     else
                     {
                        ETG_TRACE_ERR_THR(("!!! dia_SAFeatureBluetooth::vOnRoutineCtrlResult => ERROR: No Data Available in Datamessage"));

                        dia_ErrorInfoInterface err(dia_IErrorLog::getUID());
                        pListener->vOnError(err);
                     }
                  }
                  else
                  {
                     DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadSupportedDTCResult => ERROR: oResultMsg.s32GetData FAILED");

                     //STC2HI: Error handling TODO
                  }
                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadSupportedDTCResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                  //STC2HI: Error handling TODO

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadSupportedDTCResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureErrorLog::vHandleReadLastConfirmedDTCResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleReadLastConfirmedDTCResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
          DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

          // Extract the op code of the message called
          tU8 u8OpCode = poMessage->u8GetOpCode();

          // Switch on the opCode received
          switch (u8OpCode)
          {
              case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadLastConfirmedDTCResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
                  fi_tclVisitorMessage oResultMsg(poMessage);
                  midw_diaglogfi_tclMsggetLastConfirmedDTCMethodResult oCompResp;
                  tS32 s32Result = oResultMsg.s32GetData(oCompResp);

                  if ( s32Result == OSAL_OK )
                  {
                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        std::list<tdia_IErrorLogResult> ErrorLogList;

                        for (tU32 i = 0; i < oCompResp.ReadResultList.DTCList.size(); i++)
                        {
                           tdia_IErrorLogResult dtc;

                           dtc.u8Status = oCompResp.ReadResultList.DTCList[i].StatusOfDtc;
                           dtc.u32DTC = oCompResp.ReadResultList.DTCList[i].DTC;

                           ErrorLogList.push_back(dtc);
                        }

                        pListener->vOnReadLastConfirmedDTCResult(oCompResp.StatusAvailabilityMask, ErrorLogList);
                     }
                     else
                     {
                        DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadLastConfirmedDTCResult => Error: oCompResp.Status = 0x%x", oCompResp.Status.enType);

                        //STC2HI: Error handling TODO
                     }
                  }
                  else
                  {
                     DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadLastConfirmedDTCResult => ERROR: oResultMsg.s32GetData FAILED");

                     //STC2HI: Error handling TODO
                  }
                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleReadLastConfirmedDTCResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                  //STC2HI: Error handling TODO

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleReadLastConfirmedDTCResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------


void
dia_SAFeatureErrorLog::vHandleGetExtendedDataByDTCResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::vHandleGetExtendedDataByDTCResult");
   if(0 == poMessage)
   {
      return;
   }
   dia_IErrorLogListener* pListener = OSAL_NULL;
   if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) != DIA_SUCCESS) || (0 ==pListener))
   {
      return;
   }
   // Msg must come from a server
   DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

   // Extract the op code of the message called
   tU8 u8OpCode = poMessage->u8GetOpCode();

   // Switch on the opCode received
   switch (u8OpCode)
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
      {
         DIA_TR_INF("dia_SAFeatureErrorLog::vHandleGetExtendedDataByDTCResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
         fi_tclVisitorMessage oResultMsg(poMessage);
         midw_diaglogfi_tclMsggetExtendedDataByDTCMethodResult oCompResp;
         tS32 s32Result = oResultMsg.s32GetData(oCompResp);
         if ( s32Result == OSAL_OK )
         {
            if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
            {
               DIA_TR_INF("dia_SAFeatureErrorLog::vHandleGetExtendedDataByDTCResult: FI Result OK, decoding");
               IErrorLogExtendedDataResult extendedData;
               extendedData.dtc = oCompResp.DTC;
               extendedData.status = oCompResp.StatusOfDtc;
               DIA_TR_INF("dia_SAFeatureErrorLog::vHandleGetExtendedDataByDTCResult: FFList size: %zu", oCompResp.FreezeFrameList.FreezeFrameDataList.size());
               for(size_t i = 0; i < oCompResp.FreezeFrameList.FreezeFrameDataList.size(); ++i)
               {
                  extendedData.data.push_back(oCompResp.FreezeFrameList.FreezeFrameDataList[i].ID);
                  extendedData.data.insert(extendedData.data.end(), oCompResp.FreezeFrameList.FreezeFrameDataList[i].Data.begin(), oCompResp.FreezeFrameList.FreezeFrameDataList[i].Data.end());
               }
               DIA_TR_INF("dia_SAFeatureErrorLog::vHandleGetExtendedDataByDTCResult: Calling listener with extended data");
               pListener->vOnDiaglogGetExtendedDataByDTC(extendedData);
            }
            else
            {
               DIA_TR_ERR(("!!! dia_SAFeatureErrorLog::vHandleGetExtendedDataByDTCResult => ERROR: Negative Response"));

               dia_ErrorInfoInterface err(dia_IErrorLogListener::getUID());
               err.setErrorCode(DIA_E_OUT_OF_RANGE);
               pListener->vOnError(err);
            }
         }
         else
         {
            DIA_TR_INF("dia_SAFeatureErrorLog::vHandleGetExtendedDataByDTCResult => oResultMsg.s32GetData FAILED");
         }
         break;
      }

      case AMT_C_U8_CCAMSG_OPCODE_ERROR:
      {
         DIA_TR_INF("dia_SAFeatureErrorLog::vHandleGetExtendedDataByDTCResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");
         break;
      }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

      default:
      {
         DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleProdClearResult => ERROR: unsuported u8OpCode %x", u8OpCode);
         // unkown opcode: enter error handling here ( trace or assert )
         break;
      }
    }// switch (u8OpCode)
    poMessage->bDelete();
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureErrorLog::vHandleControlResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleControlResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
          DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

          // Extract the op code of the message called
          tU8 u8OpCode = poMessage->u8GetOpCode();

          // Switch on the opCode received
          switch (u8OpCode)
          {
              case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleControlResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
                  fi_tclVisitorMessage oResultMsg(poMessage);
                  midw_diaglogfi_tclMsgControlMethodResult oCompResp;
                  tS32 s32Result = oResultMsg.s32GetData(oCompResp);

                  if ( s32Result == OSAL_OK )
                  {
                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        pListener->vOnControlResult();
                     }
                     else
                     {
                        //STC2HI: Error handling TODO
                     }
                  }
                  else
                  {
                     DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleControlResult => ERROR: oResultMsg.s32GetData FAILED");

                     //STC2HI: Error handling TODO
                  }
                  break;
              }

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleControlResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                  //STC2HI: Error handling TODO

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleControlResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}
//-----------------------------------------------------------------------------
void
dia_SAFeatureErrorLog::vHandleSendNextTestResult ( amt_tclServiceData* poMessage ) const
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::vHandleSendNextTestResult");

   if(poMessage != NULL)
   {
      // Msg must come from a server
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

      tU8 u8OpCode = poMessage->u8GetOpCode();

      switch ( u8OpCode )
      {
         case AMT_C_U8_CCAMSG_OPCODE_STATUS:
         {
            DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSendNextTestResult => AMT_C_U8_CCAMSG_OPCODE_STATUS");
            // Get method result data into fi type
            fi_tclVisitorMessage oMsg(poMessage);
            midw_diaglogfi_tclMsgSendNextTestResultStatus oResultData;
            (tVoid) oMsg.s32GetData(oResultData);

            // only handle if status = true
            if (oResultData.Status)
            {
               DIA_TR_INF("--- dia_SAFeatureErrorLog::vHandleSendNextTestResult => oResultData.status = true");
               //get the test controller pointer
               dia_TestController* pTestController = dia_TestController::getInstance();

               //run the test with condition error mem clear
               if (pTestController)
               {
                  pTestController->runTests(DIA_EN_TESTCONDITION_ERROR_MEMORY_RESET);
               }
            }
            break;
         }// case AMT_C_U8_CCAMSG_OPCODE_STATUS:


         case AMT_C_U8_CCAMSG_OPCODE_ERROR:
         {
            DIA_TR_ERR("--- In CASE: AMT_C_U8_CCAMSG_OPCODE_ERROR");
            mpSrvPlugin->vOnErrorMessage(poMessage);
            break;
         }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

         default:
         {
            DIA_TR_ERR( "dia_SAFeatureErrorLog::vHandleSendNextTestResult: Unexpected OpCode = %d", u8OpCode);
            break;
         }// default:
      }// switch ( u8OpCode )
   } //if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureErrorLog::handleActiveDTCStatus(amt_tclServiceData* poMessage, dia_IErrorLogListener* pListener) const
{
   // Get method result data into fi type
   fi_tclVisitorMessage oMsg(poMessage);
   midw_diaglogfi_tclMsgSendActiveDTCListStatus oResultData;
   (tVoid) oMsg.s32GetData(oResultData);

   size_t listSize = oResultData.ActiveDTCList.DTCList.size();
   // only handle if status = true
   if (listSize != 0)
   {
      DIA_TR_INF("--- dia_SAFeatureErrorLog::handleActiveDTCStatus => oResultData.ActiveDTCList.DTCList size = %zu", listSize);
      std::list<tU32> dtcList;
      for (size_t i = 0; i < listSize; i++)
      {
         dtcList.push_back(oResultData.ActiveDTCList.DTCList[i]);
      }
      DIA_TR_INF("dia_SAFeatureErrorLog::handleActiveDTCStatus: Calling listener with DTCs");
      pListener->vOnFailedDTCList(dtcList);
   }
   else
   {
      DIA_TR_ERR("--- dia_SAFeatureErrorLog::handleActiveDTCStatus => empty list");
#if 0 //Empty list is a valid Status message e.g. after Upreg
      dia_ErrorInfoInterface err(dia_IErrorLogListener::getUID());
      err.setErrorCode(DIA_E_CONDITIONS_NOT_CORRECT);
      pListener->vOnError(err);
#endif
   }
}

void
dia_SAFeatureErrorLog::vHandleActiveDTCListResult ( amt_tclServiceData* poMessage ) const
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::vHandleActiveDTCListResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
         DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

         tU8 u8OpCode = poMessage->u8GetOpCode();

         switch ( u8OpCode )
         {
            case AMT_C_U8_CCAMSG_OPCODE_STATUS:
            {
               DIA_TR_INF("dia_SAFeatureErrorLog::vHandleActiveDTCListResult => AMT_C_U8_CCAMSG_OPCODE_STATUS");
               handleActiveDTCStatus(poMessage, pListener);
               break;
            }// case AMT_C_U8_CCAMSG_OPCODE_STATUS:


            case AMT_C_U8_CCAMSG_OPCODE_ERROR:
            {
               DIA_TR_ERR("--- In CASE: AMT_C_U8_CCAMSG_OPCODE_ERROR");
               mpSrvPlugin->vOnErrorMessage(poMessage);
               break;
            }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

            default:
            {
               DIA_TR_ERR( "dia_SAFeatureErrorLog::vHandleActiveDTCListResult: Unexpected OpCode = %d", u8OpCode);
               break;
            }// default:
         }// switch ( u8OpCode )
      }
   } //if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}
//-----------------------------------------------------------------------------
void
dia_SAFeatureErrorLog::vHandleSnapShotIdResult ( amt_tclServiceData* poMessage ) const
{

   dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleSnapShotIdResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
         DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

         // Extract the op code of the message called
         tU8 u8OpCode = poMessage->u8GetOpCode();

         // Switch on the opCode received
         switch (u8OpCode)
         {
            case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
            {
               DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotIdResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
               fi_tclVisitorMessage oResultMsg(poMessage);
               midw_diaglogfi_tclMsgreportDTCSnapshotIdentificationMethodResult oCompResp;
               tS32 s32Result = oResultMsg.s32GetData(oCompResp);
               if ( s32Result == OSAL_OK )
               {
                  if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                  {
                     DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotIdResult: FI Result OK, decoding");


                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        size_t listSize = oCompResp.SnapshotIdentificationList.SnapshotIdentification.size();
                        std::list<IErrorLogSnapshotIdResult> snapshotIdList;

                        for (size_t i = 0; i < listSize; i++)
                        {
                           IErrorLogSnapshotIdResult snapshot;

                           snapshot.dtc   = oCompResp.SnapshotIdentificationList.SnapshotIdentification[i].DTC;
                           snapshot.recNo = oCompResp.SnapshotIdentificationList.SnapshotIdentification[i].RecordNumber;
                           DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotIdResult: receive DTC:%06x RecNo:%02x",snapshot.dtc,snapshot.recNo);

                           snapshotIdList.push_back(snapshot);
                        }
                        DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotIdResult: Calling listener with snapshot IDs");
                        pListener->vOnDiaglogGetSnapShotId(snapshotIdList);
                     }// if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     else
                     {
                        DIA_TR_ERR(("!!! dia_SAFeatureErrorLog::vHandleSnapShotIdResult => ERROR: not FI_EN_ACCEPTED"));
                        dia_ErrorInfoInterface err(dia_IErrorLogListener::getUID());
                        err.setErrorCode(DIA_E_CONDITIONS_NOT_CORRECT);
                        pListener->vOnError(err);
                     }
                  }
                  else
                  {
                     DIA_TR_ERR(("!!! dia_SAFeatureErrorLog::vHandleSnapShotIdResult => ERROR: Negative Response"));

                     dia_ErrorInfoInterface err(dia_IErrorLogListener::getUID());
                     err.setErrorCode(DIA_E_CONDITIONS_NOT_CORRECT);
                     pListener->vOnError(err);
                  }
               }
               else
               {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotIdResult => oResultMsg.s32GetData FAILED");
               }
               break;
            }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotIdResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                  //STC2HI: Error handling TODO

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleSnapShotIdResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------
void
dia_SAFeatureErrorLog::vHandleSnapShotDataResult ( amt_tclServiceData* poMessage ) const
{
      dia_tclFnctTrace oTrace("dia_SAFeatureErrorLog::vHandleSnapShotDataResult");

   if(poMessage != NULL)
   {
      dia_IErrorLogListener* pListener = OSAL_NULL;
      if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         // Msg must come from a server
         DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

         // Extract the op code of the message called
         tU8 u8OpCode = poMessage->u8GetOpCode();

         // Switch on the opCode received
         switch (u8OpCode)
         {
            case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
            {
               DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotDataResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
               fi_tclVisitorMessage oResultMsg(poMessage);
               midw_diaglogfi_tclMsgreportDTCSnapshotRecordByDTCNumberMethodResult oCompResp;
               tS32 s32Result = oResultMsg.s32GetData(oCompResp);
               if ( s32Result == OSAL_OK )
               {
                  if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                  {
                     DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotDataResult: FI Result OK, decoding");


                     if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     {
                        size_t listSize = oCompResp.SnapshotDataList.SnapshotData.size();
                        std::list<IErrorLogSnapshotDataResult> snapshotDataList;
                        tU32 DTC    = oCompResp.DTC;
                        tU8  status = oCompResp.StatusOfDtc;

                        for ( size_t i = 0; i < listSize; i++ )
                        {
                           IErrorLogSnapshotDataResult snapshot;

                           snapshot.recNo    = oCompResp.SnapshotDataList.SnapshotData[i].RecordNumber;
                           snapshot.noOfData = oCompResp.SnapshotDataList.SnapshotData[i].NumberOfData;
                           snapshot.data     = oCompResp.SnapshotDataList.SnapshotData[i].Data;

                           tU8 size = (tU8) oCompResp.SnapshotDataList.SnapshotData[i].Data.size();
                           tU8 aData[size];
                           ::memset(aData,0,size);
                           for ( tU8 j=0; j<size;j++ )
                           {
                              aData[j] = snapshot.data[j];
                           }
                           DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotDataResult: receive DTC:%06x RecNo:%02x NrOfData:%02x",DTC,snapshot.recNo,snapshot.noOfData);
                           //ETG_TRACE_COMP_CLS((TR_CLASS_DIAGNOSTICS_SM, "DTC:%06x, Status:%02x RecNo:%02x Data:%*x", DTC,status, snapshot.recNo,ETG_LIST((tU16) snapshot.noOfData, ETG_T8, aData)));

                           snapshotDataList.push_back(snapshot);
                        }
                        DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotDataResult: Calling listener with snapshot IDs");
                        pListener->vOnDiaglogGetSnapShotDataByDTC(DTC,status, snapshotDataList);
                     }// if (oCompResp.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
                     else
                     {
                        DIA_TR_ERR(("!!! dia_SAFeatureErrorLog::vHandleSnapShotDataResult => ERROR: not FI_EN_ACCEPTED"));
                        dia_ErrorInfoInterface err(dia_IErrorLogListener::getUID());
                        err.setErrorCode(DIA_E_CONDITIONS_NOT_CORRECT);
                        pListener->vOnError(err);
                     }
                  }
                  else
                  {
                     DIA_TR_ERR(("!!! dia_SAFeatureErrorLog::vHandleSnapShotDataResult => ERROR: Negative Response"));

                     dia_ErrorInfoInterface err(dia_IErrorLogListener::getUID());
                     err.setErrorCode(DIA_E_CONDITIONS_NOT_CORRECT);
                     pListener->vOnError(err);
                  }
               }
               else
               {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotDataResult => oResultMsg.s32GetData FAILED");
               }
               break;
            }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:

              case AMT_C_U8_CCAMSG_OPCODE_ERROR:
              {
                  DIA_TR_INF("dia_SAFeatureErrorLog::vHandleSnapShotDataResult => AMT_C_U8_CCAMSG_OPCODE_ERROR");

                  //STC2HI: Error handling TODO

                  break;
              }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

              default:
              {
                  DIA_TR_ERR("dia_SAFeatureErrorLog::vHandleSnapShotDataResult => ERROR: unsupported u8OpCode %x",u8OpCode);
                  // unkown opcode: enter error handling here ( trace or assert )
                  break;
              }
          }// switch (u8OpCode)
       } //if (pListener != NULL)
   }// if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void dia_SAFeatureErrorLog::vHandleGetItcInformationResult ( amt_tclServiceData* poMessage ) const
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::vHandleGetItcInformationResult");

   if(poMessage != NULL)
   {
     dia_IErrorLogListener* pListener = OSAL_NULL;
     if ((querySysAdapterListener<dia_IErrorLogListener>(&pListener) == DIA_SUCCESS) && pListener)
     {
      // Msg must come from a server
        DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

        tU8 u8OpCode = poMessage->u8GetOpCode();

        switch ( u8OpCode )
        {
          case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
          {
            DIA_TR_INF("dia_SAFeatureErrorLog::vHandleGetItcInformationResult => AMT_C_U8_CCAMSG_OPCODE_METHODRESULT");
            // Get method result data into fi type
            fi_tclVisitorMessage oMsg(poMessage);
   #ifdef MIDW_DIAGLOGFI_C_U16_GETITCINFORMATION_AVAILABLE //To be prepared when it is available
            midw_diaglogfi_tclMsggetItcInformationMethodResult oResultData;
            (tVoid) oMsg.s32GetData(oResultData);

            // only handle if status = true
            if (oResultData.Status.enType == midw_fi_tcl_e8_MethodResultReply::FI_EN_ACCEPTED)
            {
               DIA_TR_INF("dia_SAFeatureErrorLog::vHandleGetItcInformationResult => oResultData.status = true");
               DIA_TR_INF("dia_SAFeatureErrorLog::vHandleGetItcInformationResult ItcInformationList.size() = %zu", oResultData.ItcInformationList.size());

               std::vector<dia_tITCInfo> oITCInfoList;

               for(::size_t i = 0; i < oResultData.ItcInformationList.size(); ++i)
               {
                 dia_tITCInfo oITCInfo;

                 oITCInfo.m_u16ITC          = oResultData.ItcInformationList[i].ITC;
                 oITCInfo.m_u32DTC          = oResultData.ItcInformationList[i].DTC;
                 oITCInfo.m_u8Status        = oResultData.ItcInformationList[i].StatusOfItc;
                 oITCInfo.m_u32AgeingCounter = oResultData.ItcInformationList[i].AgeingCounter;
                 oITCInfo.m_u32FaultCounter  = oResultData.ItcInformationList[i].OccurenceCounter;

                 DIA_TR_INF("--- ITC:%04x DTC:%06x Status:%02x AgeCtr:%d OccCtr:%d",oITCInfo.m_u16ITC ,oITCInfo.m_u32DTC, oITCInfo.m_u8Status, oITCInfo.m_u32AgeingCounter, oITCInfo.m_u32FaultCounter);

                 oITCInfoList.push_back(oITCInfo);
               }

               pListener->vOnITCInfo(oITCInfoList);
            }
   #endif
            break;
          }// case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:


          case AMT_C_U8_CCAMSG_OPCODE_ERROR:
          {
            DIA_TR_ERR("--- In CASE: AMT_C_U8_CCAMSG_OPCODE_ERROR");
            mpSrvPlugin->vOnErrorMessage(poMessage);
            break;
          }// case AMT_C_U8_CCAMSG_OPCODE_ERROR:

          default:
          {
            DIA_TR_ERR( "dia_SAFeatureErrorLog::vHandleGetItcInformationResult: Unexpected OpCode = %d", u8OpCode);
            break;
          }// default:
        }// switch ( u8OpCode )
     }
   } //if(poMessage != NULL)

   if(poMessage != NULL)
   {
      poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::registerForActiveDtc ( )
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::registerForActiveDtc");

   tDiaResult retCode = DIA_SUCCESS;

   if ( (registerFunction(MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST) != DIA_SUCCESS))
   {
      DIA_TR_ERR("##### REGISTRATION TO DIAGLOG PROPERTY MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST FAILED #####");
      retCode = DIA_FAILED;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureErrorLog::unregisterForActiveDtc ( )
{
   dia_tclFnctTrace trc("dia_SAFeatureErrorLog::unregisterForActiveDtc");

   tDiaResult retCode = DIA_SUCCESS;

   if ( (unregisterFunction(MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST) != DIA_SUCCESS))
   {
      DIA_TR_ERR("##### UNREGISTRATION TO DIAGLOG PROPERTY MIDW_DIAGLOGFI_C_U16_SENDACTIVEDTCLIST FAILED #####");
      retCode = DIA_FAILED;
   }

   return retCode;
}
