/*!
 * \file       dia_SecurityManager.h
 *
 * \brief      Entry point to the security framework. Responsible for maintaining system security
 *             levels.
 *
 * \details    Class dia_SecurityManager is the central entry point to the security framework
 *             of the diagnosis core. maintaining system security levels.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSecurity
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#ifndef __INCLUDED_DIA_SECURITY_MANAGER__
#define __INCLUDED_DIA_SECURITY_MANAGER__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_SECURITY__
#include <common/framework/security/dia_security.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SESSION_LISTENER__
#include <common/interfaces/dia_ISessionListener.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SECURITY_LISTENER__
#include <common/interfaces/dia_ISecurityListener.h>
#endif

class dia_SecurityLevel;
class dia_SecurityTimer;

class dia_SecurityManager
   : public dia_ISessionListenerImpl
{
   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_SecurityManager);

public:
   //! install a signal handler
   virtual tBool addSecurityLevel ( dia_SecurityLevel* pSecLevel );
   //! install a signal handler
   virtual tBool removeSecurityLevel ( dia_UID uid );

   //! fill the given list with pointers to all active security levels
   virtual void getActiveLevels ( std::list<dia_SecurityLevel*>& activeLevels );
   //! return the security level object for the given UID
   virtual dia_SecurityLevel* pGetLevel ( dia_UID uid );
   //! return the signal object with the given UDS identifier
   virtual dia_SecurityLevel* pGetLevelUDS ( tU8 udsID );
   //! return status of the given security level
   virtual dia_eSecurityLevelStatus getStatus ( dia_UID uid );

   //! return the access types for this security level
   virtual tDiaResult getAccessTypes ( std::vector<tU16>& accessTypes );

   //! repository used to map access types to security level objects
   const std::map<tU16,dia_SecurityLevel*>& getAccessTypeMapping ( void ) const { return  mAccessTypeRep; }

   virtual tDiaResult lock ( void );
   virtual tDiaResult unlock ( void );

   //
   // Methods defined in SessionListener interface
   //

   //! called by the session control object to notify about a session change
   virtual void vOnSessionChanged ( tU8 newSession, tU8 oldSession );

   //! register a security level listener
   virtual tBool addListener ( dia_ISecurityListener* pListener );
   //! unregister a session listener
   virtual tBool removeListener ( dia_ISecurityListener* pListener );

   virtual tDiaResult initialize ( void );
   virtual tDiaResult finalize ( void ) { return DIA_SUCCESS; }

   virtual void onSecurityLevelUpdate ( dia_SecurityLevel& secLevel );

protected:
   //! default class constructor
   dia_SecurityManager ( void );
   //! class destructor
   virtual ~dia_SecurityManager ( void );

protected:
   //! repository of signal controlled via remote control
   std::map<dia_UID,dia_SecurityLevel*> mLevelRep;
   //! repository of signal controlled via remote control
   std::list<dia_UID> mActiveLevelRep;
   //! repository used to map access types to security level objects
   std::map<tU16,dia_SecurityLevel*> mAccessTypeRep;
   //! repository of security listeners
   std::list<dia_ISecurityListener*> mListenerRep;
};

#endif /* __INCLUDED_DIA_SECURITY_MANAGER__ */
