/*
 * dia_SecurityLevelBosch.cpp
 *
 *  Created on: 10.08.2012
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_SECURITY_LEVEL_BOSCH__
#include <common/framework/security/dia_SecurityLevelBosch.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include <common/framework/config/dia_ConfigManager.h>
#endif

#ifndef __INCLUDED_DIA_RANDOM_GENERATOR__
#include "common/framework/utils/dia_RandomGenerator.h"
#endif

//-----------------------------------------------------------------------------

dia_SecurityLevelBosch::dia_SecurityLevelBosch ( dia_SecurityLevelConfiguration& config )
   : dia_SecurityLevel(DIA_NAME_SECURITY_LEVEL_BOSCH,config),
     mKeyID(0x0000),
     mIsPassedKeyValid(false)
{}

//-----------------------------------------------------------------------------

dia_SecurityLevelBosch::~dia_SecurityLevelBosch ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelBosch::getSeed ( void /*std::vector<tU8>& seedValue*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::getSeed");

   mErrorCode = DIA_E_RESPONSE_PENDING;

   if ( mpFSM )
   {
      mpFSM->acceptEvent(dia_SecurityLevelFSM::evSeedRequested, 0 /*(Void*) &seedValue*/);
      DIA_TR_INF(" State of dia_SecurityLevelBosch-FSM: %s", mpFSM->getStateName());
   }

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelBosch::acceptKey ( std::vector<tU8>& keyValue )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::acceptKey");

   mErrorCode = DIA_E_RESPONSE_PENDING;
   mIsPassedKeyValid = false;

   if ( mpFSM )
   {
      mpFSM->acceptEvent(dia_SecurityLevelFSM::evKeyReceived,(tVoid*) &keyValue);
      DIA_TR_INF(" State of dia_SecurityLevelBosch-FSM: %s", mpFSM->getStateName());
   }

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelBosch::loadConfiguration ( void )
{
   return dia_SecurityLevel::loadConfiguration(DIA_PROP_CM_SECLEV_BOSCH_INVALID_KEY_COUNTER, DIA_PROP_CM_SECLEV_BOSCH_REMAINING_LOCK_TIME);

}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelBosch::storeConfiguration ( void )
{
   return dia_SecurityLevel::storeConfiguration(DIA_PROP_CM_SECLEV_BOSCH_INVALID_KEY_COUNTER,DIA_PROP_CM_SECLEV_BOSCH_REMAINING_LOCK_TIME);
}


//-----------------------------------------------------------------------------

void
dia_SecurityLevelBosch::generateSeed ( void )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::generateSeed");

   // initialize the seed attribute
   initializeSeed();

   // generate random values
#ifndef __DIA_UNIT_TESTING__
   dia_RandomGenerator* randgen = getInstanceOfRandomGenerator();
   tS32 random1 = randgen->getRangedRandomNumber<tS32>();
   tS32 random2 = randgen->getRangedRandomNumber<tS32>();
#else
   tS32 random1 = 0x01020304;
   tS32 random2 = 0x05060708;
#endif

   // calculate seed
   for ( tU8 j=0; j<cu8SeedLength; j++ )
   {
      if ( j < (cu8SeedLength/2) )
      {
         mSeed.push_back( (tU8) (((tU32) random1 & (cu32ByteMask >> (cu8ByteSize*((4-j)-1)))) >> (cu8ByteSize*j)) );
      }
      else
      {
         mSeed.push_back( (tU8) ( ( (tU32) random2 & ( cu32ByteMask >> (cu8ByteSize*((8-j)-1)) ) ) >> (cu8ByteSize*(j-4)) ) );
      }
   }

   mIsSeedValid = true;

   tU16 u16Code = (tU16) ((((tU16) mSeed[3]) << cu8ByteSize) + mSeed[7]);

   DIA_TR_INF("[XXXX] : 0x%08x", randomizeSeed(u16Code));
}

//-----------------------------------------------------------------------------

tU32
dia_SecurityLevelBosch::randomizeSeed ( tU16 data ) const
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::randomizeSeed");

   tU16 outCode = scrambleGeneratedSeed(data);

   dia_RandomGenerator* randgen = getInstanceOfRandomGenerator();
   tU16 randomValue = randgen->getRangedRandomNumber<tU16>();
   outCode ^= randomValue;

   tU32 retCode = ((tU32) outCode << 16) + (tU32) randomValue;

   return (retCode);
}

//-----------------------------------------------------------------------------

tU16
dia_SecurityLevelBosch::scrambleGeneratedSeed ( tU16 data ) const
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::scrambleGeneratedSeed");

   tU32 result;
   tU16 retVal;

   result  = (tU32) data * ( ( ~ (tU32) data ) - 0x1892 );
   retVal  = (tU16) (result & 0xAFFE);
   result  = (tU32) retVal * ( ~ (tU32) data );
   retVal  = (tU16) (result ^ 0x9A53);
   retVal = tU16(retVal + (retVal % 17));

   return retVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelBosch::checkKey ( std::vector<tU8>& key )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::checkKey");

   tDiaResult retCode = DIA_FAILED;

   if ( mIsSeedValid )
   {
      tU8 keyLength = (tU8) ( (key.size() > 255) ? 255 : key.size() );
      DIA_TR_INF("KeyLength : %d", keyLength);

      if ( keyLength != (cu8SeedLength/2) )
      {
         DIA_TR_INF("WRONG KEY LENGTH !!!");
         return DIA_E_INVALID_KEY;
      }

      for ( tU8 j=0; j<cu8SeedLength/2; j++ )
      {
         DIA_TR_INF("KEY[%02d] : 0x%02x", j,key[j]);
      }

      tU16 area;
      area  = (tU16)( (((tU16) key[0]) << cu8ByteSize ) + key[1] ); //lint !e864
      area = tU16(area ^ (tU16)( (((tU16) key[2]) << cu8ByteSize ) + key[3])); //lint !e864
      DIA_TR_INF("Area : 0x%04x", area);

      tU16 code;
      code  = tU16((mSeed[3]) << cu8ByteSize);
      code = tU16(code + mSeed[7]);
      DIA_TR_INF("Code : 0x%04x", code);
      DIA_TR_INF("Code : 0x%08x", scrambleGeneratedSeed(code));
      if ( area == scrambleGeneratedSeed(code) )
      {
         retCode = DIA_SUCCESS;
         DIA_TR_INF("########## SUCCESS ##########");
      }
   }
   else
   {
      // Trace out for debugging
      DIA_TR_INF("-- Seed Validity: %d", mIsSeedValid);
      DIA_TR_INF("-- Security Level State: %s", mpFSM->getStateName());
   }

   return retCode;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelBosch::vFsmCopyKeyCtrlFromNvm ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::vFsmCopyKeyCtrlFromNvm");

   // load the configuration (remaining attempts and locking time
   loadConfiguration();
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelBosch::vFsmSaveKeyCtrlToNvm ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::vFsmSetErrCode_RequestSequence");
   // empty default implementation
}

//------------------------------------------------------------------------------


void
dia_SecurityLevelBosch::vFsmCalculateSeed ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::vFsmCalculateSeed");
   generateSeed();

   if ( mpFSM ) mpFSM->acceptEvent(dia_SecurityLevelFSM::evOnSeedAvailable,0);
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelBosch::vFsmSendSeed ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::vFsmSendSeed");

   std::vector<tU8> seedValue; // = (std::vector<tU8>*) pArg;

   // CM security protocol (constant)
   seedValue.push_back(0x00);
   seedValue.push_back(0x02);

   // keyID for MAUS Protocol
   seedValue.push_back(0x00);
   seedValue.push_back(0x00);

   std::vector<tU8>::iterator iter = mSeed.begin();
   for ( ; iter != mSeed.end(); iter++ )
   {
      seedValue.push_back(*iter);
   }

   notifySeedResult(DIA_SUCCESS,seedValue);
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelBosch::vFsmSendSeedWhenActive ( void* )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::vFsmSendSeedWhenActive");

   std::vector<tU8> seedValue;

   // CM security protocol (constant)
   seedValue.push_back(0x00);
   seedValue.push_back(0x02);

   // keyID for MAUS Protocol
   seedValue.push_back(0x00);
   seedValue.push_back(0x00);

   for ( tU16 i=0; i<8; i++ )
   {
      seedValue.push_back(0x00);
   }

   notifySeedResult(DIA_SUCCESS,seedValue);
}

//------------------------------------------------------------------------------

bool
dia_SecurityLevelBosch::isKeyInvalid ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::isKeyInvalid");
   return !mIsPassedKeyValid;
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelBosch::vFsmValidateKey ( void* pArg )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::vFsmValidateKey");

   mIsPassedKeyValid = false;

   if ( !pArg )
   {
      DIA_TR_INF("##### vFsmValidateKey: NO KEY AVAILABLE. ABORTING #####");
      return;
   }

   std::vector<tU8>* pKey = (std::vector<tU8>*) pArg;

   if ( pKey->size() != 8 )
   {
      DIA_TR_INF("##### vFsmValidateKey: INVALID KEY LENGTH. ABORTING #####");
      return;
   }

   std::vector<tU8>  mausKey;
   for ( tU8 j=0; j<4; j++) mausKey.push_back(pKey->at(j));

   if ( checkKey(mausKey) == DIA_SUCCESS )
   {
      mIsPassedKeyValid = true;
   }

   if ( mpFSM ) mpFSM->acceptEvent(dia_SecurityLevelFSM::evOnKeyValidationDone,0);
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelBosch::loadLockStatus ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityManager::loadSecurityLevel()");

#ifndef __DIA_UNIT_TESTING__
   dia_UID data = 0;
   dia_UID dataInverted = 0;

   if ( dia_getProperty(DIA_PROP_SYSTEM_SECURITY_LEVEL_UID, data) == DIA_SUCCESS )
   {
      if ( dia_getProperty(DIA_PROP_SYSTEM_SECURITY_LEVEL_UID_INVERT, dataInverted) == DIA_SUCCESS )
      {
         DIA_TR_INF("Reading level : %x", data);
         DIA_TR_INF("Reading invert: %x", dataInverted);
         if ( dataInverted == (~data) )
         {
            if ( data == this->getUID() )
            {
               DIA_TR_INF("-- Security Level: %d", data);
               if ( (data == DIA_EN_SECURITY_LEVEL_UNKNOWN) || (data == DIA_EN_SECURITY_LEVEL_NOT_ACTIVE) )
               {
                  mStatus = DIA_EN_SECURITY_LEVEL_STATUS_NOT_ACTIVE;
               }
               else
               {
                  mStatus = DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE;
               }
            }
         }
      }
   }
#endif
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelBosch::saveLockStatus ( dia_UID uid ) const
{
   tDiaResult retCode = DIA_FAILED;

#ifndef __DIA_UNIT_TESTING__
   bool needToStore = true;

   dia_UID data = uid;
   dia_UID dataStored = 0;
   dia_UID dataStoredInverted = 0;

   if ( dia_getProperty(DIA_PROP_SYSTEM_SECURITY_LEVEL_UID, dataStored) == DIA_SUCCESS )
   {
      if ( dia_getProperty(DIA_PROP_SYSTEM_SECURITY_LEVEL_UID_INVERT, dataStoredInverted) == DIA_SUCCESS )
      {
         if ( ((dia_UID) dataStoredInverted) == ((dia_UID) ~dataStored) )
         {
            if ( data == dataStored )
            {
               needToStore = false;
               DIA_TR_ERR("##### NO NEED TO STORE SAME SECURITY LEVEL #####");
            }
         }
      }
   }

   if ( needToStore )
   {
      if ( dia_setProperty(DIA_PROP_SYSTEM_SECURITY_LEVEL_UID, data) == DIA_SUCCESS )
      {
         dia_UID dataInverted = ~data;
         if ( dia_setProperty(DIA_PROP_SYSTEM_SECURITY_LEVEL_UID_INVERT, dataInverted) == DIA_SUCCESS )
         {
            DIA_TR_INF("Writing level : %x", data);
            DIA_TR_INF("Writing invert: %x", dataInverted);
            retCode = DIA_SUCCESS;
         }
      }
   }
#else
   DIA_PARAMETER_INTENTIONALLY_UNUSED(uid);
#endif
   return retCode;
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelBosch::vFsmEvaluateOperationMode ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelBosch::vFsmEvaluateOperationMode");

   mIsDisabled = true;

   // if security is disabled via the HW config string we always return FALSE (master)
   if ( dia_checkPropertyValue(DIA_PROP_CM_CONFIG_INTERFACE_LOCK) == TRUE ) mIsDisabled = false;

   if ( mIsDisabled )
   {
      DIA_TR_INF("#### SECURITY LEVEL MAUSKEY: SECURITY MANAGEMENT DISABLED VIA CONFIGURATION ####");
   }

   if ( mpFSM )
   {
	   DIA_TR_INF("#### POSTING FSM EVENT (dia_SecurityLevelFSM::evOperationModeUpdate) ####");
	   mpFSM->acceptEvent(dia_SecurityLevelFSM::evOperationModeUpdate, 0 );
   }
   else
   {
	   DIA_TR_INF("#### SECURITY LEVEL MAUSKEY: UNABLE TO POST FSM EVENT (NO FSM OBJECT) ####");
   }
}

