/*!
 * \file       dia_SecurityLevelALD.h
 *
 * \brief      Security level that interacts with ALD
 *
 * \details    Class dia_SecurityLevelALD provides a seed/key based method using
 *             the ALD (Authorization Level Daemon) protocol. A seed request is called
 *             at the ALD and is returned to the tester. The tester forwards the seed
 *             to the smart card server for encryption and retrieves the corresponding
 *             key. The key is send to the ECU. The ECU encrypts the stored seed and
 *             compares the result with the received key. If the values are matching
 *             access for the security level is granted to the tester.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSecurity
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#ifndef __INCLUDED_DIA_SECURITY_LEVEL_ALD__
#define __INCLUDED_DIA_SECURITY_LEVEL_ALD__

#ifndef __INCLUDED_DIA_SECURITY_LEVEL__
#include <common/framework/security/dia_SecurityLevel.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_AUTHORIZATIONLEVEL_LISTENER__
#include <common/interfaces/dia_IAuthorizationLevelListener.h>
#endif

class dia_SecurityLevelALD
   : public dia_SecurityLevel,
     public dia_IAuthorizationLevelListener
{
   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_SecurityLevelALD);

public:
   //! class constructor
   dia_SecurityLevelALD ( dia_SecurityLevelConfiguration& config );

   //! class destructor
   virtual ~dia_SecurityLevelALD ( void );

   //! method used to retrieve the seed value
   virtual tDiaResult getSeed ( void /*std::vector<tU8>& seedValue*/ );
   //! method used to deliver a key
   virtual tDiaResult acceptKey ( std::vector<tU8>& keyValue );

   //! response on getSeed process
   virtual void vOnSeed ( const std::vector<tU8>& seedData );
   //! reply of sending key
   virtual void vOnKeyValidationResult ( tDiaResult retCode );
   //! result for the lockDevice request
   virtual void vOnLockDeviceResult ( tDiaResult /*result*/ ) { /* nothing to be done */ }
   //! currently active security level
   virtual void vOnLevel ( tU16 levelID );
   //! information about a newly requested security level (will be sent before the scripts are executed
   virtual void vOnLevelRequested ( tU16 newLevelID );
   //! notification that the level was changed to the newly requested level
   virtual void vOnLevelChanged ( tU16 newLevelID );

protected:
   //! default class constructor without implementation
   dia_SecurityLevelALD ( void );

   //! load the lock status of the security level
   virtual void loadLockStatus ( void );
   virtual tDiaResult saveLockStatus ( dia_UID uid ) const;

   //
   // FSM methods
   //
   virtual void vFsmEvaluateOperationMode ( void* /*pArg*/ );
   virtual void vFsmHandleActivation ( void* /*pArg*/ );
   virtual void vFsmSendSeed ( void* pArg );
   virtual bool isKeyInvalid ( void* pArg );
   virtual void vFsmHandleFailedActivation ( void* pArg );

   virtual tDiaResult getInitializationVector ( std::vector<tU8>& initVector );

protected:
   //! currently activated security level
   tU8  mActiveLevel;
   //! key identifier
   tU16 mKeyID;
   //! flag used to indicate if the passed key matches or not
   bool mIsPassedKeyValid;

   // some constants
   static const tU8  cu8SeedLength = 8;
   static const tU8  cu8ByteSize   = 8;
   static const tU32 cu32ByteMask  = 0xFFFFFFFF;
};

#endif /* __INCLUDED_DIA_SECURITY_LEVEL_ALD__ */
