/*!
 * \file       dia_SecurityLevel.cpp
 *
 * \brief      tbd
 *
 * \details    tbd
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSecurity
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_SECURITY_LEVEL__
#include "dia_SecurityLevel.h"
#endif

#ifndef __INCLUDED_DIA_SECURITY_MODULE__
#include "dia_SecurityModule.h"
#endif

#ifndef __INCLUDED_DIA_SECURITY_TIMER__
#include "dia_SecurityTimer.h"
#endif

#ifndef __INCLUDED_DIA_SECURITY_MANAGER__
#include "dia_SecurityManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_SRVHANDLER_GENERIC_SECURITY_ACCESS__
#include <common/services/uds/generic/dia_SrvHandlerGenericSecurityAccess.h>
#endif

#ifndef __INCLUDED_DIA_HASH_CALCULATOR__
#include <common/framework/utils/dia_HashCalculator.h>
#endif

#ifndef __INCLUDED_DIA_RANDOM_GENERATOR__
#include <common/framework/utils/dia_RandomGenerator.h>
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include "common/framework/application/dia_LockScope.h"
#endif


#define DIA_C_U16_SECURITY_TIMER_TIMEOUT_INFINITE ((tU16) 0xFFFF)

#define DIA_C_SECURITY_LEVEL_LOCK_NAME_EXTENSION  "_LK"

//-----------------------------------------------------------------------------

dia_SecurityLevel::dia_SecurityLevel ( tCString name, dia_SecurityLevelConfiguration& config )
   : dia_ObjectWithUID(name),
     mConfig(config),
     mStatus(DIA_EN_SECURITY_LEVEL_STATUS_NOT_ACTIVE),
     mIsDisabled(false),
     mWasSeedRequested(false),
     mIsSeedValid(false),
     mIsLevelUnderSessionControl(false),
     mIsSimulationEnabled(config.getSimulationMode()),
     mNeedToRunSeedKeyPolicy(false),
     mLockState(false),
     mRemainingAttempts(config.getMaxNumberOfLogins()),
     mRemainingLockTime(config.getLoginLockTime()),
     mErrorCode(DIA_E_NOERROR),
     mInvalidKeyCounter(0),
     mpFSM(0),
     mSyncObj(std::string(std::string(name) + DIA_C_SECURITY_LEVEL_LOCK_NAME_EXTENSION).c_str())
{}

//-----------------------------------------------------------------------------

dia_SecurityLevel::~dia_SecurityLevel ( void )
{
   _BP_TRY_BEGIN
   {
      OSAL_DELETE mpFSM;
      mpFSM = 0;
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SecurityLevel::~dia_SecurityLevel !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevel::initialize ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::initialize");

   dia_LockScope mScopeLock(mSyncObj);

   if ( mpFSM )
   {
      // we initialize the state machine exactly once after creation
      return;
   }

   // create the state machine object
   if ( !(dia_SecurityLevelFSM::Fsm::createFSM(&mpFSM,this)) )
   {
      DIA_TR_INF( "### STATE MACHINE INITIALIZATION FAILED ###");
      return;
   }

   mErrorCode = DIA_E_NOERROR;
}

//------------------------------------------------------------------------------

dia_eSecurityLevelStatus
dia_SecurityLevel::getStatus ( void ) const
{
   return mStatus;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevel::acceptEvent ( dia_SecurityLevelFSM::FsmEvent event, void* pArg, tDiaResult initialErrorCode )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::acceptEvent(dia_SecurityLevelFSM::FsmEvent,void*)");

   dia_LockScope mScopeLock(mSyncObj);

   if ( !mpFSM )  return DIA_E_FSM_NOT_AVAILABLE;

   mErrorCode = initialErrorCode;

   DIA_TR_INF("dia_SecurityLevel::acceptEvent - State (before): %s (%p), Event = %s", mpFSM->getStateName(),mpFSM,dia_SecurityLevelFSM::getEventName(event));
   mpFSM->acceptEvent(event,pArg);
   DIA_TR_INF("dia_SecurityLevel::acceptEvent - State (after): %s", mpFSM->getStateName());

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevel::acceptEventWithoutErrorInitialization ( dia_SecurityLevelFSM::FsmEvent event, void* pArg )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::acceptEventWithoutErrorInitialization(dia_SecurityLevelFSM::FsmEvent,void*)");

   dia_LockScope mScopeLock(mSyncObj);

   if ( !mpFSM )  return DIA_E_FSM_NOT_AVAILABLE;

   DIA_TR_INF("dia_SecurityLevel::acceptEventWithoutErrorInitialization - State (before): %s (%p), Event = %s", mpFSM->getStateName(),mpFSM,dia_SecurityLevelFSM::getEventName(event));
   mpFSM->acceptEvent(event,pArg);
   DIA_TR_INF("dia_SecurityLevel::acceptEventWithoutErrorInitialization - State (after): %s", mpFSM->getStateName());

   return mErrorCode;
}

//------------------------------------------------------------------------------

tBool
dia_SecurityLevel::vOnSecurityTimerTick ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::vOnSecurityTimerTick()");

   tBool bRetCode = FALSE;

   if ( (mRemainingLockTime > 0) && (mRemainingLockTime != DIA_C_U16_SECURITY_TIMER_TIMEOUT_INFINITE) )
   {
      mRemainingLockTime--;
      if ( mRemainingLockTime == 0 )
      {
         // locking timer is expired
         (void) acceptEvent(dia_SecurityLevelFSM::evLockTimeExpired,0);
         bRetCode = TRUE;
      }
   }

   return bRetCode;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::onSessionChange ( tU8 /*newSession*/, tU8 /*oldSession*/ )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::onSessionChange()");
   (void) acceptEventWithoutErrorInitialization(dia_SecurityLevelFSM::evSessionChanged,0);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevel::loadConfiguration ( tU32 propID_InvalidKeyCounter, tU32 propID_RemainingLockTime )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::loadConfiguration(invalidKeyCounter,remainingLockTime");

   tDiaResult retCode = DIA_FAILED;

   if ( dia_getProperty(propID_InvalidKeyCounter,mInvalidKeyCounter) != DIA_SUCCESS )
   {
      mInvalidKeyCounter = 0;
   }

   if ( dia_getProperty(propID_RemainingLockTime,mRemainingLockTime) != DIA_SUCCESS )
   {
      mRemainingLockTime = 0;
   }

   mRemainingAttempts = mConfig.getMaxNumberOfLogins() - mInvalidKeyCounter;

   DIA_TR_INF("INVALID KEY COUNTER: %d",mInvalidKeyCounter);
   DIA_TR_INF("REMAINING LOCK TIME: %d",mRemainingLockTime);
   DIA_TR_INF("REMAINING ATTEMPTS : %d",mRemainingAttempts);

   return retCode;
}

//------------------------------------------------------------------------------

#if 0 //gib2hi
tDiaResult
dia_SecurityLevel::storeConfiguration ( tU32 propID_InvalidKeyCounter, tU32 propID_RemainingLockTime )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::storeConfiguration(invalidKeyCounter,remainingLockTime");

   tDiaResult retCode = DIA_FAILED;

   if ( dia_setProperty(propID_InvalidKeyCounter,mInvalidKeyCounter) == DIA_SUCCESS )
   {
      if ( dia_getProperty(propID_RemainingLockTime,mRemainingLockTime) == DIA_SUCCESS )
      {
         retCode = DIA_SUCCESS;
      }
   }

   DIA_TR_INF("INVALID KEY COUNTER: %d",mInvalidKeyCounter);
   DIA_TR_INF("REMAINING LOCK TIME: %d",mRemainingLockTime);
   DIA_TR_INF("REMAINING ATTEMPTS : %d",mRemainingAttempts);

   return retCode;
}
#else
tDiaResult
dia_SecurityLevel::storeConfiguration ( tU32 /*propID_InvalidKeyCounter*/, tU32 /*propID_RemainingLockTime*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::storeConfiguration(invalidKeyCounter,remainingLockTime");

   DIA_TR_INF("##### PERSISTENT STORING NOT YET IMPLEMENTED #####");

   DIA_TR_INF("INVALID KEY COUNTER: %d",mInvalidKeyCounter);
   DIA_TR_INF("REMAINING LOCK TIME: %d",mRemainingLockTime);
   DIA_TR_INF("REMAINING ATTEMPTS : %d",mRemainingAttempts);

   return DIA_SUCCESS;
}
#endif

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmInitializeSecurityLevel ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmInitializeSecurityLevel");
   mIsDisabled = false;
   mStatus = DIA_EN_SECURITY_LEVEL_STATUS_NOT_ACTIVE;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmSetRemainingLockTime ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmSetRemainingLockTime");
   mRemainingLockTime = mConfig.getLoginLockTime();
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmSetErrCode_RequestSequence ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmSetErrCode_RequestSequence");
   mErrorCode = DIA_E_SEQUENCE_ERROR;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmSetErrCode_RequiredTimeDelayNotExpired ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmSetErrCode_RequiredTimeDelayNotExpired");
   mErrorCode = DIA_E_REQUIRED_TIME_DELAY_NOT_EXPIRED;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmLock ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmLock");

   if ( mConfig.getLoginLockTime() )
   {
      dia_SecurityTimer* pTimer = getInstanceOfSecurityTimer();
      if ( pTimer && (pTimer->addTimerClient(this) == DIA_SUCCESS) )
      {
         if ( getInstanceOfSecurityTimer()->start() == DIA_SUCCESS )
         {
            DIA_TR_INF("##### SECURITY ACCESS TEMPORARY LOCKED #####");
            mLockState = true;
         }
      }

      if ( !mLockState )
      {
         DIA_TR_INF("##### FAILED TO LOCK SECURITY ACCESS !!! #####");
      }
   }
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmUnlock ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmUnlock");

   // reset the locking state
   mLockState = false;
   // reset the number of remaining attempts
   mRemainingAttempts = mConfig.getMaxNumberOfLogins();
   // reset the invalid key counter
   mInvalidKeyCounter = 0;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmSetErrCode_InvalidLevel ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmSetErrCode_InvalidLevel");
   mErrorCode = DIA_E_SUBFUNCTION_NOT_SUPPORTED;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmInvalidAttempt ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmInvalidAttempt");

   if ( mRemainingAttempts  > 0 ) mRemainingAttempts--;
   if ( mRemainingAttempts == 0 )
   {
	  mErrorCode = DIA_E_EXCEED_NUMBER_OF_ATTEMPTS;
	  DIA_TR_INF("dia_SecurityLevel::vFsmInvalidAttempt mErrorCode = 0x%08x",mErrorCode);
      (void) acceptEvent(dia_SecurityLevelFSM::evNumOfAttemptsExceeded,0);
   }
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmSetErrCode_InvalidKey ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmSetErrCode_InvalidKey");
   if ( mErrorCode == DIA_E_NO_ERROR )
   {
      DIA_TR_INF("dia_SecurityLevel::vFsmSetErrCode_InvalidKey mErrorCode=DIA_E_NO_ERROR");
   mErrorCode = DIA_E_INVALID_SECURITY_KEY;
   }
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmSendKeyAccepted ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmSendKeyAccepted");
   notifyKeyValidationResult(DIA_SUCCESS);
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmSendKeyNotAccepted ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmSendKeyNotAccepted");
   notifyKeyValidationResult(mErrorCode);
}

//------------------------------------------------------------------------------

bool
dia_SecurityLevel::isLockRequired ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::isLockRequired");
   return ( mRemainingAttempts ) ?  false : true;
}

//------------------------------------------------------------------------------

bool
dia_SecurityLevel::isLevelInvalid ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::isLevelInvalid");
   return false;
}

//------------------------------------------------------------------------------

bool
dia_SecurityLevel::isKeyInvalid ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::isKeyInvalid");
   return false;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmActivate ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmActivate");

   mStatus = DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE;
   activate();
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmHandleActivation ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmHandleActivation");
   (void) acceptEvent(dia_SecurityLevelFSM::evActivate,0);
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmHandleFailedActivation ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmHandleFailedActivation");
   (void) acceptEvent(dia_SecurityLevelFSM::evDeactivate,0);
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmActivateAtStartup ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmActivateAtStartup");

   mStatus = (mIsDisabled) ? DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE : DIA_EN_SECURITY_LEVEL_STATUS_NOT_ACTIVE;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmDeactivate ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmDeactivate");

   mStatus = DIA_EN_SECURITY_LEVEL_STATUS_NOT_ACTIVE;
   deactivate(DIA_UID_SECURITY_LOCK_LEVEL_LOCKED);
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevel::initializeSeed ( void )
{
   mSeed.erase( mSeed.begin(), mSeed.end() );
   mIsSeedValid = false;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityLevel::addSecurityModule ( dia_SecurityModule* pSecModule )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::addSecurityModule()");

   tDiaResult retCode = DIA_FAILED;

   if ( pSecModule )
   {
      std::map<dia_UID,dia_SecurityModule*>::iterator iter = mModuleRep.find(pSecModule->getUID());

      if ( iter == mModuleRep.end() )
      {
         DIA_TR_INF("#######################################################");
         DIA_TR_INF("#");
         DIA_TR_INF("# ADDING SECURITY MODULE \"%s\"", pSecModule->getName());
         DIA_TR_INF("#");
         DIA_TR_INF("#######################################################");

         mModuleRep[pSecModule->getUID()] = pSecModule;
         retCode = DIA_SUCCESS;
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityLevel::removeSecurityModule ( dia_UID moduleID )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::removeSecurityModule()");

   tDiaResult retCode = DIA_FAILED;

   std::map<dia_UID,dia_SecurityModule*>::iterator iter = mModuleRep.find(moduleID);

   if ( iter != mModuleRep.end() )
   {
      dia_SecurityModule* pModule = mModuleRep[moduleID];
      mModuleRep.erase(moduleID);
      OSAL_DELETE pModule;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityLevel::activate ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::activate()");

   if ( isEnabled() )
   {
      (void) unlock();
//
//      std::map<dia_UID,dia_SecurityModule*>::iterator iter = mModuleRep.begin();
//      for ( ; iter != mModuleRep.end(); ++iter )
//      {
//         if ( iter->second->unlockAccess() != DIA_SUCCESS )
//         {
//            DIA_TR_INF("#######################################################");
//            DIA_TR_INF("#");
//            DIA_TR_INF("# FAILED TO UNLOCK MODULE \"%s\"", iter->second->getName());
//            DIA_TR_INF("#");
//            DIA_TR_INF("#######################################################");
//         }
//      }
   }

   // inform the security manager that the security level was entered
   getInstanceOfSecurityManager()->onSecurityLevelUpdate(*this); //setLevel(mUID);

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityLevel::deactivate ( dia_SecurityLockLevelID lockLevelID )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::deactivate()");

   if ( isEnabled() && (this->getStatus() == DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE))
   {
//      (void) lock();
      std::map<dia_UID,dia_SecurityModule*>::iterator iter = mModuleRep.begin();
      for ( ; iter != mModuleRep.end(); iter++ )
      {
         if ( iter->second->lockAccess(lockLevelID) != DIA_SUCCESS )
         {
            DIA_TR_INF("#######################################################");
            DIA_TR_INF("#");
            DIA_TR_INF("# FAILED TO LOCK MODULE \"%s\"", iter->second->getName());
            DIA_TR_INF("#");
            DIA_TR_INF("#######################################################");
         }
      }
   }
   else
   {
      DIA_TR_INF("dia_SecurityLevel::deactivate: NO OPERATION PERFORMED (LevelMode=\"%s\",LevelStatus=\"%s\")",
            (isEnabled() ? "ENABLED" : "DISABLED"),
            ((getStatus() == DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE) ? "DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE" : "DIA_EN_SECURITY_LEVEL_STATUS_INACTIVE")
            );
   }

   // inform the security manager that the security level was entered
   getInstanceOfSecurityManager()->onSecurityLevelUpdate(*this); //setLevel(DIA_EN_SECURITY_LEVEL_NOT_ACTIVE);

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityLevel::lock ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::lock()");

   (void) saveLockStatus(this->getUID());

   if ( mStatus == DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE )
   {
      (void) acceptEventWithoutErrorInitialization(dia_SecurityLevelFSM::evSessionChanged,0);
   }

   std::map<dia_UID,dia_SecurityModule*>::iterator iter = mModuleRep.begin();
   for ( ; iter != mModuleRep.end(); ++iter )
   {
      if ( iter->second->lockAccess() != DIA_SUCCESS )
      {
         DIA_TR_INF("#######################################################");
         DIA_TR_INF("#");
         DIA_TR_INF("# FAILED TO LOCK MODULE \"%s\"", iter->second->getName());
         DIA_TR_INF("#");
         DIA_TR_INF("#######################################################");
      }
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityLevel::unlock ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevel::unlock()");

   (void) saveLockStatus(this->getUID());

   std::map<dia_UID,dia_SecurityModule*>::iterator iter = mModuleRep.begin();
   for ( ; iter != mModuleRep.end(); ++iter )
   {
      if ( iter->second->unlockAccess() != DIA_SUCCESS )
      {
         DIA_TR_INF("#######################################################");
         DIA_TR_INF("#");
         DIA_TR_INF("# FAILED TO UNLOCK MODULE \"%s\"", iter->second->getName());
         DIA_TR_INF("#");
         DIA_TR_INF("#######################################################");
      }
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevel::notifySeedResult ( tDiaResult resultCode, const std::vector<tU8>& seedValue )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::notifySeedResult");

   // the seed was successfully generated and is sent back to the requester
   mWasSeedRequested = true;

   dia_SrvHandlerGenericSecurityAccess::onSeedResult(resultCode,seedValue,*this);
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevel::notifyKeyValidationResult ( tDiaResult resultCode )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::notifyKeyValidationResult");
   dia_SrvHandlerGenericSecurityAccess::onKeyValidationResult(resultCode,*this);
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevel::vFsmSimulateCalcSeed ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevel::vFsmSimulateCalcSeed");

   // initialize the seed attribute
   initializeSeed();

   // calculate seed
   for ( tU16 j=0; j<mConfig.getSeedSize(); j++ )
   {
      mSeed.push_back((tU8) getInstanceOfRandomGenerator()->getRandomNumber());
   }

   mIsSeedValid = true;
}
