/*!
 * \file       dia_RoutineSimple.h
 *
 * \brief      Template class for a simple Routine-Control
 *
 * \details    Provide a template for all routines controls that work as follows:
 *             - receive uds-message
 *             - forward request to a system-adapter.
 *             - receive answer from system-adapter or timeout
 *             - send uds-answer accordingly
 *             
 *             All interfaces are overloaded and detected by the data-types.
 *             Only pure virtual  methods are 
 *                parseRequest(): convert received uds-request into internal data-type
 *                assembleResult(): convert received answer from system-adapter to uds-answer.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaServicesCommon
 *
 * \copyright  (c) 2018 Robert Bosch Car Multimedia
 *
 */


#ifndef __INCLUDED_DIA_ROUTINE_SIMPLE__
#define __INCLUDED_DIA_ROUTINE_SIMPLE__

#include "common/framework/trace/dia_trace.h"
#include "common/framework/protocols/uds/rtctrl/dia_Routine.h"
#include "common/framework/protocols/uds/rtctrl/dia_RoutineCtrl.h"
#include "common/services/uds/generic/dia_ToUdsHelpers.h"
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#include "common/framework/protocols/uds/rtctrl/dia_RoutineCtrlManager.h"
#include "common/framework/protocols/uds/rtctrl/dia_RoutineCtrlManager.h"
#include "common/services/uds/generic/dia_ListToUds.h"

namespace dia {
template<class ADAPTER, class LISTENER, class REQUEST, class RESULT>
class dia_RoutineSimple : public dia_Routine,
                          public LISTENER
{

   DECL_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_RoutineSimple);

public:
   dia_RoutineSimple(char const *name, uint16_t did, dia_eRoutineID id): 
      dia_Routine(name, 
                  did, 
                  id, 
                  DIA_EN_RTCTRL_TYPE_SHORT_TERM)
   {
      dia_ScopeTraceVarg oTrace("%s::%s", getName(), getName());
   }
   
   ~dia_RoutineSimple()
   {
      dia_ScopeTraceVarg oTrace("%s::~%s", getName(), getName());
      (void) unsetSysAdapterListener<LISTENER>(this);

   }
   
   virtual tDiaResult start ( std::vector<tU8>& params, tU8 /*timerValue*/ )
   {
      dia_ScopeTraceVarg oTrace("%s::start", getName());
      
      tDiaResult retCode = DIA_FAILED;
      
      // prepare processing of the routine
      vInitialize();
      
      ADAPTER* pSysAdapter = NULL;
      
      if ( (querySysAdapterInterface<ADAPTER>(&pSysAdapter) != DIA_SUCCESS) || !pSysAdapter )
      {
         DIA_TR_ERR("%s::start => ERROR: pSysAdapter == NULL", getName());
         return retCode;
      }

      (void) setSysAdapterListener<LISTENER>(this);
      REQUEST request;
      if (! parseRequest(params, request)) {
         DIA_TR_ERR("%s::start => ERROR could not parse request", getName());
      }
      else if ( pSysAdapter->handleRequest(&request) != DIA_SUCCESS)
      {
         DIA_TR_ERR("%s::start => ERROR calling adapter != DIA_SUCCESS", getName());
      }
      else
      {
         DIA_TR_ERR("%s::start => Send Successfully to Server", getName());
         eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
         retCode = DIA_SUCCESS;
      }
      
      if ( retCode != DIA_SUCCESS )
      {
         (void) unsetSysAdapterListener<LISTENER>(this);
      }
      
      return retCode;
   }

   virtual void vOnAdaperResult (RESULT *resultData)
   {
      dia_ScopeTraceVarg oTrace("%s::vOnAdaperResult",getName());
      
      DIA_TR_INF("%s::vOnAdaperResult => Result", getName());
      (void) unsetSysAdapterListener<LISTENER>(this);
      eSetStatus(assembleResult(*resultData, mResults) == DIA_SUCCESS ? 
                 DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK: DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
      mIsResultReady = TRUE;
      dia_RoutineCtrlManager::getInstance()->vOnRoutineUpdate(*this);
   }

   tDiaResult
   requestResult ( std::vector<tU8>& results )
   {
      dia_ScopeTraceVarg oTrace("%s::requestResult", getName());

      tDiaResult retCode = DIA_E_SEQUENCE_ERROR;

      results.clear();
      results.push_back(diaRoutineStatusToUds(mStatus));

      if ( mResults.size() )
      {
         DIA_TR_INF("%s::requestResult --- 1", getName());
         std::vector<tU8>::iterator iter = mResults.begin();
         for ( ; iter != mResults.end(); iter++ )
         {
            DIA_TR_INF("%s::requestResult: push %02x", getName(), *iter);
            results.push_back(*iter);
         }
         mResults.clear();
         retCode = DIA_SUCCESS;
      }

      if ( mStatus ==  DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK || 
           mStatus ==  DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK)
      {
         retCode = DIA_SUCCESS;
      }
   
      return retCode;
   }

   virtual tU32 getMaxProcessingTime ( void ) const { return 5000; }
   virtual tU8 diaRoutineStatusToUds(dia_eRoutineStatus diaStatus) {
      return dia_mapRoutineStatusType1ToUds(diaStatus);
   }

   virtual void vOnServiceTimeout () {
      dia_ScopeTraceVarg oTrace("%s::vOnServiceTimeout",getName());
      DIA_TR_ERR("%s::vOnServiceTimeout", getName());
      (void) unsetSysAdapterListener<LISTENER>(this);
   }
   
protected:
   virtual bool parseRequest( std::vector<tU8> const &uds, REQUEST &dia)=0;
   virtual tDiaResult assembleResult(RESULT  const &dia, std::vector<tU8>& uds)=0;



};

}
#endif

