/**
 * \file    dia_IOCtrlSignal.h
 *
 * \brief   TBD
 *
 * \author  gib2hi
 * \date    2012/04/11
 */

#ifndef __INCLUDED_DIA_IOCTRL_SIGNAL__
#define __INCLUDED_DIA_IOCTRL_SIGNAL__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_IOCTRL__
#include "common/framework/protocols/uds/ioctrl/dia_IOCtrl.h"
#endif

#ifndef __INCLUDED_DIA_IOCTRL_TIMER_CLIENT__
#include "common/framework/protocols/uds/ioctrl/dia_IOCtrlTimerClient.h"
#endif

#define DIA_C_U16_IOCTRL_DEFAULT_TIMER_VALUE             ((tU8)   0xFF)
#define DIA_C_U16_IOCTRL_DEFAULT_PAYLOAD_SIZE            ((tU16)     1) // size in bytes
#define DIA_C_U16_IOCTRL_DEFAULT_PROCESSING_TIMEOUT      ((tU16) 10000) // 10 secs
#define DIA_C_U16_IOCTRL_PAYLOAD_START_BYTE              ((tU16)     5) //! Position of 1st byte of payload
#define DIA_C_U16_IOCTRL_MIN_MESSAGE_LENGTH              ((tU16)     5) //! Minimum length for IO Control message


////////////////////////////////////////////////////////////////////////////////
//
//
//
////////////////////////////////////////////////////////////////////////////////

class dia_IOCtrlSignal
   : public virtual dia_Object,
     public dia_IOCtrlTimerClient
{
public:
   //! class constructor
   dia_IOCtrlSignal (
         tCString name,
         tU16 udsID,
         tU8  ctrlValueSize = DIA_C_U16_IOCTRL_DEFAULT_PAYLOAD_SIZE,
         dia_eIOCtrlMode mode = DIA_EN_IOCTRL_CTRLMODE_ECU,
         dia_eIOCtrlMonitoringMode monitoring = DIA_EN_IOCTRL_MONITORING_MODE_NOT_SUPPORTED
         );

   //! class destructor
   virtual ~dia_IOCtrlSignal ( void );

   //! return the name of the ioctrl signal
   tCString getName ( void ) const { return mName; }
   //! return the DID (diagnosis data identifier) of the signal
   virtual tU16 getDID ( void ) const;
   //! return the signals' unique identifier (type of the signal
//   dia_eIOCtrlSignalType eGetType ( void ) const;
   //! return the unique identifier of the signal
   dia_UID getUID ( void ) const;

   //! return the control mode of the signal
   dia_eIOCtrlMode getMode /*eGetMode*/ ( void ) const;
   //! set the control mode for the signal
   void setMode /*eSetMode*/ ( dia_eIOCtrlMode mode );

   //! number of payload bytes
   tU16 sizeOfCtrlValue ( void ) const;

   //! get the status of the IOCtrl signal
   dia_eIOCtrlStatus getStatus ( void ) const;
   //! set the status of the IOCtrl signal
   virtual void setStatus ( dia_eIOCtrlStatus newStatus );

   //! called to request the current state of the corresponding service
   virtual void getCurrentState ( std::vector<tU8>& results );

   //! return the timer mode for the signal
   virtual dia_eIOCtrlMonitoringMode getMonitoringMode ( void ) const;
   //! set the control mode for the signal
   virtual void setMonitoringMode ( dia_eIOCtrlMonitoringMode mode, tU8 timerValue = IOCTRL_U8_TIMER_TIMEOUT_INFINITE );

   //! check if the result of the request is available
   void setResultReady ( bool mode=true ) { mIsResultReady = (mode) ? TRUE : FALSE; }
   //! check if the result of the request is available
   bool isResultReady ( void ) const;

   //! check if the signal is currently frozen
   virtual bool isFrozen ( void ) const;

   //! check if all preconditions for using the iocontrol signal are matched
   virtual tDiaResult checkPreconditions ( void ) const { return DIA_SUCCESS; }

   //! check if the length for freeze request is matched
   virtual tDiaResult checkFreezeLength ( tU16 length ) const;

   //! set the timer for this signal (specified in seconds
   void setTimer ( tU16 seconds );
   //! return the timer value for this signal
   tU8 getTimerValue ( void ) const;
   //! handle the elapsed short term adjustment timer
   virtual void handleTimerElapsed ( void );
   //! overloaded method called by the timer after a timer tick (resolution 1s)
   virtual tDiaResult onTimerTickIOCtrl ( void );

   //! return the control value for this signal
   const std::vector<tU8>& getControlValue ( void ) const;
   //! checks the payload in its correctness - overloaded by some IOCtrls with special handling
   virtual tDiaResult checkPayloadCorrectness ( tU16 realLength, std::vector<tU8>* payload );

   //! prepare processing of a request
   virtual void initializeRequest ( void );
   //! prepare processing of a request
   virtual void initializeRequest ( tU8 timerValue, std::vector<tU8>* ctrlValue ); //lint !e1735: checked that default param is not changed by subclasses
   //! handle the IOCtrl request from the tester (implemented by concrete subclasses)
   virtual tDiaResult handleFreeze ( void );
   //! handle the IOCtrl request from the tester (implemented by concrete subclasses)
   virtual tDiaResult handleRequest ( tU8 timerValue, std::vector<tU8>* ctrlValue ) = 0;
   //! handle the return control to ECU
   virtual tDiaResult handleReturnControlToECU ( void );
   //! handle timeout from UDS session object
   virtual void handleTimeout ( void ) {}

   //! retrieve the value of the maximum processing time in ms
   virtual tU32 getMaxProcessingTime ( void ) const { return DIA_C_U16_IOCTRL_DEFAULT_PROCESSING_TIMEOUT; }

   //! set error info
   void setErrorInfo ( tDiaResult errInfo = DIA_E_NOERROR ) { mErrorInfo = errInfo; }
   //! return the error information for the last action
   tDiaResult getErrorInfo ( void ) const { return mErrorInfo; }

   //##########################################################################
   //#
   //# DEPRECATED METHODS (DO NOT USE !!)
   //#
   //##########################################################################

   //! get the status of the IOCtrl signal
   dia_eIOCtrlStatus eGetStatus ( void ) const { return getStatus(); }
   //! set the status of the IOCtrl signal
   virtual void eSetStatus ( dia_eIOCtrlStatus newStatus ) { return setStatus(newStatus); }
   //! set the timer for this signal (specified in seconds
   void vSetTimer ( tU16 seconds ) { setTimer(seconds); }

protected:
   //##########################################################################
   //#
   //# DEPRECATED METHODS (DO NOT USE !!)
   //#
   //##########################################################################

   //! deprecated class constructor. do not use anymore !!
   dia_IOCtrlSignal (
         tU16 udsID,
         dia_eIOCtrlSignalType type,
         tU8  ctrlValueSize = DIA_C_U16_IOCTRL_DEFAULT_PAYLOAD_SIZE,
         dia_eIOCtrlMode mode = DIA_EN_IOCTRL_CTRLMODE_ECU,
         dia_eIOCtrlMonitoringMode monitoring = DIA_EN_IOCTRL_MONITORING_MODE_NOT_SUPPORTED
         );

   //! return the control mode of the signal
   dia_eIOCtrlMode eGetMode ( void ) const { return getMode(); }
   //! set the control mode for the signal
   void eSetMode ( dia_eIOCtrlMode mode ) { setMode(mode); }

protected:
   //! deprecated default constructor
   dia_IOCtrlSignal ( tCString name );
   //! deprecated default constructor
   dia_IOCtrlSignal ( void );

   //! switch off the ioctrl signal
   virtual tDiaResult vOnTerminate ( dia_eIOCtrlStatus status ) = 0;

protected:
   //! ioctrl name
   tCString mName;
   //! unique ioctrl identifier
   dia_UID mUID;
   //! uds identifier
   tU16 mUdsID;
   //! number of payload bytes
   tU16 mCtrlValueSize;
   //! status of the signal
   dia_eIOCtrlStatus mStatus;
   //! control mode of the signal (either controlled by ECU or by the tester
   dia_eIOCtrlMode mMode;
   //! specifies if remote control is only used for a short term adjustment
   dia_eIOCtrlMonitoringMode mTimerMode;
   //! timer value delivered via the UDS request
   tU8 mTimerValue;
   //! timer counter decremented by the IOCtrlTimer
   tU16 mTimerCounter;
   //! control value
   std::vector<tU8> mCtrlValue;
   //! flag to indicate if we are waiting for a result (e.g. from a component)
   tBool mIsResultReady;
   //! last error info
   tDiaResult mErrorInfo;
};

#endif /* __INCLUDED_DIA_IOCTRL_SIGNAL__ */
