/*!
 * \file       dia_TesterPresentHandler.cpp
 *
 * \brief      Handling of Tester-Present as Message-Handler of udd-comm-channel
 *
 * \details    
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreUDD
 *
 * \copyright  (c) 2019 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif


#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif


#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_DIAGSESSION_UDS__
#include "common/depricated/dia_tclDiagSessionUds.h"
#endif


static const uint8_t NACK_CODE_RESPONSE_PENDING =(uint8_t)0x78;


#include "dia_TesterPresentHandler.h"

#define DIA_C_U32_S3_TIMER_VAL_MS 50000

#define DATA_START 3

namespace dia {

TesterPresentHandler::TesterPresentHandler (  ):
   mS3ValMs(DIA_C_U32_S3_TIMER_VAL_MS),
   mIsSetupDone(false),
   mNonDefaultSessionActive(false),
   mSelfTimer(this, this->mRefCounter)
{
   dia_tclFnctTrace oTrace("dia::TesterPresentHandler::CTOR default");
   if ( TesterPresentHandler::setup() != DIA_SUCCESS )
   {
      DIA_TR_INF("### SETUP OF dia::TesterPresentHandler FAILED ###");
   }

}

TesterPresentHandler::TesterPresentHandler ( dia_MessageHandler* nextHandler ):
   dia_MessageHandler(nextHandler),
   mS3ValMs(DIA_C_U32_S3_TIMER_VAL_MS),
   mIsSetupDone(false),
   mNonDefaultSessionActive(false),
   mSelfTimer(this, this->mRefCounter)
{
   dia_tclFnctTrace oTrace("dia::TesterPresentHandler::CTOR");
   if ( TesterPresentHandler::setup() != DIA_SUCCESS )
   {
      DIA_TR_INF("### SETUP OF dia::TesterPresentHandler FAILED ###");
   }
}

TesterPresentHandler::~TesterPresentHandler ( void ) {
   dia_tclFnctTrace oTrace("dia::TesterPresentHandler::DTOR");
   tearDown();
}

tDiaResult TesterPresentHandler::handleMessage ( dia_MessageBuffer& msg ) {
   dia_tclFnctTrace oTrace("dia::TesterPresentHandler::handleMessage");
   
   // basic checks on message
   if ( msg.getProtocol() != DIA_EN_PROTOCOL_UDS ) {
      // check protocoll.
      DIA_TR_INF("dia::TesterPresentHandler::handleMessage forward, no uds");
      return dia_MessageHandler::handleMessage(msg);
   }
   // check minimum len
   uint8_t offsetUdsStart=(msg.getFormat() == dia_MessageBuffer::format_length_and_data) ? 1 : 0;
   uint32_t numUdsBytes=msg.u16GetDataLength() - offsetUdsStart;
   if (numUdsBytes < 1) {
      // msg too short
      DIA_TR_ERR("dia::TesterPresentHandler::handleMessage() empty message");
      return DIA_FAILED;
   }

   // message has at least an sid, while processing the message, S3 timer is stopped
   mSelfTimer.stop();

   uint8_t sid=msg.getDataU8(offsetUdsStart);

   // check for session-change. we need to know if a non-default-session has been requested on this channel
   if (sid==DIA_C_U8_UDS_SID_SESSION_CONTROL) {
      DIA_TR_ERR("dia::TesterPresentHandler::handleMessage() sid=DIA_C_U8_UDS_SID_SESSION_CONTROL");

      if (numUdsBytes != 2) {
         // do noting,  pass responsibility
      }
      else {
         uint8_t sessionType=msg.getDataU8((tU16)(offsetUdsStart + 1));
         DIA_TR_ERR("dia::TesterPresentHandler::handleMessage() sessionType=%02x", sessionType);
      }
      DIA_TR_INF("dia::TesterPresentHandler::handleMessage forward DIA_C_U8_UDS_SID_SESSION_CONTROL");
   }
   else if (sid==DIA_C_U8_UDS_SID_TESTER_PRESENT /* 0x3E, missing in dia_defsCommon.h */) {
      DIA_TR_ERR("dia::TesterPresentHandler::handleMessage() sid=DIA_C_U8_UDS_SID_TESTER_PRESENT");
      if (numUdsBytes != 2) {
         msg.vSetNegResp(getInstanceOfFactory()->makeNRC(DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT));
         msg.vSendResponse();
         return DIA_FAILED;
      }
      uint8_t subFunction=msg.getDataU8((tU16)(offsetUdsStart + 1));
      DIA_TR_ERR("dia::TesterPresentHandler::handleMessage() subFunction=%02x", subFunction);
      if (subFunction == 0x80) {
         // supress positive response
         return DIA_SUCCESS;
      }
      else if (subFunction == 0x00) {
         msg.vSetPosResp();
         msg.vSetDataLength(1); // only sid
         msg.vSendResponse();
         return DIA_SUCCESS;
      }
      else {
         msg.vSetNegResp(getInstanceOfFactory()->makeNRC(DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT));
         msg.vSendResponse();
         return DIA_FAILED;
      }
   }
   else {
      DIA_TR_INF("dia::TesterPresentHandler::handleMessage forward other");
   }
   return dia_MessageHandler::handleMessage(msg);
}

void TesterPresentHandler::onTimeout(MsgTimeoutS3 * /* msg */) {
   dia_tclFnctTrace oTrace("dia::TesterPresentHandler::onTimeout");
   // forceDefaultSession
   mNonDefaultSessionActive=false;

   tU8 defSession = 0;
   tU32 retCode = dia_getProperty(DIA_PROP_DEFAULT_SESSION, defSession);
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_INF("!!!  dia_CommChannelINC::onTesterDisconnected => ERROR: dia_getProperty FAILED retCode=0x%08X", retCode);
      return;
   }

   tU8 tempBuffer[] = { 3, DIA_C_U8_UDS_SID_SESSION_CONTROL, defSession };

   dia_MessageBufferUDS* pMsgBuffer = DIA_NEW dia_MessageBufferUDS (
                                                                    &tempBuffer[0],
                                                                    tU16(sizeof tempBuffer),
                                                                    dia_tclDiagSessionUds::vNullResponse
                                                                    );
   // create and send a ReqRx event to uds engine
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgBuffer));
   // trigger IDLE event to inform LCM that diagnosis activity has been terminated
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventIdle());
}


tVoid TesterPresentHandler::onDiagnosisResponse (  tU8 const *au8MsgBuffer, tU16 u16Length ) {
   DIA_TR_INF("dia::TesterPresentHandler::onDiagnosisResponse START "
              "u16Length=%u  mNonDefaultSessionActive=%u",
                           u16Length,  mNonDefaultSessionActive);
   if (!u16Length) {
      return;
   }
   uint8_t nackCode=0;
   // check for ack of SESSION_CONTROL
   uint8_t sid=au8MsgBuffer[0];
   if (sid==DIA_C_U8_UDS_SID_SESSION_CONTROL + 0x40) {
      // todo: extract timing-parameters
      mNonDefaultSessionActive=true;
   } else if (u16Length==3 && sid==0x7F) {
      /* uint8_t sidNack=au8MsgBuffer[1]; */
      nackCode=au8MsgBuffer[2];
   }

   if (mNonDefaultSessionActive && nackCode!=DIA_E_U8_UDS_RESPONSE_PENDING) {
      mSelfTimer.start(mS3ValMs);
   }
   DIA_TR_INF("dia::TesterPresentHandler::onDiagnosisResponse END "
              "u16Length=%u mNonDefaultSessionActive=%u",
              u16Length, mNonDefaultSessionActive);
}


tDiaResult TesterPresentHandler::setup ( void ) {
   dia_tclFnctTrace oTrace("dia::TesterPresentHandler::setup");
   if (mIsSetupDone) {
      return DIA_SUCCESS;
   }
   mIsSetupDone=true;
   return DIA_SUCCESS;
}

tDiaResult TesterPresentHandler::tearDown ( void ) {
   dia_tclFnctTrace oTrace("dia::TesterPresentHandler::tearDown");
   return DIA_SUCCESS;
}

}
