/*!
* \file       dia::OpModeHandler.cpp
*
* \brief      Pre-service handler for all diag services
*             defer processing of UDS requests until SCC sends a valid Operation Mode
*
*
* \component  Diagnosis
*
* \ingroup    diaCoreProtocols
*
* \copyright  (c) 2018 Robert Bosch GmbH
*
* The reproduction, distribution and utilization of this file as
* well as the communication of its contents to others without express
* authorization is prohibited. Offenders will be held liable for the
* payment of damages. All rights reserved in the event of the grant
* of a patent, utility model or design.
*/

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include "common/framework/application/dia_AppController.h"
#endif

#ifndef __INCLUDED_DIA_OPMODE_HANDLER__
#include <common/framework/protocols/uds/dia_OpModeHandler.h>
#endif

#ifndef __INCLUDED_DIA_CMD_START_TEST__
#include "common/framework/test/dia_CmdStartTest.h"
#endif

#ifndef __INCLUDED_DIA_PREDICATE_RUNLEVEL__
#include "common/framework/application/dia_PredicateOpMode.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER__
#include <common/framework/application/dia_CommandController.h>
#endif

using namespace dia;

//-----------------------------------------------------------------------------

OpModeHandler::OpModeHandler(void)
{
   ScopeTrace trc("dia::OpModeHandler::OpModeHandler");
}

//-----------------------------------------------------------------------------

OpModeHandler::OpModeHandler(dia_MessageHandler* nextHandler)
         : dia_MessageHandler(nextHandler)
{
   ScopeTrace trc("dia::OpModeHandler::OpModeHandler(dia_MessageHandler*)");
}

//-----------------------------------------------------------------------------

OpModeHandler::~OpModeHandler(void)
{
}

//-----------------------------------------------------------------------------

tDiaResult
OpModeHandler::handleMessage(dia_MessageBuffer& msg)
{
   ScopeTrace trc("dia::OpModeHandler::handleMessage(dia_MessageBuffer&)");

   tDiaResult retCode = DIA_FAILED;

   bool bDeferProcessing = false;

   dia_Factory* pFactory = getInstanceOfFactory();
   if (pFactory)
   {
      dia_enInitLevel activeInitLevel = pFactory->getInitializationLevel();
      DIA_TR_INF("##### dia::OpModeHandler::handleMessage: Current Init Level : %d #####",activeInitLevel );
   }
   else
   {
      DIA_TR_ERR("dia::OpModeHandler::handleMessage - UNABLE TO GET POINTER TO FACTORY!");
   }

   if (! OpMode::isSynchronized())
   {
      Command* pCmd = new Command;
      if (pCmd)
      {
         pCmd->addFunctor(new dia_FunctorOneArgNoReturnValue<OpModeHandler, dia_MessageBuffer*>(this, &OpModeHandler::postprocessMessage, &msg));
         bDeferProcessing = true;

         //add predicate
         (void)pCmd->addPredicate(new PredicateOpMode(OpMode::getActiveOpMode()));
         // add command to cmdController
         getInstanceOfCommandController()->addCommand(pCmd);

         retCode = DIA_SUCCESS;
         DIA_TR_INF("dia::OpModeHandler::handleMessage DIA_SUCCESS!");
      }
      else
      {
         DIA_TR_ERR("dia::OpModeHandler::handleMessage TRIGGER FAILED!");
      }
      pCmd = nullptr;
   }

   if (!bDeferProcessing)
   {
      //Go on processing.
      retCode = dia_MessageHandler::handleMessage(msg);
      DIA_TR_INF("dia::OpModeHandler::handleMessage forward message to next handler! retCode=0x%08X.", retCode);
   }

   return retCode;
}

void
OpModeHandler::postprocessMessage(dia_MessageBuffer* msg)
{
   ScopeTrace trc("dia::OpModeHandler::postprocessMessage(dia_MessageBuffer&)");
   if (msg && (dia_MessageHandler::handleMessage(*msg) == DIA_E_NOT_AVAILABLE))
   {
      getInstanceOfApplication()->postMessage(new dia_tclDiagSession::tclEventReqRx(msg));
      DIA_TR_INF("--- OpModeHandler::postprocessMessage => Posted as UDS message");
   }
}

