/*
 * dia_MultipleReadDataByIDHandler.cpp
 *
 *  Created on: 22.07.2015
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_MULTIPLE_READ_DATA_BY_ID_HANDLER__
#include <common/framework/protocols/uds/dia_MultipleReadDataByIDHandler.h>
#endif

#define DIA_U16_READDATA_BY_ID_REQUEST_LENGTH   ((tU16) (1 /* SID */ + 2 /* DID */))
#define DIA_U16_SIZEOF_DID                      ((tU16) 2)

static const tU32 DIA_C_U32_MULTIPLE_DID_READ_TIMEOUT = 15000;

using namespace dia;
//-----------------------------------------------------------------------------

dia_MultipleReadDataByIDHandler::dia_DIDWorkItem::dia_DIDWorkItem( tU16 did )
   : mDID(did)
{}

//-----------------------------------------------------------------------------

dia_MultipleReadDataByIDHandler::dia_MultipleReadDataByIDHandler ( void )
   : mIsSetupDone(false),
     mpFSM(0),
     mProcessingTimeout(DIA_C_U32_MULTIPLE_DID_READ_TIMEOUT),
     mErrorCode(DIA_SUCCESS),
     mpActiveMessageBuffer(0),
     mNumOfDIDs(0),
     mpActiveWorkItem(0)
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::dia_MultipleReadDataByIDHandler");

   if ( dia_MultipleReadDataByIDHandler::setup() != DIA_SUCCESS )
   {
      DIA_TR_INF("### SETUP OF MULTIPLE DID READER FAILED ###");
   }
}

//-----------------------------------------------------------------------------

dia_MultipleReadDataByIDHandler::dia_MultipleReadDataByIDHandler ( dia_MessageHandler* nextHandler )
   : dia_MessageHandler(nextHandler),
     mIsSetupDone(false),
     mpFSM(0),
     mProcessingTimeout(DIA_C_U32_MULTIPLE_DID_READ_TIMEOUT),
     mErrorCode(DIA_SUCCESS),
     mpActiveMessageBuffer(0),
     mNumOfDIDs(0),
     mpActiveWorkItem(0)
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::dia_MultipleReadDataByIDHandler(dia_MessageHandler*)");

   if ( dia_MultipleReadDataByIDHandler::setup() != DIA_SUCCESS )
   {
      DIA_TR_INF("### SETUP OF MULTIPLE DID READER FAILED ###");
   }
}

//-----------------------------------------------------------------------------

dia_MultipleReadDataByIDHandler::~dia_MultipleReadDataByIDHandler ( void )
{
   _BP_TRY_BEGIN
   {
      (void) dia_MultipleReadDataByIDHandler::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_MultipleReadDataByIDHandler::~dia_MultipleReadDataByIDHandler !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_MultipleReadDataByIDHandler::setup ( void )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::setup");

   // we setup the engine object only once
   if ( mIsSetupDone )
   {
      DIA_TR_INF("### MULTIPLE DID READER WAS ALREADY SET UP. RETURNING... ###");
      return DIA_SUCCESS;
   }

   mIsSetupDone = true;

   mTimer.s32Create();
   // create the state machine object
   if ( dia_MultipleReadFSM::Fsm::createFSM (&mpFSM,this) != true )
   {
      DIA_TR_INF("### FAILED TO CREATE MULTIPLE DID READER STATE MACHINE ###");
      return DIA_FAILED;
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_MultipleReadDataByIDHandler::tearDown ( void )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::tearDown");

   clearWorkItems();
   clearProcessedWorkItems();

   mTimer.s32Delete();
   mTimer.removeTimerListener(this);

   OSAL_DELETE mpFSM;
   mpFSM = 0;

   OSAL_DELETE mpActiveMessageBuffer;
   mpActiveMessageBuffer = 0;

   OSAL_DELETE mpActiveWorkItem;
   mpActiveWorkItem = 0;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_MultipleReadDataByIDHandler::handleMessage ( dia_MessageBuffer& msg )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::handleMessage(dia_MessageBuffer&)");

   tDiaResult retCode = DIA_FAILED;

   if ( msg.getProtocol() == DIA_EN_PROTOCOL_UDS )
   {
      tU16 offset = (msg.getFormat() == dia_MessageBuffer::format_length_and_data) ? 2 : 1; /* one length byte plus one byte SID */
      if ( msg.getDataU8(tU16(offset-1)) == DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER )
      {
         tU16 dataLength = msg.u16GetDataLength();
         if ( dataLength > (offset + DIA_U16_SIZEOF_DID) ) // must contain more than one DID
         {
            tU16 dataLengthStripped = tU16(dataLength - offset);
            if ( (dataLengthStripped % sizeof(tU16)) == 0 )
            {
               clearWorkItems();
               clearProcessedWorkItems();

               // populate the workitem list (one entry per DID)
               tU16 numOfDIDs = tU16(dataLengthStripped / sizeof(tU16));
               int j=0;
               for ( tU16 i=0; i<numOfDIDs; ++i, j+=2 )
               {
                  tU16 nextOffset = tU16(offset + j);
                  tU16 did = 0;
                  did = tU16(msg.getDataU8(nextOffset) << 8);
                  did = tU16(did | msg.getDataU8(++nextOffset));
                  dia_DIDWorkItem* pWorkItem = new dia_DIDWorkItem(did);
                  if ( pWorkItem )
                  {
                     DIA_TR_INF("##### CREATED WORKITEM FOR DID 0x%04x #####",did);
                     mWorkItems.push_back(pWorkItem);
                  }
               } //lint !e429 custodial pointer is freed by framework after message was processed

               if ( mWorkItems.size() > 1 ) retCode = DIA_SUCCESS;
            }
            else
            {
               DIA_TR_INF("##### dia_MultipleReadDataByIDHandler: FAILED TO EXTRACT DIDS #####");

#ifndef __DIA_UNIT_TESTING__
               msg.vSetNegResp(getInstanceOfFactory()->makeNRC(DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT));
               msg.vSendResponse();
               return retCode;
#endif
            }
         }
         else
         {
            DIA_TR_INF("##### dia_MultipleReadDataByIDHandler: NO MULTIPLE DIDS DETECTED #####");
         }
      }
   }
   else
   {
      DIA_TR_INF("##### dia_MultipleReadDataByIDHandler: INVALID PROTOCOL (ONLY UDS SUPPORTED) #####");
   }

   if ( retCode == DIA_SUCCESS )
   {
      // number of DIDs we have extracted from the UDS service request
      mNumOfDIDs = (tU16) mWorkItems.size();
      // store the original message as it contains the callback method used to deliver the final result
      mpActiveMessageBuffer = &msg;
      boost::any ParamVal = 0;
      if ( mpFSM ) mpFSM->acceptEvent(dia_MultipleReadFSM::evStartProcessing,ParamVal);
   }
   else
   {
      retCode = dia_MessageHandler::handleMessage(msg);
   }

   return retCode;
}

//-------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vOnTimerElapsed ( dia_TimerID /*id*/ )
{
  	boost::any ParamVal = 0;
   DIA_TR_INF("### dia_MultipleReadDataByIDHandler::vOnTimerElapsed ###");

   if ( mpFSM ) mpFSM->acceptEvent(dia_MultipleReadFSM::evTimeout,ParamVal);
}


//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::clearWorkItems ( void )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::clearWorkItems()");

   std::list<dia_DIDWorkItem*>::iterator iter = mWorkItems.begin();
   for ( ; iter != mWorkItems.end(); iter++ )
   {
      delete (*iter);
   }
   mWorkItems.clear();
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::clearProcessedWorkItems ( void )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::clearProcessedWorkItems()");

   std::list<dia_DIDWorkItem*>::iterator iter = mPocessedWorkItems.begin();
   for ( ; iter != mPocessedWorkItems.end(); iter++ )
   {
      delete (*iter);
   }
   mPocessedWorkItems.clear();
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::onDIDResponse ( const tU8 msgBuffer[], tU16 length, tCookieType cookie )
{
   if ( isCookieEmpty(cookie) ) return;
   dia_MultipleReadDataByIDHandler* pObj = UnsafeVoidCast<dia_MultipleReadDataByIDHandler>(cookie);
   if(pObj == 0) return;
   std::vector<tU8> vec;
   std::copy(msgBuffer, msgBuffer + length, std::back_inserter(vec));
   boost::any data(std::move(vec));

   DIA_TR_INF("### onDIDResponse ###");
   for ( tU16 k=0; k<length; k++ )
   {
      DIA_TR_INF("... msg[%d] = 0x%02x",k,msgBuffer[k]);
   }

   if ( pObj->mpFSM )pObj->mpFSM->acceptEvent(dia_MultipleReadFSM::evResponseReceived,data);

   // sent trigger to the UDS state machine that the response was sent
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vFsmEvaluateResponse ( boost::any& pArg )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::vFsmEvaluateResponse()");

   if ( pArg.empty() ) 	return;

   std::vector<tU8> pDIDResponse;
   _BP_TRY_BEGIN
   {
      pDIDResponse = std::move(boost::any_cast<std::vector<tU8> >(pArg));
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_MultipleReadDataByIDHandler::vFsmEvaluateResponse ( ) !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END

   if ( pDIDResponse.size() >= 2 )
   {
      if ( mpActiveWorkItem )
      {
         mpActiveWorkItem->mResponse = std::move(pDIDResponse);
         mPocessedWorkItems.push_back(mpActiveWorkItem);
         mpActiveWorkItem = 0;
      }
   }
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vFsmPrepareProcessing ( boost::any& /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::vFsmPrepareProcessing()");

   if ( mWorkItems.size() )
   {
      mpActiveWorkItem = mWorkItems.front();
      mWorkItems.pop_front();
   }
   else
   {
      mpActiveWorkItem = 0;
   }
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vFsmProcessDID ( boost::any& /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::vFsmProcessDID()");

   if ( mpActiveWorkItem )
   {
      tU16 did = mpActiveWorkItem->mDID;

      tU8 txBuffer[3] = { /*4,*/ DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 0x00, 0x00 };
      txBuffer[1] = (tU8) (did >> 8);
      txBuffer[2] = (tU8)  did;

      dia_MessageBufferUDS* pMsgBuffer = new dia_MessageBufferUDS (
            &txBuffer[0],
            tU16(sizeof(txBuffer)),
            dia_MultipleReadDataByIDHandler::onDIDResponse,
            dia_MessageBuffer::holds_request,
            dia_MessageBuffer::format_raw,
            tCookieType(this)
      );

      getInstanceOfApplication()->postMessage(new dia_tclDiagSession::tclEventReqRx(pMsgBuffer));
      DIA_TR_INF("##### FORWARDED UDS MESSAGE TO UDS ENGINE #####");
   } //lint !e429 custodial pointer is freed by framework after message was processed
   else
   {
      DIA_TR_ERR("##### NO MESSAGE FOUND TO BE FORWARDED TO UDS ENGINE #####");
   }
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vFsmReset ( boost::any& /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::vFsmReset()");

   mpActiveMessageBuffer = 0;
   mErrorCode = DIA_SUCCESS;
   mNumOfDIDs = 0;

   clearWorkItems();
   clearProcessedWorkItems();
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vFsmSendResponse ( boost::any& /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::vFsmSendResponse()");

   if ( !mpActiveMessageBuffer ) return;

   if ( mErrorCode != DIA_SUCCESS )
   {
      mpActiveMessageBuffer->vSetNegResp(getInstanceOfFactory()->makeNRC(mErrorCode));
      return;
   }

   tU16 countPosResponses = 0;
   std::vector<tU8> responseData;

   DIA_TR_INF("### evaluate processed work items ###");

   std::list<dia_DIDWorkItem*>::iterator iter = mPocessedWorkItems.begin();
   for ( ; iter != mPocessedWorkItems.end(); iter++ )
   {
      dia_DIDWorkItem* pWorkItem = (*iter);
      if ( pWorkItem )
      {
         DIA_TR_INF("### evaluate work items (0x%04x) ###",pWorkItem->mDID);

         tU16 offset = 0;

         if ( pWorkItem->mResponse[offset] != 0x7F )
         {
            DIA_TR_INF("### positive response received ###");
            for(size_t k = offset+3; k < pWorkItem->mResponse.size(); ++k)
            {
               DIA_TR_INF("... msg[%zu] = 0x%02x",k,pWorkItem->mResponse[k]);
            }
            tU8 didHiByte = (tU8) (pWorkItem->mDID >> 8);
            tU8 didLoByte = (tU8) (pWorkItem->mDID);
            DIA_TR_INF("... didHiByte = 0x%02x",didHiByte);
            DIA_TR_INF("... didLoByte = 0x%02x",didLoByte);
            if ( pWorkItem->mResponse[offset]   != 0x62      ) continue;
            DIA_TR_INF("### 1 ###");
            if ( pWorkItem->mResponse[offset+1] != didHiByte ) continue;
            DIA_TR_INF("### 2 ###");
            if ( pWorkItem->mResponse[offset+2] != didLoByte ) continue;
            DIA_TR_INF("### 3 ###");
            responseData.push_back(didHiByte);
            responseData.push_back(didLoByte);
            for ( size_t i=offset+3; i<pWorkItem->mResponse.size(); ++i )
            {
               responseData.push_back(pWorkItem->mResponse[i]);
            }
            ++countPosResponses;
         }
         else if ( pWorkItem->mResponse.size() == 3 )
         {
            DIA_TR_INF("### negative response 0x%02x received ###", pWorkItem->mResponse[offset+2]);
         }
      }
   }

   // we need at least one positive response from all processed DIDs
   if ( countPosResponses )
   {
      for ( tU16 k=0; k<responseData.size(); k++ )
      {
         DIA_TR_INF("... responseData[%02d] = 0x%02x",k,responseData[k]);
      }
      vSendPositiveResponse(mpActiveMessageBuffer,responseData);
   }
   else
   {
      mpActiveMessageBuffer->vSetNegResp(getInstanceOfFactory()->makeNRC(DIA_E_OUT_OF_RANGE));
   }

   mpActiveMessageBuffer->vSendResponse();
   delete mpActiveMessageBuffer;
   mpActiveMessageBuffer = 0;
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vSendPositiveResponse ( dia_MessageBuffer* pMsgBuffer, const std::vector<tU8>& dataVec ) const
{
   dia_tclFnctTrace oTrace("dia_MultipleReadDataByIDHandler::vSendPositiveResponse");

   if ( !pMsgBuffer ) return;

   DIA_TR_INF("dataVec.size() = %zu",dataVec.size());
   pMsgBuffer->vSetPosResp();
   tU16 offset = 1; // 1st byte contains the SID
   pMsgBuffer->vSetDataLength((tU16) (dataVec.size()+1)); // 1 extra byte for the SID
   for ( tU16 i=0; i < dataVec.size(); i++ )
   {
      pMsgBuffer->setDataU8((tU16)(offset+i), dataVec[i]);
   }
   for ( tU16 i=0; i < pMsgBuffer->u16GetDataLength(); i++ )
   {
      //DIA_TR_INF("[%d] = 0x%02X", i, pMsgBuffer->getDataU8(i));
   }
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vFsmSetErrCode_Timeout ( boost::any& /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::vFsmSetErrCode_Timeout()");
   mErrorCode = DIA_E_TIMEOUT;
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vFsmSetNextDID ( boost::any& /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::vFsmSetNextDID()");

   if ( mWorkItems.size() )
   {
      mpActiveWorkItem = mWorkItems.front();
      mWorkItems.pop_front();
   }
   else
   {
      mpActiveWorkItem = 0;
   }
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vFsmStartProcessingTimer ( boost::any& /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::vFsmStartProcessingTimer()");

   mTimer.addTimerListener(this);
   mTimer.s32SetTime(0,0);
   mTimer.s32SetTime(mProcessingTimeout, 0);
}

//-----------------------------------------------------------------------------

void
dia_MultipleReadDataByIDHandler::vFsmStopProcessingTimer ( boost::any& /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::vFsmStopProcessingTimer()");

   mTimer.s32SetTime(0,0);
   mTimer.removeTimerListener(this);
}

//-----------------------------------------------------------------------------

bool
dia_MultipleReadDataByIDHandler::bIsProcessingComplete ( boost::any& /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_MultipleReadDataByIDHandler::bIsProcessingComplete()");
   return ( mNumOfDIDs == mPocessedWorkItems.size() ) ? true : false;
}
