/*!
 * \file       dia::AppRunLevelHandler.cpp
 *
 * \brief      Pre-service handler for all diag services (check if service is allowed in current runlevel, if not defer
 *             processing until required runlevel is reached
 *
 * \details
 *
 * \component  Diagnosis
 *
 * \ingroup    diaServicesProject
 *
 * \copyright  (c) 2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER_CONFIGURATION__
#include "common/framework/engine/dia_EngineServerConfiguration.h"
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include <common/framework/engine/dia_LookupKey.h>
#endif

#ifndef __INCLUDED_DIA_SERVICE_ACCESS_INFO__
#include "common/framework/engine/dia_ServiceAccessInfo.h"
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include "common/framework/application/dia_AppController.h"
#endif

#ifndef __INCLUDED_DIA_APP_RUNLEVEL_HANDLER__
#include <common/framework/protocols/uds/dia_AppRunLevelHandler.h>
#endif

#ifndef __INCLUDED_DIA_CMD_START_TEST__
#include "common/framework/test/dia_CmdStartTest.h"
#endif

#ifndef __INCLUDED_DIA_PREDICATE_RUNLEVEL__
#include <common/framework/application/dia_PredicateRunlevel.h>
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER__
#include <common/framework/application/dia_CommandController.h>
#endif

namespace dia {

//-----------------------------------------------------------------------------

AppRunLevelHandler::AppRunLevelHandler ( void )
{
   ScopeTrace trc("dia::LevelHandler::AppRunLevelHandler");

}

//-----------------------------------------------------------------------------

AppRunLevelHandler::AppRunLevelHandler ( dia_MessageHandler* nextHandler )
   : dia_MessageHandler(nextHandler)
{
   ScopeTrace trc("dia::AppRunLevelHandler::AppRunLevelHandler(dia_MessageHandler*)");
}

//-----------------------------------------------------------------------------

AppRunLevelHandler::~AppRunLevelHandler ( void )
{
}

//-----------------------------------------------------------------------------

tDiaResult
AppRunLevelHandler::handleMessage ( dia_MessageBuffer& msg )
{
   ScopeTrace trc("dia::AppRunLevelHandler::handleMessage(dia_MessageBuffer&)");

   tDiaResult retCode = DIA_FAILED;

   bool bDeferProcessing = false;

   // first we need to get access to te diagnostic service configuration of our server engine
   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine )
   {
      const std::map< tU32, std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> > >& config = pEngine->getServerConfiguration().getSubFunctionConfiguration();

      // extract the service and subfunction identifier of the given message and find the associated configuration
      dia_LookupKey lookupKey(msg);
      tU32 queryKey = (((tU32) lookupKey.getSID()) << 16) | lookupKey.getSubFuncID();
      const std::map< tU32, std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> > >::const_iterator iter1 = config.find(queryKey);
      if ( iter1 != config.end() )
      {
         // get the session specific configuration
         dia_Session* pSession = pEngine->getSessionController()->getActiveSession();
         if ( pSession )
         {
            std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> >::const_iterator iter2 = iter1->second.find(pSession->getID());
            if ( iter2 != iter1->second.end() && iter2->second )
            {
               dia_enRunlevel runLevel = getInstanceOfAppController()->getRunLevel();

               ::boost::shared_ptr<dia_ServiceAccessInfo> pInfoItem = iter2->second;
               if ( !(pInfoItem->checkRunLevel((tU8) runLevel)) )
               {
                  DIA_TR_INF("##### dia::AppRunLevelHandler::handleMessage: EXPECTED RUNLEVEL NOT REACHED. DEFERRING PROCESSING OF MESSAGE! #####");

                  // the expected run level is not yet reached
                  Command* pCmd = new Command;
                  if ( pCmd )
                  {
                     pCmd->addFunctor( new dia_FunctorOneArgNoReturnValue<AppRunLevelHandler,dia_MessageBuffer*>(this,&AppRunLevelHandler::postprocessMessage,&msg) );

                     bDeferProcessing = true;

                     // add predicate
                     dia_enRunlevel expectedRunLevel = (pInfoItem->getRunLevel() < DIA_EN_RUNLEVEL_COUNT) ? ((dia_enRunlevel) pInfoItem->getRunLevel()) : DIA_EN_RUNLEVEL_UNDEFINED;
                     (void) pCmd->addPredicate( new dia_PredicateRunlevel(expectedRunLevel) );
                      // add command to cmdController
                      getInstanceOfCommandController()->addCommand(pCmd);

                      retCode = DIA_SUCCESS;
                      DIA_TR_INF("dia::AppRunLevelHandler::handleMessage DIA_SUCCESS!");
                   }
                   else
                   {
                      DIA_TR_ERR("dia::AppRunLevelHandler::handleMessage TRIGGER FAILED!");
                   }
                   pCmd = 0;  //lint !e423 Warning: Creation of memory leak in assignment to 'pCmd'. --> lifetime is controlled by command controller
               }
               else
               {
                  DIA_TR_INF("dia::AppRunLevelHandler::handleMessage: Required Runlevel Reached!");
               }
            }
            else
            {
               DIA_TR_INF("dia::AppRunLevelHandler::handleMessage: No configuration available for active session!");
            }
         }
         else
         {
            DIA_TR_INF("dia::AppRunLevelHandler::handleMessage: Unable to retrieve active session!");
         }
      }
      else
      {
         DIA_TR_INF("dia::AppRunLevelHandler::handleMessage: Unable to retrieve configuration for given service request (key = 0x%08x)!",queryKey);
      }
   }
   else
   {
      DIA_TR_INF("dia::AppRunLevelHandler::handleMessage: Unable to get access to UDS server engine!");
   }

   if ( !bDeferProcessing )
   {
      //Go on processing, do not wait for vehicle speed.
      retCode = dia_MessageHandler::handleMessage(msg);
   }

   return retCode;
}

void
AppRunLevelHandler::postprocessMessage ( dia_MessageBuffer* msg )
{
   ScopeTrace trc("dia::AppRunLevelHandler::postprocessMessage(dia_MessageBuffer&)");

   if ( msg && (dia_MessageHandler::handleMessage(*msg) == DIA_E_NOT_AVAILABLE) )
   {
      getInstanceOfApplication()->postMessage(new dia_tclDiagSession::tclEventReqRx(msg));
      DIA_TR_INF("--- AppRunLevelHandler::postprocessMessage => Posted as UDS message");
   }
}

}
