/**
 * \file      dia_ServiceTracker.cpp
 *
 * \brief     {insert brief description here}
 *
 * \details   {insert file description here}
 *
 * \author    gib2hi
 * \date      Oct 17, 2013
 *
 * \copyright Robert Bosch Car Multimedia 2013
 */


#ifndef __INCLUDED_DIA_INTERFACE_SERVICE_LISTENER__
#include "common/interfaces/dia_IServiceListener.h"
#endif

#ifndef __INCLUDED_DIA_SERVICE_TRACKER__
#include "common/framework/platform/cca/dia_ServiceTracker.h"
#endif

#ifndef __INCLUDED_DIA_SERVICE_INFO_CCA__
#include "common/framework/platform/cca/dia_ServiceInfoCCA.h"
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

DIA_IMPL_SINGLETON(dia_ServiceTracker)

using namespace dia;

#ifndef __DIA_UNIT_TESTING__

dia_ServiceTracker*
getInstanceOfServiceTracker ( void )
{
   return dia_ServiceTracker::getInstance();
}

#endif

//-----------------------------------------------------------------------------

dia_ServiceTracker::dia_ServiceTracker ( void )
   : mSyncObj("dia_ServiceTracker_LK")
{
   ScopeTrace trc("dia_ServiceTracker::dia_ServiceTracker()");
}

//-----------------------------------------------------------------------------

dia_ServiceTracker::~dia_ServiceTracker ( void )
{}

//-----------------------------------------------------------------------------

void
dia_ServiceTracker::destroy ( void )
{
   LockScope lock(mSyncObj);

   mListenerRep.clear();
   reset();
}

//-----------------------------------------------------------------------------

void
dia_ServiceTracker::reset ( void )
{
   LockScope lock(mSyncObj);

   std::map<tU32,dia_ServiceInfoCCA*>::iterator mapIter = mRep.begin();
   for ( ; mapIter != mRep.end(); mapIter++ ) delete mapIter->second;
   mRep.clear();
}

//-----------------------------------------------------------------------------

void
dia_ServiceTracker::onServiceState ( tU16 serviceID, tU16 serverID, dia_enServiceState state )
{
   ScopeTrace trc("dia_ServiceTracker::onServiceState()");

   LockScope lock(mSyncObj);

   DIA_TR_INF( "# Service State (0x%0x,0x%0x,%s)", serviceID, serverID, ((state == DIA_EN_SERVICE_STATE_AVAILABLE) ? "AVAILABLE" : "NOT_AVAILABLE"));

   tU32 key = (((tU32) serviceID) << 16) | serverID;
   std::map<tU32,dia_ServiceInfoCCA*>::iterator mapIter = mRep.find(key);
   if ( mapIter != mRep.end() )
   {
      dia_ServiceInfoCCA* pInfo = mapIter->second;
      if ( pInfo )
      {
         dia_enServiceState oldState = pInfo->getState();
         pInfo->setState(state);
         DIA_TR_INF(" --> STATE = %d", pInfo->getState());
         std::list<dia_IServiceListener*>::const_iterator listIter = pInfo->getListeners().begin();
         for ( ; listIter != pInfo->getListeners().end(); listIter++ )
         {
            (*listIter)->vOnServiceStateChange(serviceID, static_cast<tU16>(oldState), static_cast<tU16>(pInfo->getState()) );
         }
      }
   }
   else
   {
      DIA_TR_INF( "### Creating new ServiceInfo Element !!!");

      // we have to set up a new info item
      dia_ServiceInfoCCA* pInfo = new dia_ServiceInfoCCA(serviceID,serverID,state);
      if ( pInfo )
      {
         mRep[key] = pInfo;
         DIA_TR_INF(" --> STATE = %d", pInfo->getState());
      }
   }

   std::list<dia_IServiceListener*>::iterator listIter = mListenerRep.begin();
   for ( ; listIter != mListenerRep.end(); listIter++ )
   {
      (*listIter)->vOnServiceState(serviceID,serverID, (state == DIA_EN_SERVICE_STATE_AVAILABLE) ? TRUE : FALSE );
   }
}

//-----------------------------------------------------------------------------

void
dia_ServiceTracker::onServiceRegistration ( tU16 serviceID, tU16 serverID, tU16 regID )
{
   ScopeTrace trc("dia_ServiceTracker::onServiceRegistration()");

   LockScope lock(mSyncObj);

   DIA_TR_INF( "Service Registration (0x%0x,0x%0x,0x%0x)", serviceID, serverID, regID);

   tU32 key = (((tU32) serviceID) << 16) | serverID;
   std::map<tU32,dia_ServiceInfoCCA*>::iterator mapIter = mRep.find(key);
   if ( mapIter != mRep.end() )
   {
      dia_ServiceInfoCCA* pInfo = mapIter->second;
      if ( pInfo )
      {
         pInfo->setRegID(regID);

         std::list<dia_IServiceListener*>::const_iterator listIter = pInfo->getListeners().begin();
         for ( ; listIter != pInfo->getListeners().end(); listIter++ )
         {
            (*listIter)->vOnServiceRegistration(serviceID,serverID,regID);
         }
      }
   }
   else
   {
      DIA_TR_INF( "Creating new ServiceInfo Element !!!");

      // we have to set up a new info item
      dia_ServiceInfoCCA* pInfo = new dia_ServiceInfoCCA(serviceID,serverID);
      if ( pInfo )
      {
         pInfo->setRegID(regID);
         mRep[key] = pInfo;
      }
   }

   std::list<dia_IServiceListener*>::iterator listIter = mListenerRep.begin();
   for ( ; listIter != mListenerRep.end(); listIter++ )
   {
      (*listIter)->vOnServiceRegistration(serviceID,serverID,regID);
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_ServiceTracker::addListener ( dia_IServiceListener* pListener )
{
   ScopeTrace trc("dia_ServiceTracker::addListener()");

   LockScope lock(mSyncObj);

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   std::list<dia_IServiceListener*>::iterator listIter = mListenerRep.begin();
   for ( ; listIter != mListenerRep.end(); listIter++ )
   {
      if ( (*listIter) == pListener )
      {
         // already registered
         retCode = DIA_E_ALREADY_EXISTS;
         break;
      }
   }

   if ( listIter == mListenerRep.end() )
   {
      mListenerRep.push_back(pListener);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_ServiceTracker::addListener ( tU16 serviceID, tU16 serverID, dia_IServiceListener* pListener )
{
   ScopeTrace trc("dia_ServiceTracker::addListener(serviceID,serverID,pListener)");

   LockScope lock(mSyncObj);

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   tU32 key = (((tU32) serviceID) << 16) | serverID;
   std::map<tU32,dia_ServiceInfoCCA*>::iterator itr = mRep.find(key);
   if ( itr != mRep.end() )
   {
      dia_ServiceInfoCCA* pInfo = itr->second;
      if ( pInfo )
      {
         retCode = pInfo->addListener(pListener);
      }
   }
   else
   {
      // we have to set up a new info item
      dia_ServiceInfoCCA* pInfo = new dia_ServiceInfoCCA(serviceID,serverID);
      if ( pInfo )
      {
         mRep[key] = pInfo;
         retCode = pInfo->addListener(pListener);
      }
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_ServiceTracker::removeListener ( const dia_IServiceListener* pListener )
{
   ScopeTrace trc("dia_ServiceTracker::removeListener()");

   LockScope lock(mSyncObj);

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   std::list<dia_IServiceListener*>::iterator listIter = mListenerRep.begin();
   for ( ; listIter != mListenerRep.end(); listIter++ )
   {
      if ( (*listIter) == pListener )
      {
         mListenerRep.erase(listIter);
         retCode = DIA_SUCCESS;
         break;
      }
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_ServiceTracker::removeListener ( tU16 serviceID, tU16 serverID, dia_IServiceListener* pListener )
{
   ScopeTrace trc("dia_ServiceTracker::removeListener(serviceID,serverID,pListener)");

   LockScope lock(mSyncObj);

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   tU32 key = (((tU32) serviceID) << 16) | serverID;
   std::map<tU32,dia_ServiceInfoCCA*>::iterator itr = mRep.find(key);
   if ( itr != mRep.end() )
   {
      if ( itr->second ) retCode = itr->second->removeListener(pListener);
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tU16
dia_ServiceTracker::getServiceRegistration ( tU16 serviceID, tU16 serverID )
{
   ScopeTrace trc("dia_ServiceTracker::getServiceRegistration()");

   LockScope lock(mSyncObj);

   tU16 regID = DIA_C_U16_INVALID_REGISTRATION_ID;

   // create key and try to find the corresponding entry
   tU32 key = (((tU32) serviceID) << 16) | serverID;
   std::map<tU32,dia_ServiceInfoCCA*>::iterator itr = mRep.find(key);
   if ( itr != mRep.end() )
   {
      regID = itr->second->getRegID();
   }

   return regID;
}

//-----------------------------------------------------------------------------

dia_enServiceState
dia_ServiceTracker::getServiceState ( tU16 serviceID, tU16 serverID )
{
   ScopeTrace trc("dia_tclServiceTracker::enGetServiceState()");

   LockScope lock(mSyncObj);

   dia_enServiceState state = DIA_EN_SERVICE_STATE_UNKNOWN;

   // create key and try to find the corresponding entry
   tU32 key = (((tU32) serviceID) << 16) | serverID;
   std::map<tU32,dia_ServiceInfoCCA*>::iterator itr = mRep.find(key);
   if ( itr != mRep.end() )
   {
      state = itr->second->getState();
   }

   DIA_TR_INF( " --> STATE=%d", state);

   return state;
}

void dia_ServiceTracker::printStatistic ( void )
{
   LockScope lock(mSyncObj);

   DIA_TR_INF( "# Print statistic of service registration listeners (item number = %zu)", mRep.size());

   std::map<tU32,dia_ServiceInfoCCA*>::iterator itr = mRep.begin();
   tU32 i = 0;
   while (itr != mRep.end())
   {
      dia_ServiceInfoCCA* pInfo = itr->second;
      if ( pInfo )
      {
         DIA_TR_INF("[%04d] AppID = 0x%04x, ServiceID = 0x%04x, RegID = 0x%04x", i, pInfo->getServerID(), pInfo->getID(), pInfo->getRegID());
      }
      else
      {
         DIA_TR_ERR("Error in mRep. NULL ptr at key = 0x%08x", itr->first );
      }

      ++itr;
      ++i;
   }
}


