/*!
 * \file       dia_ASFComponentSystemAdapterFactory.cpp
 *
 * \brief      ASF component providing a service to create other ASF system adapters
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCorePlatform
 *
 * \copyright  (c) 2019 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#include "common/framework/application/dia_ThreadMonitor.h"
#endif

#ifndef __INCLUDED_DIA_SAFEATURE_SYSTEM_ADAPTER_FACTORY__
#include "common/framework/sysadapters/dia_SAFeatureSystemAdapterFactory.h"
#endif

#ifndef __INCLUDED_DIA_DYNAMIC_LOADER__
#include "common/framework/dynamicloader/dia_DynamicLoader.h"
#endif

#ifndef __INCLUDED_DIA_DYNAMIC_LOADER_MODULE__
#include "common/framework/dynamicloader/dia_DynamicLoaderModule.h"
#endif

#ifndef __INCLUDED_DIA_DYNAMIC_LOADER_PLUGIN_SYSTEMADAPTER_ASF__
#include "common/framework/dynamicloader/dia_DynamicLoaderPluginSystemAdapterASF.h"
#endif

#ifndef __INCLUDED_DIA_DYNAMIC_LOADER_PLUGIN_SYSTEMADAPTER_CCA__
#include "common/framework/dynamicloader/dia_DynamicLoaderPluginSystemAdapterCCA.h"
#endif

#ifndef __INCLUDED_DIA_ASF_COMPONENT_SYSTEM_ADAPTER_FACTORY__
#include "dia_ASFComponentSystemAdapterFactory.h"
#endif

/*
@startuml

== Initialization ==

create dia_ASFComponentSystemAdapterFactory
-> dia_ASFComponentSystemAdapterFactory : new

create DynamicLoaderPluginSystemAdapterASF
dia_ASFComponentSystemAdapterFactory -> DynamicLoaderPluginSystemAdapterASF : new
dia_ASFComponentSystemAdapterFactory -> DynamicLoader : addDynamicLoaderPlugin

...
== Registration ==

-> dia_ASFComponentSystemAdapterFactory : onRegisterModuleRequest
create DynamicLoaderModule
dia_ASFComponentSystemAdapterFactory -> DynamicLoaderModule : new
<- dia_ASFComponentSystemAdapterFactory  : sendRegisterModuleResponse
...
== Creation ==

-> dia_ASFComponentSystemAdapterFactory : onCreateSystemAdapterRequest
dia_ASFComponentSystemAdapterFactory -> DynamicLoaderModule : open

create DynamicLoaderModuleManifestSAxxx
DynamicLoaderModule -> DynamicLoaderModuleManifestSAxxx : new

create DynamicLoaderItemSystemAdapterASF
DynamicLoaderModuleManifestSAxxx -> DynamicLoaderItemSystemAdapterASF : new

DynamicLoaderModuleManifestSAxxx -> DynamicLoaderModuleManifestSAxxx : addItem
DynamicLoaderModuleManifestSAxxx -> DynamicLoader : factoryMethodsLoaderManifest

dia_ASFComponentSystemAdapterFactory -> DynamicLoader : loadModule

loop For all Plugins

DynamicLoader -> DynamicLoaderPluginSystemAdapterASF : getItemUID

DynamicLoader -> DynamicLoaderModuleManifestSAxxx : getItems

loop For all Items

DynamicLoader -> DynamicLoaderPluginSystemAdapterASF : loadItem

DynamicLoaderPluginSystemAdapterASF -> DynamicLoaderItemSystemAdapterASF : createSystemAdapterFeature

end

end

<- dia_ASFComponentSystemAdapterFactory  : sendCreateSystemAdapterResponse

@enduml
*/


using namespace asf::SystemAdapterFactoryService;

namespace asf {

//------------------------------------------------------------------------------

class dia_SystemAdapterServicePluginFactory : public dia_SystemAdapterServicePluginASF< SystemAdapterFactoryServiceProxy >
{
public:

   //! default constructor
   dia_SystemAdapterServicePluginFactory ( tCString name, const boost::shared_ptr<SystemAdapterFactoryServiceProxy>& proxy, const std::shared_ptr<dia_SystemAdapterASF>& adapter )
      : dia_SystemAdapterServicePluginASF(name, proxy, adapter)
   {}

   //! class destructor
   virtual ~dia_SystemAdapterServicePluginFactory ( void )
   {}

   //! ServiceAvailableIF implementation
   void onAvailable ( const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange ) final;
   void onUnavailable ( const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange ) final;
};

//------------------------------------------------------------------------------

void
dia_SystemAdapterServicePluginFactory::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   if (proxy == mProxy)
   {
      switch (stateChange.getCurrentState())
      {
         case ServiceState__Available:
            {
               dia::registerThread("DIA::ASF");

               mCondVar.signal(DIA_C_CONDVAR_EVENT_01,TRUE);

               mIsAvailable = true;

               std::vector<tU32> args;
               args.push_back(mIsAvailable);
               getInstanceOfCommandController()->update(dia_PredicateServiceASF<SystemAdapterFactoryServiceProxy>::getUID(*this), args);
            }
            break;

         default:
            break;
      }
   }
}

//----------------------------------------------------------------------------------------

void
dia_SystemAdapterServicePluginFactory::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   if (proxy == mProxy)
   {
      switch (stateChange.getCurrentState())
      {
         case ServiceState__Disconnected:
         case ServiceState__Suspended:
            {
               mCondVar.signal(DIA_C_CONDVAR_EVENT_01,FALSE);

               mIsAvailable = false;

               std::vector<tU32> args;
               args.push_back(mIsAvailable);
               getInstanceOfCommandController()->update(dia_PredicateServiceASF<SystemAdapterFactoryServiceProxy>::getUID(*this), args);
            }
            break;

         default:
            break;
      }
   }
}

//----------------------------------------------------------------------------------------
dia_ASFComponentSystemAdapterFactory::dia_ASFComponentSystemAdapterFactory()
   : SystemAdapterFactoryServiceStub("diagSystemAdapterFactoryStubPort"),
     mpFeature(nullptr),
     mpDynamicLoader(getInstanceOfDynamicLoader())
{
   dia_tclFnctTrace oTrace("dia_ASFComponentSystemAdapterFactory::dia_ASFComponentSystemAdapterFactory");

   //
   // System Adapter object
   //
   std::shared_ptr< dia_SystemAdapterASF > pSystemAdapter(dia_ASFComponent::createSystemAdapter());
   if (!pSystemAdapter)
   {
      return;
   }

   if (mpDynamicLoader)
   {
      mpDynamicLoader->addDynamicLoaderPlugin( new dia::DynamicLoaderPluginSystemAdapterASF(pSystemAdapter) );

      mpDynamicLoader->addDynamicLoaderPlugin( new dia::DynamicLoaderPluginSystemAdapterCCA("dia_DynamicLoaderPluginSystemAdapterCCA") );
   }

   ::boost::shared_ptr< SystemAdapterFactoryServiceProxy > proxy =
         SystemAdapterFactoryServiceProxy::createProxy("diagSystemAdapterFactoryProxyPort", *pSystemAdapter);

   //
   // Service Plugin
   //
   dia_SystemAdapterServicePluginFactory* pService = new dia_SystemAdapterServicePluginFactory("DIA_SAFactoryService", proxy, pSystemAdapter);

   if (pService && (pSystemAdapter->addServicePlugin(proxy, *pService) == DIA_SUCCESS))
   {
      //
      // Feature Plugin
      //
      mpFeature = dia_ASFComponent::createSystemAdapterFeature< dia_SAFeatureSystemAdapterFactory, SystemAdapterFactoryServiceProxy >(pService);
   }

   dia_SystemAdapterFacade* pSAFacade = createInstanceOfSystemAdapterFacade();
   if ( !pSAFacade || (DIA_SUCCESS !=
         assignInterfaceAndNotifier<dia_ISystemAdapterFactory, dia_ISystemAdapterFactoryListener, dia_ISystemAdapterFactoryNotifier>(mpFeature)))
   {
      DIA_TR_ERR("Error: assignInterfaceAndNotifier failed!");
   }
}

//------------------------------------------------------------------------------

dia_ASFComponentSystemAdapterFactory::~dia_ASFComponentSystemAdapterFactory()
{
}

//------------------------------------------------------------------------------

void
dia_ASFComponentSystemAdapterFactory::onRegisterModuleRequest (const ::boost::shared_ptr< RegisterModuleRequest >& request)
{
   dia_tclFnctTrace oTrace("dia_ASFComponentSystemAdapterFactory::onRegisterModuleRequest");

   std::string moduleName = request->getModuleName();
   std::vector<uint32> iidList = request->getIidList();

   tDiaResult result = DIA_SUCCESS;

   // Check if any IID is already registered
   for (auto iid: iidList)
   {
      if (mDynamicLoaderModules.find(iid) != mDynamicLoaderModules.end())
      {
         result = DIA_E_ALREADY_EXISTS;
         break;
      }
   }

   if (result == DIA_SUCCESS)
   {
      // Register the same Module for all IIDs
      for (auto iid: iidList)
      {
         mDynamicLoaderModules[iid] = std::unique_ptr<dia::DynamicLoaderModule>(new dia::DynamicLoaderModule(moduleName));
      }
   }

   if (result != DIA_SUCCESS)
   {
      DIA_TR_ERR("dia_ASFComponentSystemAdapterFactory::onRegisterModuleRequest: Name='%s', result=0x%x", moduleName.c_str(), result);
   }

   sendRegisterModuleResponse(moduleName, result);
}
 //------------------------------------------------------------------------------

void
dia_ASFComponentSystemAdapterFactory::onCreateSystemAdapterRequest (const ::boost::shared_ptr< CreateSystemAdapterRequest >& request)
{
   dia_tclFnctTrace oTrace("dia_ASFComponentSystemAdapterFactory::onCreateSystemAdapterRequest");

   dia::IID iid = (dia::IID) request->getIid();

   tDiaResult result = DIA_E_NOT_FOUND;

   auto moduleIter = mDynamicLoaderModules.find(iid);
   if (moduleIter != mDynamicLoaderModules.end())
   {
      if ((moduleIter->second->isOpen() == false) &&
            (moduleIter->second->open("/opt/bosch/base/lib") == DIA_SUCCESS))
      {
         result = mpDynamicLoader->loadModule(*(moduleIter->second));
      }
   }

   if (result != DIA_SUCCESS)
   {
      DIA_TR_ERR("dia_ASFComponentSystemAdapterFactory::onCreateSystemAdapterRequest: iid=0x%x, result=0x%x", iid, result);
   }

   sendCreateSystemAdapterResponse (iid, result);
}

//------------------------------------------------------------------------------

} //namespace asf

