/*!
 * \file       dia_PDXParserStrategyTinyXML.cpp
 *
 * \brief      TBD
 *
 * \details    TBD
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreODX
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_PDX_PARSER_STRATEGY_TINYXML__
#include <common/framework/odx/pdx/dia_PDXParserStrategyTinyXML.h>
#endif

#include <common/framework/xml/tinyxml2.h>

#include <common/framework/odx/dia_ODX.h>
#include <common/framework/odx/dia_ODXDataBlock.h>
#include <common/framework/odx/dia_ODXDataBlockRef.h>
#include <common/framework/odx/dia_ODXEcuMem.h>
#include <common/framework/odx/dia_ODXExpectedIdent.h>
#include <common/framework/odx/dia_ODXFlash.h>
#include <common/framework/odx/dia_ODXFlashData.h>
#include <common/framework/odx/dia_ODXFlashDataExtern.h>
#include <common/framework/odx/dia_ODXFlashDataIntern.h>
#include <common/framework/odx/dia_ODXFlashDataRef.h>
#include <common/framework/odx/dia_ODXIdentValue.h>
#include <common/framework/odx/dia_ODXMem.h>
#include <common/framework/odx/dia_ODXOwnIdent.h>
#include <common/framework/odx/dia_ODXSegment.h>
#include <common/framework/odx/dia_ODXSession.h>
#include <common/framework/odx/pdx/dia_PDXABlock.h>
#include <common/framework/odx/pdx/dia_PDXCatalog.h>
#include <common/framework/odx/pdx/dia_PDXFile.h>

#include <sstream>

using namespace tinyxml2;

namespace dia {
namespace odx {

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::parseIndexXML ( const std::string& indexFile )
{
   ScopeTrace oTrace("dia::odx::PDXParserStrategyTinyXML::parseIndexXML");

   tDiaResult parseResult = DIA_FAILED;

   XMLDocument xmlDoc;
   XMLError retCode = xmlDoc.LoadFile(indexFile.c_str());
   if ( retCode == XML_SUCCESS )
   {
      size_t found = indexFile.find_last_of("/\\");
      mIndexFileName = indexFile.substr(found+1);
      mPath          = indexFile.substr(0,found);

      DIA_TR_INF("+---------------------------------------+");
      DIA_TR_INF("| START PARSING OF PDX CONTENT          |");
      DIA_TR_INF("+---------------------------------------+");
      DIA_TR_INF("");
      DIA_TR_INF("  +--> PATH: \"%s\"",mPath.c_str());
      DIA_TR_INF("  +--> FILE: \"%s\"",mIndexFileName.c_str());

      XMLNode* pNode = xmlDoc.FirstChild();
      while ( pNode )
      {
         std::string nodeName = pNode->Value();
         DIA_TR_INF("XML-Node: %s", nodeName.c_str());

         if ( nodeName == "CATALOG" )
         {
            dia::odx::PDXCatalog* pPDXCatalog = new dia::odx::PDXCatalog;
            if ( pPDXCatalog )
            {
               pPDXCatalog->setRootFolder(mPath);
               DIA_TR_INF("  +--> READING CATALOG");
               parseResult = readCatalog(pNode,pPDXCatalog);
               if ( parseResult == DIA_SUCCESS )
               {
                  DIA_TR_INF("  +--> EVALUATING CATALOG");
                  parseResult = evaluateCatalog(*pPDXCatalog);
                  if ( parseResult == DIA_SUCCESS )
                  {
                     mpPDXCatalog = pPDXCatalog;
                  }
                  else
                  {
                     DIA_TR_ERR("### INCONSISTENT PDX CATALOG DETECTED ###");
                     delete pPDXCatalog;
                  }
               }
               else
               {
                  DIA_TR_ERR("### ERROR DETECTED WHILE PARSING PDX INDEX FILE \"%s\"",indexFile.c_str());
               }
            }
            break;
         }
         else
         {
            pNode = pNode->NextSibling();
         }
      }
   }

   return parseResult;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::parseOdxFlashContainer ( const std::string& odxFile )
{
   tDiaResult parseResult = DIA_FAILED;

   XMLDocument xmlDoc;
   XMLError retCode = xmlDoc.LoadFile(odxFile.c_str());
   if ( retCode == XML_SUCCESS )
   {
      size_t found = odxFile.find_last_of("/\\");
      mODXFileName   = odxFile.substr(found+1);
      mPath          = odxFile.substr(0,found);

      DIA_TR_INF("+---------------------------------------+");
      DIA_TR_INF("| START PARSING OF ODX-F CONTENT        |");
      DIA_TR_INF("+---------------------------------------+");
      DIA_TR_INF("");
      DIA_TR_INF("  +--> PATH: \"%s\"",mPath.c_str());
      DIA_TR_INF("  +--> FILE: \"%s\"",mODXFileName.c_str());

      XMLNode* pNode = xmlDoc.FirstChild();
      while ( pNode )
      {
         std::string nodeName = pNode->Value();

         if ( nodeName == "ODX" )
         {
            DIA_TR_INF("XML-Node: %s", nodeName.c_str());

            dia::odx::ODX* pODX = new dia::odx::ODX;
            if ( pODX )
            {
               pODX->setRootFolder(mPath);
               DIA_TR_INF("  +--> READING ODX-F");
               parseResult = readODX(pNode,pODX);
               if ( parseResult == DIA_SUCCESS )
               {
                  DIA_TR_INF("  +--> EVALUATING ODX-F");
                  parseResult = evaluateODXF(*pODX);
                  if ( parseResult == DIA_SUCCESS )
                  {
                     mpODX = pODX;
                  }
                  else
                  {
                     DIA_TR_ERR("### INCONSISTENT ODX-F DETECTED ###");
                     delete pODX;
                  }
               }
               else
               {
                  DIA_TR_ERR("### ERROR DETECTED WHILE PARSING ODX-F FILE \"%s\"",odxFile.c_str());
               }
            }
            break;
         }
         else
         {
            pNode = pNode->NextSibling();
         }
      }
   }

   return parseResult;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::evaluateCatalog ( dia::odx::PDXCatalog& oPDXCatalog )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::evaluateCatalog");

   if ( !oPDXCatalog.hasABlocks() ) return DIA_FAILED;

   return oPDXCatalog.evaluate();
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::evaluateODXF ( dia::odx::ODX& oODX )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::evaluateODXF");

   return oODX.evaluate();
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readCatalog ( XMLNode* pNode, dia::odx::PDXCatalog* pPDXCatalog )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::readCatalog");

   if ( !pNode ) return DIA_FAILED;
   if ( !pPDXCatalog ) return DIA_FAILED;

   (void) readShortName(pNode,pPDXCatalog);
   (void) readLongName(pNode,pPDXCatalog);
   (void) /*tDiaResult*/ readABlocks(pNode,*pPDXCatalog);
   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readODX ( tinyxml2::XMLNode* pNode, dia::odx::ODX* pODX )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::readODX");

   if ( !pNode ) return DIA_FAILED;
   if ( !pODX ) return DIA_FAILED;

   return readFlash(pNode,*pODX);
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readABlocks ( tinyxml2::XMLNode* pElement, dia::odx::PDXCatalog& oPDXCatalog )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::readABlocks");

   if ( !pElement ) return DIA_FAILED;
//   if ( !pPDXCatalog ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pABlocks = pElement->FirstChildElement("ABLOCKS");
   if ( pABlocks )
   {
      XMLElement* pABlock = pABlocks->FirstChildElement("ABLOCK");
      while ( pABlock )
      {
         dia::odx::PDXABlock* pPDXABlock = 0;
         retCode = readABlock(pABlock,&pPDXABlock,oPDXCatalog);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pPDXABlock )
         {
            retCode = oPDXCatalog.addABlock(*pPDXABlock);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD ABLOCK (NO_ABLOCK)");
            }
         }

         pABlock = pABlock->NextSiblingElement("ABLOCK");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readABlock ( tinyxml2::XMLElement* pElement, dia::odx::PDXABlock** ppPDXABlock, dia::odx::PDXCatalog& oPDXCatalog )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::readABlock");

   if ( !pElement ) return DIA_FAILED;
   if ( !ppPDXABlock ) return DIA_FAILED;

   std::string elemName = pElement->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::PDXABlock* pODXABlock = new dia::odx::PDXABlock(oPDXCatalog);
   if ( pODXABlock )
   {
      (void) readShortName(pElement,pODXABlock);
      (void) readLongName(pElement,pODXABlock);
      (void) /*tDiaResult*/ readFiles(pElement,pODXABlock,oPDXCatalog);

//      retCode = oPDXCatalog.addABlock(*pODXABlock);
//      if ( retCode != DIA_SUCCESS )
//      {
//         DIA_TR_ERR("FAILED TO ADD ABLOCK !!");
//      }

      *ppPDXABlock = pODXABlock;
   }

   return (*ppPDXABlock) ? DIA_SUCCESS : DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readFiles ( XMLElement* pElement, dia::odx::PDXABlock* pPDXABlock, dia::odx::PDXCatalog& oPDXCatalog )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::readFiles");

   if ( !pElement ) return DIA_FAILED;
   if ( !pPDXABlock ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pFiles = pElement->FirstChildElement("FILES");
   if ( pFiles )
   {
      std::string elemName = pFiles->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pFile = pFiles->FirstChildElement("FILE");
      while ( pFile )
      {
         dia::odx::PDXFile* pPDXFile = 0;
         retCode = readFile(pFile,&pPDXFile,oPDXCatalog);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pPDXFile )
         {
            retCode = pPDXABlock->addFile(*pPDXFile);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD FILE (NO_FILE)");
            }
         }

         pFile = pFile->NextSiblingElement("FILE");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readFile ( XMLElement* pFileElement, dia::odx::PDXFile** ppPDXFile, dia::odx::PDXCatalog& oPDXCatalog )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::readFile");

   if ( !pFileElement ) return DIA_FAILED;
   if ( !ppPDXFile ) return DIA_FAILED;

   std::string elemName = pFileElement->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::PDXFile* pPDXFile = new dia::odx::PDXFile(oPDXCatalog);
   if ( pPDXFile )
   {
      std::string fileName = pFileElement->GetText();
      DIA_TR_INF("  +--> FILE-NAME = \"%s\"", fileName.c_str());
      pPDXFile->setFileName(std::string(fileName));

      (void) readAttributeCreationDate(pFileElement,pPDXFile);
      (void) readAttributeMimeType(pFileElement,pPDXFile);
      *ppPDXFile = pPDXFile;
   }

   return (*ppPDXFile) ? DIA_SUCCESS : DIA_FAILED;
}

//<FILES>
//  <FILE CREATION-DATE="2016-11-17T14:19:10" MIME-TYPE="text/xml">VID_1B234X23456.odx-f</FILE>
//  <FILE CREATION-DATE="2016-11-17T14:21:52" MIME-TYPE="application/octet-stream">AUDIO_1.BIN</FILE>
//</FILES>

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readAttributeCreationDate ( XMLElement* pFileElement, dia::odx::PDXFile* pPDXFile )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::readAttributeCreationDate");

   if ( !pFileElement ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   const char* attrID = pFileElement->Attribute("CREATION-DATE");
   if ( attrID ) {
      DIA_TR_INF("XML-ATTRIBUTE:  CREATION-DATE = \"%s\"", attrID);
      pPDXFile->setCreationDate(std::string(attrID));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readAttributeMimeType ( XMLElement* pFileElement, dia::odx::PDXFile* pPDXFile )
{
   dia::ScopeTrace oTrc("PDXParserStrategyTinyXML::readAttributeMimeType");

   if ( !pFileElement ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   const char* attrID = pFileElement->Attribute("MIME-TYPE");
   if ( attrID ) {
      DIA_TR_INF("XML-ATTRIBUTE:  MIME-TYPE = \"%s\"", attrID);
      pPDXFile->setMimeType(std::string(attrID));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readShortName ( XMLNode* pNode, dia::odx::NamedObject* pODXNamedObject )
{
   if ( !pNode ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   XMLElement* pShortName = pNode->FirstChildElement("SHORT-NAME");
   if ( pShortName )
   {
      std::string shortName = pShortName->GetText();
      DIA_TR_INF("XML-ELEMENT: SHORT-NAME = \"%s\"", shortName.c_str());
      pODXNamedObject->setShortName(std::string(shortName));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readLongName ( XMLNode* pNode, dia::odx::NamedObject* pODXNamedObject )
{
   if ( !pNode ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   XMLElement* pLongName = pNode->FirstChildElement("LONG-NAME");
   if ( pLongName )
   {
      std::string longName = pLongName->GetText();
      DIA_TR_INF("XML-ELEMENT: LONG-NAME = \"%s\"", longName.c_str());
      pODXNamedObject->setLongName(std::string(longName));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readFlash ( XMLNode* pNode, dia::odx::ODX& oODX )
{
   if ( !pNode ) return DIA_FAILED;
// if ( !pODX ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pFlash = pNode->FirstChildElement("FLASH");
   if ( pFlash )
   {
      std::string elemName = pFlash->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      dia::odx::Flash* pODXFlash = new dia::odx::Flash(oODX);
      if ( pODXFlash )
      {
         (void) readAttributeID(pFlash,pODXFlash);
         (void) readAttributeOID(pFlash,pODXFlash);
         (void) readShortName(pFlash,pODXFlash);
         (void) readLongName(pFlash,pODXFlash);
         (void) /*tDiaResult*/ readEcuMems(pFlash,pODXFlash,oODX);

         oODX.setFlash(*pODXFlash);
         retCode = DIA_SUCCESS;
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readEcuMems ( XMLElement* pElement, dia::odx::Flash* pODXFlash, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXFlash ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pEcuMems = pElement->FirstChildElement("ECU-MEMS");
   if ( pEcuMems )
   {
      XMLElement* pEcuMem = pEcuMems->FirstChildElement("ECU-MEM");
      while ( pEcuMem )
      {
         dia::odx::EcuMem* pODXEcuMem = 0;
         retCode = readEcuMem(pEcuMem,&pODXEcuMem,oODX);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pODXEcuMem )
         {
            retCode = pODXFlash->addEcuMem(*pODXEcuMem);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD DATABLOCK \"%s\"", pODXEcuMem->getID().c_str());
               }
            }

         pEcuMem = pEcuMem->NextSiblingElement("ECU-MEM");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readEcuMem ( XMLElement* pEcuMem, dia::odx::EcuMem** ppODXEcuMem, dia::odx::ODX& oODX )
{
   if ( !pEcuMem ) return DIA_FAILED;
   if ( !ppODXEcuMem ) return DIA_FAILED;

   std::string elemName = pEcuMem->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::EcuMem* pODXEcuMem = new dia::odx::EcuMem(oODX);
   if ( pODXEcuMem )
   {
      (void) readAttributeID(pEcuMem,pODXEcuMem);
      (void) readAttributeOID(pEcuMem,pODXEcuMem);
      (void) readShortName(pEcuMem,pODXEcuMem);
      (void) readLongName(pEcuMem,pODXEcuMem);
      (void) /*tDiaResult*/ readMem(pEcuMem,pODXEcuMem,oODX);
      *ppODXEcuMem = pODXEcuMem;
   }

   return (*ppODXEcuMem) ? DIA_SUCCESS : DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readMem ( XMLElement* pEcuMem, dia::odx::EcuMem* pODXEcuMem, dia::odx::ODX& oODX )
{
   if ( !pEcuMem ) return DIA_FAILED;
   if ( !pODXEcuMem ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pMem = pEcuMem->FirstChildElement("MEM");
   if ( pMem )
   {
      std::string elemName = pMem->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      dia::odx::Mem* pODXMem = new dia::odx::Mem(oODX);
      if ( pODXMem )
      {
         (void) /*tDiaResult*/ readSessions(pMem,pODXMem,oODX);
         (void) /*tDiaResult*/ readDataBlocks(pMem,pODXMem,oODX);
         (void) /*tDiaResult*/ readFlashDatas(pMem,pODXMem,oODX);
         retCode = pODXEcuMem->setMem(*pODXMem);
         if ( retCode != DIA_SUCCESS )
         {
            DIA_TR_ERR("FAILED TO ADD MEM");
         }
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readDataBlockRefs ( XMLElement* pElement, dia::odx::Session* pODXSession, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXSession ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pDataBlockRefs = pElement->FirstChildElement("DATABLOCK-REFS");
   if ( pDataBlockRefs )
   {
      std::string elemName = pDataBlockRefs->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pDataBlockRef = pDataBlockRefs->FirstChildElement("DATABLOCK-REF");
      while (  pDataBlockRef )
      {
         dia::odx::DataBlockRef* pODXDataBlockRef = 0;
         retCode = readDataBlockRef(pDataBlockRef,&pODXDataBlockRef,*pODXSession,oODX);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pODXDataBlockRef )
         {
            retCode = pODXSession->addDataBlockRef(*pODXDataBlockRef);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD DATABLOCK \"%s\"", pODXDataBlockRef->getRefID().c_str());
            }
         }

         pDataBlockRef = pDataBlockRef->NextSiblingElement("DATABLOCK-REF");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readDataBlockRef ( XMLElement* pDataBlockRef, dia::odx::DataBlockRef** ppODXDataBlockRef, dia::odx::Session& oSession, dia::odx::ODX& oODX )
{

   if ( !pDataBlockRef ) return DIA_FAILED;
   if ( !ppODXDataBlockRef ) return DIA_FAILED;

   std::string elemName = pDataBlockRef->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::DataBlockRef* pODXDataBlockRef = new dia::odx::DataBlockRef(oSession,oODX);
   if ( pODXDataBlockRef )
   {
      const char* attrIDREF = pDataBlockRef->Attribute("ID-REF");
      if ( attrIDREF ) {
         DIA_TR_INF("XML-ATTRIBUTE:  ID-REF = \"%s\"", attrIDREF);
         pODXDataBlockRef->setIdRef(std::string(attrIDREF));
      }
      *ppODXDataBlockRef = pODXDataBlockRef;
   }

   return (*ppODXDataBlockRef) ? DIA_SUCCESS : DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readAttributeID ( XMLElement* pElement, dia::odx::NamedObject* pODXNamedObject )
{
   if ( !pElement ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   const char* attrID = pElement->Attribute("ID");
   if ( attrID ) {
      DIA_TR_INF("XML-ATTRIBUTE:  ID = \"%s\"", attrID);
      pODXNamedObject->setID(std::string(attrID));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readAttributeOID ( XMLElement* pElement, dia::odx::NamedObject* pODXNamedObject )
{
   if ( !pElement ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   const char* attrOID = pElement->Attribute("OID");
   if ( attrOID ) {
      DIA_TR_INF("XML-ATTRIBUTE: OID = \"%s\"", attrOID);
      pODXNamedObject->setID(std::string(attrOID));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readShortName ( XMLElement* pElement, dia::odx::NamedObject* pODXNamedObject )
{
   if ( !pElement ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   XMLElement* pShortName = pElement->FirstChildElement("SHORT-NAME");
   if ( pShortName )
   {
      std::string shortName = pShortName->GetText();
      DIA_TR_INF("XML-ELEMENT: SHORT-NAME = \"%s\"", shortName.c_str());
      pODXNamedObject->setShortName(std::string(shortName));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readLongName ( XMLElement* pElement, dia::odx::NamedObject* pODXNamedObject )
{
   if ( !pElement ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   XMLElement* pLongName = pElement->FirstChildElement("LONG-NAME");
   if ( pLongName )
   {
      std::string longName = pLongName->GetText();
      DIA_TR_INF("XML-ELEMENT: LONG-NAME = \"%s\"", longName.c_str());
      pODXNamedObject->setLongName(std::string(longName));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readDataFormat ( XMLElement* pElement, dia::odx::FlashData* pODXFlashData, dia::odx::ODX& /*oODX*/ )
{
   if ( !pElement ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   XMLElement* pDataFormat = pElement->FirstChildElement("DATAFORMAT");
   if ( pDataFormat )
   {
      std::string elemName = pDataFormat->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      const char* attrSelection = pDataFormat->Attribute("SELECTION");
      if ( attrSelection ) {
         DIA_TR_INF("XML-ATTRIBUTE: SELECTION = \"%s\"", attrSelection);
         pODXFlashData->setDataFormat(attrSelection);
         retCode = DIA_SUCCESS;
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readSessions ( XMLElement* pElement, dia::odx::Mem* pODXMem, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXMem ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pSessions = pElement->FirstChildElement("SESSIONS");
   if ( pSessions )
   {
      std::string elemName = pSessions->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pSession = pSessions->FirstChildElement("SESSION");
      while ( pSession )
      {
         dia::odx::Session* pODXSession = 0;
         retCode = readSession(pSession,&pODXSession,*pODXMem,oODX);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pODXSession )
         {
            retCode = pODXMem->addSession(*pODXSession);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD DATABLOCK \"%s\"", pODXSession->getID().c_str());
            }
         }

         pSession = pSession->NextSiblingElement("SESSION");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readSession ( XMLElement* pSession, dia::odx::Session** ppODXSession, dia::odx::Mem& oODXMem, dia::odx::ODX& oODX )
{
   if ( !pSession ) return DIA_FAILED;
   if ( !ppODXSession ) return DIA_FAILED;

   std::string elemName = pSession->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::Session* pODXSession = new dia::odx::Session(oODXMem,oODX);
   if ( pODXSession )
   {
      (void) readAttributeID(pSession,pODXSession);
      (void) readAttributeOID(pSession,pODXSession);
      (void) readShortName(pSession,pODXSession);
      (void) readLongName(pSession,pODXSession);
      (void) /*tDiaResult*/ readExpectedIdents(pSession,pODXSession,oODX);
      (void) /*tDiaResult*/ readDataBlockRefs(pSession,pODXSession,oODX);
      *ppODXSession = pODXSession;
   }

   return (*ppODXSession) ? DIA_SUCCESS : DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readDataBlocks ( XMLElement* pElement, dia::odx::Mem* pODXMem, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXMem ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pDataBlocks = pElement->FirstChildElement("DATABLOCKS");
   if ( pDataBlocks )
   {
      std::string elemName = pDataBlocks->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pDataBlock = pDataBlocks->FirstChildElement("DATABLOCK");
      while ( pDataBlock )
      {
         dia::odx::DataBlock* pODXDataBlock = 0;
         retCode = readDataBlock(pDataBlock,&pODXDataBlock,*pODXMem,oODX);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pODXDataBlock )
         {
            retCode = pODXMem->addDataBlock(*pODXDataBlock);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD DATABLOCK \"%s\"", pODXDataBlock->getID().c_str());
            }
         }

         pDataBlock = pDataBlock->NextSiblingElement("DATABLOCK");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readDataBlock ( XMLElement* pDataBlock, dia::odx::DataBlock** ppODXDataBlock, dia::odx::Mem& oMem, dia::odx::ODX& oODX )
{
   if ( !pDataBlock ) return DIA_FAILED;
   if ( !ppODXDataBlock ) return DIA_FAILED;

   std::string elemName = pDataBlock->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::DataBlock* pODXDataBlock = new dia::odx::DataBlock(oMem,oODX);
   if ( pODXDataBlock )
   {
      (void) readAttributeID(pDataBlock,pODXDataBlock);
      (void) readAttributeOID(pDataBlock,pODXDataBlock);
      (void) readShortName(pDataBlock,pODXDataBlock);
      (void) readLongName(pDataBlock,pODXDataBlock);
      (void) readLogicalBlockIndex(pDataBlock,pODXDataBlock);
      (void) readAttributeDataBlockType(*pDataBlock,*pODXDataBlock);
      (void) /*tDiaResult*/ readOwnIdents(pDataBlock,pODXDataBlock,oODX);
      (void) /*tDiaResult*/ readFlashDataRef(*pDataBlock,*pODXDataBlock,oODX);
      (void) /*tDiaResult*/ readSegments(pDataBlock,pODXDataBlock,oODX);
      *ppODXDataBlock = pODXDataBlock;
   }

   return (*ppODXDataBlock) ? DIA_SUCCESS : DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readAttributeDataBlockType ( XMLElement& oElement, dia::odx::DataBlock& oDataBlock )
{
   tDiaResult retCode = DIA_E_NOT_FOUND;

   const char* attrID = oElement.Attribute("TYPE");
   if ( attrID ) {
      DIA_TR_INF("XML-ATTRIBUTE:  TYPE = \"%s\"", attrID);
      oDataBlock.setType(std::string(attrID));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readSegments ( XMLElement* pElement, dia::odx::DataBlock* pODXDataBlock, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXDataBlock ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pSegments = pElement->FirstChildElement("SEGMENTS");
   if ( pSegments )
   {
      std::string elemName = pSegments->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pSegment = pSegments->FirstChildElement("SEGMENT");
      while ( pSegment )
      {
         dia::odx::Segment* pODXSegment = 0;
         retCode = readSegment(pSegment,&pODXSegment,*pODXDataBlock,oODX);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pODXDataBlock )
         {
            retCode = pODXDataBlock->addSegment(*pODXSegment);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED ADD SEGMENT \"%s\"", ((pODXSegment) ? pODXSegment->getID().c_str() : "NO_SEGMENT"));
            }
         }

         pSegment = pSegment->NextSiblingElement("SEGMENT");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readSegment ( XMLElement* pSegment, dia::odx::Segment** ppODXSegment, dia::odx::DataBlock& oODXDataBlock, dia::odx::ODX& oODX )
{
   if ( !pSegment ) return DIA_FAILED;
   if ( !ppODXSegment ) return DIA_FAILED;

   std::string elemName = pSegment->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::Segment* pODXSegment = new dia::odx::Segment(oODXDataBlock, oODX);
   if ( pODXSegment )
   {
      (void) readAttributeID(pSegment,pODXSegment);
      (void) readAttributeOID(pSegment,pODXSegment);
      (void) readShortName(pSegment,pODXSegment);
      (void) readLongName(pSegment,pODXSegment);
      (void) readSourceStartAddress(pSegment,*pODXSegment,oODX);
      (void) readUncompressedSize(pSegment,*pODXSegment,oODX);
      *ppODXSegment = pODXSegment;
   }

   return (*ppODXSegment) ? DIA_SUCCESS : DIA_FAILED;
}

tDiaResult
PDXParserStrategyTinyXML::readSourceStartAddress ( XMLElement* pSegment, dia::odx::Segment& oODXSegment, dia::odx::ODX& /*oODX*/ )
{
   if ( !pSegment ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   XMLElement* pStartAddr = pSegment->FirstChildElement("SOURCE-START-ADDRESS");
   if ( pStartAddr )
   {
      std::string strStartAddr = pStartAddr->GetText();
      DIA_TR_INF("XML-ELEMENT: SOURCE-START-ADDRESS = \"%s\" (DATABLOCK-TYPE = \"%s\")", strStartAddr.c_str(),oODXSegment.getParent()->getType().c_str());
      if ( (strStartAddr == "FFFFFFFF") && (oODXSegment.getParent()->getType() == DIA_ODX_DATABLOCK_TYPE_WDBI) )
      {
         DIA_TR_INF("  +--> SOURCE ADDRESS NOT REQUIRED (\"WDBI_TYPE\"-DATABLOCK)");
         oODXSegment.setSourceStartAddress(0);
         retCode = DIA_SUCCESS;
      }
      else
      {
         std::istringstream streamData(strStartAddr.c_str());
         int startAddr = 0;
         streamData >> std::hex >> startAddr;
         DIA_TR_INF("  +--> decValue = %d, hexValue = %x",startAddr,startAddr);
         oODXSegment.setSourceStartAddress((tU32) startAddr);
         retCode = DIA_SUCCESS;
      }
   }

   return retCode;
}

tDiaResult
PDXParserStrategyTinyXML::readUncompressedSize ( XMLElement* pSegment, dia::odx::Segment& oODXSegment, dia::odx::ODX& /*oODX*/ )
{
   if ( !pSegment ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   XMLElement* pUncompressedSize = pSegment->FirstChildElement("UNCOMPRESSED-SIZE");
   if ( pUncompressedSize )
   {
      std::string strUncompressedSize = pUncompressedSize->GetText();
      std::istringstream streamData(strUncompressedSize.c_str());

      DIA_TR_INF("XML-ELEMENT: UNCOMPRESSED-SIZE = \"%s\"", strUncompressedSize.c_str());
      int uncompressedSize = 0;
      streamData >> uncompressedSize;
      DIA_TR_INF("  +--> decValue = %d, hexValue = %x",uncompressedSize,uncompressedSize);
      oODXSegment.setUncompressedSize((tU32) uncompressedSize);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readOwnIdents ( XMLElement* pElement, dia::odx::DataBlock* pODXDataBlock, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXDataBlock ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pOwnIdents = pElement->FirstChildElement("OWN-IDENTS");
   if ( pOwnIdents )
   {
      std::string elemName = pOwnIdents->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pOwnIdent = pOwnIdents->FirstChildElement("OWN-IDENT");
      while ( pOwnIdent )
      {
         dia::odx::OwnIdent* pODXOwnIdent = 0;
         retCode = readOwnIdent(pOwnIdent,&pODXOwnIdent,oODX);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pODXOwnIdent )
         {
            retCode = pODXDataBlock->addOwnIdent(*pODXOwnIdent);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD OWN-IDENT \"%s\"", pODXOwnIdent->getID().c_str());
            }
         }

         pOwnIdent = pOwnIdent->NextSiblingElement("OWN-IDENT");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readLogicalBlockIndex ( XMLElement* pElement, dia::odx::DataBlock* pODXDataBlock )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXDataBlock ) return DIA_FAILED;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   XMLElement* pLogicalBlockIndex = pElement->FirstChildElement("LOGICAL-BLOCK-INDEX");
   if ( pLogicalBlockIndex )
   {
      std::string logBlockIndx = pLogicalBlockIndex->GetText();
      DIA_TR_INF("LOGICAL-BLOCK-INDEX: \"%s\"", logBlockIndx.c_str());
      pODXDataBlock->setLogicalBlockIndex(std::string(logBlockIndx));
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readOwnIdent ( XMLElement* pOwnIdent, dia::odx::OwnIdent** ppODXOwnIdent, dia::odx::ODX& oODX )
{
   if ( !pOwnIdent ) return DIA_FAILED;
   if ( !ppODXOwnIdent ) return DIA_FAILED;

   std::string elemName = pOwnIdent->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::OwnIdent* pODXOwnIdent = new dia::odx::OwnIdent(oODX);
   if ( pODXOwnIdent )
   {
      (void) readAttributeID(pOwnIdent,pODXOwnIdent);
      (void) readAttributeOID(pOwnIdent,pODXOwnIdent);
      (void) readShortName(pOwnIdent,pODXOwnIdent);
      (void) readLongName(pOwnIdent,pODXOwnIdent);
      (void) /*tDiaResult*/ readIdentValues(pOwnIdent,pODXOwnIdent,oODX);
      (*ppODXOwnIdent) = pODXOwnIdent;
   }

   return (*ppODXOwnIdent) ? DIA_SUCCESS : DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readExpectedIdents ( XMLElement* pElement, dia::odx::Session* pODXSession, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXSession ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pExpectedIdents = pElement->FirstChildElement("EXPECTED-IDENTS");
   if ( pExpectedIdents )
   {
      std::string elemName = pExpectedIdents->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pExpectedIdent = pExpectedIdents->FirstChildElement("EXPECTED-IDENT");
      while (  pExpectedIdent )
      {
         dia::odx::ExpectedIdent* pODXExpectedIdent = 0;
         retCode = readExpectedIdent(pExpectedIdent,&pODXExpectedIdent,oODX);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pODXExpectedIdent )
         {
            retCode = pODXSession->addExpectedIdent(*pODXExpectedIdent);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD EXPECTED-IDENT \"%s\"", pODXExpectedIdent->getID().c_str());
            }
         }

         pExpectedIdent = pExpectedIdent->NextSiblingElement("EXPECTED-IDENT");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readExpectedIdent ( XMLElement* pExpectedIdent, dia::odx::ExpectedIdent** ppODXExpectedIdent, dia::odx::ODX& oODX )
{
   if ( !pExpectedIdent ) return DIA_FAILED;
   if ( !ppODXExpectedIdent ) return DIA_FAILED;

   std::string elemName = pExpectedIdent->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::ExpectedIdent* pODXExpectedIdent = new dia::odx::ExpectedIdent(oODX);
   if ( pODXExpectedIdent )
   {
      (void) readAttributeID(pExpectedIdent,pODXExpectedIdent);
      (void) readAttributeOID(pExpectedIdent,pODXExpectedIdent);
      (void) readShortName(pExpectedIdent,pODXExpectedIdent);
      (void) readLongName(pExpectedIdent,pODXExpectedIdent);
      (void) /*tDiaResult*/ readIdentValues(pExpectedIdent,pODXExpectedIdent,oODX);
      (*ppODXExpectedIdent) = pODXExpectedIdent;
   }

   return (*ppODXExpectedIdent) ? DIA_SUCCESS : DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readIdentValues ( XMLElement* pElement, dia::odx::ExpectedIdent* pODXExpectedIdent, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXExpectedIdent ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pIdentValues = pElement->FirstChildElement("IDENT-VALUES");
   if ( pIdentValues )
   {
      std::string elemName = pIdentValues->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pIdentValue = pIdentValues->FirstChildElement("IDENT-VALUE");
      while ( pIdentValue )
      {
         dia::odx::IdentValue* pODXIdentValue = new dia::odx::IdentValue(oODX);
         if(pODXIdentValue == 0) break;
         retCode = readIdentValue(pIdentValue,&pODXIdentValue,oODX);
         if ( retCode != DIA_SUCCESS ) 
         {
            delete pODXIdentValue;
            pODXIdentValue = 0;
            break;
         }
         if ( pODXIdentValue )
         {
            retCode = pODXExpectedIdent->addIdentValue(*pODXIdentValue);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD IDENT-VALUE");
            }
         }

         pIdentValue = pIdentValue->NextSiblingElement("IDENT-VALUE");
      }
   }
   else
   {
      XMLElement* pIdentValue = pElement->FirstChildElement("IDENT-VALUE");
      while ( pIdentValue )
      {
         dia::odx::IdentValue* pODXIdentValue = new dia::odx::IdentValue(oODX);
         if(pODXIdentValue == 0) break;
         retCode = readIdentValue(pIdentValue,&pODXIdentValue,oODX);
         if ( retCode != DIA_SUCCESS ) 
         {
            delete pODXIdentValue;
            pODXIdentValue = 0;
            break;
         }
         if ( pODXIdentValue )
         {
            retCode = pODXExpectedIdent->addIdentValue(*pODXIdentValue);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD IDENT-VALUE");
            }
         }

         pIdentValue = pIdentValue->NextSiblingElement("IDENT-VALUE");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readIdentValues ( XMLElement* pElement, dia::odx::OwnIdent* pODXOwnIdent, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXOwnIdent ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pIdentValues = pElement->FirstChildElement("IDENT-VALUES");
   if ( pIdentValues )
   {
      std::string elemName = pIdentValues->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pIdentValue = pIdentValues->FirstChildElement("IDENT-VALUE");
      while (  pIdentValue )
      {
         dia::odx::IdentValue* pODXIdentValue = new dia::odx::IdentValue(oODX);
         if(pODXIdentValue == 0) break;
         retCode = readIdentValue(pIdentValue,&pODXIdentValue,oODX);
         if ( retCode != DIA_SUCCESS ) 
         {
            delete pODXIdentValue;
            pODXIdentValue = 0;
            break;
         }
         if ( pODXIdentValue )
         {
            retCode = pODXOwnIdent->addIdentValue(*pODXIdentValue);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD IDENT-VALUE");
            }
         }

         pIdentValue = pIdentValue->NextSiblingElement("IDENT-VALUE");
      }
   }
   else
   {
      XMLElement* pIdentValue = pElement->FirstChildElement("IDENT-VALUE");
      while (  pIdentValue )
      {
         dia::odx::IdentValue* pODXIdentValue = new dia::odx::IdentValue(oODX);
         if(pODXIdentValue == 0) break;
         retCode = readIdentValue(pIdentValue,&pODXIdentValue,oODX);
         if ( retCode != DIA_SUCCESS ) 
         {
            delete pODXIdentValue;
            pODXIdentValue = 0;
            break;
         }
         if ( pODXIdentValue )
         {
            retCode = pODXOwnIdent->addIdentValue(*pODXIdentValue);
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("FAILED TO ADD IDENT-VALUE");
            }
         }

         pIdentValue = pIdentValue->NextSiblingElement("IDENT-VALUE");
      }

   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readIdentValue ( XMLElement* pIdentValue, dia::odx::IdentValue** ppODXIdentValue, dia::odx::ODX& /*oODX*/ )
{
   if ( !pIdentValue ) return DIA_FAILED;
   if ( !ppODXIdentValue ) return DIA_FAILED;

   std::string elemName = pIdentValue->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   dia::odx::IdentValue* pODXIdentValue = *ppODXIdentValue;
   if ( pODXIdentValue )
   {
      std::string elemName = pIdentValue->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      const char* attrType = pIdentValue->Attribute("TYPE");
      if ( attrType ) {
         DIA_TR_INF("XML-ATTRIBUTE:  TYPE = \"%s\"", attrType);
         pODXIdentValue->setType(std::string(attrType));
      }

      std::string textValue = pIdentValue->GetText();
      DIA_TR_INF("XML-ELEMENT: IDENT-VALUE = \"%s\"", textValue.c_str());
      pODXIdentValue->setTextValue(textValue);

      (*ppODXIdentValue) = pODXIdentValue;
   }

   return (*ppODXIdentValue) ? DIA_SUCCESS : DIA_FAILED;
}

//const std::string& getTextValue ( void ) const { return mTextValue; }
//void setTextValue ( const std::string& textValue ) { mTextValue = textValue; }
//dia::UID getHashCodeOfTextValue ( void ) { return dia_getHashCodeFromString(mTextValue); }
//
//const std::string& getType ( void ) const { return mType; }
//void setType ( const std::string& type ) { mType = type; }
//dia::UID getHashCodeOfType ( void ) { return dia_getHashCodeFromString(mType); }
//

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readFlashDataRef ( XMLElement& oElement, dia::odx::DataBlock& oODXDataBlock, dia::odx::ODX& oODX )
{
//   if ( !pDataBlock ) return DIA_FAILED;
//   if ( !pODXDataBlock ) return DIA_FAILED;

   dia::odx::FlashDataRef* pODXFlashDataRef = 0;
   XMLElement* pFlashDataRef = oElement.FirstChildElement("FLASHDATA-REF");
   if (  pFlashDataRef )
   {
      std::string elemName = pFlashDataRef->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      pODXFlashDataRef = new dia::odx::FlashDataRef(oODXDataBlock,oODX);
      if ( pODXFlashDataRef )
      {
         const char* attrIDREF = pFlashDataRef->Attribute("ID-REF");
         if ( attrIDREF ) {
            DIA_TR_INF("XML-ATTRIBUTE:  ID-REF = \"%s\"", attrIDREF);
            pODXFlashDataRef->setIdRef(std::string(attrIDREF));
         }

         tDiaResult retCode = oODXDataBlock.setFlashDataRef(*pODXFlashDataRef);
         if ( retCode != DIA_SUCCESS )
         {
            DIA_TR_ERR("FAILED TO ADD DATABLOCK-REF: ID-REF = \"%s\"", pODXFlashDataRef->getIdRef().c_str());
         }
      }
   }

   return (pODXFlashDataRef) ? DIA_SUCCESS : DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readFlashDatas ( XMLElement* pElement, dia::odx::Mem* pODXMem, dia::odx::ODX& oODX )
{
   if ( !pElement ) return DIA_FAILED;
   if ( !pODXMem ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   XMLElement* pFlashDatas = pElement->FirstChildElement("FLASHDATAS");
   if ( pFlashDatas )
   {
      std::string elemName = pFlashDatas->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      XMLElement* pFlashData = pFlashDatas->FirstChildElement("FLASHDATA");
      while ( pFlashData )
      {
         dia::odx::FlashData* pODXFlashData = 0;
         retCode = readFlashData(pFlashData,&pODXFlashData,oODX);
         if ( retCode != DIA_SUCCESS ) break;
         if ( pODXFlashData )
         {
            (void) pODXMem->addFlashData(*pODXFlashData);
         }

         pFlashData = pFlashData->NextSiblingElement("FLASHDATA");
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readFlashData ( XMLElement* pFlashData, dia::odx::FlashData** ppODXFlashData, dia::odx::ODX& oODX )
{
   if ( !pFlashData ) return DIA_FAILED;
   if ( !ppODXFlashData ) return DIA_FAILED;

   std::string elemName = pFlashData->Value();
   DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

   const char* attrXsiType = pFlashData->Attribute("xsi:type");
   if ( attrXsiType )
   {
      std::string xsiType(attrXsiType);
      if ( xsiType == std::string("EXTERN-FLASHDATA") )
      {
         (void) readFlashDataExtern(pFlashData,ppODXFlashData,oODX);
      }
      else if ( xsiType == std::string("INTERN-FLASHDATA") )
      {
         (void) readFlashDataIntern(pFlashData,ppODXFlashData,oODX);
      }
      else
      {
         DIA_TR_ERR("UNKNOWN FLASH DATA TYPE !!");
         return DIA_FAILED;
      }
   }

   dia::odx::FlashData* pODXFlashData = (*ppODXFlashData);

   if ( pODXFlashData )
   {
      (void) readAttributeID(pFlashData,pODXFlashData);
      (void) readAttributeOID(pFlashData,pODXFlashData);
      (void) readShortName(pFlashData,pODXFlashData);
      (void) readLongName(pFlashData,pODXFlashData);
      (void) readDataFormat(pFlashData,pODXFlashData,oODX);
   }

   return (*ppODXFlashData) ? DIA_SUCCESS : DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readFlashDataExtern ( XMLElement* pFlashData, dia::odx::FlashData** ppODXFlashData, dia::odx::ODX& oODX )
{
   if ( !pFlashData ) return DIA_FAILED;
   if ( !ppODXFlashData ) return DIA_FAILED;

   dia::odx::FlashDataExtern* pODXFlashDataExtern = new dia::odx::FlashDataExtern(oODX);

   XMLElement* pDataFile = pFlashData->FirstChildElement("DATAFILE");
   if ( pDataFile )
   {
      std::string elemName = pDataFile->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      std::string fileName = pDataFile->GetText();
      DIA_TR_INF("XML-ELEMENT: DATAFILE = \"%s\"", fileName.c_str());
      pODXFlashDataExtern->setDataFileName(fileName);

      const char* attrLateBoundDataFile = pDataFile->Attribute("LATEBOUND-DATAFILE");
      if ( attrLateBoundDataFile ) {
         DIA_TR_INF("XML-ATTRIBUTE: LATEBOUND-DATAFILE = \"%s\"", attrLateBoundDataFile);
         pODXFlashDataExtern->setLateBound((std::string(attrLateBoundDataFile) == "true") ? true : false);
      }
   }

   *ppODXFlashData = pODXFlashDataExtern;
   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
PDXParserStrategyTinyXML::readFlashDataIntern ( XMLElement* pFlashData, dia::odx::FlashData** ppODXFlashData, dia::odx::ODX& oODX )
{
   if ( !pFlashData ) return DIA_FAILED;
   if ( !ppODXFlashData ) return DIA_FAILED;

   dia::odx::FlashDataIntern* pODXFlashDataIntern = new dia::odx::FlashDataIntern(oODX);

   XMLElement* pData = pFlashData->FirstChildElement("DATA");
   if ( pData )
   {
      std::string elemName = pData->Value();
      DIA_TR_INF("XML-ELEMENT: %s", elemName.c_str());

      std::string data = pData->GetText();
      std::istringstream dataStream(data.c_str());
////      unsigned foundWhiteSpace = data.find_first_of(" \t");
////      unsigned found = odxFile.find_last_of("/\\");
////      mODXFileName   = odxFile.substr(found+1);
////      mPath          = odxFile.substr(0,found);
//
////      bool errorDetected =
//      DIA_TR_INF("XML-ELEMENT: DATA = \"%s\"", data.c_str());
//      std::vector<tU8> dataVec;
////      for ( tU16 i=0; i < data.size(); i++ )
//      while ( !dataStream.eof() )
//      {
////         if ( data[i] != ' ' ) dataVec.push_back(data[i]);
////         tU8 dataByte = 0;
//         int dataByte = 0;
//         dataStream.g >> setprecision(2) /*setw(2)*/ >> hex >> dataByte;
//         DIA_TR_INF("##### DATA-BYTE = 0x%02x", dataByte);
//         dataVec.push_back((tU8) dataByte);
//      }

      std::vector<tU8> dataVec;
      const char* pStringByte = data.c_str();
      unsigned int byteValue = 0;
      while ( *pStringByte != 0 )
      {
          sscanf(pStringByte, "%2X", &byteValue);
//          DIA_TR_INF("##### DATA-BYTE = 0x%02x", byteValue);
          dataVec.push_back((tU8) byteValue);
          pStringByte += 2;
      }

      pODXFlashDataIntern->setData(dataVec);
   }

   *ppODXFlashData = pODXFlashDataIntern;
   return DIA_SUCCESS;
}

} // namespace odx
} // namespace dia

