#include <string.h>
#include "dia_SelftestControllerFSM.h"

#define MAX_TRIGGER_EVENTS  20

namespace dia_SelftestControllerFSM
{

///////////////////////////////////////////////////////////////////////////////
//
// Forward Declarations
//
///////////////////////////////////////////////////////////////////////////////

class Fsm;
class FsmSuperState;

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM states.
//
///////////////////////////////////////////////////////////////////////////////

class FsmState
{
public:
    //! name of the FSM's state
    virtual const char* getStateName ( void ) const = 0;

    //! resetting state internals
    virtual void reset ( void ) {}

    // event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequest ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStop ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evConfirmedSuccess ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evConfirmedFailure ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStartFailed ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evResultReady ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequestResultFailed ( FsmBody& b, FsmBehaviour& s, void* pArg );

    //! default implementation for the Entry action
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! default implementation for the Exit action
    virtual void exit  ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! method used to propagate the event to the superstate
    virtual const FsmState* propagate ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! return the parent state
    virtual FsmSuperState* getParent ( void );


protected:
    //! default constructor is protected to prevent usage of this constructor
    FsmState ( void );
    //! constructor for a substate of the given superstate
    FsmState ( FsmSuperState& parent );
    //! class destructor
    virtual ~FsmState ( void );

    //! pointer to the superstate
    FsmSuperState* mpParent;
};

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM superstates.
//
///////////////////////////////////////////////////////////////////////////////

class FsmSuperState
    : public FsmState
{
public:
    //! class constructor for a superstate without parent
    FsmSuperState ( FsmState& defState, bool histEnable=false );
    //! class constructor for a superstate with parent
    FsmSuperState ( FsmSuperState& parent, FsmState& defState, bool histEnable=false );

    //! set a savepoint for the history
    virtual void savepoint ( FsmBody& b );
    //! default action
    virtual void defaultAction ( FsmBody& b, FsmBehaviour& s, void* pArg );

    //! resetting state internals
    virtual void reset ( void ) { _history = 0; }

    //! method used to propagate the event to the superstate
    virtual const FsmState* propagate ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    //! default constructor is protected to prevent usage of this constructor
    FsmSuperState ( void );

    //! pointer to the default substate
    FsmState* _default;
    //! pointer to the history substate
    const FsmState* _history;
    //! flag that indicates if history is enabled or not
    bool _historyEnable;
};

///////////////////////////////////////////////////////////////////////////////
//
// Class: FinishedState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class FinishedState
    : public FsmState
{
public:
    //! default constructor
    FinishedState ( void );

    FinishedState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequest ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStop ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStartFailed ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequestResultFailed ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* FinishedState::mStateName = "Finished";

///////////////////////////////////////////////////////////////////////////////
//
// Class: NotStartedState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class NotStartedState
    : public FsmState
{
public:
    //! default constructor
    NotStartedState ( void );

    NotStartedState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequest ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStop ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* NotStartedState::mStateName = "NotStarted";

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestedState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class RequestedState
    : public FsmState
{
public:
    //! default constructor
    RequestedState ( void );

    RequestedState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequest ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evConfirmedSuccess ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evConfirmedFailure ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStartFailed ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* RequestedState::mStateName = "Requested";

///////////////////////////////////////////////////////////////////////////////
//
// Class: StartedState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class StartedState
    : public FsmState
{
public:
    //! default constructor
    StartedState ( void );

    StartedState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequest ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStop ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStartFailed ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evResultReady ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* StartedState::mStateName = "Started";

///////////////////////////////////////////////////////////////////////////////
//
// Class: TopLevelState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class TopLevelState
    : public FsmSuperState
{
public:
    //! default constructor
    TopLevelState ( void );

    TopLevelState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* TopLevelState::mStateName = "TopLevel";

///////////////////////////////////////////////////////////////////////////////
//
// Class: FsmBody
//
// This is the finite state machine's body class 'FsmBody'.
//
///////////////////////////////////////////////////////////////////////////////

class FsmBody
{
    struct FsmDeferredEvent
    {
          FsmDeferredEvent() : mEvent(evUnknown), mArg(0) {}
          FsmEvent mEvent;
          void*    mArg;
    };

public:
    //! class constructor
    FsmBody ( void );

    //! connect the FSM body with its frontend
    void connect ( Fsm* frontend );

    //! initialize the FSM: set initial state and execute initial actions
    void init ( void );

    //! this method is used to stimulate the FSM with events
    void acceptEvent ( FsmEvent event, void* pArg );

    //! return the name of the active state
    const char* getStateName ( void ) const;

    //! push trigger events to the event queue
    bool addEvent ( const FsmEvent& event, void* arg  );
    //! get event from the event queue
    bool getEvent ( FsmEvent& event, void** arg  );
    //! get event from the event queue
    bool isQueueEmpty ( void ) const;

    // set the FSM's internal state
    void setState ( FsmState& state, void* pArg );
    // return the active state object
    const FsmState* getState ( void ) const;

    //! static state objects
    FinishedState oFinishedState;
    NotStartedState oNotStartedState;
    RequestedState oRequestedState;
    StartedState oStartedState;
    TopLevelState oTopLevelState;

    //! frontend for this body
    Fsm* mpFrontend;
    //! link to the active state object
    FsmState* mpState;

    //! busy indicator flag
    bool mBusyFlag;

    //! queue of trigger events
    FsmEvent mEventQueue[MAX_TRIGGER_EVENTS];
    //! queue of arguments for trigger events
    void*    mParamQueue[MAX_TRIGGER_EVENTS];
    //! index of the queue's front element
    unsigned int mQueueHead;
    //! index of the queue's back element
    unsigned int mQueueTail;
    //! number of queued events
    unsigned int mQueueSize;
};

///////////////////////////////////////////////////////////////////////////////
//
// Default implementation for all events.
//
///////////////////////////////////////////////////////////////////////////////

void
FsmState::evError ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evError ***
}

void
FsmState::evRequest ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evRequest ***
}

void
FsmState::evRequestResult ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evRequestResult ***
}

void
FsmState::evStop ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evStop ***
}

void
FsmState::evConfirmedSuccess ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evConfirmedSuccess ***
}

void
FsmState::evConfirmedFailure ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evConfirmedFailure ***
}

void
FsmState::evStartFailed ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evStartFailed ***
}

void
FsmState::evResultReady ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evResultReady ***
}

void
FsmState::evTimeout ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTimeout ***
}

void
FsmState::evRequestResultFailed ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evRequestResultFailed ***
}

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM states components.
//
///////////////////////////////////////////////////////////////////////////////

FsmState::FsmState ( void )
    : mpParent(0)
{}

FsmState::FsmState ( FsmSuperState& parent )
    : mpParent(&parent)
{}

FsmState::~FsmState ( void )
{
    // make lint happy
    mpParent = 0;
}

void
FsmState::entry ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    //... empty as default ...
}

void
FsmState::exit ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    //... empty as default ...
}

const FsmState*
FsmState::propagate ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    entry(b,s,pArg);
    return this;
}

FsmSuperState*
FsmState::getParent ( void )
{
    return (mpParent) ? mpParent : 0;
}

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM superstates.
//
///////////////////////////////////////////////////////////////////////////////

FsmSuperState::FsmSuperState ( FsmState& defState, bool histEnable )
    : _default(&defState),
      _history(0),
      _historyEnable(histEnable)
{}

FsmSuperState::FsmSuperState ( FsmSuperState& parent, FsmState& defState, bool histEnable )
    : FsmState(parent),
      _default(&defState),
      _history(0),
      _historyEnable(histEnable)
{}

FsmSuperState::FsmSuperState ( void )
    : _default(0),
      _history(0),
      _historyEnable(false)
{}

const FsmState*
FsmSuperState::propagate ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    const FsmState* pState = 0;

    // execute the entry actions for this superstate
    entry(b,s,pArg);

    if ( _history && _historyEnable )
    {
        pState = const_cast<FsmState*>(_history)->propagate(b,s,pArg);
    }
    else
    {
        defaultAction(b,s,pArg);
        pState = _default->propagate(b,s,pArg);
    }

    return pState;
}

void
FsmSuperState::savepoint ( FsmBody& b )
{
    if ( _historyEnable )
    {
        _history = b.getState();
    }
}

void
FsmSuperState::defaultAction ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /*pArg*/ )
{
    //... empty as default ...
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: FinishedState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

FinishedState::FinishedState ( FsmBody& b )
    : FsmState(b.oTopLevelState)
{}

void
FinishedState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vFsmSelftestDone(pArg);
}

void
FinishedState::evError ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmHandleError(pArg);
    // this is an internal transition
}

void
FinishedState::evRequest ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oRequestedState,pArg);
}

void
FinishedState::evRequestResult ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmRequestResult(pArg);
    // this is an internal transition
}

void
FinishedState::evStop ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmStopDone(pArg);
    // this is an internal transition
}

void
FinishedState::evStartFailed ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotStartedState,pArg);
}

void
FinishedState::evRequestResultFailed ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmRequestResultFailed(pArg);
    // this is an internal transition
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: NotStartedState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

NotStartedState::NotStartedState ( FsmBody& b )
    : FsmState(b.oTopLevelState)
{}

void
NotStartedState::evError ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmHandleError(pArg);
    // this is an internal transition
}

void
NotStartedState::evRequest ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oRequestedState,pArg);
}

void
NotStartedState::evRequestResult ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmRequestResultFailed(pArg);
    // this is an internal transition
}

void
NotStartedState::evStop ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmStopFailed(pArg);
    // this is an internal transition
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestedState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

RequestedState::RequestedState ( FsmBody& b )
    : FsmState(b.oTopLevelState)
{}

void
RequestedState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vFsmRequestSelftest(pArg);
}

void
RequestedState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmHandleError(pArg);
    b.setState(b.oNotStartedState,pArg);
}

void
RequestedState::evRequest ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oRequestedState,pArg);
}

void
RequestedState::evConfirmedSuccess ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmRequestSuccess(pArg);
    b.setState(b.oStartedState,pArg);
}

void
RequestedState::evConfirmedFailure ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmRequestFailure(pArg);
    b.setState(b.oFinishedState,pArg);
}

void
RequestedState::evStartFailed ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmStartFailed(pArg);
    b.setState(b.oNotStartedState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: StartedState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

StartedState::StartedState ( FsmBody& b )
    : FsmState(b.oTopLevelState)
{}

void
StartedState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmHandleError(pArg);
    b.setState(b.oNotStartedState,pArg);
}

void
StartedState::evRequest ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oRequestedState,pArg);
}

void
StartedState::evRequestResult ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmRequestResultInProgress(pArg);
    // this is an internal transition
}

void
StartedState::evStop ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmStopSelftest(pArg);
    s.vFsmStopDone(pArg);
    b.setState(b.oNotStartedState,pArg);
}

void
StartedState::evStartFailed ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmStartFailed(pArg);
    b.setState(b.oNotStartedState,pArg);
}

void
StartedState::evResultReady ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oFinishedState,pArg);
}

void
StartedState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmSelftestFailed(pArg);
    b.setState(b.oNotStartedState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: TopLevelState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

TopLevelState::TopLevelState ( FsmBody& b )
    : FsmSuperState(b.oNotStartedState, false)
{}

void
TopLevelState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmHandleError(pArg);
    b.setState(b.oNotStartedState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// This is the finite state machine class implementation
//
///////////////////////////////////////////////////////////////////////////////

Fsm::Fsm ( void )
    : mpBehaviour(0),
      mpBody(0)
{
}

Fsm::Fsm ( FsmBehaviour* server )
    : mpBehaviour(server),
      mpBody(0)
{
    mpBody = new FsmBody();
}

Fsm::~Fsm ( void )
{
//    _BP_TRY_BEGIN
//    {
        if ( mpBody )
        {
            delete mpBody;
            mpBody = 0;
        }
        mpBehaviour = 0;
//    }
//    _BP_CATCH_ALL
//    {
//       DIA_TR_ERR("EXCEPTION CAUGHT: Fsm::~Fsm !!!");
//       DIA_ASSERT_ALWAYS();
//    }
//    _BP_CATCH_END
}

void
Fsm::init ( void )
{

    if ( mpBody )
    {
        // connect the body with its frontend
        mpBody->connect(this);
        {
            mpBody->init();
        }
    }
}

const char*
Fsm::getStateName (void) const
{
    return ( mpBody ) ? mpBody->getStateName() : 0;
}

void
Fsm::acceptEvent ( FsmEvent event, void* pArg )
{
    if ( mpBody )
    {
        // forward the event to the FSM body (handle-body pattern)
        mpBody->acceptEvent(event,pArg);
    }
}
///////////////////////////////////////////////////////////////////////////////
//
// This is the finite state machine class implementation
//
///////////////////////////////////////////////////////////////////////////////

FsmBody::FsmBody ( void ) :
      oFinishedState(*this),
      oNotStartedState(*this),
      oRequestedState(*this),
      oStartedState(*this),
      oTopLevelState(*this),
      mpFrontend(0),
      mpState(0),
      mBusyFlag(false),
      mQueueHead(0),
      mQueueTail(0),
      mQueueSize(0)
{
    (void) ::memset(mEventQueue,0,sizeof(FsmEvent)*MAX_TRIGGER_EVENTS);
    (void) ::memset(mParamQueue,0,sizeof(void*)*MAX_TRIGGER_EVENTS);
}

void
FsmBody::init ( void )
{
    oFinishedState.reset();
    oNotStartedState.reset();
    oRequestedState.reset();
    oStartedState.reset();
    oTopLevelState.reset();

    setState(oTopLevelState,0);
}

void
FsmBody::connect ( Fsm* frontend )
{
    mpFrontend = frontend;
}

const char*
FsmBody::getStateName ( void ) const
{
    return ( mpState ) ? mpState->getStateName() : 0;
}

bool
FsmBody::addEvent ( const FsmEvent& event, void* param )
{
    if ( mQueueSize < MAX_TRIGGER_EVENTS )
    {
        mQueueSize++;
        mEventQueue[mQueueTail] = event;
        mParamQueue[mQueueTail] = param;
        mQueueTail++;
        if ( mQueueTail == MAX_TRIGGER_EVENTS )
        {
            mQueueTail = 0;
        }
        return true;
    }
    return false;
}

bool
FsmBody::getEvent ( FsmEvent& event, void** param )
{
    if ( !mQueueSize || !param )
    {
        return false;
    }
    mQueueSize--;
    event  = mEventQueue[mQueueHead];
    *param = mParamQueue[mQueueHead];
    mQueueHead++;
    if ( mQueueHead == MAX_TRIGGER_EVENTS )
    {
        mQueueHead = 0;
    }
    return true;
}

bool
FsmBody::isQueueEmpty ( void ) const
{
    return ( mQueueSize ) ? false : true;
}

void
FsmBody::acceptEvent ( FsmEvent event, void* pArg )
{
    (void) addEvent(event,pArg);

    if ( mBusyFlag )
    {
        return;
    }

    mBusyFlag = true;
    while ( !isQueueEmpty() )
    {
        FsmEvent nextEvent = evUnknown;
        void*    nextArg   = 0;
        if ( !getEvent(nextEvent,&nextArg) )
        {
            break;
        }

        switch (nextEvent)
        {
        case evError:
            mpState->evError(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evRequest:
            mpState->evRequest(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evRequestResult:
            mpState->evRequestResult(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evStop:
            mpState->evStop(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evConfirmedSuccess:
            mpState->evConfirmedSuccess(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evConfirmedFailure:
            mpState->evConfirmedFailure(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evStartFailed:
            mpState->evStartFailed(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evResultReady:
            mpState->evResultReady(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTimeout:
            mpState->evTimeout(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evRequestResultFailed:
            mpState->evRequestResultFailed(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        default:
            //... unknown event ...
            break;
        } //lint !e788: not all items used within defaulted switch
    }
    mBusyFlag = false;
}

void
FsmBody::setState ( FsmState& state, void* pArg )
{
    if ( mpFrontend )
    {
        mpState = const_cast<FsmState*>(state.propagate(*this,*(mpFrontend->mpBehaviour),pArg));
    }
}

const FsmState*
FsmBody::getState ( void ) const
{
    return mpState;
}


} // namespace
