#include <string.h>
#include "dia_BluetoothModeFSM.h"

#define MAX_TRIGGER_EVENTS  20

namespace dia_BluetoothModeFSM
{

///////////////////////////////////////////////////////////////////////////////
//
// Forward Declarations
//
///////////////////////////////////////////////////////////////////////////////

class Fsm;
class FsmSuperState;

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM states.
//
///////////////////////////////////////////////////////////////////////////////

class FsmState
{
public:
    //! name of the FSM's state
    virtual const char* getStateName ( void ) const = 0;

    //! resetting state internals
    virtual void reset ( void ) {}

    // event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evNormalMode ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStartSetMode ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestModeLoopback ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestModeLoopbackLocal ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestModeI2S ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneStopped ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evLoopbackStopped ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evProdModeActive ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evDUTModeActive ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evLoopbackStarted ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneStarted ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evModeRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evModeStatus ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneStatus ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evLoopbackRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );

    //! default implementation for the Entry action
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! default implementation for the Exit action
    virtual void exit  ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! method used to propagate the event to the superstate
    virtual const FsmState* propagate ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! return the parent state
    virtual FsmSuperState* getParent ( void );


protected:
    //! default constructor is protected to prevent usage of this constructor
    FsmState ( void );
    //! constructor for a substate of the given superstate
    FsmState ( FsmSuperState& parent );
    //! class destructor
    virtual ~FsmState ( void );

    //! pointer to the superstate
    FsmSuperState* mpParent;
};

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM superstates.
//
///////////////////////////////////////////////////////////////////////////////

class FsmSuperState
    : public FsmState
{
public:
    //! class constructor for a superstate without parent
    FsmSuperState ( FsmState& defState, bool histEnable=false );
    //! class constructor for a superstate with parent
    FsmSuperState ( FsmSuperState& parent, FsmState& defState, bool histEnable=false );

    //! set a savepoint for the history
    virtual void savepoint ( FsmBody& b );
    //! default action
    virtual void defaultAction ( FsmBody& b, FsmBehaviour& s, void* pArg );

    //! resetting state internals
    virtual void reset ( void ) { _history = 0; }

    //! method used to propagate the event to the superstate
    virtual const FsmState* propagate ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    //! default constructor is protected to prevent usage of this constructor
    FsmSuperState ( void );

    //! pointer to the default substate
    FsmState* _default;
    //! pointer to the history substate
    const FsmState* _history;
    //! flag that indicates if history is enabled or not
    bool _historyEnable;
};

///////////////////////////////////////////////////////////////////////////////
//
// Class: BTModeControlState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class BTModeControlState
    : public FsmSuperState
{
public:
    //! default constructor
    BTModeControlState ( void );

    BTModeControlState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! set a savepoint for the history
    virtual void savepoint ( FsmBody& b );

protected:
    static const char* mStateName;
};

const char* BTModeControlState::mStateName = "BTModeControl";

///////////////////////////////////////////////////////////////////////////////
//
// Class: BTModeToplevelState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class BTModeToplevelState
    : public FsmSuperState
{
public:
    //! default constructor
    BTModeToplevelState ( void );

    BTModeToplevelState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* BTModeToplevelState::mStateName = "BTModeToplevel";

///////////////////////////////////////////////////////////////////////////////
//
// Class: IdleState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class IdleState
    : public FsmState
{
public:
    //! default constructor
    IdleState ( void );

    IdleState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evNormalMode ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestModeLoopback ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestModeLoopbackLocal ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestModeI2S ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneStopped ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evLoopbackStopped ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* IdleState::mStateName = "Idle";

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestI2SModeState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class RequestI2SModeState
    : public FsmState
{
public:
    //! default constructor
    RequestI2SModeState ( void );

    RequestI2SModeState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStartSetMode ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evDUTModeActive ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneStarted ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* RequestI2SModeState::mStateName = "RequestI2SMode";

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestLocalLoopbackModeState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class RequestLocalLoopbackModeState
    : public FsmState
{
public:
    //! default constructor
    RequestLocalLoopbackModeState ( void );

    RequestLocalLoopbackModeState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStartSetMode ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evDUTModeActive ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evLoopbackStarted ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* RequestLocalLoopbackModeState::mStateName = "RequestLocalLoopbackMode";

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestLoopbackModeState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class RequestLoopbackModeState
    : public FsmState
{
public:
    //! default constructor
    RequestLoopbackModeState ( void );

    RequestLoopbackModeState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStartSetMode ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evProdModeActive ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* RequestLoopbackModeState::mStateName = "RequestLoopbackMode";

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestNormalModeState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class RequestNormalModeState
    : public FsmState
{
public:
    //! default constructor
    RequestNormalModeState ( void );

    RequestNormalModeState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evStartSetMode ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* RequestNormalModeState::mStateName = "RequestNormalMode";

///////////////////////////////////////////////////////////////////////////////
//
// Class: SetDUTModeState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class SetDUTModeState
    : public FsmState
{
public:
    //! default constructor
    SetDUTModeState ( void );

    SetDUTModeState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evModeRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evModeStatus ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* SetDUTModeState::mStateName = "SetDUTMode";

///////////////////////////////////////////////////////////////////////////////
//
// Class: SetNormalModeState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class SetNormalModeState
    : public FsmState
{
public:
    //! default constructor
    SetNormalModeState ( void );

    SetNormalModeState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evModeRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evModeStatus ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* SetNormalModeState::mStateName = "SetNormalMode";

///////////////////////////////////////////////////////////////////////////////
//
// Class: SetProductionModeState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class SetProductionModeState
    : public FsmState
{
public:
    //! default constructor
    SetProductionModeState ( void );

    SetProductionModeState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evModeRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evModeStatus ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* SetProductionModeState::mStateName = "SetProductionMode";

///////////////////////////////////////////////////////////////////////////////
//
// Class: StartI2STestToneState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class StartI2STestToneState
    : public FsmState
{
public:
    //! default constructor
    StartI2STestToneState ( void );

    StartI2STestToneState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneStatus ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* StartI2STestToneState::mStateName = "StartI2STestTone";

///////////////////////////////////////////////////////////////////////////////
//
// Class: StartLoopbackState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class StartLoopbackState
    : public FsmState
{
public:
    //! default constructor
    StartLoopbackState ( void );

    StartLoopbackState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evLoopbackRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* StartLoopbackState::mStateName = "StartLoopback";

///////////////////////////////////////////////////////////////////////////////
//
// Class: StopI2STestToneState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class StopI2STestToneState
    : public FsmState
{
public:
    //! default constructor
    StopI2STestToneState ( void );

    StopI2STestToneState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTestToneStatus ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* StopI2STestToneState::mStateName = "StopI2STestTone";

///////////////////////////////////////////////////////////////////////////////
//
// Class: StopLoopbackState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class StopLoopbackState
    : public FsmState
{
public:
    //! default constructor
    StopLoopbackState ( void );

    StopLoopbackState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evError ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evLoopbackRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* StopLoopbackState::mStateName = "StopLoopback";

///////////////////////////////////////////////////////////////////////////////
//
// Class: FsmBody
//
// This is the finite state machine's body class 'FsmBody'.
//
///////////////////////////////////////////////////////////////////////////////

class FsmBody
{
    struct FsmDeferredEvent
    {
          FsmDeferredEvent() : mEvent(evUnknown), mArg(0) {}
          FsmEvent mEvent;
          void*    mArg;
    };

public:
    //! class constructor
    FsmBody ( void );

    //! connect the FSM body with its frontend
    void connect ( Fsm* frontend );

    //! initialize the FSM: set initial state and execute initial actions
    void init ( void );

    //! this method is used to stimulate the FSM with events
    void acceptEvent ( FsmEvent event, void* pArg );

    //! return the name of the active state
    const char* getStateName ( void ) const;

    //! push trigger events to the event queue
    bool addEvent ( const FsmEvent& event, void* arg  );
    //! get event from the event queue
    bool getEvent ( FsmEvent& event, void** arg  );
    //! get event from the event queue
    bool isQueueEmpty ( void ) const;

    // set the FSM's internal state
    void setState ( FsmState& state, void* pArg );
    // return the active state object
    const FsmState* getState ( void ) const;

    //! static state objects
    BTModeControlState oBTModeControlState;
    BTModeToplevelState oBTModeToplevelState;
    IdleState oIdleState;
    RequestI2SModeState oRequestI2SModeState;
    RequestLocalLoopbackModeState oRequestLocalLoopbackModeState;
    RequestLoopbackModeState oRequestLoopbackModeState;
    RequestNormalModeState oRequestNormalModeState;
    SetDUTModeState oSetDUTModeState;
    SetNormalModeState oSetNormalModeState;
    SetProductionModeState oSetProductionModeState;
    StartI2STestToneState oStartI2STestToneState;
    StartLoopbackState oStartLoopbackState;
    StopI2STestToneState oStopI2STestToneState;
    StopLoopbackState oStopLoopbackState;

    //! frontend for this body
    Fsm* mpFrontend;
    //! link to the active state object
    FsmState* mpState;

    //! busy indicator flag
    bool mBusyFlag;

    //! queue of trigger events
    FsmEvent mEventQueue[MAX_TRIGGER_EVENTS];
    //! queue of arguments for trigger events
    void*    mParamQueue[MAX_TRIGGER_EVENTS];
    //! index of the queue's front element
    unsigned int mQueueHead;
    //! index of the queue's back element
    unsigned int mQueueTail;
    //! number of queued events
    unsigned int mQueueSize;
};

///////////////////////////////////////////////////////////////////////////////
//
// Default implementation for all events.
//
///////////////////////////////////////////////////////////////////////////////

void
FsmState::evError ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evError ***
}

void
FsmState::evTimeout ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTimeout ***
}

void
FsmState::evNormalMode ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evNormalMode ***
}

void
FsmState::evStartSetMode ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evStartSetMode ***
}

void
FsmState::evTestModeLoopback ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTestModeLoopback ***
}

void
FsmState::evTestModeLoopbackLocal ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTestModeLoopbackLocal ***
}

void
FsmState::evTestModeI2S ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTestModeI2S ***
}

void
FsmState::evTestToneStopped ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTestToneStopped ***
}

void
FsmState::evLoopbackStopped ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evLoopbackStopped ***
}

void
FsmState::evProdModeActive ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evProdModeActive ***
}

void
FsmState::evDUTModeActive ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evDUTModeActive ***
}

void
FsmState::evLoopbackStarted ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evLoopbackStarted ***
}

void
FsmState::evTestToneStarted ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTestToneStarted ***
}

void
FsmState::evModeRequestResult ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evModeRequestResult ***
}

void
FsmState::evModeStatus ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evModeStatus ***
}

void
FsmState::evTestToneRequestResult ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTestToneRequestResult ***
}

void
FsmState::evTestToneStatus ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTestToneStatus ***
}

void
FsmState::evLoopbackRequestResult ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evLoopbackRequestResult ***
}

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM states components.
//
///////////////////////////////////////////////////////////////////////////////

FsmState::FsmState ( void )
    : mpParent(0)
{}

FsmState::FsmState ( FsmSuperState& parent )
    : mpParent(&parent)
{}

FsmState::~FsmState ( void )
{
    // make lint happy
    mpParent = 0;
}

void
FsmState::entry ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    //... empty as default ...
}

void
FsmState::exit ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    //... empty as default ...
}

const FsmState*
FsmState::propagate ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    entry(b,s,pArg);
    return this;
}

FsmSuperState*
FsmState::getParent ( void )
{
    return (mpParent) ? mpParent : 0;
}

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM superstates.
//
///////////////////////////////////////////////////////////////////////////////

FsmSuperState::FsmSuperState ( FsmState& defState, bool histEnable )
    : _default(&defState),
      _history(0),
      _historyEnable(histEnable)
{}

FsmSuperState::FsmSuperState ( FsmSuperState& parent, FsmState& defState, bool histEnable )
    : FsmState(parent),
      _default(&defState),
      _history(0),
      _historyEnable(histEnable)
{}

FsmSuperState::FsmSuperState ( void )
    : _default(0),
      _history(0),
      _historyEnable(false)
{}

const FsmState*
FsmSuperState::propagate ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    const FsmState* pState = 0;

    // execute the entry actions for this superstate
    entry(b,s,pArg);

    if ( _history && _historyEnable )
    {
        pState = const_cast<FsmState*>(_history)->propagate(b,s,pArg);
    }
    else
    {
        defaultAction(b,s,pArg);
        pState = _default->propagate(b,s,pArg);
    }

    return pState;
}

void
FsmSuperState::savepoint ( FsmBody& b )
{
    if ( _historyEnable )
    {
        _history = b.getState();
    }
}

void
FsmSuperState::defaultAction ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /*pArg*/ )
{
    //... empty as default ...
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: BTModeControlState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

BTModeControlState::BTModeControlState ( FsmBody& b )
    : FsmSuperState(b.oBTModeToplevelState, b.oIdleState, true)
{}

void
BTModeControlState::savepoint ( FsmBody& b )
{
    if ( _historyEnable )
    {
        _history = b.getState();
        while (  _history && (_history->getStateName() != b.oIdleState.getStateName()) && (_history->getStateName() != b.oRequestLoopbackModeState.getStateName()) && (_history->getStateName() != b.oRequestLocalLoopbackModeState.getStateName()) && (_history->getStateName() != b.oRequestI2SModeState.getStateName()) && (_history->getStateName() != b.oRequestNormalModeState.getStateName()) && (_history->getStateName() != b.oSetNormalModeState.getStateName()) )
        {
            _history = const_cast<FsmState*>(_history)->getParent();
        }
    }
}

void
BTModeControlState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    savepoint(b);
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
BTModeControlState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    savepoint(b);
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: BTModeToplevelState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

BTModeToplevelState::BTModeToplevelState ( FsmBody& b )
    : FsmSuperState(b.oBTModeControlState, false)
{}

void
BTModeToplevelState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
BTModeToplevelState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: IdleState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

IdleState::IdleState ( FsmBody& b )
    : FsmState(b.oBTModeControlState)
{}

void
IdleState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
IdleState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
IdleState::evNormalMode ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsI2STestToneActive(pArg) )
    {
        // set savepoint in superstate with enabled history
        b.oBTModeControlState.savepoint(b);

        // transition Actions
        s.vSetRequestedModeNormal(pArg);
        b.setState(b.oStopI2STestToneState,pArg);
    }
    else if ( s.bIsLoopbackActive(pArg) )
    {
        // set savepoint in superstate with enabled history
        b.oBTModeControlState.savepoint(b);

        // transition Actions
        s.vSetRequestedModeNormal(pArg);
        b.setState(b.oStopLoopbackState,pArg);
    }
    else
    {
    // transition Actions
    s.vSetRequestedModeNormal(pArg);
    // trigger events
    FsmEvent event = dia_BluetoothModeFSM::evUnknown;

    event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
    (void) b.addEvent(event,pArg);
    b.setState(b.oRequestNormalModeState,pArg);
    }
}

void
IdleState::evTestModeLoopback ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsI2STestToneActive(pArg) )
    {
        // set savepoint in superstate with enabled history
        b.oBTModeControlState.savepoint(b);

        // transition Actions
        s.vSetRequestedModeProduction(pArg);
        b.setState(b.oStopI2STestToneState,pArg);
    }
    else if ( s.bIsLoopbackActive(pArg) )
    {
        // set savepoint in superstate with enabled history
        b.oBTModeControlState.savepoint(b);

        // transition Actions
        s.vSetRequestedModeProduction(pArg);
        b.setState(b.oStopLoopbackState,pArg);
    }
    else
    {
    // transition Actions
    s.vSetRequestedModeProduction(pArg);
    // trigger events
    FsmEvent event = dia_BluetoothModeFSM::evUnknown;

    event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
    (void) b.addEvent(event,pArg);
    b.setState(b.oRequestLoopbackModeState,pArg);
    }
}

void
IdleState::evTestModeLoopbackLocal ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsI2STestToneActive(pArg) )
    {
        // set savepoint in superstate with enabled history
        b.oBTModeControlState.savepoint(b);

        // transition Actions
        s.vSetRequestedModeDUT(pArg);
        b.setState(b.oStopI2STestToneState,pArg);
    }
    else
    {
    // transition Actions
    s.vSetRequestedModeDUT(pArg);
    // trigger events
    FsmEvent event = dia_BluetoothModeFSM::evUnknown;

    event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
    (void) b.addEvent(event,pArg);
    b.setState(b.oRequestLocalLoopbackModeState,pArg);
    }
}

void
IdleState::evTestModeI2S ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsLoopbackActive(pArg) )
    {
        // set savepoint in superstate with enabled history
        b.oBTModeControlState.savepoint(b);

        // transition Actions
        s.vSetRequestedModeDUT(pArg);
        b.setState(b.oStopLoopbackState,pArg);
    }
    else
    {
    // transition Actions
    s.vSetRequestedModeDUT(pArg);
    // trigger events
    FsmEvent event = dia_BluetoothModeFSM::evUnknown;

    event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
    (void) b.addEvent(event,pArg);
    b.setState(b.oRequestI2SModeState,pArg);
    }
}

void
IdleState::evTestToneStopped ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsProductionModeRequested(pArg) )
    {
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oRequestLoopbackModeState,pArg);
    }
    else if ( s.bIsDUTModeRequested(pArg) )
    {
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oRequestLocalLoopbackModeState,pArg);
    }
    else if ( s.bIsNormalModeRequested(pArg) )
    {
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oRequestNormalModeState,pArg);
    }
}

void
IdleState::evLoopbackStopped ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsProductionModeRequested(pArg) )
    {
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oRequestLoopbackModeState,pArg);
    }
    else if ( s.bIsDUTModeRequested(pArg) )
    {
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oRequestI2SModeState,pArg);
    }
    else if ( s.bIsNormalModeRequested(pArg) )
    {
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evStartSetMode; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oRequestNormalModeState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestI2SModeState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

RequestI2SModeState::RequestI2SModeState ( FsmBody& b )
    : FsmState(b.oBTModeControlState)
{}

void
RequestI2SModeState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
RequestI2SModeState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
RequestI2SModeState::evStartSetMode ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsDUTModeActivated(pArg) )
    {
        // set savepoint in superstate with enabled history
        b.oBTModeControlState.savepoint(b);

        b.setState(b.oStartI2STestToneState,pArg);
    }
    else
    {
    // set savepoint in superstate with enabled history
    b.oBTModeControlState.savepoint(b);

    b.setState(b.oSetDUTModeState,pArg);
    }
}

void
RequestI2SModeState::evDUTModeActive ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    // set savepoint in superstate with enabled history
    b.oBTModeControlState.savepoint(b);

    b.setState(b.oStartI2STestToneState,pArg);
}

void
RequestI2SModeState::evTestToneStarted ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendPositiveResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestLocalLoopbackModeState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

RequestLocalLoopbackModeState::RequestLocalLoopbackModeState ( FsmBody& b )
    : FsmState(b.oBTModeControlState)
{}

void
RequestLocalLoopbackModeState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
RequestLocalLoopbackModeState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
RequestLocalLoopbackModeState::evStartSetMode ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsDUTModeActivated(pArg) )
    {
        // set savepoint in superstate with enabled history
        b.oBTModeControlState.savepoint(b);

        b.setState(b.oStartLoopbackState,pArg);
    }
    else
    {
    // set savepoint in superstate with enabled history
    b.oBTModeControlState.savepoint(b);

    b.setState(b.oSetDUTModeState,pArg);
    }
}

void
RequestLocalLoopbackModeState::evDUTModeActive ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    // set savepoint in superstate with enabled history
    b.oBTModeControlState.savepoint(b);

    b.setState(b.oStartLoopbackState,pArg);
}

void
RequestLocalLoopbackModeState::evLoopbackStarted ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendPositiveResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestLoopbackModeState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

RequestLoopbackModeState::RequestLoopbackModeState ( FsmBody& b )
    : FsmState(b.oBTModeControlState)
{}

void
RequestLoopbackModeState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
RequestLoopbackModeState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
RequestLoopbackModeState::evStartSetMode ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsProductionModeActivated(pArg) )
    {
        // transition Actions
        s.vSendPositiveResponse(pArg);
        b.setState(b.oIdleState,pArg);
    }
    else
    {
    // set savepoint in superstate with enabled history
    b.oBTModeControlState.savepoint(b);

    b.setState(b.oSetProductionModeState,pArg);
    }
}

void
RequestLoopbackModeState::evProdModeActive ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendPositiveResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: RequestNormalModeState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

RequestNormalModeState::RequestNormalModeState ( FsmBody& b )
    : FsmState(b.oBTModeControlState)
{}

void
RequestNormalModeState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
RequestNormalModeState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
RequestNormalModeState::evStartSetMode ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsNormalModeActivated(pArg) )
    {
        // transition Actions
        s.vSendPositiveResponse(pArg);
        b.setState(b.oIdleState,pArg);
    }
    else
    {
    b.setState(b.oSetNormalModeState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: SetDUTModeState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

SetDUTModeState::SetDUTModeState ( FsmBody& b )
    : FsmState(b.oBTModeToplevelState)
{}

void
SetDUTModeState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vSendDUTModeRequest(pArg);
}

void
SetDUTModeState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestMode(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
SetDUTModeState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestMode(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
SetDUTModeState::evModeRequestResult ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsModeReqAccepted(pArg) )
    {
        // transition Actions
        s.vSendUpRegTestMode(pArg);
        // this is an internal transition
    }
}

void
SetDUTModeState::evModeStatus ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsRequestedModeReached(pArg) )
    {
        // transition Actions
        s.vSendRelRegTestMode(pArg);
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evDUTModeActive; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oBTModeControlState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: SetNormalModeState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

SetNormalModeState::SetNormalModeState ( FsmBody& b )
    : FsmState(b.oBTModeControlState)
{}

void
SetNormalModeState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vSendNormalModeRequest(pArg);
}

void
SetNormalModeState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestMode(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
SetNormalModeState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestMode(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
SetNormalModeState::evModeRequestResult ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsModeReqAccepted(pArg) )
    {
        // transition Actions
        s.vSendUpRegTestMode(pArg);
        // this is an internal transition
    }
}

void
SetNormalModeState::evModeStatus ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsRequestedModeReached(pArg) )
    {
        // transition Actions
        s.vSendRelRegTestMode(pArg);
        s.vSendPositiveResponse(pArg);
        b.setState(b.oIdleState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: SetProductionModeState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

SetProductionModeState::SetProductionModeState ( FsmBody& b )
    : FsmState(b.oBTModeToplevelState)
{}

void
SetProductionModeState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vSendProductionModeRequest(pArg);
}

void
SetProductionModeState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestMode(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
SetProductionModeState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestMode(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
SetProductionModeState::evModeRequestResult ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsModeReqAccepted(pArg) )
    {
        // transition Actions
        s.vSendUpRegTestMode(pArg);
        // this is an internal transition
    }
}

void
SetProductionModeState::evModeStatus ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsRequestedModeReached(pArg) )
    {
        // transition Actions
        s.vSendRelRegTestMode(pArg);
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evProdModeActive; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oBTModeControlState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: StartI2STestToneState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

StartI2STestToneState::StartI2STestToneState ( FsmBody& b )
    : FsmState(b.oBTModeToplevelState)
{}

void
StartI2STestToneState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vSendStartTestToneRequest(pArg);
}

void
StartI2STestToneState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestTone(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
StartI2STestToneState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestTone(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
StartI2STestToneState::evTestToneRequestResult ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsTestToneReqAccepted(pArg) )
    {
        // transition Actions
        s.vSendUpRegTestTone(pArg);
        // this is an internal transition
    }
}

void
StartI2STestToneState::evTestToneStatus ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsTestToneReqExecuted(pArg) )
    {
        // transition Actions
        s.vSendRelRegTestTone(pArg);
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evTestToneStarted; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oBTModeControlState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: StartLoopbackState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

StartLoopbackState::StartLoopbackState ( FsmBody& b )
    : FsmState(b.oBTModeToplevelState)
{}

void
StartLoopbackState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vSendStartLoopbackRequest(pArg);
}

void
StartLoopbackState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
StartLoopbackState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
StartLoopbackState::evLoopbackRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsLoopbackReqExecuted(pArg) )
    {
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evLoopbackStarted; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oBTModeControlState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: StopI2STestToneState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

StopI2STestToneState::StopI2STestToneState ( FsmBody& b )
    : FsmState(b.oBTModeToplevelState)
{}

void
StopI2STestToneState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vSendStopTestToneRequest(pArg);
}

void
StopI2STestToneState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestTone(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
StopI2STestToneState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendRelRegTestTone(pArg);
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
StopI2STestToneState::evTestToneRequestResult ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsTestToneReqAccepted(pArg) )
    {
        // transition Actions
        s.vSendUpRegTestTone(pArg);
        // this is an internal transition
    }
}

void
StopI2STestToneState::evTestToneStatus ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsTestToneReqExecuted(pArg) )
    {
        // transition Actions
        s.vSendRelRegTestTone(pArg);
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evTestToneStopped; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oBTModeControlState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: StopLoopbackState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

StopLoopbackState::StopLoopbackState ( FsmBody& b )
    : FsmState(b.oBTModeToplevelState)
{}

void
StopLoopbackState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vSendStopLoopbackRequest(pArg);
}

void
StopLoopbackState::evError ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
StopLoopbackState::evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vSendNegativeResponse(pArg);
    b.setState(b.oIdleState,pArg);
}

void
StopLoopbackState::evLoopbackRequestResult ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsLoopbackReqExecuted(pArg) )
    {
        // trigger events
        FsmEvent event = dia_BluetoothModeFSM::evUnknown;

        event = dia_BluetoothModeFSM::evLoopbackStopped; //lint !e838 Info: Previously assigned value to variable has not been used
        (void) b.addEvent(event,pArg);
        b.setState(b.oBTModeControlState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// This is the finite state machine class implementation
//
///////////////////////////////////////////////////////////////////////////////

Fsm::Fsm ( void )
    : mpBehaviour(0),
      mpBody(0)
{
}

Fsm::Fsm ( FsmBehaviour* server )
    : mpBehaviour(server),
      mpBody(0)
{
    mpBody = new FsmBody();
}

Fsm::~Fsm ( void )
{
//    _BP_TRY_BEGIN
//    {
        if ( mpBody )
        {
            delete mpBody;
            mpBody = 0;
        }
        mpBehaviour = 0;
//    }
//    _BP_CATCH_ALL
//    {
//       DIA_TR_ERR("EXCEPTION CAUGHT: Fsm::~Fsm !!!");
//       DIA_ASSERT_ALWAYS();
//    }
//    _BP_CATCH_END
}

void
Fsm::init ( void )
{

    if ( mpBody )
    {
        // connect the body with its frontend
        mpBody->connect(this);
        {
            mpBody->init();
        }
    }
}

const char*
Fsm::getStateName (void) const
{
    return ( mpBody ) ? mpBody->getStateName() : 0;
}

void
Fsm::acceptEvent ( FsmEvent event, void* pArg )
{
    if ( mpBody )
    {
        // forward the event to the FSM body (handle-body pattern)
        mpBody->acceptEvent(event,pArg);
    }
}
///////////////////////////////////////////////////////////////////////////////
//
// This is the finite state machine class implementation
//
///////////////////////////////////////////////////////////////////////////////

FsmBody::FsmBody ( void ) :
      oBTModeControlState(*this),
      oBTModeToplevelState(*this),
      oIdleState(*this),
      oRequestI2SModeState(*this),
      oRequestLocalLoopbackModeState(*this),
      oRequestLoopbackModeState(*this),
      oRequestNormalModeState(*this),
      oSetDUTModeState(*this),
      oSetNormalModeState(*this),
      oSetProductionModeState(*this),
      oStartI2STestToneState(*this),
      oStartLoopbackState(*this),
      oStopI2STestToneState(*this),
      oStopLoopbackState(*this),
      mpFrontend(0),
      mpState(0),
      mBusyFlag(false),
      mQueueHead(0),
      mQueueTail(0),
      mQueueSize(0)
{
    (void) ::memset(mEventQueue,0,sizeof(FsmEvent)*MAX_TRIGGER_EVENTS);
    (void) ::memset(mParamQueue,0,sizeof(void*)*MAX_TRIGGER_EVENTS);
}

void
FsmBody::init ( void )
{
    oBTModeControlState.reset();
    oBTModeToplevelState.reset();
    oIdleState.reset();
    oRequestI2SModeState.reset();
    oRequestLocalLoopbackModeState.reset();
    oRequestLoopbackModeState.reset();
    oRequestNormalModeState.reset();
    oSetDUTModeState.reset();
    oSetNormalModeState.reset();
    oSetProductionModeState.reset();
    oStartI2STestToneState.reset();
    oStartLoopbackState.reset();
    oStopI2STestToneState.reset();
    oStopLoopbackState.reset();

    setState(oBTModeToplevelState,0);
}

void
FsmBody::connect ( Fsm* frontend )
{
    mpFrontend = frontend;
}

const char*
FsmBody::getStateName ( void ) const
{
    return ( mpState ) ? mpState->getStateName() : 0;
}

bool
FsmBody::addEvent ( const FsmEvent& event, void* param )
{
    if ( mQueueSize < MAX_TRIGGER_EVENTS )
    {
        mQueueSize++;
        mEventQueue[mQueueTail] = event;
        mParamQueue[mQueueTail] = param;
        mQueueTail++;
        if ( mQueueTail == MAX_TRIGGER_EVENTS )
        {
            mQueueTail = 0;
        }
        return true;
    }
    return false;
}

bool
FsmBody::getEvent ( FsmEvent& event, void** param )
{
    if ( !mQueueSize || !param )
    {
        return false;
    }
    mQueueSize--;
    event  = mEventQueue[mQueueHead];
    *param = mParamQueue[mQueueHead];
    mQueueHead++;
    if ( mQueueHead == MAX_TRIGGER_EVENTS )
    {
        mQueueHead = 0;
    }
    return true;
}

bool
FsmBody::isQueueEmpty ( void ) const
{
    return ( mQueueSize ) ? false : true;
}

void
FsmBody::acceptEvent ( FsmEvent event, void* pArg )
{
    (void) addEvent(event,pArg);

    if ( mBusyFlag )
    {
        return;
    }

    mBusyFlag = true;
    while ( !isQueueEmpty() )
    {
        FsmEvent nextEvent = evUnknown;
        void*    nextArg   = 0;
        if ( !getEvent(nextEvent,&nextArg) )
        {
            break;
        }

        switch (nextEvent)
        {
        case evError:
            mpState->evError(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTimeout:
            mpState->evTimeout(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evNormalMode:
            mpState->evNormalMode(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evStartSetMode:
            mpState->evStartSetMode(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTestModeLoopback:
            mpState->evTestModeLoopback(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTestModeLoopbackLocal:
            mpState->evTestModeLoopbackLocal(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTestModeI2S:
            mpState->evTestModeI2S(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTestToneStopped:
            mpState->evTestToneStopped(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evLoopbackStopped:
            mpState->evLoopbackStopped(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evProdModeActive:
            mpState->evProdModeActive(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evDUTModeActive:
            mpState->evDUTModeActive(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evLoopbackStarted:
            mpState->evLoopbackStarted(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTestToneStarted:
            mpState->evTestToneStarted(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evModeRequestResult:
            mpState->evModeRequestResult(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evModeStatus:
            mpState->evModeStatus(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTestToneRequestResult:
            mpState->evTestToneRequestResult(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTestToneStatus:
            mpState->evTestToneStatus(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evLoopbackRequestResult:
            mpState->evLoopbackRequestResult(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        default:
            //... unknown event ...
            break;
        } //lint !e788: not all items used within defaulted switch
    }
    mBusyFlag = false;
}

void
FsmBody::setState ( FsmState& state, void* pArg )
{
    if ( mpFrontend )
    {
        mpState = const_cast<FsmState*>(state.propagate(*this,*(mpFrontend->mpBehaviour),pArg));
    }
}

const FsmState*
FsmBody::getState ( void ) const
{
    return mpState;
}


} // namespace
