#include <string.h>
#include "dia_ADRSessionFSM.h"

#define MAX_TRIGGER_EVENTS  20

namespace dia_ADRSessionFSM
{

///////////////////////////////////////////////////////////////////////////////
//
// Forward Declarations
//
///////////////////////////////////////////////////////////////////////////////

class Fsm;
class FsmSuperState;

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM states.
//
///////////////////////////////////////////////////////////////////////////////

class FsmState
{
public:
    //! name of the FSM's state
    virtual const char* getStateName ( void ) const = 0;

    //! resetting state internals
    virtual void reset ( void ) {}

    // event functions
    virtual void evReset ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evActivate ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evDeactivate ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evADRContainerReceived ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evCheckCompletion ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequestReceived ( FsmBody& b, FsmBehaviour& s, void* pArg );

    //! default implementation for the Entry action
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! default implementation for the Exit action
    virtual void exit  ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! method used to propagate the event to the superstate
    virtual const FsmState* propagate ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! return the parent state
    virtual FsmSuperState* getParent ( void );


protected:
    //! default constructor is protected to prevent usage of this constructor
    FsmState ( void );
    //! constructor for a substate of the given superstate
    FsmState ( FsmSuperState& parent );
    //! class destructor
    virtual ~FsmState ( void );

    //! pointer to the superstate
    FsmSuperState* mpParent;
};

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM superstates.
//
///////////////////////////////////////////////////////////////////////////////

class FsmSuperState
    : public FsmState
{
public:
    //! class constructor for a superstate without parent
    FsmSuperState ( FsmState& defState, bool histEnable=false );
    //! class constructor for a superstate with parent
    FsmSuperState ( FsmSuperState& parent, FsmState& defState, bool histEnable=false );

    //! set a savepoint for the history
    virtual void savepoint ( FsmBody& b );
    //! default action
    virtual void defaultAction ( FsmBody& b, FsmBehaviour& s, void* pArg );

    //! resetting state internals
    virtual void reset ( void ) { _history = 0; }

    //! method used to propagate the event to the superstate
    virtual const FsmState* propagate ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    //! default constructor is protected to prevent usage of this constructor
    FsmSuperState ( void );

    //! pointer to the default substate
    FsmState* _default;
    //! pointer to the history substate
    const FsmState* _history;
    //! flag that indicates if history is enabled or not
    bool _historyEnable;
};

///////////////////////////////////////////////////////////////////////////////
//
// Class: ActivatingState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class ActivatingState
    : public FsmState
{
public:
    //! default constructor
    ActivatingState ( void );

    ActivatingState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evReset ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evADRContainerReceived ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evCheckCompletion ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* ActivatingState::mStateName = "Activating";

///////////////////////////////////////////////////////////////////////////////
//
// Class: ActiveState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class ActiveState
    : public FsmState
{
public:
    //! default constructor
    ActiveState ( void );

    ActiveState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evReset ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evActivate ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evDeactivate ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evADRContainerReceived ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evRequestReceived ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* ActiveState::mStateName = "Active";

///////////////////////////////////////////////////////////////////////////////
//
// Class: DeactivatingState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class DeactivatingState
    : public FsmState
{
public:
    //! default constructor
    DeactivatingState ( void );

    DeactivatingState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evReset ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evADRContainerReceived ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evCheckCompletion ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* DeactivatingState::mStateName = "Deactivating";

///////////////////////////////////////////////////////////////////////////////
//
// Class: NotActiveState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class NotActiveState
    : public FsmState
{
public:
    //! default constructor
    NotActiveState ( void );

    NotActiveState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evReset ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evActivate ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evDeactivate ( FsmBody& b, FsmBehaviour& s, void* pArg );
    //! entry actions
    virtual void entry ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* NotActiveState::mStateName = "NotActive";

///////////////////////////////////////////////////////////////////////////////
//
// Class: TopLevelState
//
// Concrete state of FSM
//
///////////////////////////////////////////////////////////////////////////////

class TopLevelState
    : public FsmSuperState
{
public:
    //! default constructor
    TopLevelState ( void );

    TopLevelState ( FsmBody& b );

    //! return state name
    virtual const char* getStateName ( void ) const { return mStateName; }

    //! event functions
    virtual void evReset ( FsmBody& b, FsmBehaviour& s, void* pArg );
    virtual void evTimeout ( FsmBody& b, FsmBehaviour& s, void* pArg );

protected:
    static const char* mStateName;
};

const char* TopLevelState::mStateName = "TopLevel";

///////////////////////////////////////////////////////////////////////////////
//
// Class: FsmBody
//
// This is the finite state machine's body class 'FsmBody'.
//
///////////////////////////////////////////////////////////////////////////////

class FsmBody
{
    struct FsmDeferredEvent
    {
          FsmDeferredEvent() : mEvent(evUnknown), mArg(0) {}
          FsmEvent mEvent;
          void*    mArg;
    };

public:
    //! class constructor
    FsmBody ( void );

    //! connect the FSM body with its frontend
    void connect ( Fsm* frontend );

    //! initialize the FSM: set initial state and execute initial actions
    void init ( void );

    //! this method is used to stimulate the FSM with events
    void acceptEvent ( FsmEvent event, void* pArg );

    //! return the name of the active state
    const char* getStateName ( void ) const;

    //! push trigger events to the event queue
    bool addEvent ( const FsmEvent& event, void* arg  );
    //! get event from the event queue
    bool getEvent ( FsmEvent& event, void** arg  );
    //! get event from the event queue
    bool isQueueEmpty ( void ) const;

    // set the FSM's internal state
    void setState ( FsmState& state, void* pArg );
    // return the active state object
    const FsmState* getState ( void ) const;

    //! static state objects
    ActivatingState oActivatingState;
    ActiveState oActiveState;
    DeactivatingState oDeactivatingState;
    NotActiveState oNotActiveState;
    TopLevelState oTopLevelState;

    //! frontend for this body
    Fsm* mpFrontend;
    //! link to the active state object
    FsmState* mpState;

    //! busy indicator flag
    bool mBusyFlag;

    //! queue of trigger events
    FsmEvent mEventQueue[MAX_TRIGGER_EVENTS];
    //! queue of arguments for trigger events
    void*    mParamQueue[MAX_TRIGGER_EVENTS];
    //! index of the queue's front element
    unsigned int mQueueHead;
    //! index of the queue's back element
    unsigned int mQueueTail;
    //! number of queued events
    unsigned int mQueueSize;
};

///////////////////////////////////////////////////////////////////////////////
//
// Default implementation for all events.
//
///////////////////////////////////////////////////////////////////////////////

void
FsmState::evReset ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evReset ***
}

void
FsmState::evTimeout ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evTimeout ***
}

void
FsmState::evActivate ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evActivate ***
}

void
FsmState::evDeactivate ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evDeactivate ***
}

void
FsmState::evADRContainerReceived ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evADRContainerReceived ***
}

void
FsmState::evCheckCompletion ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evCheckCompletion ***
}

void
FsmState::evRequestReceived ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    // *** Ignoring Event evRequestReceived ***
}

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM states components.
//
///////////////////////////////////////////////////////////////////////////////

FsmState::FsmState ( void )
    : mpParent(0)
{}

FsmState::FsmState ( FsmSuperState& parent )
    : mpParent(&parent)
{}

FsmState::~FsmState ( void )
{
    // make lint happy
    mpParent = 0;
}

void
FsmState::entry ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    //... empty as default ...
}

void
FsmState::exit ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /* pArg */ )
{
    //... empty as default ...
}

const FsmState*
FsmState::propagate ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    entry(b,s,pArg);
    return this;
}

FsmSuperState*
FsmState::getParent ( void )
{
    return (mpParent) ? mpParent : 0;
}

///////////////////////////////////////////////////////////////////////////////
//
// Abstract base class for all FSM superstates.
//
///////////////////////////////////////////////////////////////////////////////

FsmSuperState::FsmSuperState ( FsmState& defState, bool histEnable )
    : _default(&defState),
      _history(0),
      _historyEnable(histEnable)
{}

FsmSuperState::FsmSuperState ( FsmSuperState& parent, FsmState& defState, bool histEnable )
    : FsmState(parent),
      _default(&defState),
      _history(0),
      _historyEnable(histEnable)
{}

FsmSuperState::FsmSuperState ( void )
    : _default(0),
      _history(0),
      _historyEnable(false)
{}

const FsmState*
FsmSuperState::propagate ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    const FsmState* pState = 0;

    // execute the entry actions for this superstate
    entry(b,s,pArg);

    if ( _history && _historyEnable )
    {
        pState = const_cast<FsmState*>(_history)->propagate(b,s,pArg);
    }
    else
    {
        defaultAction(b,s,pArg);
        pState = _default->propagate(b,s,pArg);
    }

    return pState;
}

void
FsmSuperState::savepoint ( FsmBody& b )
{
    if ( _historyEnable )
    {
        _history = b.getState();
    }
}

void
FsmSuperState::defaultAction ( FsmBody& /* b */, FsmBehaviour& /* s */, void* /*pArg*/ )
{
    //... empty as default ...
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: ActivatingState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

ActivatingState::ActivatingState ( FsmBody& b )
    : FsmState(b.oTopLevelState)
{}

void
ActivatingState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vFsmSendActivation(pArg);
}

void
ActivatingState::evReset ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

void
ActivatingState::evTimeout ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

void
ActivatingState::evADRContainerReceived ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmEvalADRContainer(pArg);
    // trigger events
    FsmEvent event = dia_ADRSessionFSM::evUnknown;

    event = dia_ADRSessionFSM::evCheckCompletion; //lint !e838 Info: Previously assigned value to variable has not been used
    (void) b.addEvent(event,pArg);
    // this is an internal transition
}

void
ActivatingState::evCheckCompletion ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsActivationComplete(pArg) )
    {
        b.setState(b.oActiveState,pArg);
    }
    else
    {
    b.setState(b.oNotActiveState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: ActiveState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

ActiveState::ActiveState ( FsmBody& b )
    : FsmState(b.oTopLevelState)
{}

void
ActiveState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vFsmOnSessionActivated(pArg);
}

void
ActiveState::evReset ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

void
ActiveState::evTimeout ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

void
ActiveState::evActivate ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmOnSessionActivated(pArg);
    // this is an internal transition
}

void
ActiveState::evDeactivate ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oDeactivatingState,pArg);
}

void
ActiveState::evADRContainerReceived ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmForwardResponse(pArg);
    // this is an internal transition
}

void
ActiveState::evRequestReceived ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmForwardRequest(pArg);
    // this is an internal transition
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: DeactivatingState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

DeactivatingState::DeactivatingState ( FsmBody& b )
    : FsmState(b.oTopLevelState)
{}

void
DeactivatingState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vFsmSendDeactivation(pArg);
}

void
DeactivatingState::evReset ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

void
DeactivatingState::evTimeout ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

void
DeactivatingState::evADRContainerReceived ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmEvalADRContainer(pArg);
    // trigger events
    FsmEvent event = dia_ADRSessionFSM::evUnknown;

    event = dia_ADRSessionFSM::evCheckCompletion; //lint !e838 Info: Previously assigned value to variable has not been used
    (void) b.addEvent(event,pArg);
    // this is an internal transition
}

void
DeactivatingState::evCheckCompletion ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    if ( s.bIsDeactivationComplete(pArg) )
    {
        // transition Actions
        s.vFsmOnSessionDeactivated(pArg);
        b.setState(b.oNotActiveState,pArg);
    }
    else
    {
    // transition Actions
    s.vFsmDeactivationFailed(pArg);
    b.setState(b.oNotActiveState,pArg);
    }
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: NotActiveState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

NotActiveState::NotActiveState ( FsmBody& b )
    : FsmState(b.oTopLevelState)
{}

void
NotActiveState::entry ( FsmBody& /* b */, FsmBehaviour& s, void* pArg )
{
    s.vFsmReset(pArg);
}

void
NotActiveState::evReset ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

void
NotActiveState::evTimeout ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

void
NotActiveState::evActivate ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oActivatingState,pArg);
}

void
NotActiveState::evDeactivate ( FsmBody& b, FsmBehaviour& s, void* pArg )
{
    // transition Actions
    s.vFsmOnSessionDeactivated(pArg);
    b.setState(b.oNotActiveState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// Class: TopLevelState
//
// Implementation of concrete FSM state
//
///////////////////////////////////////////////////////////////////////////////

TopLevelState::TopLevelState ( FsmBody& b )
    : FsmSuperState(b.oNotActiveState, false)
{}

void
TopLevelState::evReset ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

void
TopLevelState::evTimeout ( FsmBody& b, FsmBehaviour& /* s */, void* pArg )
{
    b.setState(b.oNotActiveState,pArg);
}

///////////////////////////////////////////////////////////////////////////////
//
// This is the finite state machine class implementation
//
///////////////////////////////////////////////////////////////////////////////

Fsm::Fsm ( void )
    : mpBehaviour(0),
      mpBody(0)
{
}

Fsm::Fsm ( FsmBehaviour* server )
    : mpBehaviour(server),
      mpBody(0)
{
    mpBody = new FsmBody();
}

Fsm::~Fsm ( void )
{
//    _BP_TRY_BEGIN
//    {
        if ( mpBody )
        {
            delete mpBody;
            mpBody = 0;
        }
        mpBehaviour = 0;
//    }
//    _BP_CATCH_ALL
//    {
//       DIA_TR_ERR("EXCEPTION CAUGHT: Fsm::~Fsm !!!");
//       DIA_ASSERT_ALWAYS();
//    }
//    _BP_CATCH_END
}

void
Fsm::init ( void )
{

    if ( mpBody )
    {
        // connect the body with its frontend
        mpBody->connect(this);
        {
            mpBody->init();
        }
    }
}

const char*
Fsm::getStateName (void) const
{
    return ( mpBody ) ? mpBody->getStateName() : 0;
}

void
Fsm::acceptEvent ( FsmEvent event, void* pArg )
{
    if ( mpBody )
    {
        // forward the event to the FSM body (handle-body pattern)
        mpBody->acceptEvent(event,pArg);
    }
}
///////////////////////////////////////////////////////////////////////////////
//
// This is the finite state machine class implementation
//
///////////////////////////////////////////////////////////////////////////////

FsmBody::FsmBody ( void ) :
      oActivatingState(*this),
      oActiveState(*this),
      oDeactivatingState(*this),
      oNotActiveState(*this),
      oTopLevelState(*this),
      mpFrontend(0),
      mpState(0),
      mBusyFlag(false),
      mQueueHead(0),
      mQueueTail(0),
      mQueueSize(0)
{
    (void) ::memset(mEventQueue,0,sizeof(FsmEvent)*MAX_TRIGGER_EVENTS);
    (void) ::memset(mParamQueue,0,sizeof(void*)*MAX_TRIGGER_EVENTS);
}

void
FsmBody::init ( void )
{
    oActivatingState.reset();
    oActiveState.reset();
    oDeactivatingState.reset();
    oNotActiveState.reset();
    oTopLevelState.reset();

    setState(oTopLevelState,0);
}

void
FsmBody::connect ( Fsm* frontend )
{
    mpFrontend = frontend;
}

const char*
FsmBody::getStateName ( void ) const
{
    return ( mpState ) ? mpState->getStateName() : 0;
}

bool
FsmBody::addEvent ( const FsmEvent& event, void* param )
{
    if ( mQueueSize < MAX_TRIGGER_EVENTS )
    {
        mQueueSize++;
        mEventQueue[mQueueTail] = event;
        mParamQueue[mQueueTail] = param;
        mQueueTail++;
        if ( mQueueTail == MAX_TRIGGER_EVENTS )
        {
            mQueueTail = 0;
        }
        return true;
    }
    return false;
}

bool
FsmBody::getEvent ( FsmEvent& event, void** param )
{
    if ( !mQueueSize || !param )
    {
        return false;
    }
    mQueueSize--;
    event  = mEventQueue[mQueueHead];
    *param = mParamQueue[mQueueHead];
    mQueueHead++;
    if ( mQueueHead == MAX_TRIGGER_EVENTS )
    {
        mQueueHead = 0;
    }
    return true;
}

bool
FsmBody::isQueueEmpty ( void ) const
{
    return ( mQueueSize ) ? false : true;
}

void
FsmBody::acceptEvent ( FsmEvent event, void* pArg )
{
    (void) addEvent(event,pArg);

    if ( mBusyFlag )
    {
        return;
    }

    mBusyFlag = true;
    while ( !isQueueEmpty() )
    {
        FsmEvent nextEvent = evUnknown;
        void*    nextArg   = 0;
        if ( !getEvent(nextEvent,&nextArg) )
        {
            break;
        }

        switch (nextEvent)
        {
        case evReset:
            mpState->evReset(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evTimeout:
            mpState->evTimeout(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evActivate:
            mpState->evActivate(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evDeactivate:
            mpState->evDeactivate(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evADRContainerReceived:
            mpState->evADRContainerReceived(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evCheckCompletion:
            mpState->evCheckCompletion(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        case evRequestReceived:
            mpState->evRequestReceived(*this,*(mpFrontend->mpBehaviour),nextArg);
            break;

        default:
            //... unknown event ...
            break;
        } //lint !e788: not all items used within defaulted switch
    }
    mBusyFlag = false;
}

void
FsmBody::setState ( FsmState& state, void* pArg )
{
    if ( mpFrontend )
    {
        mpState = const_cast<FsmState*>(state.propagate(*this,*(mpFrontend->mpBehaviour),pArg));
    }
}

const FsmState*
FsmBody::getState ( void ) const
{
    return mpState;
}


} // namespace
