/*!
 * \file       dia_TemperatureHistogramPlugin.h
 *
 * \brief      Field Load Data plugin for temperature sensors
 *
 * \details    -
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreFrw
 *
 * \copyright  (c) 2019 Robert Bosch Car Multimedia
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

//#define TEMPERATURE_HISTOGRAM_TEST
#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#ifndef __INCLUDED_DIA_TEMPERATUREHISTOGRAM_PLUGIN__
#include "dia_TemperatureHistogramPlugin.h"
#endif

#ifndef __INCLUDED_DIA_TEMPERATUREHISTOGRAM__
#include "dia_TemperatureHistogram.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_CONFIG__
#include "common/framework/config/dia_common_config.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_TEMPERATURE_HISTOGRAM_LISTENER__
#include "common/interfaces/dia_ITemperatureHistogramListener.h"
#endif


namespace dia{

TemperatureHistogramPlugin::TemperatureHistogramPlugin(uint32_t maxNumHistograms):
   mMaxNumHistograms(maxNumHistograms)
{
   dia::ScopeTrace trc("TemperatureHistogramPlugin::TemperatureHistogramPlugin");
}

TemperatureHistogramPlugin::~TemperatureHistogramPlugin()
{
      (void) unsetSysAdapterListener<dia_IHeatControlListener>(this);
}

void 
TemperatureHistogramPlugin::ProcessPluginStart(void)
{
   dia::ScopeTrace trc("TemperatureHistogramPlugin::ProcessPluginStart");
   
   mHistogramRepo.reserve(mMaxNumHistograms);
   std::vector<uint8_t> rawBuf;
   do {
      tDiaResult diaRes=dia_getProperty(DIA_PROP_TEMPERATURE_HISTOGRAMS, rawBuf);
      if (diaRes != DIA_SUCCESS) {
         DIA_TR_INF("TemperatureHistogramPlugin::ProcessPluginStart: could not read property DIA_PROP_TEMPERATURE_HISTOGRAMS");
         break;
      }
      DIA_TR_INF("TemperatureHistogramPlugin::ProcessPluginStart: rawBuf=%u", (uint32_t)rawBuf.size());

      if (rawBuf.size() % sizeof(TemperatureHistogram)) {
         DIA_TR_INF("TemperatureHistogramPlugin::ProcessPluginStart: unexpected property-size: %u", (uint32_t)rawBuf.size());
         break;
      }
      uint32_t numHistograms=(uint32_t) (rawBuf.size() / sizeof(TemperatureHistogram));
      DIA_TR_INF("TemperatureHistogramPlugin::ProcessPluginStart: numHistograms=%u", numHistograms);

      mHistogramRepo.resize(numHistograms);
      if (numHistograms != 0) {
         memcpy(&(mHistogramRepo[0]), &(rawBuf[0]), numHistograms * sizeof(TemperatureHistogram));
         for (std::vector<TemperatureHistogram>::iterator iter = mHistogramRepo.begin(); iter != mHistogramRepo.end(); ++iter) {
            TemperatureHistogram &histogram = *iter;
            DIA_TR_INF("have id=%u", histogram.getId());
         }
      }

   } while (false);
   // todo: check where to unset the interface
   (void) assignInterfaceAndNotifier<ITemperatureHistogram,ITemperatureHistogramListener,ITemperatureHistogramNotifier>(this);



}

TemperatureHistogram *TemperatureHistogramPlugin::getHistogram(dia_eHeatCtrlTemperatureID id) {
   dia_ScopeTraceVarg trc("TemperatureHistogramPlugin::getHistogram(%u) respoSize)%u", id, (uint32_t)mHistogramRepo.size());
   for (std::vector<TemperatureHistogram>::iterator it=mHistogramRepo.begin(); it!=mHistogramRepo.end();++it) {
      DIA_TR_INF("TemperatureHistogramPlugin::getHistogram: have: id=%u", (uint32_t)((*it).getId()));      
      
   }
   std::vector<TemperatureHistogram>::iterator iter= find_if(mHistogramRepo.begin(), mHistogramRepo.end(), [&id] (const TemperatureHistogram& histogram) {return (histogram.getId()==id);});
   if (iter!= mHistogramRepo.end()) {
      DIA_TR_INF("TemperatureHistogramPlugin::getHistogram: found");      
      return &(*iter);
   }
   else  {
      DIA_TR_INF("TemperatureHistogramPlugin::getHistogram: not found");      
      return nullptr;
   }
}

TemperatureHistogram &TemperatureHistogramPlugin::accessHistogram(dia_eHeatCtrlTemperatureID id) {
   dia_ScopeTraceVarg trc("TemperatureHistogramPlugin::accessHistogram(%u)", id);
   TemperatureHistogram *existingHistogram=getHistogram(id);
   if (existingHistogram) {
      return *existingHistogram;
   }
   DIA_TR_INF("TemperatureHistogramPlugin::accessHistogram: add new");      
   TemperatureHistogram newHistogram(id); 
   mHistogramRepo.push_back(newHistogram);
   return mHistogramRepo.back();
   
}

void 
TemperatureHistogramPlugin::ProcessPluginUpdate(void)
{
   dia::ScopeTrace trc("TemperatureHistogramPlugin::ProcessPluginUpdate");
   dia_IHeatControl* pHeatControl = nullptr;
   tDiaResult retval = DIA_FAILED;
   if ( querySysAdapterInterface<dia_IHeatControl>(&pHeatControl) != DIA_SUCCESS ) return;
   if( pHeatControl )
   {
      (void) setSysAdapterListener<dia_IHeatControlListener>(this);
      retval = pHeatControl->getTemperatures(DIA_EN_HEATCTRL_TEMPERATURE_CLASS_REAL);
   }
   if(retval != DIA_SUCCESS)
   {
      DIA_TR_ERR("TemperatureHistogramPlugin::Update ERROR! sending message to Heat Control Failed");
   }
}

void
TemperatureHistogramPlugin::vOnTemperatureInfo ( std::list<dia_tHeatCtrlTemperature>& tempList )
{
   dia::ScopeTrace trc("TemperatureHistogramPlugin::vOnTemperatureInfo");
   (void) unsetSysAdapterListener<dia_IHeatControlListener>(this);

   //Update all Temperature Histograms
   (void)UpdateTemperatureHistograms(tempList);
}


void 
TemperatureHistogramPlugin::ProcessPluginStop(void)
{
   dia::ScopeTrace trc("TemperatureHistogramPlugin::ProcessPluginStop");
}



tDiaResult
TemperatureHistogramPlugin::UpdateTemperatureHistograms(std::list<dia_tHeatCtrlTemperature>& sensordata)
{
   dia::ScopeTrace trc("TemperatureHistogramPlugin::UpdateTemperatureHistograms");
   std::list<dia_tHeatCtrlTemperature>::iterator iter = sensordata.begin();
   for (; iter != sensordata.end();++iter)
   {
      
      dia_tHeatCtrlTemperature const &measurement=*iter;
      DIA_TR_INF("TemperatureHistogramPlugin::UpdateTemperatureHistograms: mDiaQuality=%u id=%u temperature=%i temperatureMin=%i temperatureMax=%i",
                 measurement.mDiaQuality, measurement.mID, measurement.mTemperature,measurement.mTemperatureMin,measurement.mTemperatureMax);
      if (measurement.mDiaQuality==DIA_EN_HEATCTRL_TEMPERATURE_QUALITY_VALID) {
         TemperatureHistogram &id=accessHistogram(measurement.mID);
         id.addMeasurement(measurement.mTemperature);
         id.addTemperatureMin(measurement.mTemperatureMin);
         id.addTemperatureMax(measurement.mTemperatureMax);
      }
   }
   tDiaResult diaRes=dia_setProperty(DIA_PROP_TEMPERATURE_HISTOGRAMS, 
                                     (uint8_t *)(&(mHistogramRepo[0])), 
                                     (uint16_t)(mHistogramRepo.size() * sizeof(TemperatureHistogram)));
   if (diaRes != DIA_SUCCESS) {
      DIA_TR_ERR("TemperatureHistogramPlugin::UpdateTemperatureHistograms: failed to store to data-pool, diaRes=%u", diaRes);
   }
   return DIA_SUCCESS;
}


tDiaResult
TemperatureHistogramPlugin::getTemperatureHistogram (dia_eHeatCtrlTemperatureID id ) {
   DIA_TR_INF("TemperatureHistogramPlugin::getTemperatureHistogram: id=%u", id);
   ITemperatureHistogramListener* pListener = 0;
   if ((querySysAdapterListener<ITemperatureHistogramListener>(&pListener) != DIA_SUCCESS) || !pListener) {
      DIA_TR_INF("TemperatureHistogramPlugin::getTemperatureHistogram: no listeer found");
      return DIA_FAILED;
   }
   
   DIA_TR_INF("TemperatureHistogramPlugin::getTemperatureHistogram: pListener=%p", pListener);
   TemperatureHistogram *histogram=getHistogram(id);
   dia_tTemperatureHistogram interfacesTemperatureHistogram(id, 0, 0,nullptr);
   if (!histogram) {
      DIA_TR_INF("TemperatureHistogramPlugin::getTemperatureHistogram: no histogram found for id=%u",id);
      pListener->onTemperatureHistogram(interfacesTemperatureHistogram);
      return DIA_FAILED;
   }
   interfacesTemperatureHistogram.mData=&(histogram->getData());
   interfacesTemperatureHistogram.mTemperatureMin=histogram->getTemperatureMin();
   interfacesTemperatureHistogram.mTemperatureMax=histogram->getTemperatureMax();
   DIA_TR_INF("TemperatureHistogramPlugin::getTemperatureHistogram: histogram found, call listener");

   pListener->onTemperatureHistogram(interfacesTemperatureHistogram);
   return DIA_SUCCESS;
}


}
