/*!
 * \file       dia_FactoryPluginCore.cpp
 *
 * \brief      Factory used to instantiate/shutdown fundamental core objects (e.g. singletons)
 *
 * \details    Factory used to instantiate/shutdown fundamental core objects (e.g. singletons)
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreFactory
 *
 * \copyright  (c) 2011-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include "dia_FactoryPluginCore.h"

#ifndef __INCLUDED_DIA_COMMON_FACTORY__
#include "common/framework/factory/dia_common_factory.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_UDD__
#include "common/framework/udd/dia_common_udd.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY_METHODS__
#include "common/framework/factory/dia_factoryMethods.h"
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include "common/framework/application/dia_AppController.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER__
#include "common/framework/application/dia_CommandController.h"
#endif

#ifndef __INCLUDED_DIA_SERVICE_TRACKER__
#include "common/framework/platform/cca/dia_ServiceTracker.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER_PLUGIN_APPSTATECHANGE__
#include "common/framework/application/dia_CommandControllerPluginAppStateChange.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER_PLUGIN_ALDSTATUS__
#include "common/framework/application/dia_CommandControllerPluginALDStatus.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER_PLUGIN_RUNLEVEL__
#include "common/framework/application/dia_CommandControllerPluginRunlevel.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER_PLUGIN_SERVICE_CCA__
#include "common/framework/platform/cca/dia_CommandControllerPluginServiceCCA.h"
#endif

#ifndef __INCLUDED_DIA_PROPERTY_BAG_RAM__
#include "common/framework/config/dia_PropertyBagRAM.h"
#endif

#ifndef __INCLUDED_DIA_PROPERTY_BAG_FILE_SWITCHES__
#include "common/framework/config/dia_PropertyBagFileSwitches.h"
#endif

#ifndef __INCLUDED_DIA_PROPERTY_BAG_REGISTRY__
#include "common/framework/config/dia_PropertyBagRegistry.h"
#endif

#ifndef __INCLUDED_DIA_PROPERTY_BAG_DATAPOOL__
#include "common/framework/config/dia_PropertyBagDatapool.h"
#endif

#ifndef __INCLUDED_DIA_PROPERTY_BAG_SCD__
#include "common/framework/config/dia_PropertyBagSCD.h"
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include "common/framework/application/dia_LockScope.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY_PLUGIN_PDX_FLASHING__
#include <common/framework/factory/dia_FactoryPluginPDXFlashing.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_APPLICATION_STATE_CONTROL_LISTENER__
#include <common/interfaces/dia_IApplicationStateControlListener.h>
#endif

#ifndef __DIA_UNIT_TESTING__
#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER_PLUGIN_OPMODE__
#include "common/framework/application/dia_CommandControllerPluginOpMode.h"
#endif
#endif

// implementation of the singleton methods
DIA_IMPL_SINGLETON(dia_FactoryPluginCore)

//-----------------------------------------------------------------------------

dia_FactoryPluginCore::dia_FactoryPluginCore ( void )
   : dia_FactoryPlugin("dia_FactoryPluginCore",DIA_EN_FACTORY_PLUGIN_TYPE_CORE),
     mpApp(0),
#if !defined(VARIANT_S_FTR_DONOTUSE_APPLICATION_LOCK) && !defined(__DIA_UNIT_TESTING__)
     mpAppLock(0),
#endif
     mpConfigMgr(0),
     mpAppCtrl(0),
     mpEngMgr(0)
{}

//-----------------------------------------------------------------------------

dia_FactoryPluginCore::~dia_FactoryPluginCore ( void )
{
   _BP_TRY_BEGIN
   {
      (void) tearDownObjects();

      mpApp = 0;
#if !defined(VARIANT_S_FTR_DONOTUSE_APPLICATION_LOCK) && !defined(__DIA_UNIT_TESTING__)
      mpAppLock = 0;
#endif
      mpConfigMgr = 0;
      mpAppCtrl = 0;
      mpEngMgr = 0;
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_FactoryPluginCore::~dia_FactoryPluginCore !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_FactoryPluginCore::setup ( dia_enInitLevel level )
{
   dia_tclFnctTrace oTrace("dia_FactoryPluginCore::setup");

   dia_LockScope mScopeLock(mSyncObj);

   if ( level <= getInitializationLevel() )
   {
      DIA_TR_INF("##### REQUESTED LEVEL OF CORE FACTORY ALREADY INITIALIZED. REQUEST IGNORED #####");
      return DIA_SUCCESS;
   }

   if ( level == DIA_EN_INITLEVEL_0 || level == DIA_EN_INITLEVEL_ALL )
   {
      // initialize the AMT/CCA framework for this component
#ifndef __DIA_UNIT_TESTING__
	   (void)amt_bInit();
#endif

#if !defined(VARIANT_S_FTR_DONOTUSE_APPLICATION_LOCK) && !defined(__DIA_UNIT_TESTING__)
      mpAppLock = dia::createInstanceOfApplicationLock();
      if ( !mpAppLock)
      {
         DIA_TR_ERR("dia_FactoryPluginCore::setup(DIA_EN_INITLEVEL_0) - creation of application lock failed !!");
      }
#endif
     // load common NRC mapping
     getInstanceOfFactory()->loadNRCMappings(aCommonNRCMapping, static_cast<tU16>(sizeof(aCommonNRCMapping)/sizeof(dia_NRCConfig)));

      mpApp = getInstanceOfApplication();
      if ( !mpApp )
      {
         DIA_TR_ERR("dia_FactoryPluginCore::setup(DIA_EN_INITLEVEL_0) - creation of application failed !!");
         return DIA_FAILED;
      }

      dia_SystemAdapterFacade* pSAFacade = createInstanceOfSystemAdapterFacade();
      if ( !pSAFacade )
      {
         DIA_TR_ERR("dia_FactoryPluginCore::setup(DIA_EN_INITLEVEL_0) - creation of system adapter facade failed !!");
         return DIA_FAILED;
      }

      mpConfigMgr = getInstanceOfConfigManager();
      if ( !mpConfigMgr )
      {
         DIA_TR_ERR("dia_FactoryPluginCore::setup(DIA_EN_INITLEVEL_0) - creation of config manager failed !!");
         return DIA_FAILED;
      }
      mpConfigMgr->addPropertyBag(OSAL_NEW dia_PropertyBagRAM);
      mpConfigMgr->addPropertyBag(OSAL_NEW dia_PropertyBagFileSwitches);
      mpConfigMgr->addPropertyBag(OSAL_NEW dia_PropertyBagDatapool);
      mpConfigMgr->addPropertyBag(OSAL_NEW dia_PropertyBagSCD);

      mpAppCtrl = getInstanceOfAppController();
      if ( !mpAppCtrl ) {
         DIA_TR_ERR(("dia_FactoryPluginCore::setup(DIA_EN_INITLEVEL_0) - creation of application controller failed !!"));
         return DIA_FAILED;
      }
      (void) assignInterfaceAndNotifier<dia::IApplicationStateControl,dia::IApplicationStateControlListener,dia::IApplicationStateControlNotifier>(mpAppCtrl);

#ifndef __DIA_UNIT_TESTING__
      udd_DriverFacade* pUDDFacade = getInstanceOfFacadeUDD();
      if ( !pUDDFacade )
      {
         DIA_TR_ERR("dia_FactoryPluginCore::setup(DIA_EN_INITLEVEL_0) - creation of UDD Facade failed");
         return DIA_FAILED;
      }
#endif

      mpEngMgr = getInstanceOfEngineManager();
      if ( !mpEngMgr )
      {
         DIA_TR_ERR(("dia_FactoryPluginCore::setup(DIA_EN_INITLEVEL_0) - creation of engine manager failed"));
         return DIA_FAILED;
      }

      // create even more framework objects
      DIA_TR_INF("dia_FactoryPluginCore::setup(DIA_EN_INITLEVEL_0) - create other framework classes");
      dia_ServiceTracker* pSrvCtrl = getInstanceOfServiceTracker();
      dia_CommandController* pCmdCtrl = getInstanceOfCommandController();
      dia_AppController* pAppCtrl = getInstanceOfAppController();

      dia_CommandControllerPluginRunlevel* pCmdCtrlPluginRunlevel = OSAL_NEW dia_CommandControllerPluginRunlevel;
      if ( pCmdCtrl && pCmdCtrlPluginRunlevel )
      {
         pCmdCtrl->addCommandControllerPlugin(pCmdCtrlPluginRunlevel);
         mpAppCtrl->addRunlevelListener(pCmdCtrlPluginRunlevel);
      }

      dia_CommandControllerPluginServiceCCA* pCmdCtrlPluginCCAServices = OSAL_NEW dia_CommandControllerPluginServiceCCA;
      if ( pCmdCtrl && pSrvCtrl && pCmdCtrlPluginCCAServices )
      {
         pCmdCtrl->addCommandControllerPlugin(pCmdCtrlPluginCCAServices);
         pSrvCtrl->addListener(pCmdCtrlPluginCCAServices);
      }

      dia_CommandControllerPluginAppStateChange* pCmdCtrlPluginAppStateChange = OSAL_NEW dia_CommandControllerPluginAppStateChange;
      if ( pCmdCtrl && pAppCtrl && pCmdCtrlPluginAppStateChange )
      {
         pCmdCtrl->addCommandControllerPlugin(pCmdCtrlPluginAppStateChange);
         pAppCtrl->addAppStateListener(pCmdCtrlPluginAppStateChange);
      }

      dia::CommandControllerPluginALDStatus* pCmdCtrlPluginALDStatus = OSAL_NEW dia::CommandControllerPluginALDStatus;
      if ( pCmdCtrl && pCmdCtrlPluginALDStatus )
      {
         pCmdCtrl->addCommandControllerPlugin(pCmdCtrlPluginALDStatus);
      }
#ifndef __DIA_UNIT_TESTING__
      dia::CommandControllerPluginOpMode* pCmdCtrlPluginOpMode = DIA_NEW dia::CommandControllerPluginOpMode;
      if ( pCmdCtrl && pCmdCtrlPluginOpMode )
      {
         pCmdCtrl->addCommandControllerPlugin(pCmdCtrlPluginOpMode);
         //TO DO: Add OpMode Listener here
      }
#endif
      dia::FactoryPluginPDXFlashing* pdxFlashingFactory = getInstanceOfFactoryPluginPDXFlashing();
      if ( pdxFlashingFactory )
      {
         pdxFlashingFactory->addPDXFolder(std::string("/var/opt/bosch/persistent/diagnosis/swu/pdconfig"));
         pdxFlashingFactory->addPDXFolder(std::string("/var/opt/bosch/persistent/diagnosis/swu/cddefconfig"));
      }

      //lint -e429 custodial pointer is freed by this factory class
   }

   if ( level == DIA_EN_INITLEVEL_1 || level == DIA_EN_INITLEVEL_ALL )
   {
      if ( mpApp     && (mpApp->setup()      != DIA_SUCCESS) ) return DIA_FAILED;
      if ( mpAppCtrl && (mpAppCtrl->setup()  != DIA_SUCCESS) ) return DIA_FAILED;
      if ( mpEngMgr  && (mpEngMgr->setup()   != DIA_SUCCESS) ) return DIA_FAILED;

      // extract important information from CMVariantCoding
      evaluateCMVariantCoding();
   }

   setInitializationLevel(level);
   DIA_TR_INF("##### INITIALIZATION OF LEVEL %d DONE #####",(((tS16) level)-1));

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_FactoryPluginCore::tearDownObjects ( dia_enInitLevel level )
{
   dia_tclFnctTrace oTrace("dia_FactoryPluginCore::tearDownObjects");

   dia_LockScope mScopeLock(mSyncObj);

   if ( level == DIA_EN_INITLEVEL_1 || level == DIA_EN_INITLEVEL_ALL )
   {
      if ( mpEngMgr && (mpEngMgr->shutdown() != DIA_SUCCESS) ) return DIA_FAILED;
      if ( mpAppCtrl && (mpAppCtrl->shutdown() != DIA_SUCCESS) ) return DIA_FAILED;
//      if ( mpConfigMgr && (mpConfigMgr->shutdown() != DIA_SUCCESS) ) return DIA_FAILED;
      if ( mpApp && (mpApp->shutdown() != DIA_SUCCESS) ) return DIA_FAILED;
   }

   if ( level == DIA_EN_INITLEVEL_0 || level == DIA_EN_INITLEVEL_ALL )
   {
      if ( mpEngMgr ) {
         releaseInstanceOfEngineManager();
         mpEngMgr = 0;
      }

      if ( mpAppCtrl ) {
         mpAppCtrl->deleteInstance();
         mpAppCtrl = 0;
      }

      if ( mpConfigMgr ) {
         mpConfigMgr->deleteInstance();
         mpConfigMgr = 0;
      }

#if !defined(VARIANT_S_FTR_DONOTUSE_APPLICATION_LOCK) && !defined(__DIA_UNIT_TESTING__)
      if ( mpAppLock )
      {
         dia::releaseInstanceOfApplicationLock();
         mpAppLock = 0;
      }
#endif
      
      if ( mpApp ) {
         mpApp->deleteInstance();
         mpApp = 0;
      }
      // destroy all routine instances
      DIA_IMPL_LIST_REPOSITORY_TEAR_DOWN(dia_CommandControllerPlugin,mCmdCtrlPluginRep);
   }

   if ( getInitializationLevel() > DIA_EN_INITLEVEL_UNKNOWN )
   {
      setInitializationLevel((dia_enInitLevel) (((tS16) level) - 1));
      DIA_TR_INF("##### SHUTDOWN OF LEVEL %d DONE #####",(((tS16) level)-1));
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_FactoryPluginCore::evaluateCMVariantCoding ( void ) const
{
   dia_tclFnctTrace oTrace("dia_FactoryPluginCore::evaluateCMVariantCoding");

   std::vector<tU8> variantConfigStr((size_t)dia_getPropertySize(DIA_PROP_CM_VARIANT_CODING), 0x00);
   if ( dia_getProperty(DIA_PROP_CM_VARIANT_CODING, variantConfigStr) != DIA_SUCCESS )
   {
       DIA_TR_ERR("##### VARIANT CODING NOT AVAILABLE !!! #####");
       return;
   }

// tU8 dataCheck     =  variantConfigStr[0]       & 0x01;
   tU8 interfaceLock = (variantConfigStr[0] >> 1) & 0x01; // 0x02
// tU8 displayType   =  variantConfigStr[2]       & 0x0F;
// tU8 oemType       =  variantConfigStr[3];
   tU8 tunerModeXM   = (variantConfigStr[4] >> 4) & 0x01;
   tU8 tunerModeDAB  = (variantConfigStr[4] >> 5) & 0x01; // 0x20

   if ( tunerModeXM )   DIA_TR_INF("### CM-VARIANT-CODING: XM TUNER");
   if ( tunerModeDAB )  DIA_TR_INF("### CM-VARIANT-CODING: DAB TUNER");
   if ( interfaceLock ) DIA_TR_INF("### CM-VARIANT-CODING: LOCKING ENABLED"); else DIA_TR_INF("### CM-VARIANT-CODING: LOCKING DISABLED");

   if ( dia_setProperty(DIA_PROP_CM_CONFIG_TUNER_SUPPORT_XM,tunerModeXM) != DIA_SUCCESS )
   {
       DIA_TR_INF("### CM-VARIANT-CODING: UNABLE TO UPDATE PROPERTY DIA_PROP_CM_CONFIG_TUNER_SUPPORT_XM");
   }
   if ( dia_setProperty(DIA_PROP_CM_CONFIG_TUNER_SUPPORT_DAB,tunerModeDAB) != DIA_SUCCESS )
   {
       DIA_TR_INF("### CM-VARIANT-CODING: UNABLE TO UPDATE PROPERTY DIA_PROP_CM_CONFIG_TUNER_SUPPORT_DAB");
   }
   if ( dia_setProperty(DIA_PROP_CM_CONFIG_INTERFACE_LOCK,interfaceLock) != DIA_SUCCESS )
   {
       DIA_TR_INF("### CM-VARIANT-CODING: UNABLE TO UPDATE PROPERTY DIA_PROP_CM_CONFIG_INTERFACE_LOCK");
   }
}
