/*!
 * \file       die_error.h
 *
 * \brief      Error and return codes for the diagnosis application
 *
 * \details    Error and return codes for the diagnosis application
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2011-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_ERROR__
#define __INCLUDED_DIA_ERROR__

typedef tU32 tDiaResult;
typedef tU32 tDiaError;

#define DIA_SUCCESS                                         ((tU32) 0x00000000)
#define DIA_DONE                                            ((tU32) 0x00000000)
#define DIA_E_NOERROR                                       ((tU32) 0x00000000)
#define DIA_E_NO_ERROR                                      ((tU32) 0x00000000)

#define DIA_FAILED                                          ((tU32) 0x00000001)
#define DIA_NOT_DONE                                        ((tU32) 0x00000001)
#define DIA_E_ERROR                                         ((tU32) 0x00000001)

#define DIA_E_NOT_SUPPORTED                                 ((tU32) 0x00010002)
#define DIA_E_NOT_IMPLEMENTED                               ((tU32) 0x00010003)
#define DIA_E_NOT_AVAILABLE                                 ((tU32) 0x00010004)
#define DIA_E_NOT_FOUND                                     ((tU32) 0x00010005)
#define DIA_E_TIMEOUT                                       ((tU32) 0x00010006)
#define DIA_E_ALREADY_EXISTS                                ((tU32) 0x00010007)
#define DIA_E_ALREADY_STARTED                               ((tU32) 0x00010008)
#define DIA_E_OBSOLETE                                      ((tU32) 0x00010009)
#define DIA_E_TEMPORARY_NOT_AVAILABLE                       ((tU32) 0x00010001)

#define DIA_E_INVALID_POINTER                               ((tU32) 0x00010010)
#define DIA_E_INVALID_IDENTIFIER                            ((tU32) 0x00010011)
#define DIA_E_INVALID_KEY                                   ((tU32) 0x00010012)
#define DIA_E_INVALID_PROP_TYPE                             ((tU32) 0x00010013)
#define DIA_E_INVALID_LENGTH                                ((tU32) 0x00010014)
#define DIA_E_INVALID_MESSAGE                               ((tU32) 0x00010015)
#define DIA_E_INVALID_STATUS_MASK                           ((tU32) 0x00010016)
#define DIA_E_INVALID_FILE_NAME                             ((tU32) 0x00010017)
#define DIA_E_INVALID_SOURCE                                ((tU32) 0x00010018)
#define DIA_E_INVALID_FILE_HANDLE                           ((tU32) 0x00010021)
#define DIA_E_INVALID_HANDLE                                ((tU32) 0x00010022)

#define DIA_E_FILE_DOES_NOT_EXIST                           ((tU32) 0x00010023)
#define DIA_E_FILE_OPEN_FAILED                              ((tU32) 0x00010024)
#define DIA_E_FILE_CLOSE_FAILED                             ((tU32) 0x00010025)
#define DIA_E_FILE_READ_FAILED                              ((tU32) 0x00010026)
#define DIA_E_FILE_WRITE_FAILED                             ((tU32) 0x00010027)

#define DIA_E_RESPONSE_PENDING                              ((tU32) 0x00010019)
#define DIA_E_UID_COLLISION                                 ((tU32) 0x00010020)

#define DIA_E_IDENTIFIER_NOT_FOUND                          ((tU32) 0x00010030)
#define DIA_E_KEY_NOT_FOUND                                 ((tU32) 0x00010031)
#define DIA_E_KEY_NOT_SUPPORTED                             ((tU32) 0x00010032)
#define DIA_E_KEY_NOT_AVAILABLE                             ((tU32) 0x00010033)
#define DIA_E_KEY_ALREADY_EXISTS                            ((tU32) 0x00010034)

#define DIA_E_CHECKSUM_ERROR                                ((tU32) 0x00000050)

#define DIA_E_DRIVER_NOT_AVAILABLE                          ((tU32) 0x00010060)
#define DIA_E_DRIVER_OPEN_FAILED                            ((tU32) 0x00010061)
#define DIA_E_DRIVER_CLOSE_FAILED                           ((tU32) 0x00010062)
#define DIA_E_DRIVER_IOCTRL_FAILED                          ((tU32) 0x00010063)
#define DIA_E_DRIVER_FLUSH_FAILED                           ((tU32) 0x00010064)
#define DIA_E_DRIVER_FLUSH_DISABLED                         ((tU32) 0x00010065)

#define DIA_E_OPEN_FAILED_TRACE                             ((tU32) 0x00010066)

#define DIA_E_OPEN_FAILED_REGISTRY                          ((tU32) 0x00010070)
#define DIA_E_OPEN_FAILED_KDS                               ((tU32) 0x00010071)
#define DIA_E_OPEN_FAILED_GPIO                              ((tU32) 0x00010072)
#define DIA_E_OPEN_FAILED_ADC                               ((tU32) 0x00010073)

#define DIA_E_CLOSE_FAILED_REGISTRY                         ((tU32) 0x00010074)
#define DIA_E_CLOSE_FAILED_KDS                              ((tU32) 0x00010075)
#define DIA_E_CLOSE_FAILED_GPIO                             ((tU32) 0x00010076)
#define DIA_E_CLOSE_FAILED_ADC                              ((tU32) 0x00010077)

#define DIA_E_READ_FAILED_REGISTRY                          ((tU32) 0x00010078)
#define DIA_E_READ_FAILED_KDS                               ((tU32) 0x00010079)
#define DIA_E_READ_FAILED_GPIO                              ((tU32) 0x0001007A)
#define DIA_E_READ_FAILED_ADC                               ((tU32) 0x0001007B)

#define DIA_E_WRITE_FAILED_REGISTRY                         ((tU32) 0x0001007C)
#define DIA_E_WRITE_FAILED_KDS                              ((tU32) 0x0001007D)
#define DIA_E_WRITE_FAILED_GPIO                             ((tU32) 0x0001007E)
#define DIA_E_WRITE_FAILED_ADC                              ((tU32) 0x0001007F)

#define DIA_E_DELETE_FAILED_KDS                             ((tU32) 0x00010080)

#define DIA_E_CONFMGR_INV_TYPE                              ((tU32) 0x00010090)
#define DIA_E_CONFMGR_INV_PROPBAG                           ((tU32) 0x00010091)
#define DIA_E_CONFMGR_NOT_AVAILABLE                         ((tU32) 0x00010092)

#define DIA_E_INTERFACE_ID_MISMATCH                         ((tU32) 0x000100B0)
#define DIA_E_INTERFACE_PTR_MISMATCH                        ((tU32) 0x000100B1) // Not matching pointer provided to SysAdapterFacade.unsetInterface
#define DIA_E_INTERFACE_NOT_AVAILABLE                       ((tU32) 0x000100B2)

#define DIA_E_THREAD_ALREADY_REGISTERED                     ((tU32) 0x000100D0)
#define DIA_E_THREAD_ID_INVALID                             ((tU32) 0x000100D1)

#define DIA_E_MEMALLOC_FAILED                               ((tU32) 0x000100E0)

#define DIA_E_ACCESS_DENIED                                 ((tU32) 0x000100F0)
#define DIA_E_ACCESS_INVALID_FORMAT                         ((tU32) 0x000100F1)
#define DIA_E_ACCESS_SERVICE_NOT_SUPPORTED                  ((tU32) 0x000100F2)
#define DIA_E_ACCESS_SERVICE_NOT_SUPPORTED_BY_SESSION       ((tU32) 0x000100F3)
#define DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED                  ((tU32) 0x000100F4)
#define DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED_BY_SESSION       ((tU32) 0x000100F5)
#define DIA_E_ACCESS_REJECTED_BY_SECURITY_LEVEL             ((tU32) 0x000100F6)
#define DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED_BY_CALIBRATION   ((tU32) 0x000100F7)
#define DIA_E_ACCESS_GENERAL_PROGRAMMING_FAILURE            ((tU32) 0x000100F8)
#define DIA_E_ACCESS_DID_NOT_SUPPORTED                      ((tU32) 0x000100F9)
#define DIA_E_ACCESS_DID_NOT_SUPPORTED_BY_SESSION           ((tU32) 0x000100FA)

#define DIA_E_LOCK_FAILED                                   ((tU32) 0x00010100)
#define DIA_E_LOCK_TIMEOUT                                  ((tU32) 0x00010101)
#define DIA_E_CONDVAR_FAILED                                ((tU32) 0x00010102)
#define DIA_E_CONDVAR_TIMEOUT                               ((tU32) 0x00010103)
#define DIA_E_CONDVAR_BROADCAST                             ((tU32) 0x00010104)

#define DIA_E_TEST_EXECUTION_FAILED                         ((tU32) 0x00010105)

#define DIA_E_CONDITIONS_NOT_CORRECT                        ((tU32) 0x00010110)
#define DIA_E_OUT_OF_RANGE                                  ((tU32) 0x00010111)
#define DIA_E_SEQUENCE_ERROR                                ((tU32) 0x00010112)
#define DIA_E_BUSY_REPEAT                                   ((tU32) 0x00010113)
#define DIA_E_ROUTINE_NOT_COMPLETE                          ((tU32) 0x00010114)
#define DIA_E_SECURITY_ACCESS_DENIED                        ((tU32) 0x00010115)
#define DIA_E_INVALID_SECURITY_KEY                          ((tU32) 0x00010116)
#define DIA_E_EXCEED_NUMBER_OF_ATTEMPTS                     ((tU32) 0x00010117)
#define DIA_E_SUBFUNCTION_NOT_SUPPORTED                     ((tU32) 0x00010118)
#define DIA_E_SUBFUNCTION_NOT_SUPPORTED_IN_ACTIVE_SESSION   ((tU32) 0x00010119)
#define DIA_E_SERVICE_NOT_SUPPORTED                         ((tU32) 0x0001011A)
#define DIA_E_SERVICE_NOT_SUPPORTED_IN_ACTIVE_SESSION       ((tU32) 0x0001011B)
#define DIA_E_REQUIRED_TIME_DELAY_NOT_EXPIRED               ((tU32) 0x0001011C)
#define DIA_E_GENERAL_REJECT                                ((tU32) 0x0001011D)
#define DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT      ((tU32) 0x0001011E)
#define DIA_E_GENERAL_PROGRAMMING_FAILURE                   ((tU32) 0x0001011F)
#define DIA_E_WRONG_BLOCK_SEQUENCE_NUMBER                   ((tU32) 0x00010120)
#define DIA_E_TRANSFER_DATA_SUSPENDED                       ((tU32) 0x00010121)
#define DIA_E_ENGINE_IS_RUNNING                             ((tU32) 0x00010122)
#define DIA_E_VEHICLE_SPEED_TOO_HIGH                        ((tU32) 0x00010123)
#define DIA_E_FAILURE_PREVENTS_EXECUTION_OF_REQUESTED_ACTION ((tU32)0x00010124)
#define DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED                  ((tU32) 0x00010125)
#define DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_DIR_READ         ((tU32) 0x00010126)
#define DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_ADD         ((tU32) 0x00010127)
#define DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_DELETE      ((tU32) 0x00010128)
#define DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_READ        ((tU32) 0x00010129)
#define DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_REPLACE     ((tU32) 0x0001012A)

#define DIA_E_TIMER_NOT_ACTIVE                              ((tU32) 0x00010200)
#define DIA_E_TIMER_HAS_NO_LISTENERS                        ((tU32) 0x00010201)

#define DIA_E_DATASET_NOT_SUPPORTED                         ((tU32) 0x00010300)
#define DIA_E_DATASET_NOT_AVAILABLE                         ((tU32) 0x00010301)
#define DIA_E_DATASET_NOT_FOUND                             ((tU32) 0x00010302)
#define DIA_E_DATASET_INVALID_LENGTH                        ((tU32) 0x00010303)
#define DIA_E_DATASET_INVALID_HASH                          ((tU32) 0x00010304)
#define DIA_E_DATASET_INVALID_DATA_POINTER                  ((tU32) 0x00010305)
#define DIA_E_DATASET_INVALID_DATA_RANGE                    ((tU32) 0x00010306)
#define DIA_E_DATASET_INVALID_DATA_RANGE_START_BYTE         ((tU32) 0x00010307)
#define DIA_E_DATASET_INVALID_DATA_RANGE_STOP_BYTE          ((tU32) 0x00010308)
#define DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER         ((tU32) 0x00010309)
#define DIA_E_DATASET_LOAD_ERROR                            ((tU32) 0x0001030A)
#define DIA_E_DATASET_STORE_ERROR                           ((tU32) 0x0001030B)
#define DIA_E_DATASET_ALREADY_COMPLETED                     ((tU32) 0x0001030C)
#define DIA_E_DATASET_INVALID_OPERATION_MODE                ((tU32) 0x0001030D)
#define DIA_E_DATASET_DATA_UPLOAD_FAILED                    ((tU32) 0x0001030E)
#define DIA_E_DATA_DOWNLOAD_NO_DATA                         ((tU32) 0x0001030F)
#define DIA_E_DATA_DOWNLOAD_INVALID_DATA                    ((tU32) 0x00010310)
#define DIA_E_UPLOAD_DOWNLOAD_REJECTED_DUE_TO_POWER_LEVEL   ((tU32) 0x00010312)
#define DIA_E_UPLOAD_DOWNLOAD_REQUEST_NO_STRATEGY           ((tU32) 0x00010313)
#define DIA_E_UPLOAD_DOWNLOAD_TRANSFER_NO_STRATEGY          ((tU32) 0x00010314)
#define DIA_E_UPLOAD_DOWNLOAD_EXIT_NO_STRATEGY              ((tU32) 0x00010315)
#define DIA_E_FILE_TRANSFER_NO_STRATEGY                     ((tU32) 0x00010316)
#define DIA_E_FILE_TRANSFER_REJECTED_DUE_TO_POWER_LEVEL     ((tU32) 0x00010317)

#define DIA_E_MEDIA_NOT_FOUND                               ((tU32) 0x00010400)
#define DIA_E_MEDIA_INVALID                                 ((tU32) 0x00010401)

#define DIA_E_ROUTINE_COMPLETE                              ((tU32) 0x00000000)
#define DIA_E_ROUTINE_INCOMPLETE                            ((tU32) 0x00010410)
#define DIA_E_ROUTINE_FAILED                                ((tU32) 0x00010411)
#define DIA_E_ROUTINE_ABORTED                               ((tU32) 0x00010412)
#define DIA_E_ROUTINE_BUSY                                  ((tU32) 0x00010413)
#define DIA_E_ROUTINE_RESULT_PENDING                        ((tU32) 0x00010414)
#define DIA_E_ROUTINE_SUBFUNCTION_NOT_SUPPORTED             ((tU32) 0x00010415)
#define DIA_E_ROUTINE_START_FAILED                          ((tU32) 0x00010416)
#define DIA_E_ROUTINE_TIMEOUT                               ((tU32) 0x00010417)

#define DIA_E_FILE_OPERATION_FAILED_OPEN                    ((tU32) 0x00010440)
#define DIA_E_FILE_OPERATION_FAILED_CLOSE                   ((tU32) 0x00010441)
#define DIA_E_FILE_OPERATION_FAILED_READ                    ((tU32) 0x00010442)
#define DIA_E_FILE_OPERATION_FAILED_WRITE                   ((tU32) 0x00010443)

#define DIA_E_VOLTAGE_TOO_HIGH                              ((tU32) 0x00010500)
#define DIA_E_VOLTAGE_TOO_LOW                               ((tU32) 0x00010501)

#define DIA_E_SERVICE_HANDLER_NOT_AVAILABLE                 ((tU32) 0x00010600)

#define DIA_E_IOCTRL_PRECONDITIONS_NOT_MATCHED              ((tU32) 0x00010700)

#define DIA_E_PROCESSING_FAILED                             ((tU32) 0x00010800)
#define DIA_E_INVALID_REQUEST                               ((tU32) 0x00010801)

#define DIA_E_SYSSET_PLUGIN_PREPARATION_FAILED              ((tU32) 0x00010900)
#define DIA_E_SYSSET_PLUGIN_EXECUTION_FAILED                ((tU32) 0x00010901)
#define DIA_E_SYSSET_PLUGIN_FINALIZATION_FAILED             ((tU32) 0x00010902)
#define DIA_E_SYSSET_INVALID_TYPE                           ((tU32) 0x00010903)
#define DIA_E_SYSSET_INVALID_AREA                           ((tU32) 0x00010904)
#define DIA_E_SYSSET_INVALID_MODE                           ((tU32) 0x00010905)
#define DIA_E_SYSSET_REQUEST_REJECTED                       ((tU32) 0x00010906)
#define DIA_E_SYSSET_REQUEST_REJECTED_INVALID_MODE          ((tU32) 0x00010907)

#define DIA_E_SOCKET_CONFIG_NOT_AVAILABLE                   ((tU32) 0x00010A00)
#define DIA_E_SOCKET_CREATION_BYPASSED                      ((tU32) 0x00010A01)
#define DIA_E_SOCKET_CREATION_FAILED                        ((tU32) 0x00010A02)
#define DIA_E_SOCKET_DGRAM_INIT_FAILED                      ((tU32) 0x00010A03)
#define DIA_E_SOCKET_LOCAL_HOST_NOT_AVAILABLE               ((tU32) 0x00010A04)
#define DIA_E_SOCKET_LOCAL_BIND_FAILED                      ((tU32) 0x00010A05)
#define DIA_E_SOCKET_SERVER_HOST_NOT_AVAILABLE              ((tU32) 0x00010A06)
#define DIA_E_SOCKET_CONNECT_FAILED                         ((tU32) 0x00010A07)
#define DIA_E_SOCKET_NOT_AVAILABLE                          ((tU32) 0x00010A08)
#define DIA_E_SOCKET_WRITE_DATA_FAILED                      ((tU32) 0x00010A09)
#define DIA_E_SOCKET_WRITE_DATA_FAILED_NO_DATA              ((tU32) 0x00010A0A)
#define DIA_E_SOCKET_READ_DATA_FAILED                       ((tU32) 0x00010A0B)

#define DIA_E_DYNLOAD_MANIFEST_NOT_AVAILABLE                ((tU32) 0x00010B00)
#define DIA_E_DYNLOAD_INVALID_HANDLE                        ((tU32) 0x00010B01)
#define DIA_E_DYNLOAD_INVALID_POINTER                       ((tU32) 0x00010B02)
#define DIA_E_DYNLOAD_INVALID_SYMBOL                        ((tU32) 0x00010B03)
#define DIA_E_DYNLOAD_INVALID_ITEM_TYPE                     ((tU32) 0x00010B04)
#define DIA_E_DYNLOAD_OPEN_FAILED                           ((tU32) 0x00010B05)
#define DIA_E_DYNLOAD_CLOSE_FAILED                          ((tU32) 0x00010B06)
#define DIA_E_DYNLOAD_ITEM_NOT_FOUND                        ((tU32) 0x00010B07)

#define DIA_E_ASF_PROXY_NOT_AVAILABLE                       ((tU32) 0x00010C00)
#define DIA_E_OUT_OF_MEMORY                                 ((tU32) 0x00010C01)

#define DIA_E_FSM_NOT_AVAILABLE                             ((tU32) 0x00010D00)

#define DIA_E_NO_INSTANCE_OF_APPLICATION                    ((tU32) 0x00010E00)
#define DIA_E_NO_INSTANCE_OF_APPLICATION_CONTROLLER         ((tU32) 0x00010E01)
#define DIA_E_NO_INSTANCE_OF_COMMUNICATION_CHANNEL_INC      ((tU32) 0x00010E02)
#define DIA_E_NO_INSTANCE_OF_CONFIG_MANAGER                 ((tU32) 0x00010E03)
#define DIA_E_NO_INSTANCE_OF_DATA_DOWNLOAD_MANAGER          ((tU32) 0x00010E04)
#define DIA_E_NO_INSTANCE_OF_DYNAMIC_LOADER                 ((tU32) 0x00010E05)
#define DIA_E_NO_INSTANCE_OF_ENGINE_MANAGER                 ((tU32) 0x00010E06)
#define DIA_E_NO_INSTANCE_OF_FACTORY                        ((tU32) 0x00010E07)
#define DIA_E_NO_INSTANCE_OF_IO_CONTROL_MANAGER             ((tU32) 0x00010E08)
#define DIA_E_NO_INSTANCE_OF_ROUTINE_CONTROL_MANAGER        ((tU32) 0x00010E09)
#define DIA_E_NO_INSTANCE_OF_SECURITY_MANAGER               ((tU32) 0x00010E0A)
#define DIA_E_NO_INSTANCE_OF_SYSTEM_SETTINGS_MANAGER        ((tU32) 0x00010E0B)

#define DIA_E_AUDIO_CONNECTION_FAILED                       ((tU32) 0x00010F00)
#define DIA_E_AUDIO_CONNECTION_NOT_SUPPORTED                ((tU32) 0x00010F01)
#define DIA_E_AUDIO_SINK_NOT_SUPPORTED                      ((tU32) 0x00010F02)
#define DIA_E_AUDIO_SOURCE_NOT_SUPPORTED                    ((tU32) 0x00010F03)

#define DIA_E_EMPTY_STRING                                  ((tU32) 0x00010F04)

#define DIA_E_FLASH_ENGINE_UNKNOWN_JOB                      ((tU32) 0x00011000)
#define DIA_E_FLASH_ENGINE_UNPACKING_FAILED                 ((tU32) 0x00011001)
#define DIA_E_FLASH_ENGINE_WDBI_ERROR                       ((tU32) 0x00011002)
#define DIA_E_FLASH_ENGINE_SESSION_CHANGE_ERROR             ((tU32) 0x00011003)
#define DIA_E_FLASH_ENGINE_REQUEST_DOWNLOAD_ERROR           ((tU32) 0x00011004)
#define DIA_E_FLASH_ENGINE_TRANSFER_DATA_ERROR              ((tU32) 0x00011005)
#define DIA_E_FLASH_ENGINE_TRANSFER_EXIT_ERROR              ((tU32) 0x00011006)
#define DIA_E_FLASH_ENGINE_FLASH_CONTAINER_ERROR            ((tU32) 0x00011007)
#define DIA_E_FLASH_ENGINE_EXPECTED_IDENT_ERROR             ((tU32) 0x00011008)
#define DIA_E_FLASH_ENGINE_DISABLE_FLUSH_ERROR              ((tU32) 0x00011009)
#define DIA_E_FLASH_ENGINE_ENABLE_FLUSH_ERROR               ((tU32) 0x0001100A)
#define DIA_E_FLASH_ENGINE_DATA_ERROR                       ((tU32) 0x0001100B)
#define DIA_E_FLASH_ENGINE_CATALOG_ERROR                    ((tU32) 0x0001100C)

#define DIA_E_SESSION_CONTROLLER_SETUP_LOAD_FAILED          ((tU32) 0x00012000)
#define DIA_E_SESSION_CONTROLLER_SETUP_DEFAULT_SESSION_NOK  ((tU32) 0x00012001)

#ifndef __DIA_UNIT_TESTING__

#define DIA_ASSERT_ALWAYS()   \
   DIA_TR_ERR("!!! DIA_ASSERT %s line#%d !!!", __FILE__, __LINE__); \
   NORMAL_M_ASSERT_ALWAYS()

#define DIA_FATAL_ASSERT_ALWAYS()   \
   DIA_TR_ERR("!!! DIA_FATAL_ASSERT %s line#%d !!!", __FILE__, __LINE__); \
   FATAL_M_ASSERT_ALWAYS()

#define DIA_ASSERT(a)       \
   if (!(a))                \
   {                           \
      DIA_ASSERT_ALWAYS(); \
   }
#else

#include <gtest/gtest.h>
#define DIA_ASSERT_ALWAYS() \
   { \
      DIA_TR_ERR("!!! DIA_ASSERT (UNIT_TEST_MODE) %s line#%d !!!", __FILE__, __LINE__); \
      ADD_FAILURE(); \
      assert(0); \
   }

#define DIA_FATAL_ASSERT_ALWAYS() \
   { \
      DIA_TR_ERR("!!! DIA_FATAL_ASSERT (UNIT_TEST_MODE) %s line#%d !!!", __FILE__, __LINE__); \
      ADD_FAILURE(); \
      assert(0); \
   }

#define DIA_ASSERT(a) \
   if (!(a))                   \
   {                           \
      DIA_ASSERT_ALWAYS();     \
   }


#endif

struct dia_ResultInfo
{
    tDiaResult  mResultValue;
    const char* mResultString;
};

const dia_ResultInfo
aCommonResultData[] =
{
//    Value                                                       String
   { DIA_SUCCESS,                                                 "DIA_SUCCESS" },
   { DIA_FAILED,                                                  "DIA_FAILED" },
   { DIA_E_NOT_SUPPORTED,                                         "DIA_E_NOT_SUPPORTED" },
   { DIA_E_NOT_IMPLEMENTED,                                       "DIA_E_NOT_IMPLEMENTED" },
   { DIA_E_NOT_AVAILABLE,                                         "DIA_E_NOT_AVAILABLE" },
   { DIA_E_NOT_FOUND,                                             "DIA_E_NOT_FOUND" },
   { DIA_E_TIMEOUT,                                               "DIA_E_TIMEOUT" },
   { DIA_E_ALREADY_EXISTS,                                        "DIA_E_ALREADY_EXISTS" },
   { DIA_E_ALREADY_STARTED,                                       "DIA_E_ALREADY_STARTED" },
   { DIA_E_OBSOLETE,                                              "DIA_E_OBSOLETE" },
   { DIA_E_TEMPORARY_NOT_AVAILABLE,                               "DIA_E_TEMPORARY_NOT_AVAILABLE" },
   { DIA_E_INVALID_POINTER,                                       "DIA_E_INVALID_POINTER" },
   { DIA_E_INVALID_IDENTIFIER,                                    "DIA_E_INVALID_IDENTIFIER" },
   { DIA_E_INVALID_KEY,                                           "DIA_E_INVALID_KEY" },
   { DIA_E_INVALID_PROP_TYPE,                                     "DIA_E_INVALID_PROP_TYPE" },
   { DIA_E_INVALID_LENGTH,                                        "DIA_E_INVALID_LENGTH" },
   { DIA_E_INVALID_MESSAGE,                                       "DIA_E_INVALID_MESSAGE" },
   { DIA_E_INVALID_STATUS_MASK,                                   "DIA_E_INVALID_STATUS_MASK" },
   { DIA_E_INVALID_FILE_NAME,                                     "DIA_E_INVALID_FILE_NAME" },
   { DIA_E_INVALID_SOURCE,                                        "DIA_E_INVALID_SOURCE" },
   { DIA_E_RESPONSE_PENDING,                                      "DIA_E_RESPONSE_PENDING" },
   { DIA_E_UID_COLLISION,                                         "DIA_E_UID_COLLISION" },
   { DIA_E_IDENTIFIER_NOT_FOUND,                                  "DIA_E_IDENTIFIER_NOT_FOUND" },
   { DIA_E_KEY_NOT_FOUND,                                         "DIA_E_KEY_NOT_FOUND" },
   { DIA_E_KEY_NOT_SUPPORTED,                                     "DIA_E_KEY_NOT_SUPPORTED" },
   { DIA_E_KEY_NOT_AVAILABLE,                                     "DIA_E_KEY_NOT_AVAILABLE" },
   { DIA_E_KEY_ALREADY_EXISTS,                                    "DIA_E_KEY_ALREADY_EXISTS" },
   { DIA_E_CHECKSUM_ERROR,                                        "DIA_E_CHECKSUM_ERROR" },
   { DIA_E_DRIVER_NOT_AVAILABLE,                                  "DIA_E_DRIVER_NOT_AVAILABLE" },
   { DIA_E_DRIVER_OPEN_FAILED,                                    "DIA_E_DRIVER_OPEN_FAILED" },
   { DIA_E_DRIVER_CLOSE_FAILED,                                   "DIA_E_DRIVER_CLOSE_FAILED" },
   { DIA_E_DRIVER_IOCTRL_FAILED,                                  "DIA_E_DRIVER_IOCTRL_FAILED" },
   { DIA_E_DRIVER_FLUSH_FAILED,                                   "DIA_E_DRIVER_FLUSH_FAILED" },
   { DIA_E_DRIVER_FLUSH_DISABLED,                                 "DIA_E_DRIVER_FLUSH_DISABLED" },
   { DIA_E_OPEN_FAILED_TRACE,                                     "DIA_E_OPEN_FAILED_TRACE" },
   { DIA_E_OPEN_FAILED_REGISTRY,                                  "DIA_E_OPEN_FAILED_REGISTRY" },
   { DIA_E_OPEN_FAILED_KDS,                                       "DIA_E_OPEN_FAILED_KDS" },
   { DIA_E_OPEN_FAILED_GPIO,                                      "DIA_E_OPEN_FAILED_GPIO" },
   { DIA_E_OPEN_FAILED_ADC,                                       "DIA_E_OPEN_FAILED_ADC" },
   { DIA_E_CLOSE_FAILED_REGISTRY,                                 "DIA_E_CLOSE_FAILED_REGISTRY" },
   { DIA_E_CLOSE_FAILED_KDS,                                      "DIA_E_CLOSE_FAILED_KDS" },
   { DIA_E_CLOSE_FAILED_GPIO,                                     "DIA_E_CLOSE_FAILED_GPIO" },
   { DIA_E_CLOSE_FAILED_ADC,                                      "DIA_E_CLOSE_FAILED_ADC" },
   { DIA_E_READ_FAILED_REGISTRY,                                  "DIA_E_READ_FAILED_REGISTRY" },
   { DIA_E_READ_FAILED_KDS,                                       "DIA_E_READ_FAILED_KDS" },
   { DIA_E_READ_FAILED_GPIO,                                      "DIA_E_READ_FAILED_GPIO" },
   { DIA_E_READ_FAILED_ADC,                                       "DIA_E_READ_FAILED_ADC" },
   { DIA_E_WRITE_FAILED_REGISTRY,                                 "DIA_E_WRITE_FAILED_REGISTRY" },
   { DIA_E_WRITE_FAILED_KDS,                                      "DIA_E_WRITE_FAILED_KDS" },
   { DIA_E_WRITE_FAILED_GPIO,                                     "DIA_E_WRITE_FAILED_GPIO" },
   { DIA_E_WRITE_FAILED_ADC,                                      "DIA_E_WRITE_FAILED_ADC" },
   { DIA_E_DELETE_FAILED_KDS,                                     "DIA_E_DELETE_FAILED_KDS" },
   { DIA_E_CONFMGR_INV_TYPE,                                      "DIA_E_CONFMGR_INV_TYPE" },
   { DIA_E_CONFMGR_INV_PROPBAG,                                   "DIA_E_CONFMGR_INV_PROPBAG" },
   { DIA_E_CONFMGR_NOT_AVAILABLE,                                 "DIA_E_CONFMGR_NOT_AVAILABLE" },
   { DIA_E_INTERFACE_ID_MISMATCH,                                 "DIA_E_INTERFACE_ID_MISMATCH" },
   { DIA_E_INTERFACE_PTR_MISMATCH,                                "DIA_E_INTERFACE_PTR_MISMATCH" },
   { DIA_E_THREAD_ALREADY_REGISTERED,                             "DIA_E_THREAD_ALREADY_REGISTERED" },
   { DIA_E_THREAD_ID_INVALID,                                     "DIA_E_THREAD_ID_INVALID" },
   { DIA_E_MEMALLOC_FAILED,                                       "DIA_E_MEMALLOC_FAILED" },
   { DIA_E_ACCESS_DENIED,                                         "DIA_E_ACCESS_DENIED" },
   { DIA_E_ACCESS_INVALID_FORMAT,                                 "DIA_E_ACCESS_INVALID_FORMAT" },
   { DIA_E_ACCESS_SERVICE_NOT_SUPPORTED,                          "DIA_E_ACCESS_SERVICE_NOT_SUPPORTED" },
   { DIA_E_ACCESS_SERVICE_NOT_SUPPORTED_BY_SESSION,               "DIA_E_ACCESS_SERVICE_NOT_SUPPORTED_BY_SESSION" },
   { DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED,                          "DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED" },
   { DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED_BY_SESSION,               "DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED_BY_SESSION" },
   { DIA_E_ACCESS_REJECTED_BY_SECURITY_LEVEL,                     "DIA_E_ACCESS_REJECTED_BY_SECURITY_LEVEL" },
   { DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED_BY_CALIBRATION,           "DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED_BY_CALIBRATION" },
   { DIA_E_ACCESS_GENERAL_PROGRAMMING_FAILURE,                    "DIA_E_ACCESS_GENERAL_PROGRAMMING_FAILURE" },
   { DIA_E_ACCESS_DID_NOT_SUPPORTED,                              "DIA_E_ACCESS_DID_NOT_SUPPORTED" },
   { DIA_E_ACCESS_DID_NOT_SUPPORTED_BY_SESSION,                   "DIA_E_ACCESS_DID_NOT_SUPPORTED_BY_SESSION" },
   { DIA_E_LOCK_FAILED,                                           "DIA_E_LOCK_FAILED" },
   { DIA_E_LOCK_TIMEOUT,                                          "DIA_E_LOCK_TIMEOUT" },
   { DIA_E_CONDVAR_FAILED,                                        "DIA_E_CONDVAR_FAILED" },
   { DIA_E_CONDVAR_TIMEOUT,                                       "DIA_E_CONDVAR_TIMEOUT" },
   { DIA_E_CONDITIONS_NOT_CORRECT,                                "DIA_E_CONDITIONS_NOT_CORRECT" },
   { DIA_E_OUT_OF_RANGE,                                          "DIA_E_OUT_OF_RANGE" },
   { DIA_E_SEQUENCE_ERROR,                                        "DIA_E_SEQUENCE_ERROR" },
   { DIA_E_BUSY_REPEAT,                                           "DIA_E_BUSY_REPEAT" },
   { DIA_E_ROUTINE_NOT_COMPLETE,                                  "DIA_E_ROUTINE_NOT_COMPLETE" },
   { DIA_E_SECURITY_ACCESS_DENIED,                                "DIA_E_SECURITY_ACCESS_DENIED" },
   { DIA_E_INVALID_SECURITY_KEY,                                  "DIA_E_INVALID_SECURITY_KEY" },
   { DIA_E_EXCEED_NUMBER_OF_ATTEMPTS,                             "DIA_E_EXCEED_NUMBER_OF_ATTEMPTS" },
   { DIA_E_SUBFUNCTION_NOT_SUPPORTED,                             "DIA_E_SUBFUNCTION_NOT_SUPPORTED" },
   { DIA_E_SUBFUNCTION_NOT_SUPPORTED_IN_ACTIVE_SESSION,           "DIA_E_SUBFUNCTION_NOT_SUPPORTED_IN_ACTIVE_SESSION" },
   { DIA_E_SERVICE_NOT_SUPPORTED,                                 "DIA_E_SERVICE_NOT_SUPPORTED" },
   { DIA_E_SERVICE_NOT_SUPPORTED_IN_ACTIVE_SESSION,               "DIA_E_SERVICE_NOT_SUPPORTED_IN_ACTIVE_SESSION" },
   { DIA_E_REQUIRED_TIME_DELAY_NOT_EXPIRED,                       "DIA_E_REQUIRED_TIME_DELAY_NOT_EXPIRED" },
   { DIA_E_GENERAL_REJECT,                                        "DIA_E_GENERAL_REJECT" },
   { DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT,              "DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT" },
   { DIA_E_GENERAL_PROGRAMMING_FAILURE,                           "DIA_E_GENERAL_PROGRAMMING_FAILURE" },
   { DIA_E_WRONG_BLOCK_SEQUENCE_NUMBER,                           "DIA_E_WRONG_BLOCK_SEQUENCE_NUMBER" },
   { DIA_E_TRANSFER_DATA_SUSPENDED,                               "DIA_E_TRANSFER_DATA_SUSPENDED" },
   { DIA_E_ENGINE_IS_RUNNING,                                     "DIA_E_ENGINE_IS_RUNNING" },
   { DIA_E_VEHICLE_SPEED_TOO_HIGH,                                "DIA_E_VEHICLE_SPEED_TOO_HIGH" },
   { DIA_E_TIMER_NOT_ACTIVE,                                      "DIA_E_TIMER_NOT_ACTIVE" },
   { DIA_E_TIMER_HAS_NO_LISTENERS,                                "DIA_E_TIMER_HAS_NO_LISTENERS" },
   { DIA_E_DATASET_NOT_SUPPORTED,                                 "DIA_E_DATASET_NOT_SUPPORTED" },
   { DIA_E_DATASET_NOT_AVAILABLE,                                 "DIA_E_DATASET_NOT_AVAILABLE" },
   { DIA_E_DATASET_NOT_FOUND,                                     "DIA_E_DATASET_NOT_FOUND" },
   { DIA_E_DATASET_INVALID_LENGTH,                                "DIA_E_DATASET_INVALID_LENGTH" },
   { DIA_E_DATASET_INVALID_HASH,                                  "DIA_E_DATASET_INVALID_HASH" },
   { DIA_E_DATASET_INVALID_DATA_POINTER,                          "DIA_E_DATASET_INVALID_DATA_POINTER" },
   { DIA_E_DATASET_INVALID_DATA_RANGE,                            "DIA_E_DATASET_INVALID_DATA_RANGE" },
   { DIA_E_DATASET_INVALID_DATA_RANGE_START_BYTE,                 "DIA_E_DATASET_INVALID_DATA_RANGE_START_BYTE" },
   { DIA_E_DATASET_INVALID_DATA_RANGE_STOP_BYTE,                  "DIA_E_DATASET_INVALID_DATA_RANGE_STOP_BYTE" },
   { DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER,                 "DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER" },
   { DIA_E_DATASET_LOAD_ERROR,                                    "DIA_E_DATASET_LOAD_ERROR" },
   { DIA_E_DATASET_STORE_ERROR,                                   "DIA_E_DATASET_STORE_ERROR" },
   { DIA_E_DATASET_ALREADY_COMPLETED,                             "DIA_E_DATASET_ALREADY_COMPLETED" },
   { DIA_E_DATASET_INVALID_OPERATION_MODE,                        "DIA_E_DATASET_INVALID_OPERATION_MODE" },
   { DIA_E_DATASET_DATA_UPLOAD_FAILED,                            "DIA_E_DATASET_DATA_UPLOAD_FAILED" },
   { DIA_E_DATA_DOWNLOAD_NO_DATA,                                 "DIA_E_DATA_DOWNLOAD_NO_DATA" },
   { DIA_E_DATA_DOWNLOAD_INVALID_DATA,                            "DIA_E_DATA_DOWNLOAD_INVALID_DATA" },
   { DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED,                          "DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED" },
   { DIA_E_UPLOAD_DOWNLOAD_REJECTED_DUE_TO_POWER_LEVEL,           "DIA_E_UPLOAD_DOWNLOAD_REJECTED_DUE_TO_POWER_LEVEL" },
   { DIA_E_MEDIA_NOT_FOUND,                                       "DIA_E_MEDIA_NOT_FOUND" },
   { DIA_E_MEDIA_INVALID,                                         "DIA_E_MEDIA_INVALID" },
   { DIA_E_ROUTINE_COMPLETE,                                      "DIA_E_ROUTINE_COMPLETE" },
   { DIA_E_ROUTINE_INCOMPLETE,                                    "DIA_E_ROUTINE_INCOMPLETE" },
   { DIA_E_ROUTINE_FAILED,                                        "DIA_E_ROUTINE_FAILED" },
   { DIA_E_ROUTINE_ABORTED,                                       "DIA_E_ROUTINE_ABORTED" },
   { DIA_E_ROUTINE_BUSY,                                          "DIA_E_ROUTINE_BUSY" },
   { DIA_E_ROUTINE_RESULT_PENDING,                                "DIA_E_ROUTINE_RESULT_PENDING" },
   { DIA_E_ROUTINE_SUBFUNCTION_NOT_SUPPORTED,                     "DIA_E_ROUTINE_SUBFUNCTION_NOT_SUPPORTED" },
   { DIA_E_ROUTINE_START_FAILED,                                  "DIA_E_ROUTINE_START_FAILED" },
   { DIA_E_ROUTINE_TIMEOUT,                                       "DIA_E_ROUTINE_TIMEOUT" },
   { DIA_E_FILE_OPERATION_FAILED_OPEN,                            "DIA_E_FILE_OPERATION_FAILED_OPEN" },
   { DIA_E_FILE_OPERATION_FAILED_CLOSE,                           "DIA_E_FILE_OPERATION_FAILED_CLOSE" },
   { DIA_E_FILE_OPERATION_FAILED_READ,                            "DIA_E_FILE_OPERATION_FAILED_READ" },
   { DIA_E_FILE_OPERATION_FAILED_WRITE,                           "DIA_E_FILE_OPERATION_FAILED_WRITE" },
   { DIA_E_VOLTAGE_TOO_HIGH,                                      "DIA_E_VOLTAGE_TOO_HIGH" },
   { DIA_E_VOLTAGE_TOO_LOW,                                       "DIA_E_VOLTAGE_TOO_LOW" },
   { DIA_E_SERVICE_HANDLER_NOT_AVAILABLE,                         "DIA_E_SERVICE_HANDLER_NOT_AVAILABLE" },
   { DIA_E_IOCTRL_PRECONDITIONS_NOT_MATCHED,                      "DIA_E_IOCTRL_PRECONDITIONS_NOT_MATCHED" },
   { DIA_E_PROCESSING_FAILED,                                     "DIA_E_PROCESSING_FAILED" },
   { DIA_E_INVALID_REQUEST,                                       "DIA_E_INVALID_REQUEST" },
   { DIA_E_SYSSET_PLUGIN_PREPARATION_FAILED,                      "DIA_E_SYSSET_PLUGIN_PREPARATION_FAILED" },
   { DIA_E_SYSSET_PLUGIN_EXECUTION_FAILED,                        "DIA_E_SYSSET_PLUGIN_EXECUTION_FAILED" },
   { DIA_E_SYSSET_PLUGIN_FINALIZATION_FAILED,                     "DIA_E_SYSSET_PLUGIN_FINALIZATION_FAILED" },
   { DIA_E_SYSSET_INVALID_TYPE,                                   "DIA_E_SYSSET_INVALID_TYPE" },
   { DIA_E_SYSSET_INVALID_AREA,                                   "DIA_E_SYSSET_INVALID_AREA" },
   { DIA_E_SYSSET_INVALID_MODE,                                   "DIA_E_SYSSET_INVALID_MODE" },
   { DIA_E_SYSSET_REQUEST_REJECTED,                               "DIA_E_SYSSET_REQUEST_REJECTED" },
   { DIA_E_SYSSET_REQUEST_REJECTED_INVALID_MODE,                  "DIA_E_SYSSET_REQUEST_REJECTED_INVALID_MODE" },
   { DIA_E_FILE_DOES_NOT_EXIST,                                   "DIA_E_FILE_DOES_NOT_EXIST" },
   { DIA_E_FILE_OPEN_FAILED,                                      "DIA_E_FILE_OPEN_FAILED" },
   { DIA_E_FILE_CLOSE_FAILED,                                     "DIA_E_FILE_CLOSE_FAILED" },
   { DIA_E_FILE_READ_FAILED,                                      "DIA_E_FILE_READ_FAILED" },
   { DIA_E_FILE_WRITE_FAILED,                                     "DIA_E_FILE_WRITE_FAILED" },
   { DIA_E_AUDIO_CONNECTION_FAILED,                               "DIA_E_AUDIO_CONNECTION_FAILED" },
   { DIA_E_AUDIO_CONNECTION_NOT_SUPPORTED,                        "DIA_E_AUDIO_CONNECTION_NOT_SUPPORTED" },
   { DIA_E_AUDIO_SINK_NOT_SUPPORTED,                              "DIA_E_AUDIO_SINK_NOT_SUPPORTED" },
   { DIA_E_AUDIO_SOURCE_NOT_SUPPORTED,                            "DIA_E_AUDIO_SOURCE_NOT_SUPPORTED" },
   { DIA_E_FLASH_ENGINE_UNKNOWN_JOB,                              "DIA_E_FLASH_ENGINE_UNKNOWN_JOB" },
   { DIA_E_FLASH_ENGINE_UNPACKING_FAILED,                         "DIA_E_FLASH_ENGINE_UNPACKING_FAILED" },
   { DIA_E_FLASH_ENGINE_WDBI_ERROR,                               "DIA_E_FLASH_ENGINE_WDBI_ERROR" },
   { DIA_E_FLASH_ENGINE_SESSION_CHANGE_ERROR,                     "DIA_E_FLASH_ENGINE_SESSION_CHANGE_ERROR" },
   { DIA_E_FLASH_ENGINE_REQUEST_DOWNLOAD_ERROR,                   "DIA_E_FLASH_ENGINE_REQUEST_DOWNLOAD_ERROR" },
   { DIA_E_FLASH_ENGINE_TRANSFER_DATA_ERROR,                      "DIA_E_FLASH_ENGINE_TRANSFER_DATA_ERROR" },
   { DIA_E_FLASH_ENGINE_TRANSFER_EXIT_ERROR,                      "DIA_E_FLASH_ENGINE_TRANSFER_EXIT_ERROR" },
   { DIA_E_FLASH_ENGINE_FLASH_CONTAINER_ERROR,                    "DIA_E_FLASH_ENGINE_FLASH_CONTAINER_ERROR" },
   { DIA_E_FLASH_ENGINE_EXPECTED_IDENT_ERROR,                     "DIA_E_FLASH_ENGINE_EXPECTED_IDENT_ERROR" },
   { DIA_E_FLASH_ENGINE_DISABLE_FLUSH_ERROR,                      "DIA_E_FLASH_ENGINE_DISABLE_FLUSH_ERROR" },
   { DIA_E_FLASH_ENGINE_ENABLE_FLUSH_ERROR,                       "DIA_E_FLASH_ENGINE_ENABLE_FLUSH_ERROR" },
   { DIA_E_FLASH_ENGINE_DATA_ERROR,                               "DIA_E_FLASH_ENGINE_DATA_ERROR" },
   { DIA_E_FLASH_ENGINE_CATALOG_ERROR,                            "DIA_E_FLASH_ENGINE_CATALOG_ERROR" },
   { DIA_E_EMPTY_STRING,                                          "DIA_E_EMPTY_STRING" }
};

#endif
