/*!
 * \file       dia_ServiceHandler.h
 *
 * \brief      This file contains the definition of the base class for all kinds of
 *             service handlers
 *
 * \details    This file contains the definition of the base class for all kinds of
 *             service handlers
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreEngine
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SERVICE_HANDLER__
#define __INCLUDED_DIA_SERVICE_HANDLER__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

class dia_EngineServer;
class dia_LookupKey;
class amt_tclServiceData; // needed as long as we don't have removed methods vOnCompRes and vOnCompResPpg

#define DIA_C_U32_MAX_PROCESSING_TIME_SERVICE   ((tU32) 0xFFFFFFFF)

/**
 * \class   dia_ServiceHandler
 *
 * \brief   Base class for all kinds of service handlers
 *
 *          A service handler is responsible to process an incoming service request.
 *          Subclasses will be defined to handle protocol specific diagnostic services
 *          (like UDS).
 *
 * \ingroup diaCoreEngine
 */

class dia_ServiceHandler
   : public virtual dia_Object
{
public:
   //! default class constructor
   dia_ServiceHandler ( void );
   //! class constructor ( that takes a name )
   dia_ServiceHandler ( tCString name, tU32 timeoutVal=DIA_C_U32_MAX_PROCESSING_TIME_SERVICE );
   //! class destructor
   virtual ~dia_ServiceHandler ( void );

   //! assign the engine that processes this service
   tDiaResult setEngine ( dia_EngineServer& engine );
   //! method called to start processing of the given request
   virtual void vProcessRequest ( const std::vector<tArgsType>& vecArgs );
   //! method case if the processing timer has been elapsed
   virtual void vOnTimeout() = 0;

   //! retrieve the service handlers name
   virtual tCString getName ( void ) const;
   //! retrieve the service handlers name
   virtual tDiaResult setName ( tCString name );

   //! retrieve the value of the processing timeout
   virtual tU32  getTimeoutValue ( void ) const;
   //! assign a new value to the timeout value
   virtual void setTimeoutValue ( tU32 value );

   //! setup a lookup key for the lookup table
   virtual tDiaResult makeLookupKeys ( std::vector<dia_LookupKey*>& keys ) = 0;

   //
   // deprecated methods (this class provides default implementations in order to ensure backward compatibility)
   //

   //! old method used to start processing of a request \deprecated this method is obsolete
   virtual void vOnReqRx ( void );
   //! old method used to deliver a response from a CCA component \deprecated this method is obsolete
   virtual void vOnCompRes ( amt_tclServiceData* msg );
   //! obsolete method \deprecated this method is obsolete
   virtual void vOnCallBack ( tU8 u8MsgBuffer[], const tU16 u16MsgLength );
   //! obsolete method \deprecated this method is obsolete
   virtual tBool bOnNewReq ( tU8* pu8Buffer, tU16 u16MsgLen );
   //! deprecated method used to begin post processing \deprecated this method is obsolete
   virtual void vOnBeginPpg ( void );
   //! deprecated method used to deliver a response from a CCA component for post processing \deprecated this method is obsolete
   virtual void vOnCompResPpg(amt_tclServiceData* msg);
   //! obsolete method \deprecated this method is obsolete
   virtual void vOnTimeoutPpg ( void );
   //! obsolete method \deprecated this method is obsolete
   virtual tU32 u32MaxPgTime ( void ) const;
   //! obsolete method \deprecated this method is obsolete
   virtual tU32 u32MaxPpgTime ( void ) const;
   //! obsolete method \deprecated this method is obsolete
   virtual tCString cszClassName() const;

protected:
   //! name of the service handler
   tCString mName;
   //! max processing time
   tU32 mMaxTime;
   //! engine that processes this service
   dia_EngineServer* mpEngine;
   //! repository of lookup keys generated for this service handler
   std::vector<dia_LookupKey*> mLookupKeys;
};

#endif /* __INCLUDED_DIA_SERVICE_HANDLER__ */
