/*
 * dia_MessageBuffer.cpp
 *
 *  Created on: 12.05.2012
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#include "common/framework/engine/dia_MessageBuffer.h"

using namespace dia;

//-----------------------------------------------------------------------------

dia_MessageBuffer::dia_MessageBuffer (
    const tU8 byteBuffer[],
    tU16 length,
    tFuncSendResponse funcResponseSend,
    tCookieType cookie
    )
    : mpBuffer(OSAL_NULL),
      mBufferSize(0),
      mState(holds_request),
      mFormat(format_length_and_data),
      mProtocol(DIA_EN_PROTOCOL_UNKNOWN),
      mCookie(cookie),
      mResponseSendFunc(funcResponseSend),
      mCurrentPosition(0)
{
   ScopeTrace trc("dia_MessageBuffer::dia_MessageBuffer(byteBuffer,length,funcResponseSend)");

   // First check that we actually have some data in the buffer to deal with
   if ( length  )
   {
      // Take a local copy of the message buffer size
      mBufferSize = length;
      // request some storage from OSAL
      mpBuffer = new tU8[mBufferSize];

      // Check that the pointer to the storage we have been given is not NULL
      if ( mpBuffer )
      {
         // Copy th data supplied to the given storage space
         ::memcpy(mpBuffer, byteBuffer, mBufferSize);
      }
      else  // out of memory !
      {
         DIA_TR_ERR("dia_MessageBuffer: UNABLE TO ALLOCATE MEMORY FOR THE MESSAGE BUFFER");
      }
   }
}

//-----------------------------------------------------------------------------

dia_MessageBuffer::dia_MessageBuffer (
    const tU8 byteBuffer[],
    tU16 length,
    tFuncSendResponse  funcResponseSend,
    dia_enProtocol protocol,
    enBufferState state,
    enBufferFormat format,
    tCookieType cookie
    )
    : mpBuffer(OSAL_NULL),
      mBufferSize(0),
      mState(state),
      mFormat(format),
      mProtocol(protocol),
      mCookie(cookie),
      mResponseSendFunc(funcResponseSend),
      mCurrentPosition(0)
{
   ScopeTrace trc("dia_MessageBuffer::dia_MessageBuffer");

   // First check that we actually have some data in the buffer to deal with
   if ( length  )
   {
      // Take a local copy of the message buffer size
      mBufferSize = ( mFormat != format_length_and_data ) ? (tU16)(length+1) : length;
      // request some storage from OSAL
      mpBuffer = new tU8[mBufferSize];

      // Check that the pointer to the storage we have been given is not NULL
      if ( mpBuffer )
      {
         if ( mFormat == format_length_and_data )
         {
            // Copy the data supplied to the given storage space
            ::memcpy(mpBuffer, byteBuffer, mBufferSize);
         }
         else
         {
            mpBuffer[0] = tU8(mBufferSize);
            ::memcpy(&mpBuffer[1],byteBuffer,length);
         }

         // Set the message buffer state
         mState  = state;
         mFormat = format_length_and_data;
      }
      else  // out of memory !
      {
         DIA_TR_ERR("dia_MessageBuffer: UNABLE TO ALLOCATE MEMORY FOR THE MESSAGE BUFFER");
      }
   }
}

//-----------------------------------------------------------------------------

dia_MessageBuffer::dia_MessageBuffer (
      const std::vector<tU8>& byteBuffer,
      tFuncSendResponse funcResponseSend,
      dia_enProtocol protocol,
      enBufferState state,
      enBufferFormat format,
      tCookieType cookie
   )
   : mpBuffer(OSAL_NULL),
     mBufferSize(0),
     mState(state),
     mFormat(format),
     mProtocol(protocol),
     mCookie(cookie),
     mResponseSendFunc(funcResponseSend),
     mCurrentPosition(0)
{
   ScopeTrace trc("dia_MessageBuffer::dia_MessageBuffer");

   // First check that we actually have some data in the buffer to deal with
   if ( byteBuffer.size()  )
   {
      // Take a local copy of the message buffer size
      mBufferSize = ( mFormat != format_length_and_data ) ? tU16((byteBuffer.size()+1)) : tU16(byteBuffer.size());
      // request some storage from OSAL
      mpBuffer = new tU8[mBufferSize];

      // Check that the pointer to the storage we have been given is not NULL
      if ( mpBuffer )
      {
         if ( mFormat == format_length_and_data )
         {
            // Copy the data supplied to the given storage space
            ::memcpy(mpBuffer, &byteBuffer[0], mBufferSize);
         }
         else
         {
            mpBuffer[0] = tU8(mBufferSize);
            ::memcpy(&mpBuffer[1],&byteBuffer[0],byteBuffer.size());
         }

         // Set the message buffer state
         mState  = state;
         mFormat = format_length_and_data;
      }
      else  // out of memory !
      {
         DIA_TR_ERR("dia_MessageBuffer: UNABLE TO ALLOCATE MEMORY FOR THE MESSAGE BUFFER");
      }
   }
}


//-----------------------------------------------------------------------------

dia_MessageBuffer::~dia_MessageBuffer()
{
   delete [] mpBuffer;
   mpBuffer = 0;
   mCookie  = 0;
}

//-----------------------------------------------------------------------------

tBool
dia_MessageBuffer::hasDataU8 ( tU16 pos ) const
{
   return pos < mBufferSize;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_MessageBuffer::setDataU8 ( tU16 pos, tU8 data )
{
   if ( (!mpBuffer) || (pos >= mBufferSize) )
   {
      return DIA_FAILED;
   }

   mCurrentPosition = pos;
   mpBuffer[mCurrentPosition++] = data;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_MessageBuffer::setDataU16 ( tU16 pos, tU16 data )
{
   if ( (!mpBuffer) || (pos >= (mBufferSize-1)) )
   {
      return DIA_FAILED;
   }

   mCurrentPosition = pos;
   // Copy 16 bits of data to the position required
   (void) ::memcpy( (mpBuffer + mCurrentPosition), &data, sizeof(tU16) );
   mCurrentPosition = (tU16)(mCurrentPosition + sizeof(tU16));

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_MessageBuffer::setDataU32 ( tU16 pos, tU32 data )
{
   if ( (!mpBuffer) || (pos >= (mBufferSize-1)) )
   {
      return DIA_FAILED;
   }

   mCurrentPosition = pos;
   // Copy 32 bits of data to the position required
   (void) ::memcpy( (mpBuffer + mCurrentPosition), &data, sizeof(tU32) );
   mCurrentPosition = (tU16)(mCurrentPosition + sizeof(tU32));

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_MessageBuffer::setDataU16BigEndian ( tU16 pos, tU16 data )
{
   if ( (!mpBuffer) || (pos >= (mBufferSize-1)) )
   {
      return DIA_FAILED;
   }

   // Set 16 bit in Big Endian format
   mpBuffer[pos + 0]=(tU8)((data >> 8) & 0xFF); 
   mpBuffer[pos + 1]=(tU8)((data >> 0) & 0xFF); 

   mCurrentPosition=(tU16)(pos+2);

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_MessageBuffer::setDataU32BigEndian ( tU16 pos, tU32 data )
{
   if ( (!mpBuffer) || (pos >= (mBufferSize-1)) )
   {
      return DIA_FAILED;
   }

   // Set 16 bit in Big Endian format
   mpBuffer[pos + 0]=(tU8)((data >> 24) & 0xFF);
   mpBuffer[pos + 1]=(tU8)((data >> 16) & 0xFF);
   mpBuffer[pos + 2]=(tU8)((data >> 8) & 0xFF);
   mpBuffer[pos + 3]=(tU8)((data >> 0) & 0xFF);

   mCurrentPosition=(tU16)(pos+4);

   return DIA_SUCCESS;
}
//-----------------------------------------------------------------------------

tDiaResult dia_MessageBuffer::setDataArray ( size_t position, size_t size, std::vector<tU8>::iterator data)
{
   if ((!mpBuffer) || (position+size > mBufferSize))
   {
      return DIA_FAILED;
   }
   mCurrentPosition = (tU16)position;
   this->setDataArray(size, data);
   return DIA_SUCCESS;
}
//-----------------------------------------------------------------------------

tDiaResult dia_MessageBuffer::setDataArray ( size_t size, std::vector<tU8>::iterator data)
{
   if ((!mpBuffer) || (mCurrentPosition+size > mBufferSize))
   {
      return DIA_FAILED;
   }
   std::copy(data, data + size, (mpBuffer + mCurrentPosition));
   mCurrentPosition = (tU16)(mCurrentPosition + size);
   return DIA_SUCCESS;
}
//-----------------------------------------------------------------------------

tU8
dia_MessageBuffer::getDataU8 ( tU16 pos ) const
{
   // range check
   DIA_ASSERT(pos < mBufferSize);

   return (mpBuffer) ? mpBuffer[pos] : 0x00;
}

tU16
dia_MessageBuffer::getDataU16BigEndian ( tU16 pos ) const
{
   // range check
   DIA_ASSERT(pos + 1 < mBufferSize);
   
   DIA_ASSERT(mpBuffer);
   
   return (tU16)( 
      (mpBuffer[pos] << 8) & 
      (mpBuffer[pos + 1]));
}

tU32
dia_MessageBuffer::getDataU32BigEndian ( tU16 pos ) const
{
   // range check
   DIA_ASSERT(pos + 3 < mBufferSize);
   
   DIA_ASSERT(mpBuffer);
   
   return 
      (mpBuffer[pos] << 24) & 
      (mpBuffer[pos + 1] << 16) & 
      (mpBuffer[pos + 2] << 8)  & 
      (mpBuffer[pos + 3]);
}


//-----------------------------------------------------------------------------

tU8*
dia_MessageBuffer::u8GetBuffer ( void ) //const
{
   return mpBuffer;
}

//-----------------------------------------------------------------------------

tU16
dia_MessageBuffer::u16GetDataLength ( void ) const
{
   return mBufferSize;
}

//-----------------------------------------------------------------------------

void
dia_MessageBuffer::vSetDataLength ( tU16 newLength )
{
   if (0==newLength)
   {
      //we have to avoid calls like new tU8 [0], because behavior is not defined by C++ standard.
      DIA_TR_ERR("##### dia_MessageBuffer::vSetDataLength ##### ERROR newLength is 0.");
      return;
   }

   // Now try to create a new buffer of this size
   tU8* pNewBuffer = new tU8[newLength];

   // Check that the pointer returned from the OSAL requesd is not Zero
   if ( mpBuffer && pNewBuffer )
   {
      // Calculate the data size to copy
      tU32 numberOfBytes = (newLength < mBufferSize) ? newLength : mBufferSize;

      // Now copy the data to the new buffer
      if ( mFormat == format_length_and_data )
      {
         if(numberOfBytes >= mBufferSize)
            numberOfBytes = mBufferSize - 1;

         DIA_TR_INF("dia_MessageBuffer::vSetDataLength format_length_and_data numberOfBytes=%d", numberOfBytes);

         // missing the first byte of the message
         (void) ::memcpy(pNewBuffer, mpBuffer + 1, numberOfBytes);
         mFormat = format_raw;
      }
      else
      {
         (void) ::memcpy(pNewBuffer, mpBuffer, numberOfBytes );
      }

      // Delete the old buffer
      delete [] mpBuffer;

      // Set the Msg buffer pointer to the pointer to the new buffer
      mpBuffer = pNewBuffer;

      // Set the new buffer size
      mBufferSize = newLength;
   }

   // make lint happy
   pNewBuffer = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pNewBuffer'. --> value assigned to mpBuffer and lifetime is controlled by session

} //lint !e438 Warning: last value assigned to variable 'pNewBuffer' not used

//-----------------------------------------------------------------------------

void
dia_MessageBuffer::vSendResponse ( void )
{
   mResponseSendFunc(mpBuffer,mBufferSize,mCookie);
}
