/*!
 * \file       dia_LookupKey.h
 *
 * \brief      This file contains the definition of the dia_LookupKey class that is
 *             used to check if a service handler for incoming diagnosis request is
 *             available
 *
 * \details    This file contains the definition of the dia_LookupKey class that is
 *             used to check if a service handler for incoming diagnosis request is
 *             available
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreEngine
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#define __INCLUDED_DIA_LOOKUPKEY__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

// maximum number of SIDs
#define DIA_C_U8_UDS_MAX_SID_VALUE         256


#define UDS_U16_LENGTH_NOT_USED     ((tU16) 0xFFFF)
#define UDS_U16_PARAMETER_NOT_USED  ((tU16) 0xFFFF)
#define UDS_U8_PARAMETER_NOT_USED   ((tU8 ) 0xFF)

class dia_MessageBuffer;
class dia_MessageBufferUDS;

/**
 * \class   dia_LookupKey
 *
 * \brief   This class is used to check if a service handler for incoming diagnosis
 *          request is available
 *
 * \ingroup diaCoreEngine
 */

class dia_LookupKey
{
public:
   enum eLengthType
   {
      eLengthUnknown = 0,
      eLengthSpecified,
      eLengthCount
   };

   enum eParameterType
   {
      eParameterUnknown = 0,
      eParameterByte,
      eParameterWord,
      eParameterDoubleword,
      eParameterCount
   };

   struct tsKeyFormat
   {
      //! default constructor
      tsKeyFormat ( void );
      //! constructor used to create a customized object
      tsKeyFormat ( tU8 sid, eParameterType paraType, eLengthType lengthType );

      //! reset content of the object to default values
      void vClear ( void );
      //! assign values to the attributes
      void vSet ( tU8 sid, eParameterType paraType, eLengthType lengthType );

      //! service identifier
      tU8 mSID;
      //! type of parameter typically used to specify the subfunction
      eParameterType mParaType;
      //! length of the service request
      eLengthType mLengthType;
   };

public:
   //! class constructor
   dia_LookupKey ( tU8 sid, tU16 parameter, tU16 srvLen=UDS_U16_LENGTH_NOT_USED );
   //! class constructor
   dia_LookupKey ( tU8 sid, tU8  parameter, tU16 srvLen=UDS_U16_LENGTH_NOT_USED );
   //! class constructor that calculates a key from a received message
   dia_LookupKey ( const dia_MessageBuffer& msgBuffer );
   //! copy constructor
   dia_LookupKey ( const dia_LookupKey& key );
   //! assignment operator
   dia_LookupKey& operator = (const dia_LookupKey& key);
   //! class destructor
   virtual ~dia_LookupKey ( void );

   //! overloaded equal operator
   bool operator == ( const dia_LookupKey& key ) const;
   //! overloaded not equal operator
   bool operator != ( const dia_LookupKey& key ) const;
   //! overloaded less than operator
   bool operator <  ( const dia_LookupKey& key ) const;
   //! overloaded greater than operator
   bool operator >  ( const dia_LookupKey& key ) const;
   //! overloaded less equal operator
   bool operator <= ( const dia_LookupKey& key ) const;
   //! overloaded greater equal operator
   bool operator >= ( const dia_LookupKey& key ) const;

   virtual tU8  getSID ( void ) const { return mSID; }
   virtual tU32 getSubFuncID ( void ) const { return mParameter; }
   virtual tU16 getSizeOfSubFuncID ( void ) const { return mLength; }
   virtual eParameterType getParaType( void ) const { return mParaType; }

protected:
    //! protected default class constructor to avoid usage
   dia_LookupKey ( void );

   //! checks whether this key is equal to the specified key
   bool equal ( const dia_LookupKey& key ) const;
   //! checks whether this key is less than the specified key
   bool less  ( const dia_LookupKey& key ) const;

   //! service identifier
   tU8 mSID;
   //! parameter type
   eParameterType mParaType;
   //! service parameter
   tU32 mParameter;
   //! length type
   eLengthType mLengthType;
   //! service length
   tU16 mLength;

   //! repository for storing information about the key format
   static tsKeyFormat mKeyFormatRep[DIA_C_U8_UDS_MAX_SID_VALUE];
};

#endif /* __INCLUDED_DIA_LOOKUPKEY__ */
