/*!
 * \file       dia_EngineRunIn.cpp
 *
 * \brief      Run-in engine (client engine)
 *
 * \details    Run-in engine (client engine) is used to run several diagnostic services during a run-in
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreEngine
 *
 * \copyright  (c) 2014-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_ENGINE_RUN_IN__
#include <common/framework/engine/dia_EngineRunIn.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_RUN_IN_CONFIGURATION__
#include <common/framework/engine/dia_EngineRunInConfiguration.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include <common/framework/config/dia_defsConfig.h>
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include <common/framework/application/dia_AppController.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY_PLUGIN_RUNIN__
#include "common/framework/factory/dia_FactoryPluginRunIn.h"
#endif

#ifndef __INCLUDED_DIA_FILE_DIR__
#include <common/framework/application/dia_FileDir.h>
#endif

#ifndef __INCLUDED_DIA_FILE__
#include "common/framework/application/dia_File.h"
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include "common/framework/application/dia_Application.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

using namespace std;

#define RUNIN_QUEUE_SIZE      100
#define RUNIN_STR_BUFF_SIZE   200

static tCString strRunInThreadName = "DIA_RIN";
static tCString strRunInQueueName  = "DIA_RINQ";
static tCString strDiagDir         = "/var/opt/bosch/dynamic/diagnosis";
static tCString strDiagRunInDir    = "/var/opt/bosch/dynamic/diagnosis/runin";


#define DIA_C_U32_RUNIN_INVALID_TIMER_ID                 ((tU32) 0xFFFFFFFF)

#define DIA_C_U8_RUNIN_ROUTINE_RESULT_COMPLETED_OK       ( (tU8) 0x00)
#define DIA_C_U8_RUNIN_ROUTINE_RESULT_COMPLETED_NOK      ( (tU8) 0x01)
#define DIA_C_U8_RUNIN_ROUTINE_RESULT_ABORTED            ( (tU8) 0x02)
#define DIA_C_U8_RUNIN_ROUTINE_RESULT_RUNNING            ( (tU8) 0x03)
#define DIA_C_U8_RUNIN_ROUTINE_RESULT_STOPPED            ( (tU8) 0x04)

//-----------------------------------------------------------------------------

dia_EngineRunIn::dia_EngineRunIn ( tCString name, dia_EngineRunInConfiguration& config )
   : dia_EngineClient(name,config),
     dia_ActiveObject(strRunInThreadName,DIA_PROP_ENGINE_RUNIN_THREAD_PRIO,DIA_PROP_ENGINE_RUNIN_THREAD_STACK_SIZE),
     mRunInConfig(config),
     mpFSM(0),
     mpMsgQueue(0),
     mIsRunning(false),
     mCommandRep(config.getCommands()),
     mCmdIter(mCommandRep.begin()),
     mpActiveCommand(0),
     mIsRequestValid(false),
     mWasRequestSentSuccessfully(false),
     mWasCompleted(false),
     mpResponseData(0),
     mResponseLength(0),
     mEngineTimerID(DIA_C_U32_RUNIN_INVALID_TIMER_ID),
     mPollingTimerID(DIA_C_U32_RUNIN_INVALID_TIMER_ID),
     mWaitingForResponse(false),
     mOperationCycleCounter(0),
     mIsPostProcessingActive(false),
     mIsPostProcessingCompleted(false)
{
   dia_tclFnctTrace trc("dia_EngineRunIn::dia_EngineRunIn");

   oEngineTimer.s32Create();
   mEngineTimerID = oEngineTimer.getID();

   oPollingTimer.s32Create();
   mPollingTimerID = oPollingTimer.getID();

   ::memset(mRuninCycleDirName,0,500);
}

//-----------------------------------------------------------------------------

dia_EngineRunIn::~dia_EngineRunIn ( void )
{
   // destructors should not throw exceptions (--> lint), but functions called in the
   // could possibly throw excepections. So we catch them here and raise an assert
   _BP_TRY_BEGIN
   {
      (void) dia_EngineRunIn::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_EngineRunIn::~dia_EngineRunIn !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineRunIn::setup ( void )
{
   dia_tclFnctTrace trc("dia_EngineRunIn::setup");

   // we setup the engine object only once
   if ( mIsSetupDone )
   {
      DIA_TR_INF("### ENGINE WAS ALREADY SET UP. RETURNING... ###");
      return DIA_SUCCESS;
   }

   // create the state machine object
   mpFSM = OSAL_NEW dia_EngineRunInFSM::Fsm(this);

   if ( mpFSM )
   {
       DIA_TR_INF( "### Initializing State Machine ###");
       mpFSM->init();
       DIA_TR_INF( "### Done ###");
   }
   else
   {
      DIA_TR_INF("### INVALID STATE MACHINE POINTER ###");
      return DIA_FAILED;
   }

   //
   // create message queue
   //
   mpMsgQueue = OSAL_NEW dia_Queue<dia_RunInEvent>(strRunInQueueName,RUNIN_QUEUE_SIZE);
   if ( !mpMsgQueue || (mpMsgQueue->open() != DIA_SUCCESS) ) return DIA_FAILED;

   DIA_TR_INF("### READING ENGINE CONFIGURATION (ENGINE ADDR=0x%p)... ###",this);
   mIsSetupDone = TRUE;

#ifndef __DIA_UNIT_TESTING__
   getInstanceOfAppController()->addAppStateListener(this);
#endif

   return mIsSetupDone ? DIA_SUCCESS : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineRunIn::tearDown ( void )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::tearDown()");

   if ( mpMsgQueue )
   {
      tU32 dummy = 0;

      mpMsgQueue->addElement((dia_RunInEvent*) &dummy);

      tS32 s32WaitCount = 0;
      while ( s32WaitCount < 10 )
      {
         DIA_TR_INF("Sleep for 100ms");

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
         OSAL_s32ThreadWait(100);
#else
         usleep(100*1000); // DiagOmitSleepWarning: Required for thread synchronization
#endif
         ++s32WaitCount;
      }

      mpMsgQueue->close();
   }

   OSAL_DELETE mpFSM;
   mpFSM = 0;

   oEngineTimer.s32Delete();
   oEngineTimer.removeTimerListener(this);
   mEngineTimerID = DIA_C_U32_RUNIN_INVALID_TIMER_ID;

   oPollingTimer.s32Delete();
   oPollingTimer.removeTimerListener(this);
   mPollingTimerID = DIA_C_U32_RUNIN_INVALID_TIMER_ID;

   OSAL_DELETE mpMsgQueue;
   mpMsgQueue = 0;

   mpActiveCommand = 0;
   mpResponseData = 0;

   mCmdIter = mRunInConfig.getCommands().end();

   return DIA_SUCCESS;
} //lint !e1578: Pointer member dia_EngineRunIn::mCmdIter neither freed nor zeroed by cleanup function. No heap allocated for iterator

//-----------------------------------------------------------------------------

const dia_EngineRunInConfiguration&
dia_EngineRunIn::getRunInConfiguration ( void ) const
{
//   return dynamic_cast<const dia_EngineRunInConfiguration&>(mConfig);
   return mRunInConfig;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineRunIn::startControl ( std::vector<tArgsType>& /*args*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::startControl");

   // check if the Run-In was already started
   if ( mIsRunning ) return DIA_SUCCESS;

   mIsRunning = true;
   postEvent(dia_EngineRunInFSM::evStart);
   tDiaResult retCode = startThread();

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineRunIn::stopControl ( void )
{
   postEvent(dia_EngineRunInFSM::evTerminate);
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineRunIn::sendRunInRequest ( tU8* pData, tU16 length )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::sendRunInRequest");

   DIA_TR_INF("--- Sending RunIn Request To Service Engine");

   // create an object containing the UDS msg buffer
   dia_MessageBufferUDS* pMsgUDS = OSAL_NEW dia_MessageBufferUDS(pData, length, onRunInResponse); //lint !e429: custodial pointer is freed by after engine has processed the message

   DIA_TR_SM("UDS msg RX via RunIn: Length: %d Data: %s", length, dia::utils::bin2str(pData,length,' ').c_str());

   // create and send event ReqRx containing a ptr to the reqest msg buffer
   getInstanceOfApplication()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventReqRx(pMsgUDS)); //lint !e429: custodial pointer is freed by after engine has processed the message

   // make lint happy
   pMsgUDS = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'poMsgBuffer'. --> lifetime is controlled by diagnostic session

   return DIA_SUCCESS; //lint !e438 Warning: last value assigned to variable 'poMsgBuffer' not used
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::onRunInResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::onRunInResponse");

   // send response msg to RunIn engine
// ETG_TRACE_COMP_CLS((TR_CLASS_DIAGNOSTICS_SM, "UDS msg TX via RunIn: Length: %d Data: %02x", u16Length, ETG_LIST_LEN(u16Length), ETG_LIST_PTR_T8(au8MsgBuffer)));
   DIA_TR_INF("UDS msg TX via RunIn ...");

   dia_EngineRunIn* pEngine = getInstanceOfFactoryPluginRunIn()->getEngine();

   if ( pEngine )
   {
      pEngine->mResponseLength = length;
      pEngine->mpResponseData  = OSAL_NEW tU8[pEngine->mResponseLength];

      //copy message data to the buffer
      ::memcpy(pEngine->mpResponseData, data, length);

      pEngine->postEvent(dia_EngineRunInFSM::evResponseReceived);
   }

   dia_Application::getInstance()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vThreadEntrypointObject ( void )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vThreadEntrypointObject");

   dia_RunInEvent* pEvent = 0;

   bool isDone = false;

   do
   {
      pEvent = 0;
      if ( mpMsgQueue && (mpMsgQueue->getElement(&pEvent) == DIA_SUCCESS) )
      {
         DIA_TR_INF("pEvent = %s", pEvent ? "not null" : "null");

         if ( pEvent )
         {
            dia_EngineRunInFSM::FsmEvent fsmEvent = *((dia_EngineRunInFSM::FsmEvent*) pEvent);
            DIA_TR_INF("Accept Event Id=%d", fsmEvent);
            DIA_TR_INF("Forward Message to FSM ...");
            if ( mpFSM )
            {
                DIA_TR_INF("### State Before: %s", mpFSM->getStateName());
                mpFSM->acceptEvent(fsmEvent,0);
                DIA_TR_INF("### State After : %s", mpFSM->getStateName());
            }
            else
            {
                DIA_TR_INF( "### Event Not Accepted By Run-In-Engine. No FSM !!!");
            }

            DIA_TR_INF("Message Processed !!");
            OSAL_DELETE pEvent;
            pEvent = 0;
         }
         else
         {
            isDone = true;
         }
      }
      else
      {
         isDone = true;
      }
   }
   while ( !isDone ) ;

   DIA_TR_ERR("############################################################");
   DIA_TR_ERR("#");
   DIA_TR_ERR("# ENGINE RUN IN THREAD IS GETTING TERMINATED");
   DIA_TR_ERR("#");
   DIA_TR_ERR("############################################################");

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   // Ask OSAL what the last ERROR to occur was
   tU32 u32ErrorReason = OSAL_u32ErrorCode();
   if ( u32ErrorReason != OSAL_E_NOERROR ) DIA_TR_ERR("msg queue error: %d", u32ErrorReason);
#endif
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::postEvent ( dia_EngineRunInFSM::FsmEvent event )
{
   dia_RunInEvent* pEvent = OSAL_NEW dia_RunInEvent;  //lint !e429: custodial pointer is freed by after engine has processed the message
   if ( pEvent )
   {
      *pEvent = (dia_RunInEvent) event;
      if (mpMsgQueue) mpMsgQueue->addElement(pEvent);
   }
   pEvent = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pEvent'. --> lifetime is controlled by diagnostic session
} //lint !e438 Warning: last value assigned to variable 'pEvent' not used

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmInitialize ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmInitialize");
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineRunIn::createDirectory ( const tCString dirName )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::createDirectory");

   bool doesDiagDirExist = true;

   dia_FileDir diagDir(dirName);
   if ( !diagDir.doesExist() )
   {
      DIA_TR_INF("Creating Directory \"%s\"",dirName);
      if ( diagDir.create() != DIA_SUCCESS )
      {
         DIA_TR_ERR("##### CREATION OF DIRECTORY \"%s\" FAILED !! #####",dirName);
         doesDiagDirExist = false;
      }
   }

   return ( doesDiagDirExist ) ? DIA_SUCCESS : DIA_FAILED;
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::readCycleCounter ( void )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::readCycleCounter");

   tChar fileName[200];
   sprintf(fileName,"%s/.info",strDiagRunInDir);
//   std::string fileName(strDiagRunInDir);
//   fileName = fileName + "/.info";

   mOperationCycleCounter = 0;

   dia_File file(fileName /*.c_str()*/);
   if (file.open("r") == DIA_SUCCESS)
   {
      tChar line[RUNIN_STR_BUFF_SIZE];
      tU32 charsRead = 0;

      if ( ( file.getLine( line, RUNIN_STR_BUFF_SIZE, &charsRead ) ) == DIA_SUCCESS )
      {
         tS32 value;
         sscanf(line,"%d",&value);
         mOperationCycleCounter = (tU16) value;
      }
      file.close();
   }

   mOperationCycleCounter++;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineRunIn::writeCycleCounter ( void )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::writeCycleCounter");

   tDiaResult retCode = DIA_SUCCESS;

   tChar fileName[200];
   sprintf(fileName,"%s/.info",strDiagRunInDir);
//   std::string fileName(strDiagRunInDir);
//   fileName = fileName + "/.info";

   dia_File file(fileName /*.c_str()*/);
   if (file.open("w+") == DIA_SUCCESS)
   {
      tChar line[200];
      sprintf(line,"%d",mOperationCycleCounter);

      if ( ( file.writeLine(line, (tU16) ::strlen(line)) ) != DIA_SUCCESS )
      {
         DIA_TR_ERR("##### FAILED TO WRITE RUNIN CYCLE COUNTER !!! #####");
         retCode = DIA_FAILED;
      }
      file.close();
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineRunIn::writeCompletionMarker ( void )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::writeCompletionMarker");

   tDiaResult retCode = DIA_FAILED;

   tChar fileName[200];
   sprintf(fileName,"%s/.complete",mRuninCycleDirName);
//   std::string fileName(mRuninCycleDirName);
//   fileName = fileName + "/.complete";

   dia_File file(fileName /*.c_str()*/);
   if (file.open("w+") == DIA_SUCCESS)
   {
      retCode = DIA_SUCCESS;
      file.close();
   }

   return retCode;
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmInitializeRunIn ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmInitializeRunIn");

   if ( createDirectory(strDiagDir) == DIA_SUCCESS )
   {
      if ( createDirectory(strDiagRunInDir) == DIA_SUCCESS )
      {
         readCycleCounter();
         if ( writeCycleCounter() == DIA_SUCCESS )
         {
            sprintf(mRuninCycleDirName,"%s/runin_%04d",strDiagRunInDir, mOperationCycleCounter);
            if ( createDirectory(mRuninCycleDirName) == DIA_SUCCESS )
            {
               DIA_TR_INF("RunIn File Structure Setup Properly");
            }
         }
      }
   }
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmSequenceComplete ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmSequenceComplete");
   (void) writeCompletionMarker();
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmFinalizeRunIn ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmFinalizeRunIn");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmLoadRequests ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmLoadRequests");
   const_cast<std::vector<dia_CommandInfoUDS*>&>(mCommandRep) = mRunInConfig.getCommands();
   mCmdIter = mCommandRep.begin();
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmDiagnosisActive ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmDiagnosisActive");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmStartMonitoring ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmStartMonitoring");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmDiagnosisInactive ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmDiagnosisInactive");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmStopMonitoring ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmStopMonitoring");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmGetRequest ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmGetRequest");

   mIsRequestValid = false;

   if ( (!mCommandRep.empty()) && (mCmdIter != mCommandRep.end()) )
   {
      mpActiveCommand = *mCmdIter;
      if ( mpActiveCommand )
      {
         DIA_TR_INF("Selected Command %s", mpActiveCommand->mName);
         mIsRequestValid = true;
         postEvent(dia_EngineRunInFSM::evProcessRequest);
      }
   }
   else
   {
      DIA_TR_INF("NO COMMAND AVAILABLE !!");
   }

   if ( !mIsRequestValid )
   {
      if ( mIsPostProcessingActive ) mIsPostProcessingCompleted = true;
      postEvent(dia_EngineRunInFSM::evDone);
   }
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmLogError ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmLogError");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmHandleError ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmHandleError");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmStartTimer ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmStartTimer");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmStopTimer ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmStopTimer");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmSendRequest ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmSendRequest");

   mWasRequestSentSuccessfully = false;
   mWaitingForResponse = false;

   if ( mpActiveCommand )
   {
      DIA_TR_INF("Sending Request %s to UDS Engine ...", mpActiveCommand->mName);
      if ( mpActiveCommand->mpData && mpActiveCommand->mDataLen )
      {
         if ( sendRunInRequest(mpActiveCommand->mpData,mpActiveCommand->mDataLen) == DIA_SUCCESS )
         {
            DIA_TR_INF("Successfully Sent Request %s to UDS Engine ...", mpActiveCommand->mName);
            mWasRequestSentSuccessfully = true;
            mWaitingForResponse = true;
         }
         else
         {
            DIA_TR_INF("### SENDING REQUEST \"%s\" TO UDS ENGINE FAILED ###", mpActiveCommand->mName);
         }
      }
      else
      {
         DIA_TR_INF("### SENDING REQUEST \"%s\" TO UDS ENGINE FAILED (NO DATA FOUND) ###", mpActiveCommand->mName);
      }
   }

   postEvent(dia_EngineRunInFSM::evRequestSent);
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmLogResult ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmLogResult");

   DIA_TR_ERR("############################################################");
   DIA_TR_ERR("#");
   DIA_TR_ERR("# Logging Results");
   DIA_TR_ERR("#");
   DIA_TR_ERR("############################################################");

   bool success = false;

   if ( mpResponseData && mpActiveCommand && mpActiveCommand->mLogFileName )
   {

      for ( int i=0; i<mResponseLength; i++ ) {
         DIA_TR_INF("DATA[%02d]: 0x%02x", i, mpResponseData[i]);
      }

      tChar fileName[200];
      sprintf(fileName,"%s/%s",mRuninCycleDirName,mpActiveCommand->mLogFileName);

      dia_File file(fileName /*.c_str()*/);
      if (file.open("wb") == DIA_SUCCESS)
      {
         // write length of TX message
         size_t numBytes = sizeof(mpActiveCommand->mDataLen);
         if ( file.rawWrite((tU8*) &(const_cast<dia_CommandInfoUDS*>(mpActiveCommand)->mDataLen), &numBytes) == DIA_SUCCESS )
         {
            // write TX message
            numBytes = mpActiveCommand->mDataLen;
            if ( file.rawWrite(mpActiveCommand->mpData,&numBytes) == DIA_SUCCESS )
            {
               // write length of RX message
               numBytes = sizeof(mResponseLength);
               if ( file.rawWrite((tU8*) &(mResponseLength), &numBytes) == DIA_SUCCESS )
               {
                  // write RX message
                  numBytes = mResponseLength;
                  if ( file.rawWrite(mpResponseData,&numBytes) == DIA_SUCCESS )
                  {
                     DIA_TR_INF("##### DATA LOGGED SUCCESSFULLY TO FILE \"%s\" !!! #####",fileName);
                     success = true;
                  }
               }
            }
         }

         file.close();

         if ( !success ) file.remove();
      }
   }

   OSAL_DELETE mpResponseData;
   mResponseLength = 0;

   if ( mCmdIter != mCommandRep.end() ) mCmdIter++;
}

//-----------------------------------------------------------------------------

bool
dia_EngineRunIn::bFsmIsRunInActivated ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::bFsmIsRunInActivated");
   return true;
}

//-----------------------------------------------------------------------------

bool
dia_EngineRunIn::bFsmIsValidRequest ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::bFsmIsValidRequest");
   return mIsRequestValid ? true : false;
}

//-----------------------------------------------------------------------------

bool
dia_EngineRunIn::bFsmCanProceed ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::bFsmCanProceed");
   return true;
}

//-----------------------------------------------------------------------------

bool
dia_EngineRunIn::bFsmWasSentSuccessfully ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::bFsmWasSentSuccessfully");
   return mWasRequestSentSuccessfully;
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmReloadCommands ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmReloadCommands");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmInitializeProcessing ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmInitializeProcessing");

   oEngineTimer.s32SetTime(0,0);

   mWasCompleted = false;
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmFinalizeProcessing ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmFinalizeProcessing");
   oEngineTimer.s32SetTime(0,0);
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmEvaluateResult ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmEvaluateResult");

   static tCString statusMapping[] = { "Completed_OK", "Completed_NOK", "Aborted", "Running", "Stopped" };

   if ( mpActiveCommand && mpResponseData )
   {
      if ( ( mResponseLength == 3 ) && ( mpResponseData[0] == 0x7F ) )
      {
         // negative response received
         mWasCompleted = true;
      }
      else
      {
         switch ( mpActiveCommand->mType )
         {
         case DIA_EN_UDS_CMD_TYPE_ROUTINE_CTRL_TYPE_1:
            {
               if ( mResponseLength >= 5 )
               {
                  switch ( mpResponseData[4] )
                  {
                  case DIA_C_U8_RUNIN_ROUTINE_RESULT_COMPLETED_OK:
                  case DIA_C_U8_RUNIN_ROUTINE_RESULT_COMPLETED_NOK:
                  case DIA_C_U8_RUNIN_ROUTINE_RESULT_STOPPED:
                  case DIA_C_U8_RUNIN_ROUTINE_RESULT_ABORTED:
                     {
                        DIA_TR_INF("Routine-Result (Type 1): %s", statusMapping[mpResponseData[4]]);
                        mWasCompleted = true;
                     }
                     break;

                  default:
                     DIA_TR_INF("Routine-Result (Type 1): UNKNOWN RESULT TYPE");
                     break;
                  }
               }
            }
            break;

         case DIA_EN_UDS_CMD_TYPE_ROUTINE_CTRL_TYPE_2:
            {
               if ( mResponseLength >= 5 )
               {
                  switch ( mpResponseData[4] )
                  {
                  case DIA_C_U8_RUNIN_ROUTINE_RESULT_COMPLETED_OK:
                  case DIA_C_U8_RUNIN_ROUTINE_RESULT_COMPLETED_NOK:
                  case DIA_C_U8_RUNIN_ROUTINE_RESULT_STOPPED:
                  case DIA_C_U8_RUNIN_ROUTINE_RESULT_ABORTED:
                     {
                        DIA_TR_INF("Routine-Result (Type 1): %s", statusMapping[mpResponseData[4]]);
                        mWasCompleted = true;
                     }
                     break;

                  case DIA_C_U8_RUNIN_ROUTINE_RESULT_RUNNING:
                     {
                        DIA_TR_INF("Routine-Result (Type 1): %s", statusMapping[mpResponseData[DIA_C_U8_RUNIN_ROUTINE_RESULT_RUNNING]]);
                        if ( !(mpActiveCommand->mpSubCommand) )
                        {
                           mWasCompleted = true;
                        }
                     }
                     break;

                  default:
                     DIA_TR_INF("Routine-Result (Type 2): UNKNOWN RESULT TYPE");
                     break;
                  }
               }
            }
            break;

         default:
            mWasCompleted = true;
            break;
         }
      }

   }

   if ( mWasCompleted == true )
   {
      postEvent(dia_EngineRunInFSM::evProcessed);
   }

   mWaitingForResponse = false;
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmStartPollingTimer ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmStartPollingTimer");

   oPollingTimer.addTimerListener(this);
   oPollingTimer.s32SetTime(0,0);
   oPollingTimer.s32SetTime(2000, 2000);
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmInitializePolling ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmInitializePolling");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmStopPollingTimer ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmStopPollingTimer");
   oPollingTimer.s32SetTime(0,0);
   oPollingTimer.removeTimerListener(this);
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmFinalizePolling ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmFinalizePolling");
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmSendRequestResult ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmSendRequestResult");

   if ( !mWaitingForResponse )
   {
      mWasRequestSentSuccessfully = false;

      if ( mpActiveCommand && mpActiveCommand->mpSubCommand )
      {
         DIA_TR_INF("Sending GetResult Request %s to UDS Engine ...", mpActiveCommand->mpSubCommand->mName);
         if ( mpActiveCommand->mpSubCommand->mpData && mpActiveCommand->mpSubCommand->mDataLen )
         {
            if ( sendRunInRequest(mpActiveCommand->mpSubCommand->mpData,mpActiveCommand->mpSubCommand->mDataLen) == DIA_SUCCESS )
            {
               DIA_TR_INF("Successfully Sent GetResult Request %s to UDS Engine ...", mpActiveCommand->mpSubCommand->mName);
               mWasRequestSentSuccessfully = true;
               mWaitingForResponse = true;
            }
            else
            {
               DIA_TR_INF("### SENDING GETRESULT REQUEST \"%s\" TO UDS ENGINE FAILED ###", mpActiveCommand->mpSubCommand->mName);
            }
         }
         else
         {
            DIA_TR_INF("### SENDING GETRESULT REQUEST \"%s\" TO UDS ENGINE FAILED (NO DATA FOUND) ###", mpActiveCommand->mpSubCommand->mName);
         }
      }
   }

   postEvent(dia_EngineRunInFSM::evRequestSent);
}

//-----------------------------------------------------------------------------

bool
dia_EngineRunIn::bWasCompleted ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::bWasCompleted");
   return mWasCompleted;
}

//-----------------------------------------------------------------------------

bool
dia_EngineRunIn::bFsmNeedToPollForResponse ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::bFsmNeedToPollForResponse");

   bool pollingRequired = false;

   if ( mpActiveCommand && ( mpActiveCommand->mType == DIA_EN_UDS_CMD_TYPE_ROUTINE_CTRL_TYPE_2 ) )
   {
      if ( mpActiveCommand->mpSubCommand )
      {
         pollingRequired = true;
      }
   }

   return pollingRequired;
}

//-------------------------------------------------------------------------

void
dia_EngineRunIn::vOnTimerElapsed ( dia_TimerID id )
{
   tU32 timerID = (tU32) id;

   DIA_TR_INF("### dia_EngineRunIn::vOnTimerElapsed --- ID = %d ###", timerID);

   if ( timerID == mEngineTimerID )
   {
      postEvent(dia_EngineRunInFSM::evTimeout);
   }
   else if ( timerID == mPollingTimerID )
   {
      postEvent(dia_EngineRunInFSM::evPollingTimeout);
   }
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::onApplicationStateChange ( tU32 oldState, tU32 newState )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::onApplicationStateChange");

   if ( (oldState == DIA_C_U32_APP_STATE_NORMAL) && (newState == DIA_C_U32_APP_STATE_OFF) )
   {
      DIA_TR_INF("### RUNIN: DETECTED APPSTATE CHANGE (NORMAL --> OFF). STOP AND FINALIZE RUNIN ###");
      stopControl();
   }
}

//-----------------------------------------------------------------------------

bool
dia_EngineRunIn::bFsmIsPostProcessingComplete ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::bFsmIsPostProcessingComplete");
   return mIsPostProcessingCompleted;
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmFinalize ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmFinalize");
#ifndef __DIA_UNIT_TESTING__
   getInstanceOfAppController()->onApplicationStateChanged(DIA_C_U32_APP_STATE_OFF,this);
#endif
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmInitializePostProcessing ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmInitializePostProcessing");
   mIsPostProcessingActive = true;
}

//-----------------------------------------------------------------------------

void
dia_EngineRunIn::vFsmLoadPostProcessingCommands ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_EngineRunIn::vFsmLoadPostProcessingCommands");
   const_cast<std::vector<dia_CommandInfoUDS*>&>(mCommandRep) = mRunInConfig.getPostProcessingCommands();
   mCmdIter = mCommandRep.begin();
}

