/*!
 * \file       dia_EngineFlash.h
 *
 * \brief      This file contains the definition of the dia::EngineFlash class that is responsible for executing flash
 *             programming jobs loaded from a PDX container.
 *
 * \details    This file contains the definition of the dia::EngineFlash class that is responsible for executing flash
 *             programming jobs loaded from a PDX container.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreEngine
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_ENGINE_FLASH_PDX__
#define __INCLUDED_DIA_ENGINE_FLASH_PDX__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_DEFINES__
#include <common/framework/engine/dia_EngineDefs.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_CLIENT__
#include <common/framework/engine/dia_EngineClient.h>
#endif

#ifndef __INCLUDE_DIA_ACTIVE_OBJECT__
#include <common/framework/application/dia_ActiveObject.h>
#endif

#ifndef __INCLUDED_DIA_QUEUE__
#include "common/framework/application/dia_Queue.h"
#endif

#include <common/framework/fsm/generated/dia_EngineFlashFSM.h>

#ifndef __INCLUDED_DIA_INTERFACE_TIMER_LISTENER__
#include "common/interfaces/dia_ITimerListener.h"
#endif

namespace dia {

namespace odx {
   class PDXParserStrategyTinyXML;
   class PDXCatalog;
   class PDXFile;
   class ODX;
   class Session;
}

struct IdentInfo
{
//   IdentInfo ( void ) : mUID(0), mDID(0) {}
//   IdentInfo ( const std::string& name, dia::UID uid, tU16 did ) : mName(name), mUID(uid), mDID(did) {}
//   IdentInfo ( const IdentInfo& org ) : mName(org.mName), mUID(org.mUID), mDID(org.mDID) {}
//   IdentInfo& operator = ( const IdentInfo& obj ) {
//      if ( this != &obj ) {
//         mName = obj.mName;
//         mUID = obj.mUID;
//         mDID = obj.mDID;
//      }
//      return *this;
//   }
//   ~IdentInfo ( void ) {}

   std::string mName;
   dia::UID mUID;
   tU16 mDID;
};

typedef int PDXFlashingEvent;

class EngineFlashConfiguration;
class FlashJob;

class EngineFlash
   : public dia_EngineClient,
     public dia_ActiveObject,
     public dia_ITimerListener,
     protected dia_EngineFlashFSM::FsmBehaviour
{
   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(EngineFlash);

public:
   //! class constructor
   EngineFlash ( const std::string& name, EngineFlashConfiguration& config );
   //! class destructor
   virtual ~EngineFlash ( void );

   //! we use a two level construction process
   virtual tDiaResult setup ( void );
   virtual tDiaResult tearDown ( void );

   //! return a reference to the service provider configuration
   virtual const EngineFlashConfiguration& getFlashConfiguration ( void ) const;

   //! method used to populate the engine with service handlers
   virtual tDiaResult startControl ( std::vector<tArgsType>& /*args*/ ) { return DIA_E_NOT_IMPLEMENTED; }
   //! method used to populate the engine with service handlers
   virtual tDiaResult startControl ( std::vector<tArgsType>& args, bool automaticStartMode );
   //! method used to populate the engine with service handlers
   virtual tDiaResult stopControl ( void );

   //! overloaded method from class dia_ITimerListener
   virtual void vOnTimerElapsed ( dia_TimerID id );

   virtual tDiaResult execute ( void );
   virtual void onStartFlashingEvent ( void );

   //! callback method used to send back the response
   static void onSessionChangeResponse ( const tU8 data[], tU16 length, tCookieType cookie );
   static void onReadIdentResponse ( const tU8 data[], tU16 length, tCookieType cookie );
   static void onWDBIResponse ( const tU8 data[], tU16 length, tCookieType cookie );
   static void onDownloadResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ );
   static void onResponse ( const tU8, tU16, tCookieType /*cookie*/ ) {}
   static void onTransferDataResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ );
   static void onTransferExitResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ );

   void vLoadExpectedIdents ( void );
   void vLoadDataBlocksWDBI ( void );
   void vLoadDataBlocksDownload ( void );

protected:
   //! deprecated class constructor
   EngineFlash ( void );

   //! thread entry method inherited from dia_ActiveObject
   virtual void vThreadEntrypointObject ( void );

   // FSM actions
   virtual void vFsmAssembleData ( void* pArg );
   virtual void vFsmCalcCRC ( void* pArg );
   virtual void vFsmCheckForPDXContainer ( void* pArg );
   virtual void vFsmDiagnosisActive ( void* pArg );
   virtual void vFsmDiagnosisInactive ( void* pArg );
   virtual void vFsmEvaluateJob ( void* pArg );
   virtual void vFsmEvaluateResponseDownloadRequest ( void* pArg );
   virtual void vFsmEvaluateResponseTransferData ( void* pArg );
   virtual void vFsmEvaluateResponseTransferExit ( void* pArg );
   virtual void vFsmEvaluateResponseWDBI ( void* pArg );
   virtual void vFsmEvalueteResponseExpectedIdent ( void* pArg );
   virtual void vFsmFinalize ( void* pArg );
   virtual void vFsmFinalizeFlashing ( void* pArg );
   virtual void vFsmFinalizeFlashProcess ( void* pArg );
   virtual void vFsmFlashingComplete ( void* pArg );
   virtual void vFsmGetNextBlock ( void* pArg );
   virtual void vFsmGetNextJob ( void* pArg );
   virtual void vFsmHandleCatalogError ( void* pArg );
   virtual void vFsmHandleDataError ( void* pArg );
   virtual void vFsmHandleDataTransferError ( void* pArg );
   virtual void vFsmHandleDisableFlushingError ( void* pArg );
   virtual void vFsmHandleEnableFlushingError ( void* pArg );
   virtual void vFsmHandleError ( void* pArg );
   virtual void vFsmHandleExpectedIdentError ( void* pArg );
   virtual void vFsmHandleFlashContainerError ( void* pArg );
   virtual void vFsmHandleRequestDownloadError ( void* pArg );
   virtual void vFsmHandlerTransferExitError ( void* pArg );
   virtual void vFsmHandleSessionChangeError ( void* pArg );
   virtual void vFsmHandleTimeout ( void* pArg );
   virtual void vFsmHandleUnknownJob ( void* pArg );
   virtual void vFsmHandleUnpackingError ( void* pArg );
   virtual void vFsmHandleWDBIError ( void* pArg );
   virtual void vFsmInitialize ( void* pArg );
   virtual void vFsmInitializeFlashing ( void* pArg );
   virtual void vFsmLoadJobs ( void* pArg );
   virtual void vFsmLogError ( void* pArg );
   virtual void vFsmReadData ( void* pArg );
   virtual void vFsmReadODXFlashContainer ( void* pArg );
   virtual void vFsmReadPDXCatalog ( void* pArg );
   virtual void vFsmRemovePDXContainer ( void* pArg );
   virtual void vFsmRestoreActiveSession ( void* pArg );
   virtual void vFsmSendRequestDisableFlushing ( void* pArg );
   virtual void vFsmSendRequestDownloadRequest ( void* pArg );
   virtual void vFsmSendRequestEnableFlushing ( void* pArg );
   virtual void vFsmSendRequestReadIdent ( void* pArg );
   virtual void vFsmSendRequestSessionChange ( void* pArg );
   virtual void vFsmSendRequestTransferData ( void* pArg );
   virtual void vFsmSendRequestTransferExit ( void* pArg );
   virtual void vFsmSendRequestWDBI ( void* pArg );
   virtual void vFsmSplitData ( void* pArg );
   virtual void vFsmStartTimerDownload ( void* pArg );
   virtual void vFsmStartTimerExpectedIdent ( void* pArg );
   virtual void vFsmStartTimerFlushing ( void* pArg );
   virtual void vFsmStartTimerSession ( void* pArg );
   virtual void vFsmStartTimerWDBI ( void* pArg );
   virtual void vFsmStopTimer ( void* pArg );
   virtual void vFsmStoreActiveSession ( void* pArg );
   virtual void vFsmUnpackPDXContainer ( void* pArg );

   // FSM guards
   virtual bool bFsmHasCatalogFlashContainer ( void* pArg );
   virtual bool bFsmIsDataTransferComplete ( void* pArg );
   virtual bool bFsmIsPDXContainerAvailable ( void* pArg );
   virtual bool bFsmNeedToDisableFlushing ( void* pArg );
   virtual bool bFsmWasTimerStarted ( void* pArg );

   //! post a trigger state machine event to the engine queue
   virtual void postEvent ( dia_EngineFlashFSM::FsmEvent event );
   //! post a trigger state machine event to the state machine
   virtual void acceptEvent ( dia_EngineFlashFSM::FsmEvent event, void* pArg );

   tDiaResult unpackPDXContainer ( const std::string& pdxFileName );
   void removePDXContainer ( void );
   tDiaResult executeSystemCommand ( const std::string& cmdString );

   bool isPositiveResponseReceived ( void ) const;
   tU16 getBlockSizeFromResponse ( void ) const;
   tU16 calcBlockSize ( void ) const;

protected:
   //! reference to the run in configuration
   EngineFlashConfiguration& mConfig;
   //! pointer to the FSM object
   dia_EngineFlashFSM::Fsm* mpFSM; //lint -sem(dia::EngineFlash::tearDown,cleanup)
   //! message queue used for communication with the server engine
   dia_Queue<PDXFlashingEvent>* mpMsgQueue;
   //! flag used to indicate that the run-in is running
   bool mIsRunning;

   bool mHasCatalogFlashContainer;
   bool mIsDataTransferComplete;
   bool mIsPDXContainerAvailable;
   bool mNeedToRemovePDXContainer;
   bool mIsTimerRunning;
   bool needToLoadPDXContainers;

   tU8* mpResponseData;
   tU16 mResponseLength;

   dia_Timer oEngineTimer;
   tU32 mEngineTimerID;
   std::string mPDXFolder;
   std::string mPDXFolderUnpacked;
   std::string mPDXContainer;
   std::list<std::string> mPDXContainers;
   odx::PDXParserStrategyTinyXML* mpParser;
   const odx::PDXCatalog* mpPDXCatalog;
   const odx::PDXFile* mPDXFlashContainerFile;
   const odx::ODX* mpODXF;

   std::map<dia::UID,IdentInfo*> mDIDLookupTable;
   std::list<FlashJob*> mJobList;
   FlashJob* mCurrentJob;
   std::vector<char> mFlashData;
   tU32 mBlockSize;
   std::vector<std::vector<tU8> > mBlockData;
   tU32 mBlockSizeInBytes;
   tU16 mBlockSequenceNumber;
   tU16 mCRC;
   tU8  mSessionToBeRestored;
   tU8  mSession;

   tDiaResult mErrorCode;
};

}

#endif /* __INCLUDED_DIA_ENGINE_FLASH_PDX__ */
