/*!
 * \file       dia_UploadDownloadStrategyFiles.cpp
 *
 * \brief      Upload and download strategy used for service $38 (RequestFileTransfer)
 *
 * \details    Abstract upload and download strategy used for service $34 (RequestDownload) and $35 (RequestUpload)
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreUploadDownload
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include "dia_UploadDownloadStrategyFiles.h"

#ifndef __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__
#include "common/framework/datadownload/dia_DataDownloadManager.h"
#endif

#ifndef __INCLUDED_DATA_DOWNLOAD_MANAGER_PLUGIN__
#include "common/framework/datadownload/dia_DataDownloadManagerPlugin.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#ifndef __INCLUDED_DIA_FILE__
#include <common/framework/application/dia_File.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include <common/framework/engine/dia_EngineServer.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include <common/framework/engine/dia_EngineManager.h>
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include <common/framework/engine/dia_Session.h>
#endif

#define DIA_C_U32_REQ_FILE_TRANSFER_LOOKUP_KEY                 ((tU32) 0x00010000) // 0x000abbcc --> a=1: File Transfer, bb: Format ID, cc: addrFormat ID

#define DIA_C_U8_REQDNL_MIN_LENGTH                             ((tU8) 4)

#define DIA_C_U8_REQDNL_BYTEPOS_SRV_ID                         ((tU8) 0)
#define DIA_C_U8_REQDNL_BYTEPOS_MODE_OF_OPERATION              ((tU8) 1)
#define DIA_C_U8_REQDNL_BYTEPOS_FILE_PATH_AND_NAME_LENGTH      ((tU8) 2)

#define DIA_C_U8_TRANSDATA_BYTEPOS_BSN                         ((tU8) 1)

namespace dia {

//-----------------------------------------------------------------------------

UploadDownloadStrategyFiles::UploadDownloadStrategyFiles ( const std::string& name )
   : UploadDownloadStrategy(name),
     mpFSM(0),
     mErrorCode(DIA_E_NOERROR),
     mModeOfOperation(DIA_EN_MODE_OF_OPERATION_UNKNOWN),
     mFilePathAndNameLength(0),
     mFileSizeParameterLength(0),
     mDataFormatIdentifier(0),
     mSizeInBytes(0),
     mBSN(DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER),
     mIsDataTransferActive(false),
     mIsDataTransferComplete(false),
     mpByteBufferDownload(0),
     mWritePosition(0),
     mNumOfBytesReceived(0),
     mReadPosition(0),
     mNumOfBytesTransmitted(0)

{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::UploadDownloadStrategyFiles");

   // create the state machine object
   if ( !(dia_FileTransferFSM::Fsm::createFSM(&mpFSM,this)) )
   {
      DIA_TR_INF( "### STATE MACHINE INITIALIZATION FAILED ###");
   }
}

//-----------------------------------------------------------------------------

UploadDownloadStrategyFiles::UploadDownloadStrategyFiles ( const char* name )
   : UploadDownloadStrategy(name),
     mpFSM(0),
     mErrorCode(DIA_E_NOERROR),
     mModeOfOperation(DIA_EN_MODE_OF_OPERATION_UNKNOWN),
     mFilePathAndNameLength(0),
     mFileSizeParameterLength(0),
     mDataFormatIdentifier(0),
     mSizeInBytes(0),
     mBSN(DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER),
     mIsDataTransferActive(false),
     mIsDataTransferComplete(false),
     mpByteBufferDownload(0),
     mWritePosition(0),
     mNumOfBytesReceived(0),
     mReadPosition(0),
     mNumOfBytesTransmitted(0)

{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::UploadDownloadStrategyFiles");

   // create the state machine object
   if ( !(dia_FileTransferFSM::Fsm::createFSM(&mpFSM,this)) )
   {
      DIA_TR_INF( "### STATE MACHINE INITIALIZATION FAILED ###");
   }
}

//-----------------------------------------------------------------------------

UploadDownloadStrategyFiles::~UploadDownloadStrategyFiles ( void )
{
   _BP_TRY_BEGIN
   {
      delete mpFSM;
      mpFSM = 0;

      if ( mpByteBufferDownload )
      {
         delete [] mpByteBufferDownload;
         mpByteBufferDownload = 0;
      }

   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia::UploadDownloadStrategyFiles::~UploadDownloadStrategyFiles !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::setup ( void )
{
   ScopeTrace trc("UploadDownloadStrategyFiles::setup");

   tDiaResult retVal = DIA_FAILED;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      if( pEngine->getSessionController()->addListener(this))
      {
         retVal = DIA_SUCCESS;
      }
   }

   if(DIA_SUCCESS != retVal)
   {
      DIA_TR_ERR("!!! UploadDownloadStrategyFiles::setup => ERROR: Unable to register for Session changes");
   }

   return retVal;
}

//-----------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::tearDown ( void )
{
   ScopeTrace trc("UploadDownloadStrategyFiles::tearDown");

   tDiaResult retVal = DIA_FAILED;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      if( pEngine->getSessionController()->removeListener(this))
      {
         retVal = DIA_SUCCESS;
      }
   }

   if(DIA_SUCCESS != retVal)
   {
      DIA_TR_ERR("!!! UploadDownloadStrategyFiles::tearDown => ERROR: Unable to deregister for Session changes");
   }

   return retVal;
}

//-----------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vOnSessionChanged ( tU8 newSession, tU8 oldSession )
{
   ScopeTrace trc("UploadDownloadStrategyFiles::vOnSessionChanged()");

   //terminate FileTransfer if session changed to default session
   if( DIA_C_U8_UDS_SESSION_DEFAULT == newSession)
   {
      if (mpFSM)
      {
         DIA_TR_INF("UploadDownloadStrategyFiles session change from %d to %d in state '%s'", oldSession, newSession, mpFSM->getStateName());

         mpFSM->acceptEvent(dia_FileTransferFSM::evSessionChange,0);
      }
      else
      {
         DIA_TR_ERR("UploadDownloadStrategyFiles::vOnSessionChanged => ERROR: FSM IS NULL!!!");
      }
   }
}

//-----------------------------------------------------------------------------

tU32
UploadDownloadStrategyFiles::getID ( void )
{
   return DIA_C_U32_REQ_FILE_TRANSFER_LOOKUP_KEY;
}

//-----------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::acceptEvent ( dia_FileTransferFSM::FsmEvent event, void* pArg )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::acceptEvent(dia_FileTransferFSM::FsmEvent,void*)");

   if ( !mpFSM )  return DIA_E_FSM_NOT_AVAILABLE;

   mErrorCode = DIA_E_NO_ERROR;

   DIA_TR_INF("dia::UploadDownloadStrategyFiles::acceptEvent - State (before): %s, event = '%s'", mpFSM->getStateName(),dia_FileTransferFSM::getEventName(event));
   mpFSM->acceptEvent(event,pArg);
   DIA_TR_INF("dia::UploadDownloadStrategyFiles::acceptEvent - State (after): %s", mpFSM->getStateName());

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::requestFileTransfer ( std::vector<tU8>& requestData )
{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::requestFileTransfer");

   DIA_TR_SM("RECEIVED FILE TRANSFER REQUEST: %s", dia::utils::bin2str(requestData,' ').c_str());

   tDiaResult retCode = validateDataRequestFileTransfer(requestData);

   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia::UploadDownloadStrategyFiles: FAILED (DATA VALIDATION) !!");
      return retCode;
   }

   // at this point we know that the request is valid and can be processed
   return triggerFileTransfer();
}

//-----------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::transferData ( std::vector<tU8>& requestData, std::vector<tU8>& responseData )
{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::transferData");

   DIA_TR_SM("RECEIVED DATA TRANSFER REQUEST: %s", dia::utils::bin2str(requestData,' ').c_str());

   responseData.clear();

   mErrorCode = validateDataTransferData(requestData);

   if ( mErrorCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("### transferData FAILED (DATA VALIDATION) mErrorCode=0x%08X", mErrorCode);
      return mErrorCode;
   }

   acceptEvent(dia_FileTransferFSM::evDataTransferRx,(void*) &requestData);

   if ( mErrorCode == DIA_E_NOERROR ) //lint !e774: Boolean always evaluates to True; gib2hi: Not true, because mErrorCode might have been changed in acceptEvent() call
   {
      std::copy(mResponseData.begin(), mResponseData.end(), std::back_inserter(responseData));
   }

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::transferExit ( std::vector<tU8>& requestData )
{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::transferExit");

   DIA_TR_SM("RECEIVED TRANSFER EXIT REQUEST: %s", dia::utils::bin2str(requestData,' ').c_str());

   mErrorCode = validateDataTransferExit(requestData);

   if ( mErrorCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("UploadDownloadStrategyFiles: FAILED (DATA VALIDATION) !!");
      return mErrorCode;
   }

   if ( mpFSM ) mpFSM->acceptEvent(dia_FileTransferFSM::evDataTransferExit,0);

   return mErrorCode;
}

//----------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::validateDataRequestFileTransfer ( std::vector<tU8>& data )
{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::validateDataRequestFileTransfer");

   if ( data.size() < DIA_C_U8_REQDNL_MIN_LENGTH ) return DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;

   if ( data[DIA_C_U8_REQDNL_BYTEPOS_MODE_OF_OPERATION] == DIA_EN_MODE_OF_OPERATION_UNKNOWN ) return DIA_E_OUT_OF_RANGE;
   if ( data[DIA_C_U8_REQDNL_BYTEPOS_MODE_OF_OPERATION] >= DIA_EN_MODE_OF_OPERATION_COUNT ) return DIA_E_OUT_OF_RANGE;
   mModeOfOperation = (enModeOfOperation) data[DIA_C_U8_REQDNL_BYTEPOS_MODE_OF_OPERATION];
   DIA_TR_INF("RequestFileTransfer: ModeOfOperation       = 0x%02x",mModeOfOperation);

   mFilePathAndNameLength = size_t(data[DIA_C_U8_REQDNL_BYTEPOS_FILE_PATH_AND_NAME_LENGTH] << 8);
   mFilePathAndNameLength = size_t(mFilePathAndNameLength + size_t(data[DIA_C_U8_REQDNL_BYTEPOS_FILE_PATH_AND_NAME_LENGTH+DIA_C_U8_REQDNL_BYTEPOS_MODE_OF_OPERATION]));
   DIA_TR_INF("RequestFileTransfer: FilePathAndNameLength = 0x%08zx (%zu)",mFilePathAndNameLength,mFilePathAndNameLength);

   if ( data.size() < (DIA_C_U8_REQDNL_MIN_LENGTH + mFilePathAndNameLength) ) return DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
   readFilePathAndName(data);

   tDiaResult retCode = DIA_SUCCESS;

   switch ( mModeOfOperation )
   {
   case DIA_EN_MODE_OF_OPERATION_ADD_FILE:
   case DIA_EN_MODE_OF_OPERATION_REPLACE_FILE:
      retCode = validateDataFormatIdentifier(data);
      if ( retCode == DIA_SUCCESS )
      {
         retCode = validatefileSizeParameterLength(data);
      }
      break;

   case DIA_EN_MODE_OF_OPERATION_DELETE_FILE:
   case DIA_EN_MODE_OF_OPERATION_READ_DIR:
      if ( data.size() > (DIA_C_U8_REQDNL_MIN_LENGTH + mFilePathAndNameLength) ) retCode = DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
      break;

   case DIA_EN_MODE_OF_OPERATION_READ_FILE:
      retCode = validateDataFormatIdentifier(data);
      if ( retCode == DIA_SUCCESS )
      {
         if ( data.size() > (DIA_C_U8_REQDNL_MIN_LENGTH + mFilePathAndNameLength + 1) ) retCode = DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
      }
      break;

   default:
      DIA_TR_INF("INVALID MODE OF OPERATION !!!");
      break;
   }

   DIA_TR_INF("retCode = 0x%08x",retCode);
   return retCode;
}

//----------------------------------------------------------------------

void
UploadDownloadStrategyFiles::readFilePathAndName ( std::vector<tU8>& data )
{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::readFilePathAndName");

   std::vector<tU8> vecFilePathAndName;
   for ( size_t i=0, j=DIA_C_U8_REQDNL_MIN_LENGTH; i<mFilePathAndNameLength; i++,j++ )
   {
      vecFilePathAndName.push_back(data[j]);
   }
   vecFilePathAndName.push_back('\0');
   mFilePathAndName = dia::utils::vector2string(vecFilePathAndName);

   size_t found = mFilePathAndName.find_last_of("/\\");
   std::string fileName(mFilePathAndName.substr(found+1));
   std::string filePath(mFilePathAndName.substr(0,found));

   DIA_TR_INF("RequestFileTransfer: mFilePathAndName      = \"%s\", path = \"%s\", fileName = \"%s\"",mFilePathAndName.c_str(),filePath.c_str(),fileName.c_str());
}

//----------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::validateDataFormatIdentifier ( std::vector<tU8>& data )
{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::validateDataFormatIdentifier");

   if ( data.size() < (DIA_C_U8_REQDNL_MIN_LENGTH + mFilePathAndNameLength + 1) ) return DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;

   size_t dataPos = DIA_C_U8_REQDNL_MIN_LENGTH + mFilePathAndNameLength ;
   mDataFormatIdentifier = data[dataPos];
   tU8 compressionMethod = (mDataFormatIdentifier & 0xF0) >> 4;
   tU8 encryptionMethod  =  mDataFormatIdentifier & 0x0F;

   DIA_TR_INF("RequestFileTransfer: compressionMethod     = 0x%02x, mCompressionMode = 0x%02x",compressionMethod,mCompressionMode);
   DIA_TR_INF("RequestFileTransfer: encryptionMethod      = 0x%02x, mEncryptionMode  = 0x%02x",encryptionMethod,mEncryptionMode);

   if ( mCompressionMode != compressionMethod ) return DIA_E_OUT_OF_RANGE;
   if ( mEncryptionMode  != encryptionMethod  ) return DIA_E_OUT_OF_RANGE;

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::validatefileSizeParameterLength ( std::vector<tU8>& data )
{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::validatefileSizeParameterLength");

   if ( data.size() < (DIA_C_U8_REQDNL_MIN_LENGTH + mFilePathAndNameLength + 2) ) return DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;

   size_t dataPos = DIA_C_U8_REQDNL_MIN_LENGTH + mFilePathAndNameLength + 1;
   mFileSizeParameterLength = data[dataPos];

   DIA_TR_INF("RequestFileTransfer: mFileSizeParameterLength = 0x%02x",mFileSizeParameterLength);

   if ( data.size() != (DIA_C_U8_REQDNL_MIN_LENGTH + mFilePathAndNameLength + 1 + mFileSizeParameterLength + mFileSizeParameterLength + mFileSizeParameterLength) )
   {
      return DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
   }

   mSizeInBytes = data[dataPos+1];
   size_t j=dataPos+2;
   for ( tU8 i=1; i<mFileSizeParameterLength; ++i,++j )
   {
      mSizeInBytes = mSizeInBytes + ((tU32) data[j]);
      mSizeInBytes = mSizeInBytes << 8;
   }
   DIA_TR_INF("##### SIZE IN BYTES = %d",mSizeInBytes);

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::validateDataTransferData ( std::vector<tU8>& data )
{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::validateDataTransferData");

   // check if request contains no data
   if ( data.size() <= 2 )
   {
      DIA_TR_ERR("### validateDataTransferData: DIA_E_OUT_OF_RANGE size=%zu", data.size());
      return DIA_E_OUT_OF_RANGE;
   }

   // check range of block sequence number
   tU8 bsn = data[DIA_C_U8_TRANSDATA_BYTEPOS_BSN];
   if ( bsn == 0 )
   {
      DIA_TR_ERR("### validateDataTransferData: DIA_E_WRONG_BLOCK_SEQUENCE_NUMBER");
      return DIA_E_WRONG_BLOCK_SEQUENCE_NUMBER;
   }

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::validateDataTransferExit ( std::vector<tU8>& data )
{
   ScopeTrace trc("dia::UploadDownloadStrategyFiles::validateDataTransferExit");

   tDiaResult retVal = ( data.size() != 1 ) ? DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT : DIA_SUCCESS;

   if (DIA_SUCCESS!=retVal)
   {
      DIA_TR_ERR("### validateDataTransferExit: DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT");
   }

   return retVal;
}

//------------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::triggerFileTransfer ( void )
{
   mErrorCode = DIA_SUCCESS;
   acceptEvent(dia_FileTransferFSM::evRequestFileTransfer,0);

   return mErrorCode;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::triggerReset ( void )
{
   acceptEvent(dia_FileTransferFSM::evIdle,0);
   mErrorCode = DIA_E_NOERROR;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmCalcCrcRam ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmCalcCrcRam()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmCheckForCompletion ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmCheckForCompletion()");

   if ( isDataTransferComplete() )
   {
      acceptEvent(dia_FileTransferFSM::evDataTransferComplete,0);
   }
   else
   {
      DIA_TR_INF("dia::UploadDownloadStrategyFiles::checkForCompletion nothing to be done.");
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmDeleteFile ( void* pArg )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmDeleteFile()");

   dia_File file(mFilePathAndName);
   if ( file.doesExist() )
   {
      tDiaResult retCode = file.remove();
      DIA_TR_INF("##### DELETE FILE \"%s\", retCode = 0x%08x",file.getFileName(),retCode);
      if ( retCode == DIA_SUCCESS )
      {
         acceptEvent(dia_FileTransferFSM::evFileDeleted,pArg);
      }
      else
      {
         mErrorCode = retCode;
      }
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmDownloadData ( void* pArg )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmDownloadData()");
   if ( ! mpFSM || !pArg )
   {
      DIA_TR_ERR("### downloadData DIA_E_GENERAL_PROGRAMMING_FAILURE mpFSM=%s pArg=%s", (!mpFSM? "NULL":"OK"), (!pArg? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   std::vector<tU8> dataVec = *((std::vector<tU8>*) pArg);

   if ( dataVec.size() >= 3 )
   {
      // at this point we know that byte buffer has at least 3 bytes
      tU16 bsn   =  dataVec[1];
      tU8* pData = &dataVec[0];
      tU8* pDataStripped = &pData[2]; //lint !e416 Warning: Likely creation of out-of-bounds pointer (2 beyond end of data) by operator '['

      // data was already validated so we start can safely perform the download at this point
      mErrorCode = downloadData(bsn, pDataStripped, (tU16) (dataVec.size()-2));

      if (DIA_SUCCESS!=mErrorCode)
      {
         DIA_TR_ERR("### downloadData mErrorCode is set to 0x%08X", mErrorCode);
      }

      // data is validated so we can continue processing
      acceptEvent(dia_FileTransferFSM::evDataTransferDone,pArg);
   }
}

//-----------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::downloadData ( tU16 bsn, tU8 data[], tU16 length )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::downloadData(bsn,data,length)");

   // download mode is activated
   tU16 numBytesToWrite = ( (mWritePosition + length) >= mSizeInBytes ) ? tU16(mSizeInBytes - mNumOfBytesReceived) : length;

   DIA_TR_INF("##### numBytesToWrite = %d #####", numBytesToWrite);

   tDiaResult retCode = checkTransferredDownloadData(bsn,data,numBytesToWrite);
   if ( retCode != DIA_SUCCESS ) return retCode;

   ::memcpy(&mpByteBufferDownload[mWritePosition],data,numBytesToWrite);

   mWritePosition = tU16(mWritePosition + numBytesToWrite);
   mNumOfBytesReceived = tU16(mNumOfBytesReceived + numBytesToWrite);
   mBSN = bsn;

   DIA_TR_INF("##### mWritePosition = %d #####", mWritePosition);
   DIA_TR_INF("##### mNumOfBytesReceived = %d #####", mNumOfBytesReceived);
   DIA_TR_INF("##### mSizeInBytes = %d #####", mSizeInBytes);

   if ( mNumOfBytesReceived == mSizeInBytes )
   {
      DIA_TR_INF("##### FILE TRANSFER COMPLETED (DOWNLOAD) #####");
      mIsDataTransferComplete = true;
   }

   DIA_TR_INF("##### mIsDataTransferComplete = %s #####", (mIsDataTransferComplete) ? "true" : "false");

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmPostprocessDownload ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmPostprocessDownload()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmPostprocessUpload ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmPostprocessUpload()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmPrepareDirRead ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmPrepareDirRead()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmPrepareFileAdd ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmPrepareFileAdd()");

   if ( mpByteBufferDownload )
   {
      DIA_DBG_TR_INF("dia::UploadDownloadStrategyFiles::vFsmPrepareFileAdd mpByteBufferDownload=0x%08X", mpByteBufferDownload);
      delete [] mpByteBufferDownload;
      mpByteBufferDownload = 0;
   }

   mpByteBufferDownload = new tU8[mSizeInBytes];
   if ( !mpByteBufferDownload )
   {
      DIA_DBG_TR_ERR("dia::UploadDownloadStrategyFiles::vFsmPrepareFileAdd returned DIA_FAILED");
      return /*DIA_FAILED*/;
   }

   mWritePosition = 0;
   mNumOfBytesReceived = 0;
   mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
   mIsDataTransferActive = true;
   mIsDataTransferComplete = false;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmPrepareFileDelete ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmPrepareFileDelete()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmPrepareFileRead ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmPrepareFileRead()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmPrepareFileReplace ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmPrepareFileReplace()");

   if ( mpByteBufferDownload )
   {
      DIA_DBG_TR_INF("dia::UploadDownloadStrategyFiles::vFsmPrepareFileReplace mpByteBufferDownload=0x%08X", mpByteBufferDownload);
      delete [] mpByteBufferDownload;
      mpByteBufferDownload = 0;
   }

   mpByteBufferDownload = new tU8[mSizeInBytes];
   if ( !mpByteBufferDownload )
   {
      DIA_DBG_TR_ERR("dia::UploadDownloadStrategyFiles::vFsmPrepareFileReplace returned DIA_FAILED");
      return;
   }

   mWritePosition = 0;
   mNumOfBytesReceived = 0;
   mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
   mIsDataTransferActive = true;
   mIsDataTransferComplete = false;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmReset ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmReset()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSaveFile ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSaveFile()");

   dia_File file(mFilePathAndName);

   tDiaResult retCode = file.open("wb+");
   if ( retCode == DIA_SUCCESS )
   {
      size_t numOfBytes = mSizeInBytes;
      retCode = file.rawWrite(mpByteBufferDownload,&numOfBytes);
      if ( (retCode != DIA_SUCCESS) || (numOfBytes != mSizeInBytes) )
      {
         retCode = DIA_E_FILE_WRITE_FAILED;
      }

      (void) file.close();
   }

   getInstanceOfDataDownloadManager()->setOperationMode(DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_IDLE);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendDownloadTransferDataResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendDownloadTransferDataResponse()");
   mErrorCode = DIA_SUCCESS;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorAlreadyRequested ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorAlreadyRequested()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorInvalidData ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorAlreadyRequested()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorInvalidDataTransExit ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorAlreadyRequested()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorInvalidFormat ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorAlreadyRequested()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorInvalidLength ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorInvalidLength()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorInvalidRequest ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorAlreadyRequested()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorInvalidSequence ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorAlreadyRequested()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorNoPermission ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorNoPermission()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedDirRead ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedDirRead()");

   mErrorCode = DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_DIR_READ;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedFileAdd ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedFileAdd()");

   mErrorCode = DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_ADD;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedFileDelete ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedFileDelete()");

   mErrorCode = DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_DELETE;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedFileRead ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedFileRead()");

   mErrorCode = DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_READ;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedFileReplace ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendErrorRequestRejectedFileReplace()");

   mErrorCode = DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_REPLACE;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendRequestDirReadResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendRequestDirReadResponse()");

   mErrorCode = DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_DIR_READ;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendRequestFileAddResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendRequestFileAddResponse()");

   mResponseData.clear();
   mResponseData.push_back(this->mModeOfOperation);
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockByteNum()));
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockMaxBytes() >> 8));
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockMaxBytes() & 0x00FF));
   mResponseData.push_back(this->mDataFormatIdentifier);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendRequestFileDeleteResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendRequestFileDeleteResponse()");

   mResponseData.clear();
   mResponseData.push_back(this->mModeOfOperation);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendRequestFileReadResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendRequestFileReadResponse()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendRequestFileReplaceResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendRequestFileReplaceResponse()");

   mResponseData.clear();
   mResponseData.push_back(this->mModeOfOperation);
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockByteNum()));
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockMaxBytes() >> 8));
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockMaxBytes() & 0x00FF));
   mResponseData.push_back(this->mDataFormatIdentifier);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendTransExitResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendTransExitResponse()");
   mErrorCode = DIA_SUCCESS;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSendUploadTransferDataResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSendUploadTransferDataResponse()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSetDownloadActive ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSetDownloadActive()");
   getInstanceOfDataDownloadManager()->setOperationMode(DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_DOWNLOAD_FILE_TRANSFER);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmSetUploadActive ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmSetUploadActive()");
   getInstanceOfDataDownloadManager()->setOperationMode(DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_UPLOAD_FILE_TRANSFER);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmUploadData ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmUploadData()");
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmValidateFileTransferRequest ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmValidateFileTransferRequest()");

   switch ( this->mModeOfOperation )
   {
   case DIA_EN_MODE_OF_OPERATION_ADD_FILE:
      acceptEvent(dia_FileTransferFSM::evRequestFileAdd,0);
      break;
   case DIA_EN_MODE_OF_OPERATION_DELETE_FILE:
      acceptEvent(dia_FileTransferFSM::evRequestFileDelete,0);
      break;
   case DIA_EN_MODE_OF_OPERATION_REPLACE_FILE:
      acceptEvent(dia_FileTransferFSM::evRequestFileReplace,0);
      break;
   case DIA_EN_MODE_OF_OPERATION_READ_FILE:
      acceptEvent(dia_FileTransferFSM::evRequestFileRead,0);
      break;
   case DIA_EN_MODE_OF_OPERATION_READ_DIR:
      acceptEvent(dia_FileTransferFSM::evRequestDirRead,0);
      break;
   default:
      break;
   };
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyFiles::vFsmValidateTransferredData ( void* pArg )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::vFsmValidateTransferredData()");

   if ( ! mpFSM || !pArg )
   {
      DIA_TR_ERR("### validateTransferredData DIA_E_GENERAL_PROGRAMMING_FAILURE mpFSM=%s pArg=%s", (!mpFSM? "NULL":"OK"), (!pArg? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   std::vector<tU8> dataVec = *((std::vector<tU8>*) pArg);

   if ( dataVec.size() < 3 )
   {
      DIA_TR_INF("dia_DataDownloadManager::validateTransferredData: No Data Available !!");
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   // at this point we know that byte buffer has at least 3 bytes
   tU16 bsn   =  dataVec[1];
   tU8* pData = &dataVec[0];
   tU8* pDataStripped = &pData[2]; //lint !e416 Warning: Likely creation of out-of-bounds pointer (2 beyond end of data) by operator '['

   checkTransferredDownloadData(bsn, pDataStripped, (tU16) (dataVec.size()-2));

   // data is validated so we can continue processing
   acceptEvent(dia_FileTransferFSM::evDataTransferValidated,pArg);
}

//-----------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyFiles::checkTransferredDownloadData ( tU16 bsn, tU8 /*data*/[], tU16 /*length*/ )
{
   ScopeTrace oTrace("dia_Dataset::checkTransferredData(bsn,data,length)");

   if ( isDataTransferComplete() ) {
      DIA_TR_INF("checkTransferredDownloadData: DIA_E_DATASET_ALREADY_COMPLETED");
      return DIA_E_DATASET_ALREADY_COMPLETED;
   }

   if ( bsn == 0 )
   {
      DIA_TR_INF("checkTransferredDownloadData: DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER (0)");
      return DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER;
   }

   if ( bsn != (mBSN + 1) )
   {
      DIA_TR_INF("checkTransferredDownloadData: DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER (received %u, expected %u)", bsn, (mBSN+1) );
      return DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER;
   }

   if ( mWritePosition >= mSizeInBytes )
   {
      DIA_TR_INF("checkTransferredDownloadData: DIA_E_DATASET_INVALID_DATA_RANGE (mWritePosition = %d)", mWritePosition);
      return DIA_E_DATASET_INVALID_DATA_RANGE;
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyFiles::areDirReadConditionsOK ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::areDirReadConditionsOK()");
   return true;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyFiles::areFileAddConditionsOK ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::areFileAddConditionsOK()");

   dia_File file(mFilePathAndName);
   return file.doesExist() ? false : true;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyFiles::areFileDeleteConditionsOK ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::areFileDeleteConditionsOK()");
   dia_File file(mFilePathAndName);
   return file.doesExist() ? true : false;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyFiles::areFileReadConditionsOK ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::areFileReadConditionsOK()");
   dia_File file(mFilePathAndName);
   return file.doesExist() ? true : false;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyFiles::areFileReplaceConditionsOK ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::areFileReplaceConditionsOK()");
   return true;
}

//-----------------------------------------------------------------------------

bool
UploadDownloadStrategyFiles::isDataTransferComplete ( void ) const
{
   return (mIsDataTransferActive) ? mIsDataTransferComplete : false;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyFiles::bFsmIsCrcOk ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::bFsmIsCrcOk()");
   return true;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyFiles::bFsmIsDataValid ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyFiles::bFsmIsDataValid()");
   return true;
}

}

