/*!
 * \file       dia_UploadDownloadStrategy.h
 *
 * \brief      Upload and download strategy used for service $38 (RequestFileTransfer)
 *
 * \details    Upload and download strategy used for service $38 (RequestFileTransfer)
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreUploadDownload
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_UPLOAD_DOWNLOAD_STRATEGY__
#define __INCLUDED_DIA_UPLOAD_DOWNLOAD_STRATEGY__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_DEFINES_DATA_UP_AND_DOWNLOAD__
#include <common/framework/datadownload/dia_defsDataUpDownload.h>
#endif

class dia_DataDownloadManagerPlugin;

namespace dia {

class UploadDownloadStrategy
{
   friend class dia_DataDownloadManager;

public:
   //! class constructor
   UploadDownloadStrategy ( const std::string& name );
   //! class constructor
   UploadDownloadStrategy ( const char* name );
   //! class destructor
   virtual ~UploadDownloadStrategy ( void );

   //! we use a two level construction process
   virtual tDiaResult setup ( void ) { return DIA_SUCCESS; }
   virtual tDiaResult tearDown ( void ){ return DIA_SUCCESS; }

   //! return the strategy's name
   virtual const std::string& getName ( void ) const;
   //! return the unique identifier of associated with a given strategy
   virtual tU32 getID ( void ) = 0;

   //! process a download request (service $34)
   virtual tDiaResult requestDownload ( std::vector<tU8>& requestData ) = 0;
   //! process a file transfer request (service $38)
   virtual tDiaResult requestFileTransfer ( std::vector<tU8>& requestData ) = 0;
   //! process an upload request (service $35)
   virtual tDiaResult requestUpload ( std::vector<tU8>& requestData ) = 0;
   //! process a transfer data request (service $36)
   virtual tDiaResult transferData ( std::vector<tU8>& requestData, std::vector<tU8>& responseData ) = 0;
   //! process a transfer exit request (service $37)
   virtual tDiaResult transferExit ( std::vector<tU8>& requestData ) = 0;

   //! return the response data for the processed request
   virtual const std::vector<tU8>* getResponseData ( void ) const { return &mResponseData; }

   //! data compression used for the data download
   virtual void setCompressionMode ( dia_enDataCompression mode ) { mCompressionMode = mode; }
   //! data compression used for the data download
   virtual dia_enDataCompression getCompressionMode ( void ) const { return mCompressionMode; }
   //! data encryption used for the data download
   virtual void setEncryptionnMode ( dia_enDataEncryption mode ) { mEncryptionMode = mode; }
   //! data compression used for the data download
   virtual dia_enDataEncryption getEncryptionMode ( void ) const { return mEncryptionMode; }

   //! assign the plugin used for project specific NRC handling etc.
   void setPlugin ( const dia_DataDownloadManagerPlugin& plugin ) { mpPlugin = &plugin; }

protected:
   //! deprecated default class constructor
   UploadDownloadStrategy ( void );

protected:
   //! name for this data download set
   std::string mName;
   //! configuration object to customize error codes etc.
   const dia_DataDownloadManagerPlugin* mpPlugin;
   //! response data returned by the conrete strategy after the request was processed
   std::vector<tU8> mResponseData;
   //! data compression used for the data download
   dia_enDataCompression mCompressionMode;
   //! data encryption used for the data download
   dia_enDataEncryption mEncryptionMode;
};

}

#endif /* __INCLUDED_DIA_UPLOAD_DOWNLOAD_STRATEGY__ */
