/*
 * \file        dia_Dataset.h
 *
 * \brief       Class dia_Dataset manages one single data set (load, store, CRC calculation, ...)
 *
 * \details     None
 *
 * \author      gib2hi
 * \date        11.01.2014
 *
 * \copyright   Robert Bosch Car Multimedia 2014
 *
 */

#ifndef __INCLUDED_DIA_DATASET__
#define __INCLUDED_DIA_DATASET__

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#define DIA_C_U16_DATASET_CFG_DOWNLOADABLE            ((tU16) 0x0001)
#define DIA_C_U16_DATASET_CFG_UPLOADABLE              ((tU16) 0x0002)
#define DIA_C_U16_DATASET_CFG_UPDOWNLOADABLE          ((tU16) 0x0003)
#define DIA_C_U16_DATASET_CFG_CRC_FMT_LITTLE_ENDIAN   ((tU16) 0x0004)
#define DIA_C_U16_DATASET_CFG_NO_CRC_CHECK            ((tU16) 0x0008)
#define DIA_C_U16_DATASET_CFG_VARIABLE_LENGTH         ((tU16) 0x0010)

#define DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER    ((tU16)      0)

/**
 * \struct  dia_DatasetConfig
 *
 * \brief   This structure provide configuration data for a dia_Dataset object
 *
 * \ingroup diaCoreAppFrw
 */

struct dia_DatasetConfig
{
   dia_DatasetConfig ( tCString name, tU32 startAddr, tU32 endAddr, tU32 length, tU16 settingsMask, tU32 propID, tU32 itc )
      : mName(name), mAddrStart(startAddr), mAddrEnd(endAddr), mSizeInBytes(length), mSettings(settingsMask), mPropID(propID), mITC(itc)
   {}

   //! name of the data set
   tCString mName;
   //! logical memory start address
   tU32     mAddrStart;
   //! logical memory end address
   tU32     mAddrEnd;
   //! size of the data set in bytes. In case that the dataset might have a variable length this is the maximum size
   tU32     mSizeInBytes;
   //! bit mask with settings for the data set
   tU16     mSettings;
   //! config manager property ID
   tU32     mPropID;
   //! ITC associated with this data set
   tU32     mITC;
};

/**
 * \class   dia_Dataset
 *
 * \brief   This class manages one single data set (load, store, CRC calculation, ...)
 *
 *          This class knows how to convert the logical address into a physical address. In this
 *          case the physical address is represented by a datapool (accessed via PDD).
 *
 * \ingroup diaCoreAppFrw
 */

class dia_Dataset
   : public virtual dia_Object
{
   friend class dia_UTDataset;
   friend class dia_UTDataset_constructor_Test;
   friend class dia_UTDataset_constructor_withConfigObj_Test;

   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_Dataset);

public:
   /**
    * \brief      class constructor
    *
    *             class constructor with name, service identifier and 1 byte
    *             data identifier
    *
    * \param[in]  name        name of the data set
    * \param[in]  startAddr   logical start address of the data set
    * \param[in]  length      size of the data set in bytes
    * \param[in]  propID      byte stream property that represents the datapool element used to
    *                         read/write the data
    *
    * \return     none
    *
    * \todo       replace param propID with a param vector, because other information like the
    *             number of programming attempts need to be stored as well.
    */

   dia_Dataset ( tCString name, tU32 startAddr, tU32 length, tU32 propID );

   /**
    * \brief      class constructor
    *
    *             class constructor that uses the given configuration object to setup the internal
    *             attributes
    *
    * \param[in]  dia_DatasetConfig    configuration object
    *
    * \return     none
    */

   dia_Dataset ( const dia_DatasetConfig& cfgObj );

   /**
    * \brief      class destructor
    *
    * \param[in]  dia_DatasetConfig    configuration object
    *
    * \return     none
    */

   virtual ~dia_Dataset ( void );

   /**
    * \brief      Return the name of the data set
    *
    * \return     tCString    name of the data set
    */

   virtual tCString getName ( void ) const;

   /**
    * \brief      Return the logical start address of the data set
    *
    * \return     tU32    logical start address of the data set
    */

   virtual tU32 getLogicalAddressStart ( void ) const;

   /**
    * \brief      Return the logical end address of the data set
    *
    * \return     tU32    logical end address of the data set
    */

   virtual tU32 getLogicalAddressEnd ( void ) const;

   /**
    * \brief      Return the size of the data set in bytes
    *
    * \return     tU16    size
    */

   virtual tU32 getSize ( void ) const;

   /**
    * \brief      Adjust the settings bitmask for the dataset
    *
    *             Currently the following bits in the bit mask are defined:
    *
    *                0x0001 - DIA_C_U16_DATASET_CFG_DOWNLOADABLE
    *                0x0002 - DIA_C_U16_DATASET_CFG_UPLOADABLE
    *
    * \param[in]  newSettings settings mask
    *
    * \return     tDiaResult  return code (DIA_SUCCESS in case of success)
    */

   virtual tDiaResult adjustSettings ( tU16 newSettings );

   /**
    * \brief      Return the settings bitmask of the dataset
    *
    *             Currently the following bits in the bit mask are defined:
    *
    *                0x0001 - DIA_C_U16_DATASET_CFG_DOWNLOADABLE
    *                0x0002 - DIA_C_U16_DATASET_CFG_UPLOADABLE
    *
    * \return     tU16        settings mask
    */

   virtual tU16 getSettings ( void ) const;

   //! increment the value of the programming attempt counter
   virtual void incrProgrammingAttemptCounter ( void );
   //! clear the value of the programming attempt counter
   virtual void clearProgrammingAttemptCounter ( void );
   //! assign the given value to the programming attempt counter
   virtual void setProgrammingAttemptCounter ( tU16 value );
   //! return the value of the programming attempt counter
   virtual tU16 getProgrammingAttemptCounter ( void ) const;

   //! check if dataset is not in sync with data stored in NVM
   virtual bool isDirty ( void ) const { return mIsDirty; }
   //! check if dataset was loaded from NVM
   virtual bool isLoaded ( void ) const { return mIsLoaded; }
   //! check if the dataset is downloadable
   virtual bool isDownloadable ( void ) const;
   //! check if the dataset is uploadable
   virtual bool isUploadable ( void ) const;
   //! check if the dataset might have a variable length
   virtual bool hasVariableLength ( void ) const;
   //! return the data stored in the dataset item
   virtual const tU8* getDataTransferred ( void ) const { return mpByteBufferDownload; }
   //! return the data stored in the dataset item
   virtual const tU8* getData ( void ) const { return mpByteBuffer; }
   //! return the data stored in the dataset item
   virtual tDiaResult getData ( tU8 data[], tU32 length );
   //! return the data stored in the dataset item
   virtual tDiaResult getData ( tU8 data[], tU32 length, tU32 startByte, tU32 stopByte );
   //! return the data stored in the dataset item
   virtual tDiaResult getData ( tU32* value, tU32 startByte, tU16 startBit, tU32 stopByte, tU16 stopBit );
   //! return the data stored in the dataset item
   virtual tDiaResult getData ( tU8* value, tU32 startByte, tU16 startBit, tU32 stopByte, tU16 stopBit );
   //! set the data of the dataset item
   virtual tDiaResult setData ( const tU8 data[], tU32 length, tBool autoSync = FALSE );
   //! set the data of the dataset item
   virtual tDiaResult sync ( void );

   virtual tU32 getITC ( void ) const { return mITC; }

   //! prepare the dataset for starting a new download
   virtual tDiaResult initializeDataDownload ( void );
   //! finalize the download of dataset data
   virtual tDiaResult finalizeDataDownload ( void );
   //! check a received data block
   virtual tDiaResult checkTransferredDownloadData ( tU16 bsn, tU8 data[], tU32 length );
   //! data download of a block of data
   virtual tDiaResult downloadData ( tU16 bsn, tU8 data[], tU32 length );
   //! check if a data transfer for this dataset is currently active
   virtual bool isDataTransferActive ( void ) const;
   //! check if the data download is complete. will return false in case that no data transfer is currently active
   virtual bool isDataTransferComplete ( void ) const;
   //! pointer to the byte buffer that keeps the downloaded data
   virtual tU32 getNumOfBytesReceived ( void ) const { return mNumOfBytesReceived; }

   //! prepare the dataset for starting a new upload
   virtual tDiaResult initializeDataUpload ( void );
   //! finalize the upload of dataset data
   virtual tDiaResult finalizeDataUpload ( void );
   //! check a received data block
   virtual tDiaResult checkTransferredUploadData ( tU16 bsn );
   //! data transfer of a block of data
   virtual tDiaResult uploadData ( tU16 bsn, std::vector<tU8>& data );

   virtual bool isCrcOK ( void );

   //! check if last bsn is requested again
   virtual bool isRetransmission(tU16 bsn);

   virtual tU32 getRequestedSize ( void ) const { return mSizeInBytesRequested; }
   virtual void setRequestedSize ( tU32 reqSize ) { mSizeInBytesRequested = reqSize; }

   struct ICallback
   {
      virtual void onInitializeDataDownload( void ) = 0;
      virtual void onFinalizeDataDownload( void ) = 0;

   protected:
      virtual ~ICallback() {}
   };

   virtual void setCallback(ICallback* callback) { mpCallback = callback; }

protected:
   //! deprecated default constructor
   dia_Dataset ( void );

   //! load the data associated with this dataset from NVM
   virtual tDiaResult load ( void );
   //! store the data associated with this dataset to NVM
   virtual tDiaResult store ( void );
   //! store the data associated with this dataset to NVM
   virtual tDiaResult store ( tU32 length );
   //! calculate the hash value
   virtual tDiaResult calcCRC ( void );

   virtual tDiaResult validateRange ( tU32 length, tU32 startByte, tU32 stopByte, tU32* pRange=0 );

protected:
   //! name for this data download set
   tCString mName;
   //! logical start address of data download set
   tU32 mStartAddr;
   //! length of the data download set in bytes
   tU32 mSizeInBytes;
   //! requested size for the current ongoing up-/download
   tU32 mSizeInBytesRequested;
   //! bit mask with settings for the data set
   tU16 mSettingsBitmask;
   //! ID of the associated config manager property
   tU32 mPropID;
   //! hash value
   tU32 mCRC;
   //! dirty flag (indicates that the associated data is not in sync with the data stored in persistent memory)
   bool mIsDirty;
   //! loaded flag (indicates that the data was loaded from persistent memory
   bool mIsLoaded;
   //! counter of programming attempts
   tU16 mProgAttempts;
   //! pointer to the byte buffer
   tU8* mpByteBuffer;
   //! associated ITC
   tU32 mITC;

   //! block sequence number
   tU16 mBSN;

   //! flag to indicate that a data transfer is in progress
   bool mIsDataTransferActive;
   //! flag to indicate that a data transfer is in progress
   bool mIsDataTransferComplete;

   //! callbacks for initialize/finalize download/upload
   ICallback* mpCallback;

   //
   // download related attributes
   //

   //! pointer to the byte buffer that keeps the downloaded data
   tU8* mpByteBufferDownload;
   //! pointer to the byte buffer that keeps the downloaded data
   tU32 mWritePosition;
   //! pointer to the byte buffer that keeps the downloaded data
   tU32 mNumOfBytesReceived;

   //
   // upload related attributes
   //

   //! pointer to the byte buffer that keeps the downloaded data
   tU32 mReadPosition;
   //! pointer to the byte buffer that keeps the downloaded data
   tU32 mNumOfBytesTransmitted;
};

#endif /* __INCLUDED_DIA_DATASET__ */
