/*
 * dia_DataDownloadStrategyDefault.cpp
 *
 *  Created on: 01.08.2014
 *      Author: gib2hi
 */

#include "dia_DataDownloadStrategyDefault.h"

#ifndef __INCLUDED_DIA_DATASET__
#include "common/framework/datadownload/dia_Dataset.h"
#endif

#ifndef __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__
#include "common/framework/datadownload/dia_DataDownloadManager.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include <common/framework/engine/dia_EngineServer.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include <common/framework/engine/dia_EngineManager.h>
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include <common/framework/engine/dia_Session.h>
#endif

#define DIA_C_U8_REQDNL_MSG_DATA_OFFSET         ((tU8) 1)

#define DIA_C_U8_REQDNL_BYTEPOS_SRV_ID          ((tU8) 0)
#define DIA_C_U8_REQDNL_BYTEPOS_FMT_ID          ((tU8) 1)
#define DIA_C_U8_REQDNL_BYTEPOS_ADDRFMT_ID      ((tU8) 2)
#define DIA_C_U8_REQDNL_BYTEPOS_STARTADDR       ((tU8) 3)

#define DIA_C_U8_TRANSDATA_BYTEPOS_BSN          ((tU8) 1)

using namespace dia;

//-----------------------------------------------------------------------------

dia_DataDownloadStrategyDefault::dia_DataDownloadStrategyDefault ( std::string& name , tU8 fmtID, tU8 addrFmt )
   : UploadDownloadStrategyDatasets(name, fmtID, addrFmt),
     mLengthOfMemoryAddress((addrFmt & 0x0F)),
     mLengthOfUncrompressedMemorySize(((addrFmt & 0xF0)>>4))
{}

//-----------------------------------------------------------------------------

dia_DataDownloadStrategyDefault::~dia_DataDownloadStrategyDefault ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::setup ( void )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::setup");

   return registerSessionChange();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::tearDown ( void )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::tearDown");

   return unregisterSessionChange();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::acceptEvent ( dia_DataDownloadFSM::FsmEvent event, void* pArg )
{
   ScopeTrace oTrace("dia_DataDownloadStrategyDefault::acceptEvent(dia_DataDownloadFSM::FsmEvent,void*)");

   if ( !mpFSM )  return DIA_E_FSM_NOT_AVAILABLE;

   mErrorCode = DIA_E_NO_ERROR;

   DIA_TR_INF("dia_DataDownloadStrategyDefault::acceptEvent - State (before): %s, event = '%d'", mpFSM->getStateName(),event);
   mpFSM->acceptEvent(event,pArg);
   DIA_TR_INF("dia_DataDownloadStrategyDefault::acceptEvent - State (after): %s", mpFSM->getStateName());

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::requestDownload ( std::vector<tU8>& requestData )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::requestDownload");

   tDiaResult retCode = validateDataRequestUpDownload(requestData);

   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_DataDownloadStrategyDefault: FAILED (DATA VALIDATION) !!");
      return retCode;
   }

   // extract the start address from the message buffer
   tU32 startAddress = 0;
   tU32 shiftValue   = (mLengthOfMemoryAddress - 1) * 8;
   for ( tU16 i=0; i<mLengthOfMemoryAddress; i++ )
   {
      startAddress |= ((tU32) requestData[DIA_C_U8_REQDNL_BYTEPOS_STARTADDR+i]) << shiftValue;
      shiftValue   -= 8;
   }

   // extract the dataset size from the message buffer
   tU32 memSize = 0;
   shiftValue   = (mLengthOfUncrompressedMemorySize - 1) * 8;
   for ( tU16 i=0; i<mLengthOfUncrompressedMemorySize; i++ )
   {
      memSize |= ((tU32) requestData[DIA_C_U8_REQDNL_BYTEPOS_STARTADDR + mLengthOfMemoryAddress+i]) << shiftValue;
      shiftValue   -= 8;
   }

   mRequestedStartAddress = startAddress;
   mRequestedMemSize      = memSize;
   DIA_TR_INF("StartAddress: 0x%08x, DataLength: 0x%08x", startAddress,memSize);

   // reset the state machine (required in case that a download is already in progress)
   mpFSM->acceptEvent(dia_DataDownloadFSM::evIdle,0);

   return triggerDownload();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::requestUpload ( std::vector<tU8>& requestData )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::requestUpload");

   tDiaResult retCode = validateDataRequestUpDownload(requestData);

   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_DataDownloadStrategyDefault::requestUpload: FAILED (DATA VALIDATION) !!");
      return retCode;
   }

   // extract the start address from the message buffer
   tU32 startAddress = 0;
   tU32 shiftValue   = (mLengthOfMemoryAddress - 1) * 8;
   for ( tU16 i=0; i<mLengthOfMemoryAddress; i++ )
   {
      startAddress |= ((tU32) requestData[DIA_C_U8_REQDNL_BYTEPOS_STARTADDR+i]) << shiftValue;
      shiftValue   -= 8;
   }

   // extract the dataset size from the message buffer
   tU32 memSize = 0;
   shiftValue   = (mLengthOfUncrompressedMemorySize - 1) * 8;
   for ( tU16 i=0; i<mLengthOfUncrompressedMemorySize; i++ )
   {
      memSize |= ((tU32) requestData[DIA_C_U8_REQDNL_BYTEPOS_STARTADDR + mLengthOfMemoryAddress+i]) << shiftValue;
      shiftValue   -= 8;
   }

   mRequestedStartAddress = startAddress;
   mRequestedMemSize      = memSize;
   DIA_TR_INF("StartAddress: 0x%08x, DataLength: 0x%08x", startAddress,memSize);

   return triggerUpload();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::transferData ( std::vector<tU8>& requestData, std::vector<tU8>& responseData )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::transferData");

   responseData.clear();
   mResponseData.clear();

   mErrorCode = validateDataTransferData(requestData);

   if ( mErrorCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("### transferData FAILED (DATA VALIDATION) mErrorCode=0x%08X", mErrorCode);
      return mErrorCode;
   }

   acceptEvent(dia_DataDownloadFSM::evDataTransferRx,(void*) &requestData);

   if ( mErrorCode == DIA_E_NOERROR ) //lint !e774: Boolean always evaluates to True; gib2hi: Not true, because mErrorCode might have been changed in acceptEvent() call
   {
      for ( tU16 i=0; i<mResponseData.size(); i++ ) responseData.push_back(mResponseData[i]);
   }

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::transferExit ( std::vector<tU8>& requestData )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::transferExit");

   mErrorCode = validateDataTransferExit(requestData);

   if ( mErrorCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_DataDownloadStrategyDefault: FAILED (DATA VALIDATION) !!");
      return mErrorCode;
   }

   if ( mpFSM ) mpFSM->acceptEvent(dia_DataDownloadFSM::evDataTransferExit,0);

   return mErrorCode;
}

//----------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::validateDataRequestUpDownload ( std::vector<tU8>& data )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::validateDataRequestUpDownload");

   tU8 compressionMethod = (data[DIA_C_U8_REQDNL_BYTEPOS_FMT_ID] & 0xF0) >> 4;
   tU8 encryptionMethod  =  data[DIA_C_U8_REQDNL_BYTEPOS_FMT_ID] & 0x0F;

   if ( mCompressionMode != compressionMethod ) return DIA_E_OUT_OF_RANGE;
   if ( mEncryptionMode  != encryptionMethod  ) return DIA_E_OUT_OF_RANGE;

   tU8 lengthOfUncrompressedMemorySize = (data[DIA_C_U8_REQDNL_BYTEPOS_ADDRFMT_ID] & 0xF0) >> 4;
   tU8 lengthOfMemoryAddress           =  data[DIA_C_U8_REQDNL_BYTEPOS_ADDRFMT_ID] & 0x0F;

   if ( mLengthOfUncrompressedMemorySize != lengthOfUncrompressedMemorySize ) return DIA_E_OUT_OF_RANGE;
   if ( mLengthOfMemoryAddress  != lengthOfMemoryAddress  ) return DIA_E_OUT_OF_RANGE;

   size_t length = 3 /* sid + format identifiers */ + lengthOfMemoryAddress + lengthOfUncrompressedMemorySize;

   return checkLength(data.size(),length,length);
}

//----------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::validateDataTransferData ( std::vector<tU8>& data )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::validateDataTransferData");

   // check if request contains no data
   if ( data.size() < 2 )
   {
      DIA_TR_ERR("### validateDataTransferData: DIA_E_OUT_OF_RANGE size=%zu", data.size());
      return DIA_E_OUT_OF_RANGE;
   }

   // check if a up- or download was requested before
   if ( !mRequestedDataset )
   {
      DIA_TR_ERR("### validateDataTransferData: DIA_E_SEQUENCE_ERROR");
      return DIA_E_SEQUENCE_ERROR;
   }

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::validateDataTransferExit ( std::vector<tU8>& data )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::validateDataTransferExit");

   tDiaResult retVal;
   retVal = ( data.size() != 1 ) ? DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT : DIA_SUCCESS;

   if (DIA_SUCCESS!=retVal)
   {
      DIA_TR_ERR("### validateDataTransferExit: DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT");
   }

   return retVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::checkLength ( size_t length, size_t minValue, size_t maxValue )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::checkLength");

   tDiaResult retVal;
   retVal = ((length >= minValue) && (length <= maxValue)) ? DIA_SUCCESS : DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;

   if (DIA_SUCCESS!=retVal)
   {
      DIA_TR_ERR("### checkLength: DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT");
   }

   return retVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::registerSessionChange ( void )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::registerSessionChange");

   tDiaResult retVal = DIA_FAILED;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      if( pEngine->getSessionController()->addListener(this))
      {
         retVal = DIA_SUCCESS;
      }
   }

   if(DIA_SUCCESS != retVal)
   {
      DIA_TR_ERR("!!! dia_DataDownloadStrategyDefault::registerSessionChange => ERROR: Unable to register for Session changes");
   }

   return retVal;
}

//-----------------------------------------------------------------------------

void
dia_DataDownloadStrategyDefault::vOnSessionChanged ( tU8 newSession, tU8 oldSession )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::vOnSessionChanged()");

   //terminate DataDownload if session changed to default session
   if( DIA_C_U8_UDS_SESSION_DEFAULT == newSession)
   {
      if (mpFSM)
      {
         DIA_TR_INF("dia_DataDownloadStrategyDefault session change from %d to %d in state '%s'", oldSession, newSession, mpFSM->getStateName());

         mpFSM->acceptEvent(dia_DataDownloadFSM::evSessionChange,0);
      }
      else
      {
         DIA_TR_ERR("dia_DataDownloadStrategyDefault::vOnSessionChanged => ERROR: FSM IS NULL!!!");
      }
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadStrategyDefault::unregisterSessionChange ( void )
{
   ScopeTrace trc("dia_DataDownloadStrategyDefault::unregisterSessionChange");

   tDiaResult retVal = DIA_FAILED;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      if( pEngine->getSessionController()->removeListener(this))
      {
         retVal = DIA_SUCCESS;
      }
   }

   if(DIA_SUCCESS != retVal)
   {
      DIA_TR_ERR("!!! dia_DataDownloadStrategyDefault::unregisterSessionChange => ERROR: Unable to deregister for Session changes");
   }

   return retVal;
}
