/*!
 * \file       dia_DataDownloadManager.h
 *
 * \brief      Manager class that controls dataset and file upload and download (services $34, $35, $36, $37, $38)
 *
 * \details    Manager class that controls dataset and file upload and download (services $34, $35, $36, $37, $38)
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreUploadDownload
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__
#define __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include <common/framework/config/dia_defsConfig.h>
#endif

#ifndef __INCLUDED_DIA_DATADOWNLOADFSM__
#include <common/framework/fsm/generated/dia_DataDownloadFSM.h>
#endif

#ifndef __INCLUDED_DIA_DATASET__
#include <common/framework/datadownload/dia_Dataset.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_DATA_DOWNLOAD_LISTENER__
#include "common/interfaces/dia_IDataDownloadListener.h"
#endif

enum dia_enDownloadManagerOperationMode
{
   DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_UNKNOWN = 0,
   DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_IDLE,
   DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_DOWNLOAD,
   DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_DOWNLOAD_FILE_TRANSFER,
   DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_UPLOAD,
   DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_UPLOAD_FILE_TRANSFER,
   DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_COUNT
};

class dia_DataDownloadManagerPlugin;

namespace dia {
   class UploadDownloadStrategy;
}

//-----------------------------------------------------------------------------

class dia_DataDownloadManager
   : public dia_IDataDownload
{
   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_DataDownloadManager);
   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_DataDownloadManager);

public:
   //! add a new data set
   virtual tDiaResult addDataset ( dia_Dataset* dataset );
   //! remove a data set item from the data set repository
   virtual tDiaResult removeDataset ( tU32 startAddr );
   //! query for a data set via its start address
   virtual tDiaResult queryDataset ( tU32 startAddress, dia_Dataset** ppDataset );
   //! return the number of managed datasets
   virtual tU16 numberOfDataSets ( void ) const;

   //! load a set of properties
   virtual tDiaResult loadDatasets ( const dia_DatasetConfig data[], tU16 numOfItems );

   virtual tDiaResult requestDownload ( std::vector<tU8>& requestData );
   virtual tDiaResult requestFileTransfer ( std::vector<tU8>& requestData );
   virtual tDiaResult requestUpload ( std::vector<tU8>& requestData );
   virtual tDiaResult transferData ( std::vector<tU8>& requestData, std::vector<tU8>& responseData );
   virtual tDiaResult transferExit ( std::vector<tU8>& requestData );

   virtual tU16 getDataBlockByteNum  ( void ) const;
   virtual tU16 getDataBlockMaxBytes ( void ) const;

   virtual void setPlugin ( const dia_DataDownloadManagerPlugin& plugin );
   virtual const dia_DataDownloadManagerPlugin* getPlugin ( void ) const { return mpPlugin; }

   virtual const std::vector<tU8>* getResponseData ( void ) const;

   //! activate/deactivate CRC checks for dataset downloads
   virtual void setCrcMode ( bool crcMode );
   //! return current CRC mode
   virtual bool getCrcMode ( void ) const { return mCRCActivationMode; }

   virtual dia_enDownloadManagerOperationMode getOperationMode ( void ) const { return mOperationMode; }
   virtual void setOperationMode ( dia_enDownloadManagerOperationMode mode ) { mOperationMode = mode; }

   virtual bool getPowerMonitoringMode ( void ) const { return mPowerMonitoringMode; }
   virtual void setPowerMonitoringMode ( bool monitoringMode ) { mPowerMonitoringMode = monitoringMode; }

   //! add a new download strategy
   virtual tDiaResult addDataDownloadStrategy ( dia::UploadDownloadStrategy* strategy );
   //! remove the associated download strategy from the download strategy repository
   virtual tDiaResult removeDataDownloadStrategy ( tU32 id ); //tU8 fmtID, tU8 addrFmtID );
   //! return the number of managed download strategy
   virtual tU16 numberOfDataDownloadStrategies ( void ) const;
   //!
   virtual tDiaResult queryDataDownloadStrategy ( tU32 id, dia::UploadDownloadStrategy** ppStrategy );

protected:
   //! protected class constructor
   dia_DataDownloadManager ( void );
   //! protected class destructor
   virtual ~dia_DataDownloadManager ( void );

   //! setup method for the application
   virtual tDiaResult setup ( void );
   //! shutdown method for the application
   virtual tDiaResult tearDown ( void ) const;
   //! clear the content of all repositories
   virtual void destroy ( void );

protected:
   //! repository of available datasets
   std::map<tU32,dia_Dataset*> mDatasetRep;
   //! repository of available download strategies
   std::map<tU32,dia::UploadDownloadStrategy*> mStrategyRep;

   //! coniguration object
   const dia_DataDownloadManagerPlugin* mpPlugin;
   //! currently activated strategy
   dia::UploadDownloadStrategy* mpActiveStrategy;
   //! mode for CRC checks
   bool mCRCActivationMode;
   //! if power monitoring mode is enabled, dataset download will be rejected in low or high voltage operation mode
   bool mPowerMonitoringMode;

   dia_enDownloadManagerOperationMode mOperationMode;
};

#endif /* __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__ */
