/*!
 * \file       dia_DataDownloadManager.cpp
 *
 * \brief      Manager class that controls dataset and file upload and download (services $34, $35, $36, $37, $38)
 *
 * \details    Manager class that controls dataset and file upload and download (services $34, $35, $36, $37, $38)
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreUploadDownload
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__
#include "common/framework/datadownload/dia_DataDownloadManager.h"
#endif

#ifndef __INCLUDED_DATA_DOWNLOAD_MANAGER_PLUGIN__
#include "common/framework/datadownload/dia_DataDownloadManagerPlugin.h"
#endif

#ifndef __INCLUDED_DIA_DATASET__
#include "common/framework/datadownload/dia_Dataset.h"
#endif

#ifndef __INCLUDED_DIA_DATA_DOWNLOAD_STRATEGY_DEFAULT__
#include "common/framework/datadownload/dia_DataDownloadStrategyDefault.h"
#endif

#ifndef __INCLUDED_DIA_UPLOAD_DOWNLOAD_STRATEGY_FILES__
#include <common/framework/datadownload/dia_UploadDownloadStrategyFiles.h>
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include <common/framework/application/dia_AppController.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#define DIA_C_U8_REQDNL_BYTEPOS_SRV_ID          ((tU8) 0)
#define DIA_C_U8_REQDNL_BYTEPOS_FMT_ID          ((tU8) 1)
#define DIA_C_U8_REQDNL_BYTEPOS_ADDRFMT_ID      ((tU8) 2)
#define DIA_C_U8_REQDNL_BYTEPOS_STARTADDR       ((tU8) 3)

#define DIA_C_U32_REQ_FILE_TRANSFER_LOOKUP_KEY  ((tU32) 0x00010000) // 0x000abbcc --> a=1: File Transfer, bb: Format ID, cc: addrFormat ID

using namespace dia;

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_DataDownloadManager)

#ifndef __DIA_UNIT_TESTING__

dia_DataDownloadManager*
getInstanceOfDataDownloadManager ( void )
{
   return dia_DataDownloadManager::getInstance();
}

void
releaseInstanceOfDataDownloadManager ( void )
{
   dia_DataDownloadManager::deleteInstance();
}

#endif

//-----------------------------------------------------------------------------

dia_DataDownloadManager::dia_DataDownloadManager ( void )
   : mpPlugin(0),
     mpActiveStrategy(0),
     mCRCActivationMode(true),
     mPowerMonitoringMode(false),
     mOperationMode(DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_IDLE)
{}

//-----------------------------------------------------------------------------

dia_DataDownloadManager::~dia_DataDownloadManager ( void )
{
   _BP_TRY_BEGIN
   {
      delete mpPlugin;
      mpPlugin = 0;

      mDatasetRep.clear();

      std::map<tU32,dia::UploadDownloadStrategy*>::iterator iter = mStrategyRep.begin();
      for ( ; iter != mStrategyRep.end(); ++iter )
      {
         if ( iter->second )
         {
            (void)(iter->second)->tearDown();
            delete iter->second;
         }
      }
      mStrategyRep.clear();

      mpActiveStrategy  = 0;
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_DataDownloadManager::~dia_DataDownloadManager !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::setup ( void )
{
   ScopeTrace trc("dia_DataDownloadManager::setup");

   if ( !mpPlugin )
   {
      mpPlugin = new dia_DataDownloadManagerPlugin;
   }

   if ( !mpPlugin ) return DIA_FAILED;

   std::string strategy1 = "dia_DataDownloadStrategyDefault";
   dia_DataDownloadStrategyDefault* pStrategy1 = new dia_DataDownloadStrategyDefault(strategy1);
   if (pStrategy1)
   {
      if( pStrategy1->setup() != DIA_SUCCESS )
      {
         DIA_TR_ERR("Failed to setup dia_DataDownloadStrategyDefault (ADDR=0x%p)", pStrategy1);
      }
   }
   tDiaResult retCode1 = addDataDownloadStrategy(pStrategy1);

   std::string strategy2 = "dia::UploadDownloadStrategyFiles";
   UploadDownloadStrategyFiles* pStrategy2 = new UploadDownloadStrategyFiles(strategy2);
   if (pStrategy2)
   {
      if( pStrategy2->setup() != DIA_SUCCESS )
      {
         DIA_TR_ERR("Failed to setup UploadDownloadStrategyFiles (ADDR=0x%p)", pStrategy2);
      }
   }
   tDiaResult retCode2 = addDataDownloadStrategy(pStrategy2);

   return ( (retCode1 == DIA_SUCCESS) && (retCode2 == DIA_SUCCESS) ) ? DIA_SUCCESS : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::tearDown ( void ) const
{
   ScopeTrace trc("dia_DataDownloadManager::tearDown");
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_DataDownloadManager::setPlugin ( const dia_DataDownloadManagerPlugin& plugin )
{
   ScopeTrace trc("dia_DataDownloadManager::setPlugin()");

   // delete the existing plugin
   if ( mpPlugin )
   {
      DIA_TR_INF("DataDownload Plugin to be removed: %p",mpPlugin);
      delete mpPlugin;
   }

   mpPlugin = &plugin;
   DIA_TR_INF("DataDownload Plugin installed:     %p",mpPlugin);

   // now inject the new plugin into every upload/download strategy
   std::map<tU32,dia::UploadDownloadStrategy*>::iterator iter = mStrategyRep.begin();
   for ( ; iter != mStrategyRep.end(); ++iter )
   {
      if ( iter->second ) iter->second->setPlugin(*mpPlugin);
   }
}

//-----------------------------------------------------------------------------

void
dia_DataDownloadManager::destroy ( void )
{
   mDatasetRep.clear();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::addDataset ( dia_Dataset* pDataset )
{
   ScopeTrace oTrace("dia_DataDownloadManager::addDataset()");

   if ( !pDataset ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   std::map<tU32,dia_Dataset*>::iterator iter = mDatasetRep.find(pDataset->getLogicalAddressStart());
   if ( iter == mDatasetRep.end() )
   {
      DIA_TR_INF("#######################################################");
      DIA_TR_INF("#");
      DIA_TR_INF("# ADDING DATASET \"%s\"", pDataset->getName());
      DIA_TR_INF("#");
      DIA_TR_INF("#######################################################");
      mDatasetRep[pDataset->getLogicalAddressStart()] = pDataset;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::removeDataset ( tU32 startAddr )
{
   ScopeTrace oTrace("dia_DataDownloadManager::removeDataset()");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   std::map<tU32,dia_Dataset*>::iterator iter = mDatasetRep.find(startAddr);
   if ( iter != mDatasetRep.end() )
   {
      mDatasetRep.erase(startAddr);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::queryDataset ( tU32 startAddr, dia_Dataset** ppDataset )
{
   ScopeTrace oTrace("dia_DataDownloadManager::queryDataset()");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   DIA_TR_INF("dia_DataDownloadManager::queryDataset startAddr=0x%08X", startAddr);

   if ( !ppDataset ) return DIA_E_INVALID_POINTER;

   std::map<tU32,dia_Dataset*>::iterator iter = mDatasetRep.find(startAddr);
   if ( iter != mDatasetRep.end() )
   {
      (*ppDataset) = iter->second;
      retCode = DIA_SUCCESS;
      DIA_TR_INF("dia::UploadDownloadStrategyDatasets::Data set found");
   }
   else
   {
      (*ppDataset) = 0;
      DIA_TR_ERR("dia::UploadDownloadStrategyDatasets::Data set NOT found");
   }

   return retCode;
}

//------------------------------------------------------------------------------

tU16
dia_DataDownloadManager::numberOfDataSets ( void ) const
{
   return (tU16) mDatasetRep.size();
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::loadDatasets ( const dia_DatasetConfig data[], tU16 numOfItems )
{
   ScopeTrace oTrace("dia_DataDownloadManager::loadDatasets");

   tDiaResult retCode = DIA_SUCCESS;

   if ( data )
   {
      for ( tU16 i=0; i<numOfItems; ++i )
      {
         dia_Dataset* pDataset = OSAL_NEW dia_Dataset(data[i]);
         if ( !pDataset )
         {
            retCode = DIA_FAILED;
            break;
         }
         mDatasetRep[pDataset->getLogicalAddressStart()] = pDataset;
         DIA_TR_ERR("dia_DataDownloadManager::loadDatasets pDataset=0x%04x Size=%d", pDataset->getLogicalAddressStart(), pDataset->getSize() );
      }
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::addDataDownloadStrategy ( dia::UploadDownloadStrategy* pStrategy )
{
   ScopeTrace oTrace("dia_DataDownloadManager::addDataDownloadStrategy()");

   if ( !pStrategy ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;
   DIA_TR_INF("dia_DataDownloadManager::addDataDownloadStrategy, pStrategy->getID() == 0x%08x ", pStrategy->getID() );

   std::map<tU32,dia::UploadDownloadStrategy*>::iterator iter = mStrategyRep.find(pStrategy->getID());
   if ( iter == mStrategyRep.end() )
   {
      DIA_TR_INF("#######################################################");
      DIA_TR_INF("#");
      DIA_TR_INF("# ADDING DATASET DOWNLOAD STRATEGY \"%s\"", pStrategy->getName().c_str());
      DIA_TR_INF("#");
      DIA_TR_INF("#######################################################");
      mStrategyRep[pStrategy->getID()] = pStrategy;
      if ( mpPlugin ) pStrategy->setPlugin(*mpPlugin);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::removeDataDownloadStrategy ( tU32 id )
{
   ScopeTrace oTrace("dia_DataDownloadManager::removeDataDownloadStrategy()");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   std::map<tU32,dia::UploadDownloadStrategy*>::iterator iter = mStrategyRep.find(id);
   if ( iter != mStrategyRep.end() )
   {
      (void)(iter->second)->tearDown();
      delete iter->second;
      mStrategyRep.erase(id);

      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::queryDataDownloadStrategy ( tU32 id, dia::UploadDownloadStrategy** ppStrategy )
{
   ScopeTrace oTrace("dia_DataDownloadManager::queryDataDownloadStrategy()");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   if ( !ppStrategy ) return DIA_E_INVALID_POINTER;

   (*ppStrategy) = 0;
   std::map<dia_UID,dia::UploadDownloadStrategy*>::iterator iter = mStrategyRep.find(id);
   if ( iter != mStrategyRep.end() )
   {
      (*ppStrategy) = iter->second;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tU16
dia_DataDownloadManager::numberOfDataDownloadStrategies ( void ) const
{
   return (tU16) mStrategyRep.size();
}

//------------------------------------------------------------------------------

tU16
dia_DataDownloadManager::getDataBlockByteNum ( void ) const
{
   return ( mpPlugin ) ? mpPlugin->getDataBlockByteNum() : 0;
}

//------------------------------------------------------------------------------

tU16
dia_DataDownloadManager::getDataBlockMaxBytes ( void ) const
{
   return ( mpPlugin ) ? mpPlugin->getDataBlockMaxBytes() : 0;
}

//------------------------------------------------------------------------------

const std::vector<tU8>*
dia_DataDownloadManager::getResponseData ( void ) const
{
   return (( mpActiveStrategy ) ? mpActiveStrategy->getResponseData() : 0);
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::requestDownload ( std::vector<tU8>& requestData )
{
   ScopeTrace oTrace("dia_DataDownloadManager::requestDownload()");

   if ( getPowerMonitoringMode() && (!getInstanceOfAppController()->isPowerModeNormal()) )
   {
      return DIA_E_UPLOAD_DOWNLOAD_REJECTED_DUE_TO_POWER_LEVEL;
   }

   tU8 fmtID     = requestData[DIA_C_U8_REQDNL_BYTEPOS_FMT_ID];
   tU8 addrFmtID = requestData[DIA_C_U8_REQDNL_BYTEPOS_ADDRFMT_ID];
   tU32 key      = (((tU32) addrFmtID) | (((tU32) fmtID ) << 8));

   DIA_TR_INF("requestDownload, fmtID = 0x%02x, addrFmtID = 0x%02x, key = 0x%04x", int(fmtID), int(addrFmtID), int(key) );

   std::map<tU32,dia::UploadDownloadStrategy*>::iterator iter = mStrategyRep.find(key);
   if ( iter == mStrategyRep.end() )
   {
       DIA_TR_ERR("requestDownload, strategy not found" );
       return DIA_E_OUT_OF_RANGE;
   }

   mpActiveStrategy = iter->second;
   if ( !mpActiveStrategy )
   {
       DIA_TR_ERR("requestDownload, mpActiveStrategy = NULL");
       return DIA_E_UPLOAD_DOWNLOAD_REQUEST_NO_STRATEGY;
   }

   return mpActiveStrategy->requestDownload(requestData);
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::requestFileTransfer ( std::vector<tU8>& requestData )
{
   ScopeTrace oTrace("dia_DataDownloadManager::requestFileTransfer()");

   if ( getPowerMonitoringMode() && (!getInstanceOfAppController()->isPowerModeNormal()) )
   {
      return DIA_E_FILE_TRANSFER_REJECTED_DUE_TO_POWER_LEVEL;
   }

   tU32 key = DIA_C_U32_REQ_FILE_TRANSFER_LOOKUP_KEY;

   DIA_TR_INF("requestFileTransfer, key = 0x%04x", int(key) );

   std::map<tU32,dia::UploadDownloadStrategy*>::iterator iter = mStrategyRep.find(key);
   if ( iter == mStrategyRep.end() )
   {
       DIA_TR_ERR("requestFileTransfer, strategy not found" );
       return DIA_E_OUT_OF_RANGE;
   }

   mpActiveStrategy = iter->second;
   if ( !mpActiveStrategy )
   {
       DIA_TR_ERR("requestFileTransfer, mpActiveStrategy = NULL");
       return DIA_E_FILE_TRANSFER_NO_STRATEGY;
   }

   return mpActiveStrategy->requestFileTransfer(requestData);
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::requestUpload ( std::vector<tU8>& requestData )
{
   ScopeTrace oTrace("dia_DataDownloadManager::requestUpload()");

   if ( getPowerMonitoringMode() && (!getInstanceOfAppController()->isPowerModeNormal()) )
   {
      return DIA_E_UPLOAD_DOWNLOAD_REJECTED_DUE_TO_POWER_LEVEL;
   }

   tU8 fmtID     = requestData[DIA_C_U8_REQDNL_BYTEPOS_FMT_ID];
   tU8 addrFmtID = requestData[DIA_C_U8_REQDNL_BYTEPOS_ADDRFMT_ID];
   tU32 key      = (((tU32) addrFmtID) | (((tU32) fmtID ) << 8));

   DIA_TR_INF("requestUpload, fmtID = 0x%02x, addrFmtID = 0x%02x, key = 0x%04x", int(fmtID), int(addrFmtID), int(key) );

   std::map<tU32,dia::UploadDownloadStrategy*>::iterator iter = mStrategyRep.find(key);
   if ( iter == mStrategyRep.end() )
   {
       DIA_TR_ERR("requestUpload, strategy not found" );
       return DIA_E_OUT_OF_RANGE;
   }

   mpActiveStrategy = iter->second;
   if ( !mpActiveStrategy )
   {
       DIA_TR_ERR("requestUpload, mpActiveStrategy = NULL");
       return DIA_E_UPLOAD_DOWNLOAD_REQUEST_NO_STRATEGY;
   }

   return mpActiveStrategy->requestUpload(requestData);
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::transferData ( std::vector<tU8>& requestData, std::vector<tU8>& responseData )
{
   ScopeTrace oTrace("dia_DataDownloadManager::transferData()");

   if ( getPowerMonitoringMode() && (!getInstanceOfAppController()->isPowerModeNormal()) )
   {
      return DIA_E_UPLOAD_DOWNLOAD_REJECTED_DUE_TO_POWER_LEVEL;
   }

   if ( !mpActiveStrategy )
   {
      return DIA_E_UPLOAD_DOWNLOAD_TRANSFER_NO_STRATEGY;
   }

   return mpActiveStrategy->transferData(requestData, responseData);
}

//------------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManager::transferExit ( std::vector<tU8>& requestData )
{
   ScopeTrace oTrace("dia_DataDownloadManager::transferExit()");

   if ( getPowerMonitoringMode() && (!getInstanceOfAppController()->isPowerModeNormal()) )
   {
      return DIA_E_UPLOAD_DOWNLOAD_REJECTED_DUE_TO_POWER_LEVEL;
   }

   if ( !mpActiveStrategy )
   {
      return DIA_E_UPLOAD_DOWNLOAD_EXIT_NO_STRATEGY;
   }

   return mpActiveStrategy->transferExit(requestData);
}

//------------------------------------------------------------------------------

void
dia_DataDownloadManager::setCrcMode ( bool crcMode )
{
   mCRCActivationMode = crcMode;
   DIA_TR_INF("DATADOWNLOAD CRC CHECK %s!", (mCRCActivationMode) ? "ENABLED" : "DISABLED");
}
