/*!
 * \file       dia_PropertyBagRAM.cpp
 *
 * \brief      property-bag with values only stored in RAM.
 *
 * \details    -
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreConfig 
 *
 * \copyright  (c) 2018 Robert Bosch Car Multimedia
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_PROPERTY_BAG_RAM__
#include "dia_PropertyBagRAM.h"
#endif

#define EIGHT 8
//----------------------------------------------------------------------------------------

dia_PropertyBagRAM::dia_PropertyBagRAM ( void )
    : dia_PropertyBag(DIA_C_STR_PROPBAG_RAM)
{
    dia_tclFnctTrace trc("dia_PropertyBagRAM::dia_PropertyBagRAM");
}

//----------------------------------------------------------------------------------------

dia_PropertyBagRAM::~dia_PropertyBagRAM ( void )
{
   //    dia_tclFnctTrace trc("dia_PropertyBagRAM::~dia_PropertyBagRAM");
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::getPropertyU8 ( tU32 propID, tU8& propValue )
{
   dia_tclFnctTrace trc("dia_PropertyBagRAM::getPropertyU8(tU32 propID, tU8& propValue)");

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) != DIA_SUCCESS ) return DIA_E_INVALID_KEY;
   if ( propData.mPropSize != sizeof(tU8) ) return DIA_E_INVALID_LENGTH;
 
   propValue = static_cast<tU8>(propData.mPropExtData[0]);

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::getPropertyU16 ( tU32 propID, tU16& propValue )
{
   dia_tclFnctTrace trc("dia_PropertyBagRAM::getPropertyU8(tU32 propID, tU16& propValue)");

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) != DIA_SUCCESS ) return DIA_E_INVALID_KEY;
   if ( (propData.mPropSize < sizeof(tU8)) || (propData.mPropSize > sizeof(tU16)) ) return DIA_E_INVALID_LENGTH;

   propValue = static_cast<tU16>(propData.mPropExtData[0]);

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::getPropertyU32 ( tU32 propID, tU32& propValue )
{
   dia_tclFnctTrace trc("dia_PropertyBagRAM::getPropertyU32(tU32 propID, tU32& propValue)");

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) != DIA_SUCCESS ) return DIA_E_INVALID_KEY;
   if ( (propData.mPropSize < sizeof(tU8)) || (propData.mPropSize > sizeof(tU32)) ) return DIA_E_INVALID_LENGTH;

   propValue = static_cast<tU32>(propData.mPropExtData[0]);

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::getProperty ( tU32 propID, tU8 propValue[], size_t propLength )
{
   dia_tclFnctTrace trc("dia_PropertyBagRAM::getProperty(tU32, tU8[], size_t)");

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) != DIA_SUCCESS ) return DIA_E_INVALID_KEY;
   if ( propData.mPropSize != propLength ) return DIA_E_INVALID_LENGTH;

   ::memset(propValue,0,propLength);

   switch ( propData.mPropType )
   {
   case DIA_PROP_TYPE_STR:
      if (propData.mPropExtData[1]) {
         ::memcpy(propValue,(tU8*) propData.mPropExtData[1],propData.mPropSize);
      } else if (propData.mPropExtData[0]) {
         ::memcpy(propValue,(tU8*) propData.mPropExtData[0],propData.mPropSize);
      } else {
         return DIA_E_INVALID_POINTER;
      }
      break;
   case DIA_PROP_TYPE_U8:
   case DIA_PROP_TYPE_U16:
   case DIA_PROP_TYPE_U32:
      {
         size_t propIntegerValue = propData.mPropExtData[0];
         for ( tU16 i=0; i < propData.mPropSize; i++ )
         {
            propValue[i] = (tU8) propIntegerValue;
            propIntegerValue = propIntegerValue >> 8;
         }
      }
      break;
   default:
      return DIA_E_INVALID_PROP_TYPE;
   }

   if ( propData.mStatusMask == SM_NO_STATUS_MASK ) return DIA_SUCCESS;

   tU16 statusMask = propData.mStatusMask;
   //---------------------------------------------
   // ADD NEW STATUSMASKS WITH NEW IF-STATEMENTS
   //---------------------------------------------
   if ( statusMask & SM_ZERO_PADDED )
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - statusMask is zero_padded. (-> Nothing done.)");
      //   At the moment nothing more
   }
   if ( statusMask & SM_ZERO_TERMINATED )
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - statusMask is zero_terminated.");
      propValue[propLength] = '\0';
   }

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::getProperty ( tU32 propID, tU8 propValue[], size_t* propLength )
{
   dia_tclFnctTrace trc("dia_PropertyBagRAM::getProperty(tU32, tU8[], size_t*)");

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) != DIA_SUCCESS ) return DIA_E_INVALID_KEY;
   if ( (!propLength) || (!(*propLength)) || ((*propLength) < propData.mPropSize) ) return DIA_E_INVALID_LENGTH;

   ::memset(propValue,0,*propLength);

   switch ( propData.mPropType )
   {
   case DIA_PROP_TYPE_STR:
      {
         if (propData.mPropExtData[1]) {
            ::memcpy(propValue,(tU8*) propData.mPropExtData[1],propData.mPropSize);
            *propLength = propData.mPropSize;
         } else if (propData.mPropExtData[0]) {
            ::memcpy(propValue,(tU8*) propData.mPropExtData[0],propData.mPropSize);
            *propLength = propData.mPropSize;
         } else {
            return DIA_E_INVALID_POINTER;
         }
      break;

      }
      break;
   case DIA_PROP_TYPE_U8:
   case DIA_PROP_TYPE_U16:
   case DIA_PROP_TYPE_U32:
      {
         tU32 propIntegerValue = (tU32)propData.mPropExtData[0];
         for ( tU16 i=0; i < propData.mPropSize; i++ )
         {
            propValue[i] = (tU8) propIntegerValue;
            propIntegerValue = propIntegerValue >> 8;
         }
         *propLength = propData.mPropSize;
      }
      break;
   default:
      *propLength = 0;
      return DIA_E_INVALID_PROP_TYPE;
   }

   if ( propData.mStatusMask == SM_NO_STATUS_MASK ) return DIA_SUCCESS;

   tU16 statusMask = propData.mStatusMask;
   //---------------------------------------------
   // ADD NEW STATUSMASKS WITH NEW IF-STATEMENTS
   //---------------------------------------------
   if ( statusMask & SM_ZERO_PADDED )
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - statusMask is zero_padded. (-> Nothing done.)");
      //   At the moment nothing more
   }
   if ( statusMask & SM_ZERO_TERMINATED )
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - statusMask is zero_terminated.");
      propValue[*propLength] = '\0';
      *propLength = tU16(*propLength + 1);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::getProperty ( tU32 propID, std::vector<tU8>& propValue )
{
   dia_tclFnctTrace trc("dia_PropertyBagRAM::getProperty(tU32,vector<tU8>&)");

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) != DIA_SUCCESS ) return DIA_E_INVALID_KEY;

   propValue.resize(propData.mPropSize);
   propValue.reserve(propData.mPropSize);
   propValue.resize(propData.mPropSize);
   propValue.clear();

   switch ( propData.mPropType )
   {
   case DIA_PROP_TYPE_STR:
      {
         tU8* data = (tU8*) propData.mPropExtData[1];
         if (!data) {
            data = (tU8*) propData.mPropExtData[0];
         }
         if (NULL==data) return DIA_E_INVALID_POINTER;
         for ( tU16 i=0; i<propData.mPropSize; i++ )
         {
            propValue.push_back(data[i]);
         }
      }
      break;
   case DIA_PROP_TYPE_U8:
   case DIA_PROP_TYPE_U16:
   case DIA_PROP_TYPE_U32:
      {
         tU32 propIntegerValue = (tU32)propData.mPropExtData[0];
         for ( tU16 i=0; i < propData.mPropSize; i++ )
         {
            propValue.push_back((tU8) propIntegerValue);
            propIntegerValue = propIntegerValue >> 8;
         }
      }
      break;
   default:
      return DIA_E_INVALID_PROP_TYPE;
   }

   if ( propData.mStatusMask == SM_NO_STATUS_MASK ) return DIA_SUCCESS;

   tU16 statusMask = propData.mStatusMask;
   //---------------------------------------------
   // ADD NEW STATUSMASKS WITH NEW IF-STATEMENTS
   //---------------------------------------------
   if ( statusMask & SM_ZERO_PADDED )
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - statusMask is zero_padded. (-> Nothing done.)");
      //   At the moment nothing more
   }
   if ( statusMask & SM_ZERO_TERMINATED )
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - statusMask is zero_terminated.");
      propValue.push_back('\0');
   }

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::setPropertyU8 ( tU32 propID, tU8 propValue )
{
   dia_tclFnctTrace trc("dia_PropertyBagRAM::setPropertyU8(byte)");

   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) == DIA_SUCCESS )
   {
      if ( (DIA_PROP_TYPE_U8==propData.mPropType) || (DIA_PROP_TYPE_BOOL==propData.mPropType))
      {
         if ( propData.mPropSize == sizeof(tU8) )
         {
            propData.mPropExtData[0] = propValue;
            retCode = updateProperty(propData);
         }
         else
         {
            return DIA_E_INVALID_LENGTH;
         }
      }
      else
      {
         return DIA_E_INVALID_PROP_TYPE;
      }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::setPropertyU16 ( tU32 propID, tU16 propValue )
{
   dia_tclFnctTrace trc("dia_PropertyBagRAM::setPropertyU16(short)");

   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) == DIA_SUCCESS )
   {
      if (DIA_PROP_TYPE_U16==propData.mPropType)
      {
         if ( propData.mPropSize == sizeof(tU16) )
         {
            propData.mPropExtData[0] = propValue;
            retCode = updateProperty(propData);
         }
         else
         {
            return DIA_E_INVALID_LENGTH;
         }
      }
      else
      {
         return DIA_E_INVALID_PROP_TYPE;
      }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::setPropertyU32 ( tU32 propID, tU32 propValue )
{
   dia_tclFnctTrace trc("dia_PropertyBagRAM::setPropertyU32(int)");

   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) == DIA_SUCCESS )
   {
      if (DIA_PROP_TYPE_U32==propData.mPropType)
      {
         if ( propData.mPropSize == sizeof(tU32) )
         {
            propData.mPropExtData[0] = propValue;
            retCode = updateProperty(propData);
         }
         else
         {
            return DIA_E_INVALID_LENGTH;
         }
      }
      else
      {
         return DIA_E_INVALID_PROP_TYPE;
      }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagRAM::setProperty ( tU32 propID, tU8 propValue[], size_t propLength )
{
    dia_tclFnctTrace trc("dia_PropertyBagRAM::setProperty tU32 tU8[] size_t ");

    tDiaResult retCode = DIA_FAILED;

    if ( 0==propLength ) return DIA_E_INVALID_LENGTH;
    if (NULL==(tU8*)propValue) return DIA_E_INVALID_POINTER;

    dia_PropertyInfo propData;
    if ( queryProperty(propID,propData) != DIA_SUCCESS ) return DIA_E_INVALID_KEY;

    if (DIA_PROP_TYPE_STR!=propData.mPropType)
    {
       DIA_TR_ERR("### dia_PropertyBagRAM::setProperty The method is wrong for the property type %d. ####", propData.mPropType);
       return DIA_E_INVALID_PROP_TYPE;
    }

    DIA_TR_INF("dia_PropertyBagRAM::setProperty propData.mPropSize %zu. propLength= -> %zu", propData.mPropSize, propLength);
    if (propData.mPropExtData[1]) {
       free((void *)propData.mPropExtData[1]);
    }
    
    tU8*data=nullptr;
    if  (propData.mStatusMask & SM_ZERO_TERMINATED) {
       data=new tU8[propLength + 1];
       data[propLength]=0;

    } else {
       data=new tU8[propLength];
    }
    propData.mPropExtData[1]=(intptr_t) data;
    memcpy((void *)propData.mPropExtData[1], propValue, propLength);
    propData.mPropSize = propLength;
    retCode = updateProperty(propData);

    return retCode;
}
