#ifndef __INCLUDED_DIA_EXTERN_OSAL_SYSTEM_PIF__
#include "common/framework/platform/osal/dia_system_pif.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_CONFIG__
#include "common/framework/config/dia_common_config.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_DRIVER__
#include "common/framework/platform/dia_IDriver.h"
#endif

#ifndef __INCLUDED_DIA_OSAL_DRIVER_KDS__
#include "common/framework/platform/osal/dia_OSALDriverKDS.h"
#endif

#ifndef __INCLUDED_DIA_NATIVE_KDS_LISTENER_INTERFACE__
#include "dia_IKDSListener.h"
#endif

//----------------------------------------------------------------------------------------

dia_PropertyBagKDS::dia_PropertyBagKDS ( void )
   : dia_PropertyBag(DIA_C_STR_PROPBAG_KDS),
     mpDriverKDS(0),
     mIsLengthCheckEnabled(true)
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::dia_PropertyBagKDS");
}

//----------------------------------------------------------------------------------------

dia_PropertyBagKDS::dia_PropertyBagKDS ( dia_IDriver* pDriverKDS )
   : dia_PropertyBag(DIA_C_STR_PROPBAG_KDS),
     mpDriverKDS(pDriverKDS),
     mIsLengthCheckEnabled(true)
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::dia_PropertyBagKDS");
}

//----------------------------------------------------------------------------------------

dia_PropertyBagKDS::~dia_PropertyBagKDS ( void )
{
   mpDriverKDS = 0;
}

//----------------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::open ( void )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::open");
   return ( mpDriverKDS ) ? mpDriverKDS->open() : DIA_E_OPEN_FAILED_KDS;
}

//----------------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::close ( void )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::close");
   return ( mpDriverKDS ) ? mpDriverKDS->close() : DIA_E_DRIVER_CLOSE_FAILED;
}

//----------------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::getPropertyU8(tU32 propID, tU8& propValue)
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::getPropertyU8(tU32 propID, tU8& propValue)");

   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) == DIA_SUCCESS )
   {
       if ( propData.mPropSize == sizeof(tU8) )
       {
          retCode = getProperty( propID, &propValue, (tU32)sizeof(tU8) );
       }
       else
       {
          DIA_TR_INF("dia_PropertyBagKDS::getPropertyU8 ERROR DIA_E_INVALID_KEY mPropSize=%zu", propData.mPropSize);
          retCode = DIA_E_INVALID_LENGTH;
       }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::getPropertyU16(tU32 propID, tU16& propValue)
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::getPropertyU16(tU32 propID, tU16& propValue)");

   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) == DIA_SUCCESS )
   {
       if ( propData.mPropSize == sizeof(tU8) )
       {
         tU8 value = 0;
          retCode = getProperty( propID, &value, sizeof(tU8) );
          propValue = value;
       }
       else if ( propData.mPropSize == sizeof(tU16) )
       {
          retCode = getProperty( propID, reinterpret_cast<tU8*>(&propValue), sizeof(tU16) );
       }
       else
       {
          DIA_TR_INF("dia_PropertyBagKDS::getPropertyU16 ERROR DIA_E_INVALID_KEY mPropSize=%zu", propData.mPropSize);
          retCode = DIA_E_INVALID_LENGTH;
       }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::getPropertyU32 ( tU32 propID, tU32& propValue)
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::getPropertyU32(tU32 propID, tU32& propValue)");

   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) == DIA_SUCCESS )
   {
       if ( propData.mPropSize == sizeof(tU8) )
       {
         tU8 value = 0;
          retCode = getProperty( propID, &value, sizeof(tU8) );
          propValue = value;
       }
       else if ( propData.mPropSize == sizeof(tU16) )
       {
         tU16 value = 0;
          retCode = getProperty( propID, reinterpret_cast<tU8*>(&value), sizeof(tU16) );
          propValue = value;
       }
       else if ( propData.mPropSize == sizeof(tU32) )
       {
          retCode = getProperty( propID, reinterpret_cast<tU8*>(&propValue), sizeof(tU32) );
       }
       else
       {
          DIA_TR_INF("dia_PropertyBagKDS::getPropertyU32 ERROR DIA_E_INVALID_KEY mPropSize=%zu", propData.mPropSize);
          retCode = DIA_E_INVALID_LENGTH;
       }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::getProperty ( tU32 propID, tU8 propValue[], size_t propLength )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::getProperty(tU32,tU8[],size_t) (byte stream)");

   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if (queryProperty(propID, propData) != DIA_SUCCESS)
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - DIA_E_INVALID_KEY at getProperty(propID, propValue, propLength)");
      return DIA_E_INVALID_KEY;
   }

   size_t length = propData.mPropSize;

   if ((!length) || (!propLength) || (propLength != length))
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - DIA_E_INVALID_LENGTH at getProperty(propID, propValue, propLength)");
      return DIA_E_INVALID_LENGTH;
   }

//   if (propLength < length)
//   {
//      DIA_TR_INF("dia_PropertyBagKDS::getProperty property is longer than it is requested. propLength=%d, length=%d", propLength, length);
//      length = propLength;
//   }

   size_t idx = 0;
   size_t offset = 0;
   size_t datalen = 0;

   do
   {
      // KDS key where the data is stored
      tU16 key = static_cast<tU16>(propData.mPropExtData[idx]);
      // length of the data stored in the specific KDS item
      datalen  = (int)propData.mPropExtData[idx + 1];
      auto len = (datalen) ? datalen : length - offset;

      dia_OSALDriverKDS::sDataKDS dataBuffer(key, &propValue[offset], (tU16)len);

      DIA_TR_INF("Reading KDS: Key=0x%0x, len=%d", dataBuffer.mKey, dataBuffer.mLen);
      if ( mpDriverKDS && (mpDriverKDS->read((tU8*) &dataBuffer, sizeof(dataBuffer)) >= DIA_ERR_S32_DRIVER_SUCCESS) ) {
         retCode = DIA_SUCCESS;
      }

      offset += len;
      idx += 2;
   }
   while ((retCode == DIA_SUCCESS) && (datalen != 0));

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::getProperty ( tU32 propID, tU8 propValue[], size_t* maxLength )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::getProperty(tU32,tU8[],size_t*) (byte stream)");

   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;

   if (queryProperty(propID, propData) != DIA_SUCCESS)
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty(tU32,tU8[],size_t*) ERROR: DIA_E_INVALID_KEY");
      return DIA_E_INVALID_KEY;
   }

   if ( (!maxLength) || (!(*maxLength)) || ((*maxLength) < propData.mPropSize) )
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - DIA_E_INVALID_LENGTH at getProperty(tU32,tU8[],size_t*)");
      return DIA_E_INVALID_LENGTH;
   }

   if((SM_ZERO_PADDED == propData.mStatusMask) && (DIA_PROP_TYPE_STR != propData.mPropType))
   {
     DIA_TR_INF("dia_PropertyBagKDS::getProperty zero padded mask has set for property type %d not equal to string", propData.mPropType);
     DIA_TR_INF("dia_PropertyBagKDS::getProperty(tU32,tU8[],tU16*) ERROR: DIA_E_INVALID_STATUS_MASK");
     return DIA_E_INVALID_STATUS_MASK;
   }


   size_t idx = 0;
   size_t offset = 0;
   size_t datalen = 0;

   do
   {
      // KDS key where the data is stored
      tU16 key = static_cast<tU16>(propData.mPropExtData[idx]);
      // length of the data stored in the specific KDS item
      datalen = (int)propData.mPropExtData[idx + 1];
      auto len = (datalen) ? datalen : (propData.mPropSize - offset);

      //handling for zero terminated to get rid of retrieving crc
      if(SM_ZERO_TERMINATED == propData.mStatusMask )
      {
         mpDriverKDS->setCtrlSettings(DIA_C_U32_CTRL_SETTINGS_VARLEN_ENABLE,DIA_BOOL_CTRL_SETTINGS_MODE_VOLATILE);
      }

      dia_OSALDriverKDS::sDataKDS dataBuffer(key, &propValue[offset], (tU16)len);
      DIA_TR_INF("Reading KDS: Key=0x%0x, len=%d", dataBuffer.mKey, dataBuffer.mLen);
      if ( mpDriverKDS && (mpDriverKDS->read((tU8*) &dataBuffer, sizeof(dataBuffer)) >= DIA_ERR_S32_DRIVER_SUCCESS) ) {
         retCode = DIA_SUCCESS;
      }

      offset += len;
      idx += 2;
   }
   while ((retCode == DIA_SUCCESS) && (datalen != 0));

   //---------------------------------------------
   // ADD NEW STATUSMASKS WITH NEW IF-STATEMENTS
   //---------------------------------------------
   if (propData.mStatusMask == SM_NO_STATUS_MASK)
   {
      *maxLength = propData.mPropSize;
   }
   else if (propData.mStatusMask != SM_NO_STATUS_MASK)
   {
      tU16 statusMask = propData.mStatusMask;
      dia_PropertyBagKDS::handleStatMask4Read(propID, propValue, statusMask, *maxLength);

      DIA_TR_INF("dia_PropertyBagKDS::getProperty length of data byte stream  %zu", *maxLength);
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::getProperty ( tU32 propID, std::vector<tU8>& propValue )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::getProperty(tU32,std::vector<tU8>&)");

   propValue.clear();

   dia_PropertyInfo propData;
   if (queryProperty(propID, propData) != DIA_SUCCESS)
   {
      DIA_TR_INF("dia_PropertyBagKDS::getProperty - DIA_E_INVALID_KEY at getProperty(propID, propValue, propLength)");
      return DIA_E_INVALID_KEY;
   }

   size_t length = propData.mPropSize;

   propValue.reserve( length );
   propValue.resize( length );

   tDiaResult retCode = getProperty (  propID, &propValue[0], &length );

//   propValue.resize( length );
   if(length < propValue.size())
   {
      while(length != propValue.size())
      {
        propValue.pop_back();
      }
   }

   DIA_TR_INF("dia_PropertyBagKDS::getProperty - propValue size after resize propValue.size() %zu, length %zu", propValue.size(), length);

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::setProperty ( tU32 propID, tU8 propValue[], size_t propLength )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::setProperty(tU32, tU8[], size_t)");

   tDiaResult retCode = DIA_FAILED;
   tBool statusMask_known = FALSE;

   if ( !mpDriverKDS ) return DIA_FAILED;

   // store the active flush control settings. we will disable flushing for the write operation and will flush afterwards if required
   tU16 flushCtrlSettings = mpDriverKDS->getFlushCtrl();

   dia_PropertyInfo propData;
   if (queryProperty(propID, propData) != DIA_SUCCESS)
   {
      return DIA_E_INVALID_KEY;
   }

   if((SM_ZERO_PADDED == propData.mStatusMask) && (DIA_PROP_TYPE_STR != propData.mPropType))
   {
     DIA_TR_INF("dia_PropertyBagKDS::setProperty zero padded mask has set for property type %d not equal to string", propData.mPropType);
     DIA_TR_INF("dia_PropertyBagKDS::setProperty(tU32,tU8[],tU16) ERROR: DIA_E_INVALID_STATUS_MASK");
     return DIA_E_INVALID_STATUS_MASK;
   }

   size_t length = propData.mPropSize;

   DIA_TR_INF("mPropKey=0x%08X mStatusMask=0x%04X mPropType=%u mPropSize=%zu mPropLoc=%u", propData.mPropKey, propData.mStatusMask, propData.mPropType, propData.mPropSize, propData.mPropLoc);

   tU8 propValueApplied[DIA_PROP_LENGTH_MAX] = { 0 };
   ::memcpy(propValueApplied, propValue, propLength);

   tU16 checkStatusMask = propData.mStatusMask & (static_cast<tU16>(SM_ZERO_PADDED)|static_cast<tU16>(SM_ZERO_TERMINATED));

   if (checkStatusMask == static_cast<tU16>(SM_NO_STATUS_MASK))
   {
      if ( mIsLengthCheckEnabled && (propLength != length) )
      {
         DIA_TR_ERR("dia_PropertyBagKDS::setProperty ERROR: PASSED LENGTH (%zu) != CONFIGURED LENGTH(%zu).", propLength, length);
         return DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
      }

      statusMask_known = TRUE;
   }
   else // contains SM_ZERO_PADDED | SM_ZERO_TERMINATED
   {
      DIA_TR_INF("dia_PropertyBagKDS::setProperty - statusMask, propLength %zu, length %zu", propLength, length);

      // Handle different bit statusmasks
      //---------------------------------------------
      // ADD NEW STATUSMASKS WITH NEW IF-STATEMENTS
      //---------------------------------------------
      if (checkStatusMask & static_cast<tU16>(SM_ZERO_PADDED))
      {
         // Check if payload==MAX, if not fill 0x00 until max payload
         DIA_TR_INF("dia_PropertyBagKDS::setProperty - zero_padded: (vor memset) propLength %zu, length %zu", propLength, length);

         if (length < propLength)
         {
            return DIA_FAILED;
         }

         (void) ::memset(&propValueApplied[propLength], ((tU8) 0), (length - propLength));
         propLength = length;

         statusMask_known = TRUE;
         DIA_TR_INF("dia_PropertyBagKDS::setProperty - SM zero_padded: propLength %zu, length %zu", propLength, length);
      }

      if (checkStatusMask & static_cast<tU16>(SM_ZERO_TERMINATED))
      {
         // set first value after the payload to zero
         if (length < propLength )
         {
            return DIA_FAILED;
         }

         propValueApplied[propLength+1] = 0x00;
         propLength++;

         statusMask_known = TRUE;
         DIA_TR_INF("dia_PropertyBagKDS::setProperty - SM zero_terminated: propLength %zu, length %zu", propLength, length);
      }

   }

   // flag to indicate if flushing was disabled by the configuration of this property. we assume that flushing is enabled by default
   bool flushActivated = true;

   if (propData.mStatusMask & static_cast<tU16>(SM_DISABLE_FLUSH))
   {
      // flushing is disabled by status mask
      flushActivated = false;
      statusMask_known = TRUE;
      DIA_TR_INF("dia_PropertyBagKDS::setProperty - SM_DISABLE_FLUSH enabled in status mask: propLength %zu, length %zu", propLength, length);
   }

   if (statusMask_known == FALSE)
   {
      DIA_TR_INF("dia_PropertyBagKDS::setProperty unknown statusMask 0x%04x used", propData.mStatusMask);
      return DIA_E_INVALID_STATUS_MASK;
   }

   size_t idx = 0;
   size_t offset = 0;
   size_t datalen = 0;

   // set DISABLE flag here, because dia_OSALDriverKDS::write could flush.
   mpDriverKDS->setFlushCtrlBits(DIA_C_U16_FLUSH_CTRL_DISABLE);

   std::list<dia_OSALDriverKDS::sDataKDS> kdsCollector;
   do
   {
      datalen = static_cast<tU16>(propData.mPropExtData[idx + 1]);
      tU16 key = static_cast<tU16>(propData.mPropExtData[idx]);
      auto len = (datalen) ? datalen : (propLength /*length*/- offset);

      dia_OSALDriverKDS::sDataKDS dataBuffer(key, &propValueApplied[offset], (tU16)len);
      DIA_TR_INF("Writing KDS: Key = 0x%0x, len = %d, addr = %p", dataBuffer.mKey, dataBuffer.mLen,&dataBuffer);
      retCode = (mpDriverKDS->write((tU8*) &dataBuffer, (tU16) sizeof(dataBuffer)) >= DIA_ERR_S32_DRIVER_SUCCESS) ? DIA_SUCCESS : DIA_FAILED;
      if ( retCode == DIA_SUCCESS )
      {
         DIA_TR_INF("Adding KDS key 0x%04x to notification list ...", dataBuffer.mKey);
         kdsCollector.push_back(dataBuffer);
      }

      offset += len;
      idx += 2;
   }
   while ((retCode == DIA_SUCCESS) && datalen != 0);

   // reset original flush control settings
   (void) mpDriverKDS->resetFlushCtrlBits((tU16)~flushCtrlSettings);

   if ( retCode == DIA_SUCCESS )
   {
      // check if flushing must be performed or not
      if ( flushActivated && ((flushCtrlSettings & DIA_C_U16_FLUSH_CTRL_DISABLE) != DIA_C_U16_FLUSH_CTRL_DISABLE) )
      {
         // here we know that flushing is neither disabled by master nor via the status mask of the property
         DIA_TR_INF("dia_PropertyBagKDS::setProperty - Flush is enabled.");
         retCode = mpDriverKDS->flush();
         if ( DIA_SUCCESS != retCode )
         {
            DIA_TR_ERR("### dia_PropertyBagKDS::setProperty - flush FAILED with errcode = 0x%08x ###", retCode);
         }
      }
      else if ( flushActivated && ((flushCtrlSettings & DIA_C_U16_FLUSH_CTRL_DISABLE) == DIA_C_U16_FLUSH_CTRL_DISABLE) )
      {
         // here we know that flushing is disabled by master but not disabled in the status mask of the property
         DIA_TR_INF("dia_PropertyBagKDS::setProperty - Flush is disabled (MASTER).");
         markDirty(propData.mPropKey);
         retCode = DIA_SUCCESS;
      }
      else
      {
         // here we know that flushing is disabled in the status mask
         DIA_TR_INF("dia_PropertyBagKDS::setProperty - Flush is disabled (STATUSMASK).");
         markDirty(propData.mPropKey);
         retCode = DIA_SUCCESS;
      }
   }

   // now notify updates to registered listeners
   if ( kdsCollector.size() )
   {
      DIA_TR_INF("KDS keys found to be notified.");
      std::list<dia_OSALDriverKDS::sDataKDS>::iterator iter = kdsCollector.begin();
      for ( ; iter != kdsCollector.end(); iter++ )
      {
         dia_OSALDriverKDS::sDataKDS& kdsData = (*iter);
         DIA_TR_INF("Notify about update of KDS key 0x%04x ...", kdsData.mKey);
         std::map<tU16,dia_IKDSListener*>::iterator keyIter = mKeyListenerRep.find(kdsData.mKey);
         if ( keyIter != mKeyListenerRep.end() && keyIter->second )
         {
            dia_IKDSListener* pListener = keyIter->second;
            DIA_TR_INF("### NOTIFY KDS LISTENER 0x%p ABOUT UPDATE OF KDS KEY 0x%04x",pListener,kdsData.mKey);
            pListener->vOnKDSUpdate(kdsData.mKey,kdsData.mpData,kdsData.mLen);
         }
         else
         {
            DIA_TR_INF("No listener found for KDS key 0x%04x ...", kdsData.mKey);
         }
      }
   }
   else
   {
      DIA_TR_INF("No KDS keys in notification list.");
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult dia_PropertyBagKDS::setPropertyU32 ( tU32 propID, tU32 propValue)
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::setPropertyU32(tU32 propID, tU32 propValue)");

   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if ( queryProperty(propID,propData) == DIA_SUCCESS )
   {
       if ( propData.mPropSize == sizeof(tU32) )
       {
          tU32 dataToWrite = propValue;
          retCode = setProperty( propID, reinterpret_cast<tU8*>(&dataToWrite), sizeof(tU32) );
       }
       else
       {
          DIA_TR_INF("dia_PropertyBagKDS::setPropertyU32 ERROR DIA_E_INVALID_KEY mPropSize=%zu", propData.mPropSize);
          retCode = DIA_E_INVALID_LENGTH;
       }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------
// StausMask handling takes place before the write access.
tDiaResult dia_PropertyBagKDS::handleStatMask4Write ( tU16 key, tU8* data, tU16 statMask, size_t &length ) const
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::handleStatMask4Write(tU16 key, tU8* data[], tU16 statMask, size_t &length)");
   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if (queryProperty(key, propData) != DIA_SUCCESS)
   {
      return DIA_E_INVALID_KEY;
   }
   if (length > propData.mPropSize)
   {
      return DIA_E_INVALID_LENGTH;
   }

   DIA_TR_INF("dia_PropertyBagKDS::handleStatMask4Write - received Data:");
   DIA_TR_INF("key %u statMask %u length %zu", key, statMask, length);

   if (statMask == SM_NO_STATUS_MASK)
   {
      retCode = DIA_SUCCESS;
   }
   else
   {
      if (statMask == SM_ZERO_PADDED)
      {
         DIA_TR_INF("dia_PropertyBagKDS::handleStatMask4Write - SM_ZERO_PADDED");

         // Check if payload==MAX, if not fill 0x00 until max payload
         if (length != propData.mPropSize)
         {
            (void) ::memset(&data[length], ((tU8) 0), (propData.mPropSize - length));
            length = propData.mPropSize;
         }

         statMask = tU16(statMask - SM_ZERO_PADDED);
      }

      if (statMask & SM_ZERO_TERMINATED)
      {
         DIA_TR_INF("dia_PropertyBagKDS::handleStatMask4Write - SM_ZERO_TERMINATED");
         if (length < propData.mPropSize && data[length] != 0x00)
         {
            data[length] = 0x00;
            ++length;
         }

         statMask = tU16(statMask - SM_ZERO_TERMINATED);
      }

      // check if all statusmask bits are handled and set return value
      if (statMask == SM_NO_STATUS_MASK)
      {
         retCode = DIA_SUCCESS;
      }
      else
      {
         retCode = DIA_FAILED;
      }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------
// StausMask handling takes place after the read access.
tDiaResult dia_PropertyBagKDS::handleStatMask4Read(tU32 propID, const tU8* data, tU16 statMask, size_t &length) const
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::handleStatMask4Read(tU32 propID, tU8* data[], tU16 statMask, size_t &length)");
   tDiaResult retCode = DIA_FAILED;

   dia_PropertyInfo propData;
   if (queryProperty(propID, propData) != DIA_SUCCESS)
   {
      return DIA_E_INVALID_KEY;
   }

   DIA_TR_INF("dia_PropertyBagKDS::handleStatMask4Read Received Data: propID %d statMask %d length %zu", propID, statMask, length);

   //---------------------------------------------
   // ADD NEW STATUSMASKS WITH NEW IF-STATEMENTS
   //---------------------------------------------
   if (statMask == SM_NO_STATUS_MASK)
   {
      retCode = DIA_SUCCESS;
   }
   else
   {
      int mask_zero_padded_set = statMask & SM_ZERO_PADDED;

      DIA_TR_INF("dia_PropertyBagKDS::handleStatMask4Read bit_mask 0x04%x masked_received 0x04%x", SM_ZERO_PADDED, mask_zero_padded_set);

      /*==========================*/
      /*=== SM_ZERO_PADDED     ===*/
      /*==========================*/
      if (mask_zero_padded_set == SM_ZERO_PADDED)
      {
         statMask = tU16(statMask - SM_ZERO_PADDED);

         DIA_TR_INF("dia_PropertyBagKDS::handleStatMask4Read propID %d statMask, search for zero padded byte", propID);

         for(tU16 data_iterator = 0; data_iterator < length; data_iterator++)
         {
            if(0 == data[data_iterator])
            {
               length = data_iterator;
               DIA_TR_INF("dia_PropertyBagKDS::handleStatMask4Read length of data byte stream without zero padded %zu .", length);
            }
         }
      }

      /*==========================*/
      /*=== SM_ZERO_TERMINATED ===*/
      /*==========================*/
      int mask_zero_terminated = statMask & SM_ZERO_TERMINATED;

      if (mask_zero_terminated & SM_ZERO_TERMINATED)
      {
         DIA_TR_INF("dia_PropertyBagKDS::handleStatMask4Read - SM_ZERO_TERMINATED");
         // TODO - Not tested !
         if (data[length - 1] == 0x00) {
            for (; data[length - 2] == 0x00 && length > 1; length--) { // "&& length>1" is needed to left at least one 0-Byte in a only 0-Byte String
            }
         }
         statMask = tU16(statMask - SM_ZERO_TERMINATED);
      }

      /*==========================*/
      /*=== SM_NO_STATUS_MASK  ===*/
      /*==========================*/
      // check if all statusmask bits are handled and set return value
      retCode = (statMask == SM_NO_STATUS_MASK) ? DIA_SUCCESS : DIA_FAILED;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagKDS::writeKDS ( tU16 key, tU8 data[], tU16 length )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::writeKDS(tU16, tU8[], tU16)");

   tDiaResult retCode = DIA_FAILED;

   dia_OSALDriverKDS::sDataKDS dataBuffer(key, data, length);
   if ( mpDriverKDS && (mpDriverKDS->write((tU8*) &dataBuffer, sizeof(dataBuffer)) >= DIA_ERR_S32_DRIVER_SUCCESS) ) {
      retCode = DIA_SUCCESS;
   }

   if(DIA_SUCCESS == retCode)
   {
      std::map<tU16,dia_IKDSListener*>::iterator keyIter = mKeyListenerRep.find(key);
      if ( keyIter != mKeyListenerRep.end() && keyIter->second )
      {
         dia_IKDSListener* pListener = keyIter->second;
         DIA_TR_INF("### NOTIFY KDS LISTENER 0x%p ABOUT UPDATE OF KDS KEY 0x%04x",pListener,key);
         pListener->vOnKDSUpdate(key,data,length);
      }
      else
      {
         DIA_TR_INF("No listener found for KDS key 0x%04x ...", key);
      }
   }
   else
   {
      DIA_TR_INF("Write to KDS failed ...");
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagKDS::writeKDS ( dia_OSALDriverKDS::sDataKDS& kdsData )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::writeKDS(dia_OSALDriverKDS::sDataKDS&)");

   tDiaResult retCode = DIA_FAILED;

   if ( mpDriverKDS && (mpDriverKDS->write((tU8*) &kdsData, sizeof(kdsData)) >= DIA_ERR_S32_DRIVER_SUCCESS) )
   {
      retCode = DIA_SUCCESS;
   }

   if ( DIA_SUCCESS == retCode )
   {
      std::map<tU16,dia_IKDSListener*>::iterator keyIter = mKeyListenerRep.find(kdsData.mKey);
      if ( keyIter != mKeyListenerRep.end() && keyIter->second )
      {
         dia_IKDSListener* pListener = keyIter->second;
         DIA_TR_INF("### NOTIFY KDS LISTENER 0x%p ABOUT UPDATE OF KDS KEY 0x%04x",pListener,kdsData.mKey);
         pListener->vOnKDSUpdate(kdsData.mKey,kdsData.mpData,kdsData.mLen);
      }
      else
      {
         DIA_TR_INF("No listener found for KDS key 0x%04x ...", kdsData.mKey);
      }
   }
   else
   {
      DIA_TR_INF("Write to KDS failed ...");
   }

   return retCode;
}

tDiaResult
dia_PropertyBagKDS::readKDS ( tU16 key, tU8 data[], tU16 length )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::readKDS(tU16, tU8[], tU16)");
   if(mpDriverKDS == 0)
   {
      return DIA_FAILED;
   }
   dia_OSALDriverKDS::sDataKDS dataBuffer(key, data, length);

   tDiaResult retCode = DIA_SUCCESS;
   tS32 res = mpDriverKDS->read((tU8*) &dataBuffer, sizeof(dataBuffer));
   if(res == DIA_ERR_S32_DRIVER_INVALID_LENGTH)
   {
      retCode = DIA_E_INVALID_LENGTH;
   }
   else if(res < DIA_ERR_S32_DRIVER_SUCCESS)
   {
      retCode = DIA_FAILED;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_PropertyBagKDS::flushKDS ( void )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::flushKDS()");

   if ( !mpDriverKDS )
   {
      DIA_TR_ERR("dia_PropertyBagKDS::flushKDS mpDriverKDS is NULL");
      return DIA_FAILED;
   }

   tDiaResult retCode = DIA_SUCCESS;

   if ( mpDriverKDS->resetFlushCtrlBits(DIA_C_U16_FLUSH_CTRL_DISABLE_MASTER) != DIA_SUCCESS ) retCode = DIA_FAILED;
   if ( mpDriverKDS->resetFlushCtrlBits(DIA_C_U16_FLUSH_CTRL_DISABLE) != DIA_SUCCESS ) retCode = DIA_FAILED;

   DIA_TR_INF("flushCtrlMask = 0x%04x", mpDriverKDS->getFlushCtrl());

   if ( retCode == DIA_SUCCESS )
   {
      retCode = mpDriverKDS->flush();

      if ( retCode == DIA_SUCCESS )
      {
         DIA_TR_INF("KDS MASTER FLUSH SUCCEEDED");
      }
      else
      {
         DIA_TR_ERR("KDS MASTER FLUSH FAILED (ERR = 0x%08X)", retCode);
      }
   }
   else
   {
      DIA_TR_ERR("KDS MASTER FLUSH FAILED TO SET FLUSH CTRL BITS (ERR = 0x%08X)", retCode);
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_PropertyBagKDS::update ( void )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::update()");

   std::list<dia_PropertyInfo*> propsKDS;
   if ( DIA_SUCCESS != queryProperties(DIA_PROP_SRC_KDS,propsKDS) ) return DIA_FAILED;

   std::list<dia_PropertyInfo*>::iterator iter = propsKDS.begin();
   for ( ; iter != propsKDS.end(); iter++ )
   {
      dia_PropertyInfo* pPropInfo = (*iter);

      for ( int i=0, j=1; j<DIA_PROP_EXT_DATA_ELEM_MAX; i+=2,j+=2 )
      {
         tU16 kdsKey  = (tU16) pPropInfo->mPropExtData[i];
         if ( kdsKey == 0 ) break;
         std::map<tU16,dia_PropertyInfo*>::iterator keyIter = mKeyRep.find(kdsKey);
         if ( keyIter != mKeyRep.end() ) break; // already exists
         mKeyRep[kdsKey] = pPropInfo;
      }
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_PropertyBagKDS::setFlushMode ( enFlushMode mode )
{
   if ( !mpDriverKDS ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   if ( mode == dia_IKDS::DIA_EN_KDS_FLUSH_DISABLE )
   {
      retCode = mpDriverKDS->setFlushCtrlBits(DIA_C_U16_FLUSH_CTRL_DISABLE);
   }
   else
   {
      retCode = mpDriverKDS->resetFlushCtrlBits(DIA_C_U16_FLUSH_CTRL_DISABLE);
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tU16
dia_PropertyBagKDS::getCardinality ( void ) const
{
   return (tU16) mKeyRep.size();
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagKDS::addKDSListener ( tU16 kdsKey, dia_IKDSListener* pListener )
{
   if ( (!pListener) || (!kdsKey) ) return DIA_FAILED;
   std::map<tU16,dia_IKDSListener*>::iterator iter = mKeyListenerRep.find(kdsKey);
   if ( iter != mKeyListenerRep.end() ) return DIA_E_KEY_ALREADY_EXISTS;
   mKeyListenerRep[kdsKey] = pListener;
   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagKDS::removeKDSListener ( tU16 kdsKey, dia_IKDSListener* pListener )
{
   if ( (!pListener) || (!kdsKey) ) return DIA_FAILED;

   std::map<tU16,dia_IKDSListener*>::iterator iter = mKeyListenerRep.find(kdsKey);
   if ( iter != mKeyListenerRep.end() ) mKeyListenerRep.erase(kdsKey);

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagKDS::flush ( void )
{
   dia_tclFnctTrace trc("dia_PropertyBagKDS::flush()");

   if ( !mpDriverKDS )
   {
      DIA_TR_ERR("dia_PropertyBagKDS::flush mpDriverKDS is NULL");
      return DIA_FAILED;
   }

   //reset DISABLE flag and execute flush
   (void)mpDriverKDS->resetFlushCtrlBits(DIA_C_U16_FLUSH_CTRL_DISABLE);
   tDiaResult retCode = mpDriverKDS->flush();

   if ( DIA_SUCCESS != retCode )
   {
      DIA_TR_ERR("dia_PropertyBagKDS::flush FAILED, error code = 0x%08X", retCode);
   }
   else
   {
      retCode = dia_PropertyBag::flush();
   }

   return retCode;
}
