/*!
 * \file       dia_PropertyBagEngineClient.cpp
 *
 * \brief      This file is used to define common properties to communicate with external ECU (e.g. BCM)
 *
 * \details    Get and set methods are not thread safety functions because of callback usage.
 *             The instance of the class must be used by one thread.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreConfig
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

/*
 *  Created on: 04.04.2017
 *      Author: kaa1hi
 */

#ifndef __INCLUDED_DIA_PROPERTY_ENGINE_CLIENT__
#include "dia_PropertyBagEngineClient.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include <common/framework/engine/dia_EngineManager.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_CLIENT__
#include <common/framework/engine/dia_EngineClient.h>
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#include <string>

#define DEFAULT_DID           ((tU16)0)
#define DIA_MAX_BIT_LENGTH    ((tU32)8)

//----------------------------------------------------------------------------------------

dia_PropertyBagEngineClient::dia_PropertyBagEngineClient ( /*tU32 ecuUID*/ )
    : dia_PropertyBag( (tCString) (((std::string)DIA_C_STR_PROPBAG_ENGINE_CLIENT) /*+ dia::utils::byte2hexstr((tU8)ecuUID)*/ ).c_str())
{
    dia_tclFnctTrace trc("dia_PropertyBagEngineClient::dia_PropertyBagEngineClient");
}

//----------------------------------------------------------------------------------------

dia_PropertyBagEngineClient::~dia_PropertyBagEngineClient ( void )
{
//    dia_tclFnctTrace trc("dia_PropertyBagEngineClient::~dia_PropertyBagEngineClient");
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagEngineClient::getProperty ( tU32 propID, std::vector<tU8>& /*propValue*/ )
{
   dia_tclFnctTrace trc("dia_PropertyBagEngineClient::getPropertyU8(tU32 propID, std::vector<tU8>)");
   DIA_TR_ERR("### ERROR: No synchronous read for propID=0x%x.", propID);
   return DIA_E_TEMPORARY_NOT_AVAILABLE;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagEngineClient::setProperty ( tU32 propID, tU8[] /*propValue[]*/, size_t /*propLength*/ )
{
   dia_tclFnctTrace trc("dia_PropertyBagEngineClient::setProperty(tU32 propID, tU8 propValue[], size_t propLength)");
   DIA_TR_ERR("### ERROR: No synchronous write for propID=0x%x.", propID);
   return DIA_E_TEMPORARY_NOT_AVAILABLE;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagEngineClient::getProperty ( tU32 propID, dia_IEngineClient* callback )
{
   dia_tclFnctTrace trc("dia_PropertyBagEngineClient::getPropertyU8(tU32 propID, dia_IEngineClient* callback)");

   tDiaResult result = DIA_FAILED;
   dia_PropertyInfo propData;

   if ( queryProperty(propID,propData) != DIA_SUCCESS )
   {
      DIA_TR_ERR("### ERROR: queryProperty unsuccessful.");
      return DIA_E_INVALID_KEY;
   }

   if (NULL==callback)
   {
      DIA_TR_ERR("### ERROR: callback is NULL.");
      return DIA_E_INVALID_POINTER;
   }

   dia_UID engineUID = 0;
   tCString strEngineName = (tCString)(propData.mPropExtData[0]); //static_cast<tCString>(propData.mPropExtData[0]);
   dia_enPropEngineClientKey keyType = (dia_enPropEngineClientKey) propData.mPropExtData[1];
   dia_PropertyInfo_statusMask_enum statusMask = (dia_PropertyInfo_statusMask_enum) propData.mStatusMask;

   tU16 did = 0;
   tU16* didPtr = NULL;
   tU32 numElem = 0;
   tU32 bitLen = 0;

   if (strEngineName)
   {
      engineUID = dia_getHashCodeFromString(strEngineName);
   }

   if (0==engineUID)
   {
      DIA_TR_ERR("### ERROR: Status Mask: engineUID is not set.");
      return DIA_E_INVALID_PROP_TYPE;
   }
   else
   {
      DIA_TR_INF("Status Mask: engineUID is 0x%08X.", engineUID);
   }

   if (statusMask & SM_READ_ONLY)   DIA_TR_INF("Status Mask: SM_READ_ONLY is set.");
   else                             DIA_TR_INF("Status Mask: SM_READ_ONLY is not set.");

   switch (keyType)
   {
      case DIA_PROP_SRC_ENGINE_CLIENT_KEY_SINGLE:
      {
         did = (tU16) propData.mPropExtData[2];
         bitLen = (tU32) propData.mPropExtData[3];
         DIA_TR_INF("DIA_PROP_SRC_ENGINE_CLIENT_KEY_SINGLE: did=0x%04X bitLen=%d.", did, bitLen);

         if (DEFAULT_DID==did)
         {
            DIA_TR_ERR("DIA_PROP_SRC_ENGINE_CLIENT_KEY_SINGLE: did=%d bitLen=%d. Incorrect did. Must be diff than 0.", did, bitLen);
            return DIA_E_INVALID_LENGTH;
         }
      }
      break;

      case DIA_PROP_SRC_ENGINE_CLIENT_KEY_MULTIPLE:
      {
         didPtr = (tU16*) propData.mPropExtData[2];
         numElem = (tU32) propData.mPropExtData[3];

         if (NULL==didPtr)
         {
            DIA_TR_ERR("DIA_PROP_SRC_ENGINE_CLIENT_KEY_MULTIPLE: didPtr=NULL. Incorrect didPtr.");
            return DIA_E_INVALID_POINTER;
         }

         if (0==numElem)
         {
            DIA_TR_ERR("DIA_PROP_SRC_ENGINE_CLIENT_KEY_MULTIPLE: numElem=%d. Incorrect num of elem. Must be diff than 0.", numElem);
            return DIA_E_INVALID_LENGTH;
         }
         DIA_TR_INF("DIA_PROP_SRC_ENGINE_CLIENT_KEY_MULTIPLE: numElem=%d.", numElem);
      }
      break;

      default:
      {
         DIA_TR_ERR("Unknown type of key: %d. Error.", keyType);
         return DIA_FAILED;
      }
      break;
   }

   std::vector<tU16> didVect;

   switch (keyType)
   {
      case DIA_PROP_SRC_ENGINE_CLIENT_KEY_SINGLE:
      {
         didVect.push_back(did);
         DIA_TR_INF("DIA_PROP_SRC_ENGINE_CLIENT_KEY_SINGLE: did=0x%04X", did);
      }
      break;

      case DIA_PROP_SRC_ENGINE_CLIENT_KEY_MULTIPLE:
      {
         for (tU32 i=0; i<numElem; i++)
         {
            did = didPtr[i];

            if (DEFAULT_DID!=did)
            {
               DIA_TR_INF("DIA_PROP_SRC_ENGINE_CLIENT_KEY_MULTIPLE[%02d]: did=0x%04X", i, did);
               didVect.push_back(did);
            }
            else
            {
               DIA_TR_ERR("DIA_PROP_SRC_ENGINE_CLIENT_KEY_MULTIPLE: did is zero. Break.");
               return DIA_E_INVALID_LENGTH;
            }
         }

         DIA_TR_INF("DIA_PROP_SRC_ENGINE_CLIENT_KEY_MULTIPLE: numElem=%d bitLen=%d", numElem, bitLen);
      }
      break;

      default:
      {
         DIA_TR_ERR("Unknown type of key: %d. Error.", keyType);
         return DIA_FAILED;
      }
      break;
   }

   //send message to Engine Client
   std::vector<tU8> emptyData;

   dia_EngineClient* pEngine = 0;
   if ((getInstanceOfEngineManager()->queryEngineClient(engineUID, &pEngine) == DIA_SUCCESS) && pEngine)
   {
      std::vector<tArgsType> args;
      args.push_back((tArgsType)&didVect);
      args.push_back((tArgsType)&emptyData);
      args.push_back((tArgsType)callback);
      args.push_back((tArgsType)propID);
      result = pEngine->startControl(args);
      DIA_TR_INF("%s->startControl returned 0x%08X", pEngine->getName(), result);
   }
   else
   {
      DIA_TR_ERR("queryEngineClient => pEngine = 0x%p", pEngine);
   }

   if (DIA_SUCCESS==result)
   {
#if 0
      result = addCallback(propID, callback);
      DIA_TR_INF("startControl was successful. Add callback to rep. result=0x%08X", result);
#endif
   }
   else
   {
      DIA_TR_ERR("startControl unsuccessful. result=0x%08X", result);
   }

   //Reading has been triggered successfully
   return result;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagEngineClient::setPropertyU8 ( tU32 propID, tU8 propValue, dia_IEngineClient* callback )
{
   dia_tclFnctTrace trc("dia_PropertyBagEngineClient::setPropertyTriggerU8(tU32,tU8,dia_IEngineClient*)");

   tDiaResult retCode = DIA_FAILED;
   dia_PropertyInfo propData;

   if ( queryProperty(propID,propData) != DIA_SUCCESS ) return DIA_E_INVALID_KEY;
   if ( propData.mPropSize != sizeof(tU8) ) return DIA_E_INVALID_LENGTH;
   if (NULL==callback)  return DIA_E_INVALID_POINTER;

   dia_UID engineUID = 0;
   tCString strEngineName = (tCString)(propData.mPropExtData[0]); //static_cast<tCString>(propData.mPropExtData[0]);
   dia_enPropEngineClientKey keyType = static_cast<dia_enPropEngineClientKey>(propData.mPropExtData[1]);
   dia_PropertyInfo_statusMask_enum statusMask = static_cast<dia_PropertyInfo_statusMask_enum>(propData.mStatusMask);

   tU16 did = 0;
   tU32 bitLen = 0;

   if (strEngineName)
   {
      engineUID = dia_getHashCodeFromString(strEngineName);
   }

   if (0==engineUID)
   {
      DIA_TR_ERR("### ERROR: Status Mask: engineUID is not set.");
      return DIA_E_INVALID_PROP_TYPE;
   }
   else
   {
      DIA_TR_INF("Status Mask: engineUID is 0x%08X.", engineUID);
   }

   if (statusMask & SM_READ_ONLY)
   {
      DIA_TR_ERR("#### ERROR: Status Mask SM_READ_ONLY is set. Set impossible.");
      return DIA_E_NOT_SUPPORTED;
   }

   //WDBI with multiple DIDs is impossible
   if (DIA_PROP_SRC_ENGINE_CLIENT_KEY_SINGLE!=keyType)
   {
      DIA_TR_ERR("#### ERROR: key type is not single. Set impossible.");
      return DIA_E_NOT_SUPPORTED;
   }

   did = (tU16) propData.mPropExtData[2];
   bitLen = (tU32) propData.mPropExtData[3];

   if (DEFAULT_DID==did)
   {
      DIA_TR_ERR("### ERROR did=%d bitLen=%d. Incorrect did. Must be diff than 0.", did, bitLen);
      return DIA_E_INVALID_KEY;
   }

   if (bitLen>DIA_MAX_BIT_LENGTH)
   {
      DIA_TR_ERR("### ERROR bitLen=%d is greater than max bit length %d. Use another method.", bitLen, DIA_MAX_BIT_LENGTH);
      return DIA_E_INVALID_LENGTH;
   }

   tU8 propValueBitCut = propValue & ((tU8)((1<<bitLen)-1));
   if ( propValueBitCut!=propValue )
   {
      DIA_TR_ERR("### ERROR Bit length of written value is greater than property size (%d). PropVal=0x%02X propValueBitCut=0x%02X.", bitLen, propValue, propValueBitCut);
      return DIA_E_INVALID_SOURCE;
   }

   std::vector<tU16> didVect;
   didVect.push_back(did);

   std::vector<tU8> dataVect;
   dataVect.push_back(propValue);

   dia_EngineClient* pEngine = 0;
   if ((getInstanceOfEngineManager()->queryEngineClient(engineUID, &pEngine) == DIA_SUCCESS) && pEngine)
   {
      std::vector<tArgsType> args;
      args.push_back((tArgsType)&didVect);
      args.push_back((tArgsType)&dataVect);
      args.push_back((tArgsType)callback);
      args.push_back((tArgsType)propID);
      retCode = pEngine->startControl(args);
      DIA_TR_INF("%s->startControl returned 0x%08X", pEngine->getName(), retCode);
   }
   else
   {
      DIA_TR_ERR("### queryEngineClient => pEngine = 0x%p", pEngine);
   }

   if (DIA_SUCCESS==retCode)
   {
#if 0
      retCode = addCallback(propID, callback);
      DIA_TR_INF("startControl was successful. Add callback to rep. retCode=0x%08X", retCode);
#endif
   }
   else
   {
      DIA_TR_ERR("### startControl unsuccessful. retCode=0x%08X", retCode);
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagEngineClient::addCallback ( tU32 propID, dia_IEngineClient* pCallback )
{
    dia_tclFnctTrace trc("dia_PropertyBagEngineClient::addCallback(tU32,dia_IEngineClient*)");

    if ( (!pCallback) || (0==propID) ) return DIA_FAILED;
    std::map<tU32,dia_IEngineClient*>::iterator iter = mKeyCallbackRep.find(propID);
    if ( iter != mKeyCallbackRep.end() )  return DIA_E_KEY_ALREADY_EXISTS;
    mKeyCallbackRep[propID] = pCallback;

    return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_PropertyBagEngineClient::removeCallback ( tU32 propID, dia_IEngineClient* pCallback )
{
    dia_tclFnctTrace trc("dia_PropertyBagEngineClient::removeCallback(tU32,dia_IEngineClient*)");

    if ( (!pCallback) || (0==propID) ) return DIA_FAILED;

    std::map<tU32,dia_IEngineClient*>::iterator iter = mKeyCallbackRep.find(propID);
    if ( iter != mKeyCallbackRep.end() ) mKeyCallbackRep.erase(propID);

    return DIA_SUCCESS;
}
