/*!
 * \file       dia_ConfigManager.cpp
 *
 * \brief      Manager class that is used to read/write properties from several resources
 *
 * \details    Manager class that is used to read/write properties from several resources
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreConfig
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include <string.h>

// config management
#ifndef __INCLUDED_DIA_COMMON_CONFIG__
#include "common/framework/config/dia_common_config.h"
#endif

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_ConfigManager)

#ifndef __DIA_UNIT_TESTING__

dia_ConfigManager*
getInstanceOfConfigManager ( void )
{
   return dia_ConfigManager::getInstance();
}

void
releaseInstanceOfConfigManager ( void )
{
   dia_ConfigManager::deleteInstance();
}

#endif

//----------------------------------------------------------------------------------------

dia_ConfigManager::dia_ConfigManager ( void )
{}

//----------------------------------------------------------------------------------------

dia_ConfigManager::~dia_ConfigManager ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setup ( void )
{
   dia::ScopeTrace trc("dia_ConfigManager::setup");

   //! creation of ConfigManager was successful so we can load the common properties now
   return loadProperties(aCommonPropData, (tU16)dia::getCommonPropDataSize());
}

//-----------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::tearDown ( void )
{
   std::map<dia_UID,dia_PropertyBag*>::const_iterator iter = mPropertyBagRep.begin();
   for ( ; iter != mPropertyBagRep.end(); ++iter )
   {
      dia_PropertyBag* pPropBag = iter->second;

      if ( pPropBag )
      {
         // close the property bag if it is currently opened
         if ( pPropBag->bIsOpen() )
         {
            DIA_M_RETURNCODE_INTENTIONALLY_IGNORED(pPropBag->close());
         }

         // now delete the property bag
         OSAL_DELETE pPropBag;
      }
   }
   mPropertyBagRep.clear();

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::forceFlush ( void ) const
{
   tDiaResult retCode = DIA_SUCCESS;

   std::map<dia_UID,dia_PropertyBag*>::const_iterator iter = mPropertyBagRep.begin();
   for ( ; iter != mPropertyBagRep.end(); ++iter )
   {
      dia_PropertyBag* pPropBag = iter->second;

      if ( pPropBag && pPropBag->isDirty() )
      {
         tDiaResult flushResult = pPropBag->flush();

         switch (flushResult)
         {
            case DIA_E_NOT_IMPLEMENTED:
            case DIA_SUCCESS:
               break;
            default:
               retCode = DIA_FAILED;
               break;
         }

         DIA_TR_INF( "dia_ConfigManager::forceFlush (Bag-UID=0x%08x, retCode==0x%08x", pPropBag->getUID(),flushResult);
      }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

size_t
dia_ConfigManager::getSize ( tU32 propID )
{
   size_t uSize = 0;

   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         uSize = pConfManager->mPropertyRepository[propID].mPropSize;
      }
   }

   DIA_TR_INF( "dia_ConfigManager::getSize --- ID=0x%0x, Size=%zu", propID, uSize);

   return uSize;
}

//----------------------------------------------------------------------------------------

size_t
dia_ConfigManager::getSize ( const std::string& propName )
{
   return getSize(dia_getHashCodeFromString(propName));
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( tU32 propID, tU8& propValue )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::getProperty(tU8)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::getProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->getPropertyU8(propID,propValue);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( const std::string& propName, tU8& propValue )
{
   return getProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( tU32 propID, bool& propValue )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::getProperty(bool&)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::getProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->getPropertyBoolean(propID,propValue);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( const std::string& propName, bool& propValue )
{
	return getProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( tU32 propID, tU16& propValue )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::getProperty(tU16)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::getProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->getPropertyU16(propID,propValue);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( const std::string& propName, tU16& propValue )
{
   return getProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( tU32 propID, tU32& propValue )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::getProperty(tU32)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::getProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo& propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->getPropertyU32(propID,propValue);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( const std::string& propName, tU32& propValue )
{
   return getProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( tU32 propID, tU8 propValue[], size_t propLength )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::getProperty(tU32, tU8, tU32)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::getProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo& propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->getProperty(propID,propValue,propLength);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( const std::string& propName, tU8 propValue[], size_t propLength )
{
   return getProperty(dia_getHashCodeFromString(propName),propValue,propLength);
}

//----------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty(tU32 propID, tU8 propValue[], size_t* maxLength )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::getProperty(tU32, tU8, size_t*)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF("dia_ConfigManager::getProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo& propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->getProperty(propID, propValue, maxLength);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( const std::string& propName, tU8 propValue[], size_t* maxLength )
{
   return getProperty(dia_getHashCodeFromString(propName),propValue,maxLength);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( tU32 propID, std::vector<tU8>& propValue )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::getProperty(tU32, std::vector<tU8>&)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::getProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo& propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->getProperty(propID,propValue);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( tU32 propID, dia_IEngineClient* callback )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::getProperty(tU32 propID, dia_IEngineClient* callback)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::getProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo& propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->getProperty(propID,callback);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::getProperty ( const std::string& propName, std::vector<tU8>& propValue )
{
   return getProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( tU32 propID, tU8 propValue )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::setProperty(tU8)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::setProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->setPropertyU8(propID,propValue);
               if ( retCode == DIA_SUCCESS ) (tVoid) pConfManager->notifyPropertUpdate(propID);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( const std::string& propName, tU8 propValue )
{
   return setProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( tU32 propID, tU16 propValue )
{
   dia::ScopeTrace oTrace("dia_ConfigManager::setProperty(tU16)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::setProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->setPropertyU16(propID,propValue);
               if ( retCode == DIA_SUCCESS ) (tVoid) pConfManager->notifyPropertUpdate(propID);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( const std::string& propName, tU16 propValue )
{
   return setProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( tU32 propID, tU32 propValue )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::setProperty(tU32)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "dia_ConfigManager::setProperty --- ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->setPropertyU32(propID,propValue);
               if ( retCode == DIA_SUCCESS ) (tVoid) pConfManager->notifyPropertUpdate(propID);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( const std::string& propName, tU32 propValue )
{
   return setProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( tU32 propID, tU8 propValue[], size_t propLength )
{
   dia::ScopeTrace oTrace("dia_ConfigManager::setProperty()");

   DIA_TR_INF("##### Config Manager : propLength = %zu #####", propLength);

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "Property ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo& propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;

            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->setProperty(propID, propValue,propLength);

               if ( retCode == DIA_SUCCESS ) (tVoid) pConfManager->notifyPropertUpdate(propID);
            }
            else
            {
               DIA_TR_ERR("dia_ConfigManager::setProperty() returned DIA_E_CONFMGR_INV_PROPBAG");
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            DIA_TR_ERR("dia_ConfigManager::setProperty() returned DIA_E_CONFMGR_INV_TYPE");
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      DIA_TR_ERR("dia_ConfigManager::setProperty() returned DIA_E_CONFMGR_NOT_AVAILABLE");
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("dia_ConfigManager::setProperty() returned 0x%08X", retCode);
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setPropertyU8 (tU32 propID, tU8 propValue, dia_IEngineClient* callback)
{
	dia::ScopeTrace oTrace("dia_ConfigManager::setPropertyU8()");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_ERR( "Property ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo& propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->setPropertyU8(propID,propValue,callback);
#if 0 //For asynchronous Set don't notify about a property update
               if ( retCode == DIA_SUCCESS ) (tVoid) pConfManager->notifyPropertUpdate(propID);
#endif
            }
            else
            {
               DIA_TR_ERR("dia_ConfigManager::setPropertyU8() returned DIA_E_CONFMGR_INV_PROPBAG");
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            DIA_TR_ERR("dia_ConfigManager::setPropertyU8() returned DIA_E_CONFMGR_INV_TYPE");
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      DIA_TR_ERR("dia_ConfigManager::setPropertyU8() returned DIA_E_CONFMGR_NOT_AVAILABLE");
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_ERR("dia_ConfigManager::setPropertyU8() returned 0x%08X", retCode);
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( const std::string& propName, tU8 propValue[], size_t propLength )
{
   return setProperty(dia_getHashCodeFromString(propName),propValue,propLength);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( tU32 propID, const std::vector<tU8>& vec )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::setProperty(tU32,const vector<tU8>&)");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "Property ID = 0x%0x", propID);

   // retrieve the pointer to the config manager object
   dia_ConfigManager* pConfManager = getInstanceOfConfigManager();
   if ( pConfManager )
   {
      std::map<tU32,dia_PropertyInfo>::iterator iter = pConfManager->mPropertyRepository.find(propID);

      if ( iter != pConfManager->mPropertyRepository.end() )
      {
         dia_PropertyInfo& propData = pConfManager->mPropertyRepository[propID];

         if ( propData.mPropLoc != DIA_C_UID_PROPBAG_UNKNOWN )
         {
            dia_PropertyBag* pPropertyBag = 0;
            if ( (queryPropertyBag(propData.mPropLoc,&pPropertyBag) == DIA_SUCCESS) && pPropertyBag )
            {
               retCode = pPropertyBag->setProperty(propID,&(const_cast<std::vector<tU8>&>(vec))[0],(tU16) vec.size());
               if ( retCode == DIA_SUCCESS ) (tVoid) pConfManager->notifyPropertUpdate(propID);
            }
            else
            {
               retCode = DIA_E_CONFMGR_INV_PROPBAG;
            }
         }
         else
         {
            retCode = DIA_E_CONFMGR_INV_TYPE;
         }
      }
   }
   else
   {
      retCode = DIA_E_CONFMGR_NOT_AVAILABLE;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::setProperty ( const std::string& propName, const std::vector<tU8>& vec )
{
   return setProperty(dia_getHashCodeFromString(propName),vec);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::addPropertyBag ( dia_PropertyBag* pPropBag )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::setPropertyBag()");

   tDiaResult retCode = DIA_FAILED;

   if ( pPropBag )
   {
      // retrieve the type of properties handle by the property bag
      dia_UID uid = pPropBag->getUID();

      DIA_TR_INF( "dia_ConfigManager::addPropertyBag --- UID = 0x%08x", uid);
      mPropertyBagRep[uid] = pPropBag;
      pPropBag->update();
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::loadProperty ( const dia_PropertyInfo& propData )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::loadProperty");
   mPropertyRepository[propData.mPropKey] = propData;
   return DIA_SUCCESS;
}

tDiaResult
dia_ConfigManager::loadProperties ( const dia_PropertyInfo propData[], tU16 numOfProps )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::loadProperties");

   tDiaResult retCode = DIA_FAILED;

   if ( propData )
   {
      for ( tU16 i=0; i<numOfProps; ++i )
      {
         DIA_DBG_TR_INF("dia_ConfigManager::loadProperties [%d] mPropKey=0x%08X", i, propData[i].mPropKey);
         mPropertyRepository[propData[i].mPropKey] = propData[i];
      }
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::addConfigNotifier ( dia_ConfigNotifier* pNotifier )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::addConfigNotifier()");

   if ( !pNotifier ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   std::vector<tU32> properties;
   if ( pNotifier->getProperties(properties) == DIA_SUCCESS )
   {
      for ( tU16 i=0; i<properties.size(); i++ )
      {
         tU32 propID = properties[i];
         std::map<tU32,std::list<dia_ConfigNotifier*> >::iterator iter = mNotifierRepository.find(propID);

         if ( (iter == mNotifierRepository.end()) || (iter->second.size() == 0) || (std::find(iter->second.begin(),iter->second.end(),pNotifier) == iter->second.end()) )
         {
            // here we know that this notifier pointer was not stored before
            DIA_TR_INF("#######################################################");
            DIA_TR_INF("#");
            DIA_TR_INF("# ADDING CONFIG NOTIFIER \"%s\" FOR PROPERTY 0x%08x", pNotifier->getName(), propID);
            DIA_TR_INF("#");
            DIA_TR_INF("#######################################################");
            mNotifierRepository[propID].push_back(pNotifier);
            retCode = DIA_SUCCESS;
         }
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::removeConfigNotifier ( dia_ConfigNotifier* pNotifier )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::removeConfigNotifier()");

   if ( !pNotifier ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   std::vector<tU32> properties;
   if ( pNotifier->getProperties(properties) == DIA_SUCCESS )
   {
      for ( tU16 i=0; i<properties.size(); i++ )
      {
         tU32 propID = properties[i];
         std::map<tU32,std::list<dia_ConfigNotifier*> >::iterator iter = mNotifierRepository.find(propID);
         if ( iter != mNotifierRepository.end() )
         {
            std::list<dia_ConfigNotifier*>::iterator listIter = std::find(iter->second.begin(),iter->second.end(),pNotifier);
            if ( listIter != iter->second.end() )
            {
               DIA_TR_INF("#######################################################");
               DIA_TR_INF("#");
               DIA_TR_INF("# REMOVING CONFIG NOTIFIER \"%s\" FOR PROPERTY 0x%08x", pNotifier->getName(),propID);
               DIA_TR_INF("#");
               DIA_TR_INF("#######################################################");
               iter->second.erase(listIter);
               if ( iter->second.empty() )
               {
                  mNotifierRepository.erase(propID);
               }

               retCode = DIA_SUCCESS;
            }
         }
      }
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::clearConfigNotifiers ( void )
{
   std::map<tU32,std::list<dia_ConfigNotifier*> >::iterator iter = mNotifierRepository.begin();
   for ( ; iter != mNotifierRepository.end(); iter++ )
   {
      iter->second.clear();
   }
   mNotifierRepository.clear();

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::notifyPropertUpdate ( tU32 propID )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::notifyPropertUpdate()");

   std::map<tU32,std::list<dia_ConfigNotifier*> >::iterator iter = mNotifierRepository.find(propID);
   if ( iter != mNotifierRepository.end() ) // && iter->second )
   {
      std::list<dia_ConfigNotifier*>::iterator listIter = iter->second.begin();
      for ( ; listIter != iter->second.end(); listIter++ )
      {
         if ( !(*listIter) ) continue;
         if (  (*listIter)->onPropertyUpdate(propID) != DIA_SUCCESS )
         {
            DIA_TR_ERR("##### UPDATE VIA CONFIG NOTIFIER \"%s\" FOR PROPERTY 0x%08x FAILED #####", (*listIter)->getName(),propID);
         }
         else
         {
            DIA_TR_INF("dia_ConfigManager::notifyPropertUpdate \"%s\" for property 0x%08x successfully", (*listIter)->getName(), propID);
         }
      }
   }
   else
   {
      DIA_TR_INF("##### dia_ConfigManager::notifyPropertUpdate PROPERTY 0x%08x NOT FOUND #####", propID);
   }

   return DIA_SUCCESS;
}

//----------------------------------------------------------------------------------------

size_t
dia_getPropertySize ( tU32 propID )
{
   return getInstanceOfConfigManager()->getSize(propID);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( tU32 propID, bool& propValue )
{
   return getInstanceOfConfigManager()->getProperty(propID,propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( const std::string& propName, bool& propValue )
{
   return getInstanceOfConfigManager()->getProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( tU32 propID, tU8& propValue )
{
   return getInstanceOfConfigManager()->getProperty(propID,propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( const std::string& propName, tU8& propValue )
{
   return getInstanceOfConfigManager()->getProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( tU32 propID, tU16& propValue )
{
   return getInstanceOfConfigManager()->getProperty(propID, propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( const std::string& propName, tU16& propValue )
{
   return getInstanceOfConfigManager()->getProperty(dia_getHashCodeFromString(propName), propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( tU32 propID, tU32& propValue )
{
   return getInstanceOfConfigManager()->getProperty(propID,propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( const std::string& propName, tU32& propValue )
{
   return getInstanceOfConfigManager()->getProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( tU32 propID, tU8 propValue[], size_t propLength )
{
   return getInstanceOfConfigManager()->getProperty(propID,propValue,propLength);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( const std::string& propName, tU8 propValue[], size_t propLength )
{
   return getInstanceOfConfigManager()->getProperty(dia_getHashCodeFromString(propName),propValue,propLength);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( tU32 propID, tU8 propValue[], size_t* propLength )
{
   return getInstanceOfConfigManager()->getProperty(propID,propValue,propLength);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( const std::string& propName, tU8 propValue[], size_t* propLength )
{
   return getInstanceOfConfigManager()->getProperty(dia_getHashCodeFromString(propName),propValue,propLength);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( tU32 propID, std::vector<tU8>& propValue )
{
   return getInstanceOfConfigManager()->getProperty(propID,propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_getProperty ( const std::string& propName, std::vector<tU8>& propValue )
{
   return getInstanceOfConfigManager()->getProperty(dia_getHashCodeFromString(propName),propValue);
}

//----------------------------------------------------------------------------------------
tDiaResult
dia_getProperty ( tU32 propID, dia_IEngineClient* callback)
{
   return getInstanceOfConfigManager()->getProperty(propID,callback);
}

//----------------------------------------------------------------------------------------


tDiaResult
dia_setProperty ( tU32 propID, tU8 propValue )
{
   return getInstanceOfConfigManager()->setProperty(propID,propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_setProperty ( tU32 propID, tU16 propValue )
{
   return getInstanceOfConfigManager()->setProperty(propID,propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_setProperty ( tU32 propID, tU32 propValue )
{
   return getInstanceOfConfigManager()->setProperty(propID,propValue);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_setProperty ( tU32 propID, tU8 propValue[], size_t propLength )
{

   return getInstanceOfConfigManager()->setProperty(propID,propValue,propLength);
}

//----------------------------------------------------------------------------------------

tDiaResult
dia_setProperty ( tU32 propID, const std::vector<tU8>& vec )
{
   return getInstanceOfConfigManager()->setProperty(propID,vec);
}

//----------------------------------------------------------------------------------------

tDiaResult
setPropertyU8 ( tU32 propID, tU8 propValue, dia_IEngineClient* callback)
{
   return getInstanceOfConfigManager()->setPropertyU8(propID,propValue,callback);
}
//----------------------------------------------------------------------------------------

dia_PropertyInfo*
dia_ConfigManager::getPropertyInfo ( tU32 propID )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::getPropertyInfo()");

   dia_PropertyInfo* pPropInfo = 0;

   std::map<tU32,dia_PropertyInfo>::iterator iter = mPropertyRepository.find(propID);

   if ( iter != mPropertyRepository.end() )
   {
      pPropInfo = &(mPropertyRepository[propID]);
   }

   return pPropInfo;
}

//-----------------------------------------------------------------------------------------

tBool
dia_checkPropertyValue ( tU32 propID )
{
   tBool retCode = FALSE;

   tU8 propValue = 0;
   if ( (dia_getProperty(propID,propValue) == DIA_SUCCESS) && propValue )
   {
      retCode = TRUE;
   }

   return retCode;
}

//-----------------------------------------------------------------------------------------

tDiaResult
dia_ConfigManager::queryPropertyBag ( dia_UID uid, dia_PropertyBag** ppPropBag )
{
	dia::ScopeTrace oTrace("dia_ConfigManager::queryPropertyBag(dia_UID,dia_PropertyBag**)");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   if ( !ppPropBag ) return DIA_E_INVALID_POINTER;

   (*ppPropBag) = 0;
   std::map<dia_UID,dia_PropertyBag*>::iterator iter = mPropertyBagRep.find(uid);
   if ( iter != mPropertyBagRep.end() )
   {
      (*ppPropBag) = iter->second;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

