/*
 * dia_VideoSourceControl.cpp
 *
 *  Created on: 01.07.2014
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SYSTEM_ADAPTERS__
#include "common/framework/sysadapters/dia_common_system_adapters.h"
#endif

#ifndef __INCLUDED_DIA_VIDEO_SOURCE_CONTROL__
#include "common/framework/application/dia_VideoSourceControl.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_VIDEOSOURCECONTROL_LISTENER__
#include "common/interfaces/dia_IVideoSourceControlListener.h"
#endif

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_VideoSourceControl)

#ifndef __DIA_UNIT_TESTING__

dia_VideoSourceControl*
getInstanceOfVideoSourceControl ( void )
{
   return dia_VideoSourceControl::getInstance();
}

void
releaseInstanceOfVideoSourceControl ( void )
{
   dia_VideoSourceControl::deleteInstance();
}

#endif

static dia_eVideoSourceID srcMapping[DIA_EN_VIDEO_DESTINATION_COUNT] = {
      DIA_EN_VIDEO_SOURCE_ID_UNKNOWN,
      DIA_EN_VIDEO_SOURCE_ID_UNKNOWN,
      DIA_EN_VIDEO_SOURCE_ID_DVD,
      DIA_EN_VIDEO_SOURCE_ID_RVC,
      DIA_EN_VIDEO_SOURCE_ID_RVC,
      DIA_EN_VIDEO_SOURCE_ID_UNKNOWN,
      DIA_EN_VIDEO_SOURCE_ID_UNKNOWN,
      DIA_EN_VIDEO_SOURCE_ID_HMI
};

static dia_eVideoManagerDestinationID srcDestMapping[DIA_EN_VIDEO_SOURCE_ID_COUNT] = {
      DIA_EN_VIDEO_DESTINATION_UNKNOWN,
      DIA_EN_VIDEO_DESTINATION_HMI,
      DIA_EN_VIDEO_DESTINATION_RVC_ON_HMI,
      DIA_EN_VIDEO_DESTINATION_DVD
};

//-----------------------------------------------------------------------------

dia_VideoSourceControl::dia_VideoSourceControl (void )
   : mErrorCode(DIA_E_NOERROR),
     mIsResultReady(false),
     mRequestedSourceValid(false),
     mIsSrcSwitchRequired(true),
     mIsVideoSrcActive(false),
     mIsVideoStreamStopped(false),
     mIsVideoSourceConnected(false),
     mIsVideoStreamStarted(false),
     mNeedToDisconnect(false),
     mpFSM(0)
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::dia_VideoSourceControl()");
}

//------------------------------------------------------------------------------

dia_VideoSourceControl::~dia_VideoSourceControl ( void )
{
   _BP_TRY_BEGIN
   {
     (void) unsetSysAdapterListener<dia_IVideoManagerListener>(this);
      (void) tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_VideoSourceControl::~dia_VideoSourceControl !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_VideoSourceControl::setup ( void )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::setup()");

   tDiaResult retCode = DIA_FAILED;

   // create the state machine object
   mpFSM = OSAL_NEW dia_VideoSourceControlFSM::Fsm(this);

   if ( mpFSM )
   {
       DIA_TR_INF( "### Initializing State Machine ###");
       mpFSM->init();
       DIA_TR_INF( "### Done ###");
       retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_VideoSourceControl::tearDown ( tVoid )
{
   if ( mpFSM )
   {
      OSAL_DELETE mpFSM;
      mpFSM = 0;
   }

   return DIA_SUCCESS;
}


//------------------------------------------------------------------------------

tDiaResult
dia_VideoSourceControl::adddVideoSourceControlPlugin ( dia_VideoSourceControlPlugin* pPlugin )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::adddVideoSourceControlPlugin()");

   tDiaResult retCode = DIA_FAILED;

   if ( pPlugin )
   {
      mPluginIter = mPluginRep.find(pPlugin->eGetType());
      if ( mPluginIter == mPluginRep.end() )
      {
         DIA_TR_INF("#######################################################");
         DIA_TR_INF("#");
         DIA_TR_INF("# ADDING VIDEO SOURCE CONTROL PLUGIN \"%s\"", pPlugin->getName());
         DIA_TR_INF("#");
         DIA_TR_INF("#######################################################");
         mPluginRep[pPlugin->eGetType()] = pPlugin;
         retCode = DIA_SUCCESS;
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_VideoSourceControl::removeVideoSourceControlPlugin ( dia_eVideoSourceID type )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::removeVideoSourceControlPlugin()");

   tDiaResult retCode = DIA_FAILED;

   mPluginIter = mPluginRep.find(type);
   if ( mPluginIter != mPluginRep.end() )
   {
      dia_VideoSourceControlPlugin* pPlugin = mPluginRep[type];
      mPluginRep.erase(type);
      OSAL_DELETE pPlugin;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_VideoSourceControl::activateVideoSource ( dia_eVideoSourceID srcID )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::activateVideoSource()");

   mErrorCode = DIA_SUCCESS;

   if ( mpFSM )
   {
      mPluginIter = mPluginRep.find(srcID);
      mRequestedVideoSource.mPluginRef = ( mPluginIter != mPluginRep.end() ) ? mPluginIter->second : 0;
      if ( mRequestedVideoSource.mPluginRef )
      {
         mRequestedVideoSource.mSrcID = mRequestedVideoSource.mPluginRef->eGetType();
         DIA_TR_INF( "dia_VideoSourceControl::### mRequestedVideoSource.mSrcID = 0x%02x", mRequestedVideoSource.mSrcID);
      }
      else
      {
         DIA_TR_INF( "dia_VideoSourceControl::### mRequestedVideoSource.mPluginRef = NULL");
      }

      DIA_TR_INF(("dia_VideoSourceControl::activateVideoSource() - FSM evRequestVideoSource"));
      DIA_TR_INF("State (before): %s", mpFSM->getStateName());
      mpFSM->acceptEvent(dia_VideoSourceControlFSM::evRequestVideoSource,(void*) &srcID);
      DIA_TR_INF("State (after) : %s", mpFSM->getStateName());
   }

   // send update to listener if possible
   sendUpdate();

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::sendUpdate ( void ) const
{
   if ( mIsResultReady )
   {
      dia_IVideoSourceControlListener* pListener = 0;
      if ((querySysAdapterListener<dia_IVideoSourceControlListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         if ( mErrorCode == DIA_SUCCESS )
         {
            pListener->vOnVideoSourceUpdate(mActiveVideoSource.mSrcID, mPreviousVideoSource.mSrcID);
         }
         else
         {
            pListener->vOnVideoSourceError(mErrorCode);
         }
      }
   }
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::resetFSMTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::resetFSMTimeout()");
   DIA_TR_INF("##### TRIGGER vFsmSetErrCode_TimeExpired ######");
   if ( mpFSM )
      mpFSM->acceptEvent(dia_VideoSourceControlFSM::evTimeout,0);
}

//------------------------------------------------------------------------------

dia_eVideoSourceID
dia_VideoSourceControl::getActiveVideoSource ( void )
{
   return mActiveVideoSource.mSrcID;
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vOnConnectionStatusUpdate ( dia_eVideoManagerDestinationID destID, dia_eVideoDestinationType /*destType*/, dia_eVideoSourceConnectionStatus status )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vOnConnectionStatusUpdate()");

   dia_eVideoSourceID srcID = srcMapping[destID];
   if ( mpFSM )
   {
      switch ( status )
      {
      case DIA_EN_VIDEO_SOURCE_CONNECTION_STATUS_CONNECTED:
         mIsVideoSourceConnected = true;
         mpFSM->acceptEvent(dia_VideoSourceControlFSM::evOnVideoSourceConnected,(void*) srcID);
         break;
      case DIA_EN_VIDEO_SOURCE_CONNECTION_STATUS_DISCONNECTED:
         mIsVideoSourceConnected = false;
         mpFSM->acceptEvent(dia_VideoSourceControlFSM::evOnVideoSourceDisconnected,(void*) srcID);
         break;
      default:
         break;
      }
   }
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vOnDisConnectionAllStatusUpdate ( void )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vOnDisConnectionAllStatusUpdate()");

   dia_eVideoSourceID srcID = mRequestedVideoSource.mSrcID;
   mIsVideoSourceConnected = false;
   if ( mpFSM )
      mpFSM->acceptEvent(dia_VideoSourceControlFSM::evOnVideoSourceDisconnected,(void*) srcID);
}


//------------------------------------------------------------------------------
void
dia_VideoSourceControl::vOnConnectionError ( tDiaResult errorCode )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vOnConnectionError()");

   mErrorCode = errorCode;
   DIA_TR_INF("##### DETECTED VIDEO SOURCE CONNECTION ERROR (CODE=0x%08x) #####", errorCode);

   if ( mpFSM )
      mpFSM->acceptEvent(dia_VideoSourceControlFSM::evError,0);
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vOnActiveVideoSource ( dia_eVideoDestinationType /*destType*/, dia_eVideoManagerDestinationID destID )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vOnActiveVideoSource()");

   dia_eVideoSourceID srcID = srcMapping[destID];

   mPluginIter = mPluginRep.find(srcID);
   if ( mPluginIter != mPluginRep.end() )
   {
      mActiveVideoSource.mPluginRef = mPluginIter->second;
      mActiveVideoSource.mSrcID  = mActiveVideoSource.mPluginRef->eGetType();
      mIsVideoSrcActive = true;
      if ( mpFSM ) mpFSM->acceptEvent(dia_VideoSourceControlFSM::evOnVideoSourceInfo,0);
   }
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vOnActiveVideoSourceError ( dia_eVideoDestinationType /*destType*/, tDiaResult errorCode )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vOnActiveVideoSourceError()");

   mErrorCode = errorCode;
   DIA_TR_INF("##### UNABLE TO RETRIEVE ACTIVE VIDEO SOURCE INFORMATION (CODE=0x%08x) #####", errorCode);

   if ( mpFSM )
      mpFSM->acceptEvent(dia_VideoSourceControlFSM::evError,0);
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vOnVideoStreamingStarted ( dia_VideoSourceControlPlugin* /*pPlugin*/, tDiaResult resultCode )
{
   if ( resultCode == DIA_SUCCESS )
   {
      mIsVideoStreamStopped = false;
      mIsVideoStreamStarted = true;
   }

   if ( mpFSM )
      mpFSM->acceptEvent(dia_VideoSourceControlFSM::evOnVideoStreamStarted,0);
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vOnVideoStreamingStopped ( dia_VideoSourceControlPlugin* /*pPlugin*/, tDiaResult resultCode )
{
   if ( resultCode == DIA_SUCCESS )
   {
      mIsVideoStreamStopped = true;
      mIsVideoStreamStarted = false;
   }

   if ( mpFSM )
      mpFSM->acceptEvent(dia_VideoSourceControlFSM::evOnVideoStreamStopped,0);
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmReset ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmReset()");

   mActiveVideoSource.reset();
   mRequestedVideoSource.reset();
   mPreviousVideoSource.reset();

   mErrorCode              = DIA_E_NOERROR;
   mIsResultReady          = false;
   mRequestedSourceValid   = false;
   mIsSrcSwitchRequired    = true;
   mIsVideoSrcActive       = false;
   mIsVideoStreamStopped   = false;
   mIsVideoSourceConnected = false;
   mIsVideoStreamStarted   = false;
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmSetErrCode_TimeExpired ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmSetErrCode_TimeExpired()");
   mErrorCode = DIA_E_TIMEOUT;
   mIsResultReady = true;
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmSetErrCode_InvalidVideoSource ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmSetErrCode_InvalidVideoSource()");
   mErrorCode = DIA_E_INVALID_SOURCE;
   mIsResultReady = true;
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmSetRequestedVideoSource ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmSetRequestedVideoSource()");

   // nothing to be done here. Requested source is set before trigger is sent
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmGetCurrentlyActivatedVideoSource ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmGetCurrentlyActivatedVideoSource()");

   dia_IVideoManager* pInterface = 0;
   if ((querySysAdapterInterface<dia_IVideoManager>(&pInterface) == DIA_SUCCESS) && pInterface)
   {
      (void) setSysAdapterListener<dia_IVideoManagerListener>(this);

      if (pInterface->getActiveVideoSource() != DIA_SUCCESS)
      {
         DIA_TR_ERR("dia_VideoSourceControl::vFsmGetCurrentlyActivatedVideoSource => Error: getActiveVideoSource() failed !");
         if ( mpFSM )
          mpFSM->acceptEvent(dia_VideoSourceControlFSM::evError,0);
      }
   }
   else
   {
      DIA_TR_ERR("dia_VideoSourceControl::vFsmGetCurrentlyActivatedVideoSource => Error: Cannot get dia_IVideoManager !");
      mErrorCode = DIA_E_CONDITIONS_NOT_CORRECT;
      DIA_TR_INF("##### DETECTED FAILURE IN QUERYING INTERFACE POINTER TO VIDEO MANAGER (CODE=0x%08x) #####", mErrorCode);
      if ( mpFSM )
         mpFSM->acceptEvent(dia_VideoSourceControlFSM::evError,0);
   }
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmStartRequestedVideoStream ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmStartRequestedVideoStream()");

   if ( mRequestedVideoSource.mPluginRef )
   {
      mRequestedVideoSource.mPluginRef->startVideoStreaming();
   }
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmStopActiveVideoStream ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmStopActiveVideoStream()");

   if ( !(mActiveVideoSource.mPluginRef) )
   {
      mIsVideoStreamStopped = true;
      if ( mpFSM )
        mpFSM->acceptEvent(dia_VideoSourceControlFSM::evOnVideoStreamStopped,0);
   }
   else
   {
      mActiveVideoSource.mPluginRef->stopVideoStreaming();
   }
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmConnectRequestedVideoSource ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmConnectRequestedVideoSource()");

   if ( mRequestedVideoSource.mPluginRef )
   {
      dia_IVideoManager* pInterface = 0;
      if ((querySysAdapterInterface<dia_IVideoManager>(&pInterface) == DIA_SUCCESS) && pInterface)
      {
         (void) setSysAdapterListener<dia_IVideoManagerListener>(this);

         if ( pInterface->connectVideoSource(srcDestMapping[mRequestedVideoSource.mPluginRef->eGetType()]) != DIA_SUCCESS )
         {
            DIA_TR_ERR("dia_VideoSourceControl::vFsmConnectRequestedVideoSource => Error: connectVideoSource failed !");
            if ( mpFSM ) mpFSM->acceptEvent(dia_VideoSourceControlFSM::evError,0);
         }
      }
      else
      {
         DIA_TR_ERR("dia_VideoSourceControl::vFsmConnectRequestedVideoSource => Error: No interface pointer to dia_IVideoManager !");
      }
   }
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmDisconnectActiveVideoSource ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmDisconnectActiveVideoSource()");

   if ( mActiveVideoSource.mPluginRef )
   {
      dia_IVideoManager* pInterface = 0;
      if ((querySysAdapterInterface<dia_IVideoManager>(&pInterface) == DIA_SUCCESS) && pInterface)
      {
         (void) setSysAdapterListener<dia_IVideoManagerListener>(this);

         if ( pInterface->disconnectVideoSource(srcDestMapping[mActiveVideoSource.mPluginRef->eGetType()]) != DIA_SUCCESS )
         {
            DIA_TR_ERR("dia_VideoSourceControl::vFsmDisconnectActiveVideoSource => Error: disconnectVideoSource failed !");
         }
      }
      else
      {
         DIA_TR_ERR("dia_VideoSourceControl::vFsmDisconnectActiveVideoSource => Error: No interface pointer to dia_IVideoManager !");
      }
   }
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmSetActiveVideoSource ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmSetActiveVideoSource()");

   mActiveVideoSource.mSrcID     = mRequestedVideoSource.mSrcID;
   mActiveVideoSource.mPluginRef = mRequestedVideoSource.mPluginRef;
}

//------------------------------------------------------------------------------

void
dia_VideoSourceControl::vFsmVideoSourceActivated ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::vFsmVideoSourceActivated()");

   mIsResultReady = true;
   mErrorCode     = DIA_SUCCESS;

   // we are done. source is active now
   this->sendUpdate();
}

//------------------------------------------------------------------------------

bool
dia_VideoSourceControl::bIsRequestedSourceValid ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::bIsRequestedSourceValid()");
   mPluginIter = mPluginRep.find(mRequestedVideoSource.mSrcID);
   mRequestedSourceValid = ( mPluginIter != mPluginRep.end() ) ? true : false;
   return mRequestedSourceValid;
}

//------------------------------------------------------------------------------

bool
dia_VideoSourceControl::bIsSrcSwitchRequired ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::bIsSrcSwitchRequired()");
   mIsSrcSwitchRequired = (mActiveVideoSource.mSrcID == mRequestedVideoSource.mSrcID) ? false : true;
   return mIsSrcSwitchRequired;
}

//------------------------------------------------------------------------------

bool
dia_VideoSourceControl::bIsVideoSrcActive ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::bIsVideoSrcActive()");
   return mIsVideoSrcActive;
}

//------------------------------------------------------------------------------

bool
dia_VideoSourceControl::bIsVideoSourceConnected ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::bIsVideoSourceConnected()");
   return mIsVideoSourceConnected;
}

//------------------------------------------------------------------------------

bool
dia_VideoSourceControl::bIsVideoSourceDisconnected ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::bIsVideoSourceDisconnected()");
   return (mIsVideoSourceConnected) ? false : true;
}

//------------------------------------------------------------------------------

bool
dia_VideoSourceControl::bNeedToDisconnect ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::bNeedToDisconnect()");
   return ( mActiveVideoSource.mPluginRef ) ? true : false;
}

//------------------------------------------------------------------------------

bool
dia_VideoSourceControl::bIsVideoStreamStarted ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::bIsVideoStreamStarted()");
   return mIsVideoStreamStarted;
}

//------------------------------------------------------------------------------

bool
dia_VideoSourceControl::bIsVideoStreamStopped ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_VideoSourceControl::bIsVideoStreamStopped()");
   return mIsVideoStreamStopped;
}


