/*!
 * \file       dia_ThreadMonitor.h
 *
 * \brief      Monitor diagnostic threads and provide info about them
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2018 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#define __INCLUDED_DIA_THREAD_MONITOR__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_LOCK__
#include <common/framework/application/dia_Lock.h>
#endif

#include <pthread.h>
#include <memory>

//#define DIA_C_S32_INVALID_SYSTEM_THREAD_ID ((tS32) 0xFFFFFFFF)

//--> START OF COMPATIBILITY SECTION: DO NOT MODIFY

typedef const std::string& eAppThreadID;

const std::string eThreadID_Unknown                = "eThreadID_Unknown";
const std::string eThreadID_Main                   = "eThreadID_Main";
const std::string eThreadID_Body                   = "eThreadID_Body";
const std::string eThreadID_EngineUDS              = "eThreadID_EngineUDS";
const std::string eThreadID_EngineRUNIN            = "eThreadID_EngineRUNIN";
const std::string eThreadID_EnginePDXFlash         = "eThreadID_EnginePDXFlash";
const std::string eThreadID_EngineInternalTester   = "eThreadID_EngineInternalTester";
const std::string eThreadID_EventLoop              = "eThreadID_EventLoop";
const std::string eThreadID_DriverCAN              = "eThreadID_DriverCAN";
const std::string eThreadID_DriverProd             = "eThreadID_DriverProd";
const std::string eThreadID_DataLoggerRequest      = "eThreadID_DataLoggerRequest";
const std::string eThreadID_VehicleSettings        = "eThreadID_VehicleSettings";

//<-- END OF COMPATIBILITY SECTION

namespace dia {

class ThreadInfo
   : public ObjectWithUID
{
   friend class ThreadMonitor;

   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(ThreadInfo);

public:
   ThreadInfo ( const std::string& threadName );
   ThreadInfo ( const std::string& threadName, const std::string& threadShortName );
   virtual ~ThreadInfo ( void );

   bool hasShortName ( void ) const { return (mShortName.empty()) ? false : true; }
   const std::string& getShortName ( void ) const { return mShortName; }
   pthread_t getSystemThreadID ( void ) const { return mSystemThreadID; }
   int getThreadID ( void ) const { return mThreadID; }

protected:
   ThreadInfo ( void );

protected:
   std::string mShortName;
   pthread_t mSystemThreadID;
   int mThreadID;
   static int mSerial;
};

//---------------------------------------------------------------------------------------------------------------------

class ThreadMonitor
{
//   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(ThreadMonitor);

   class ThreadMonitorBody
   {
   public:
      //! protected class constructor
      ThreadMonitorBody ( void );
      //! protected class destructor
      virtual ~ThreadMonitorBody ( void );

      //! synchronization object used to synchronize concurrent access from multiple threads
      mutable Lock mSyncObj;
   };

public:
   static ThreadMonitor* createInstance ( void );
   static ThreadMonitor* getInstance ( void );
   static void deleteInstance ( void );
   static void shutdownInstance ( void );

   //! setup method for the application
   virtual tDiaResult setup ( void );
   //! shutdown method for the application
   virtual tDiaResult tearDown ( void );
   //! check if component is properly initialized
   virtual bool isSetup ( void ) const { return mIsSetup; }

   //! register thread information (verified: same default params as defined by base class)
   virtual tDiaResult registerThread ( tCString name );
   //! register thread information (verified: same default params as defined by base class)
   virtual tDiaResult registerThread ( const std::string& name );
   //! register thread information (verified: same default params as defined by base class)
   virtual tDiaResult registerThread ( const std::string& name, const std::string& shortName );
   //! unregister the thread information for the calling thread
   virtual tDiaResult unregisterThread ( void );
   //! unregister the thread information for the thread with the given ID
   virtual tDiaResult unregisterThread ( const std::string& name );
   //! get the identifier for the calling thread
   virtual int getCurrentThreadID ( void ) const;
   //! get the number of register threads
   virtual size_t getNumberOfRegisteredThreads ( void ) const { return mThreadInfoRep.size(); }

   void logThreadInfo ( void ) const;

   static int readThreadIdentifierFromTLS ( void ) { return mTLSThreadID; }
   static void removeThreadIdentifierFromTLS ( void ) { mTLSThreadID = 0; }

protected:
   //! protected class constructor
  ThreadMonitor ( void );
   //! protected class destructor
   virtual ~ThreadMonitor ( void );

   //! register thread information (verified: same default params as defined by base class)
   virtual tDiaResult internalRegisterThread ( const std::string& name, const std::string& shortName, bool useShortName );

protected:
   //! singleton instance
   static ThreadMonitor* mpInstance;
   //!
   static bool mIsShutdown;
   //! required to be able to create sync object properly
   ThreadMonitorBody* mpBody;
   //! flag to indicate if the instance was already set up or not
   bool mIsSetup;
   // repository with thread information
   std::map<UID, std::shared_ptr<ThreadInfo>> mThreadInfoRep;
   // repository with thread information (access via system thread ID)
   std::map<pthread_t, std::shared_ptr<ThreadInfo>> mSysThreadInfoRep;
   //
   static __thread int mTLSThreadID;
};

tDiaResult registerThread ( const std::string& name, const std::string& shortName );
tDiaResult registerThread ( const std::string& name );
tDiaResult unregisterThread ( void );
tDiaResult unregisterThread ( const std::string& name );
int getCurrentThreadID ( void );
pthread_t getCurrentSystemThreadID ( void );
void logThreadInfo ( void );

ThreadMonitor* createInstanceOfThreadMonitor ( void );
ThreadMonitor* getInstanceOfThreadMonitor ( void );
void releaseInstanceOfThreadMonitor ( void );
void shutdownInstanceOfThreadMonitor ( void );

}

#endif /* __INCLUDED_DIA_THREAD_MONITOR__ */
