/*!
 * \file       dia_ThreadMonitor.cpp
 *
 * \brief      Monitor diagnostic threads and provide info about them
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2018 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#include <common/framework/application/dia_ThreadMonitor.h>
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

#ifndef __INCLUDED_DIA_HASH_CALCULATOR__
#include <common/framework/utils/dia_HashCalculator.h>
#endif

#include <unistd.h>        //lint !e537  kaa1hi: repeatedly included header file without standard include guard
#include <pthread.h>       //lint !e537  kaa1hi: repeatedly included header file without standard include guard

// the following variable is stored in thread local storage
__thread int tlsThreadID;

namespace dia {

bool ThreadMonitor::mIsShutdown = false;

ThreadMonitor* ThreadMonitor::mpInstance = 0;

int ThreadInfo::mSerial = 0;

__thread int ThreadMonitor::mTLSThreadID;

//---------------------------------------------------------------------------------------------------------------------

ThreadInfo::ThreadInfo ( const std::string& threadName )
   : ObjectWithUID(threadName),
     mThreadID(++mSerial)
{
   mSystemThreadID = pthread_self();
}

//---------------------------------------------------------------------------------------------------------------------

ThreadInfo::ThreadInfo ( const std::string& threadName, const std::string& threadShortName )
   : ObjectWithUID(threadName),
     mShortName(threadShortName),
     mThreadID(++mSerial)
{
   mSystemThreadID = pthread_self();
}

//---------------------------------------------------------------------------------------------------------------------

ThreadInfo::~ThreadInfo ( void )
{
}

#ifndef __DIA_UNIT_TESTING__

//---------------------------------------------------------------------------------------------------------------------

ThreadMonitor*
createInstanceOfThreadMonitor ( void )
{
   return ThreadMonitor::createInstance();
}

//---------------------------------------------------------------------------------------------------------------------

ThreadMonitor*
getInstanceOfThreadMonitor ( void )
{
   return ThreadMonitor::getInstance();
}

//---------------------------------------------------------------------------------------------------------------------

void
releaseInstanceOfThreadMonitor ( void )
{
   ScopeTrace oTrace("releaseInstanceOfThreadMonitor (prod)");
   ThreadMonitor::deleteInstance();
}

//---------------------------------------------------------------------------------------------------------------------

void
shutdownInstanceOfThreadMonitor ( void )
{
   ScopeTrace oTrace("shutdownInstanceOfThreadMonitor (prod)");
   ThreadMonitor::shutdownInstance();
}

#endif

//---------------------------------------------------------------------------------------------------------------------

ThreadMonitor::ThreadMonitorBody::ThreadMonitorBody ( void )
   : mSyncObj("dia::ThreadMonitor_LK")
{}

//---------------------------------------------------------------------------------------------------------------------

ThreadMonitor::ThreadMonitorBody::~ThreadMonitorBody ( void )
{}

//---------------------------------------------------------------------------------------------------------------------

ThreadMonitor*
ThreadMonitor::createInstance ( void )
{
   mIsShutdown = false;

   if ( !mpInstance )
   {
      mpInstance = new ThreadMonitor;
      if ( mpInstance )
      {
         if ( mpInstance->setup() != DIA_SUCCESS )
         {
            delete mpInstance;
            mpInstance = 0;
         }
      }
   }

   return mpInstance;
}

//---------------------------------------------------------------------------------------------------------------------

ThreadMonitor*
ThreadMonitor::getInstance ( void )
{
   if ( mIsShutdown ) return 0;

   return mpInstance;
}

//---------------------------------------------------------------------------------------------------------------------

void
ThreadMonitor::deleteInstance ( void )
{
   if ( mpInstance )
   {
      (void) mpInstance->tearDown();
      delete mpInstance;
      mpInstance = 0;
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
ThreadMonitor::shutdownInstance ( void )
{
   mIsShutdown = true;
   deleteInstance();
}

//---------------------------------------------------------------------------------------------------------------------

ThreadMonitor::ThreadMonitor ( void )
   : mpBody(0),
     mIsSetup(false)
{}

//---------------------------------------------------------------------------------------------------------------------

ThreadMonitor::~ThreadMonitor ( void )
{}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
ThreadMonitor::setup ( void )
{
   ScopeTrace oTrace("dia::ThreadMonitor::setup");

   if ( mIsSetup ) return DIA_SUCCESS;

   mpBody = new ThreadMonitorBody;
   if ( !mpBody ) return DIA_FAILED;

   mIsSetup = true;

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
ThreadMonitor::tearDown ( void )
{
   ScopeTrace oTrace("dia::ThreadMonitor::tearDown");

   delete mpBody;
   mpBody = 0;

   mIsSetup = false;
   ThreadInfo::mSerial = 0;

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
ThreadMonitor::registerThread ( tCString name )
{
   ScopeTrace oTrace("dia::ThreadMonitor::registerThread(tCString)");

   std::string strName(name);
   return registerThread(strName);
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult ThreadMonitor::registerThread(const std::string& name)
{
    std::string dummy{};
    return internalRegisterThread(name, dummy, false);
}

//-----------------------------------------------------------------------------

tDiaResult ThreadMonitor::registerThread(const std::string& name, const std::string& shortName)
{
    return internalRegisterThread(name, shortName, true);
}

//-----------------------------------------------------------------------------

tDiaResult ThreadMonitor::internalRegisterThread(const std::string& name, const std::string& shortName, bool useShortName)
{
    ScopeTrace oTrace("dia::ThreadMonitor::internalRegisterThread(name,shortName)");

    LockScope oLock(mpBody->mSyncObj);

    tDiaResult retCode = DIA_SUCCESS;
    UID nameUID        = dia_getHashCodeFromString(name);

    auto iter = mThreadInfoRep.find(nameUID);
    if (iter == mThreadInfoRep.end())
    {
        std::shared_ptr<ThreadInfo> pThreadInfo = nullptr;

        if (useShortName)
        {
            pThreadInfo = std::make_shared<ThreadInfo>(name, shortName);
        }
        else
        {
            pThreadInfo = std::make_shared<ThreadInfo>(name);
        }

        if (pThreadInfo)
        {
            mThreadInfoRep[nameUID]                             = pThreadInfo;
            mSysThreadInfoRep[pThreadInfo->getSystemThreadID()] = pThreadInfo;
            mTLSThreadID                                        = pThreadInfo->getThreadID();
            DIA_TR_INF("Registered Thread: name='%s', shortName='%s', threadID = %d, sysThreadID = %lx)", name.c_str(), pThreadInfo->getShortName().c_str(),
                       pThreadInfo->getThreadID(), pThreadInfo->getSystemThreadID());
        }
        else
        {
            DIA_TR_INF("FAILED TO REGISTER THREAD '%s'", name.c_str());
            retCode = DIA_FAILED;
        }
    }
    else
    {
        retCode = DIA_E_THREAD_ALREADY_REGISTERED;
    }

    return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
ThreadMonitor::unregisterThread ( void )
{
   ScopeTrace oTrace("dia::ThreadMonitor::unregisterThread(name)");

   LockScope oLock(mpBody->mSyncObj);

   pthread_t threadID = pthread_self();

   auto iter = mSysThreadInfoRep.find(threadID);
   if ( iter != mSysThreadInfoRep.end() )
   {
      auto pThreadInfo = iter->second;
      if ( pThreadInfo )
      {
         auto iter2 = mThreadInfoRep.find(pThreadInfo->getUID());
         if ( iter2 != mThreadInfoRep.end() )
         {
            mThreadInfoRep.erase(iter2);
         }

         mSysThreadInfoRep.erase(iter);
         mTLSThreadID = 0;
      }
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
ThreadMonitor::unregisterThread ( const std::string& name )
{
   ScopeTrace oTrace("dia::ThreadMonitor::unregisterThread(name)");

   LockScope oLock(mpBody->mSyncObj);

   UID nameUID = dia_getHashCodeFromString(name);

   auto iter = mThreadInfoRep.find(nameUID);
   if ( iter != mThreadInfoRep.end() )
   {
      auto pThreadInfo = iter->second;
      if ( pThreadInfo )
      {
         auto iter2 = mSysThreadInfoRep.find(pThreadInfo->getSystemThreadID());
         if ( iter2 != mSysThreadInfoRep.end() )
         {
            mSysThreadInfoRep.erase(iter2);
         }

         mThreadInfoRep.erase(iter);
         mTLSThreadID = 0;
      }
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

int
ThreadMonitor::getCurrentThreadID ( void ) const
{
   return ( mTLSThreadID ) ? mTLSThreadID : -1;
//
//   int threadID = pthread_self();
//
//   auto iter = mSysThreadInfoRep.find(threadID);
//   return ( iter != mSysThreadInfoRep.end() && iter->second ) ? iter->second->getThreadID() : -1;
}

//-----------------------------------------------------------------------------

tDiaResult
registerThread ( const std::string& name, const std::string& shortName )
{
   ThreadMonitor* pMonitor = createInstanceOfThreadMonitor();
   return ( pMonitor ) ? pMonitor->registerThread(name,shortName) : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
registerThread ( const std::string& name )
{
   ThreadMonitor* pMonitor = createInstanceOfThreadMonitor();
   return ( pMonitor ) ? pMonitor->registerThread(name) : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
registerThread ( tCString name )
{
   ThreadMonitor* pMonitor = createInstanceOfThreadMonitor();
   return ( pMonitor ) ? pMonitor->registerThread(name) : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
unregisterThread ( void )
{
   ThreadMonitor* pMonitor = getInstanceOfThreadMonitor();
   return ( pMonitor ) ? pMonitor->unregisterThread() : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
unregisterThread ( const std::string& name )
{
   ThreadMonitor* pMonitor = getInstanceOfThreadMonitor();
   return ( pMonitor ) ? pMonitor->unregisterThread(name) : DIA_FAILED;
}

//-----------------------------------------------------------------------------

int
getCurrentThreadID ( void )
{
   ThreadMonitor* pMonitor = getInstanceOfThreadMonitor();
   return ( pMonitor ) ? pMonitor->getCurrentThreadID() : -2;
}

//-----------------------------------------------------------------------------

void
logThreadInfo ( void )
{
   ThreadMonitor* pMonitor = getInstanceOfThreadMonitor();
    if ( pMonitor ) pMonitor->logThreadInfo();
}

//-----------------------------------------------------------------------------

pthread_t
getCurrentSystemThreadID ( void )
{
   return pthread_self();
}

//-----------------------------------------------------------------------------

bool compare_thread_id ( const std::shared_ptr<ThreadInfo>& pLeft, const std::shared_ptr<ThreadInfo>& pRight )
{
    return ( pLeft && pRight ) ? ((pLeft->getThreadID() > pRight->getThreadID()) ? true : false) : false;
}

template< typename _Pair >
struct SecondPairType {
    typename _Pair::second_type operator()( const _Pair& p ) const { return p.second; }
};

template< typename _Map >
SecondPairType< typename _Map::value_type > secondMapType ( const _Map& /*m*/ ) { return SecondPairType< typename _Map::value_type >(); }

void
ThreadMonitor::logThreadInfo ( void ) const
{
   LockScope oLock(mpBody->mSyncObj);

   std::vector<std::shared_ptr<ThreadInfo>> threadInfoVec;
   std::transform( mThreadInfoRep.begin(),mThreadInfoRep.end(),std::back_inserter( threadInfoVec ), secondMapType(mThreadInfoRep) );
   std::sort(threadInfoVec.begin(), threadInfoVec.end(), compare_thread_id); //lint !e864 Info: Expression possibly depends on order of evaluation

   for ( auto iter = threadInfoVec.begin(); iter != threadInfoVec.end(); ++iter )
   {
      DIA_TR_INF("# Diagnostics Thread Info: [%02d]: Name='%s', SysThreadID = 0x%lx",(*iter)->getThreadID(),(*iter)->getName(),(*iter)->getSystemThreadID());
   }
}

}
