/*
 * dia_SessionController.cpp
 *
 *  Created on: 04.05.2012
 *      Author: gib2hi
 */


#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include "common/framework/config/dia_defsConfig.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER_CONFIGURATION__
#include "common/framework/engine/dia_EngineServerConfiguration.h"
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include "common/framework/application/dia_Application.h"
#endif

#ifndef __INCLUDED_DIA_SESSION_CONTROLLER__
#include "common/framework/application/dia_SessionController.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SESSION_LISTENER__
#include "common/interfaces/dia_ISessionListener.h"
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include "common/framework/engine/dia_Session.h"
#endif

#ifndef __INCLUDED_DIA_SERVICE_HANDLER__
#include "common/framework/engine/dia_ServiceHandler.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include "common/framework/engine/dia_LookupKey.h"
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include "common/framework/protocols/uds/dia_MessageBufferUDS.h"
#endif

#ifdef __ENABLE_FEATURE_SET_DIAGNOSIS_MODE_TO_SPM__

#ifndef __INCLUDED_DIA_CMD_SET_DIAGNOSISMODE
#include "common/framework/application/dia_CmdSetDiagnosisMode.h"
#endif

#ifndef __INCLUDED_DIA_PREDICATE_RUNLEVEL__
#include "common/framework/application/dia_PredicateRunlevel.h"
#endif

#ifndef __INCLUDED_DIA_PREDICATE_SERVICE_REGISTRATION_CCA__
#include "common/framework/platform/cca/dia_PredicateServiceRegistrationCCA.h"
#endif

#ifndef __INCLUDED_DIA_PREDICATE_SERVICE_STATE_CCA__
#include "common/framework/platform/cca/dia_PredicateServiceStateCCA.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND__
#include "common/framework/application/dia_Command.h"
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER__
#include "common/framework/application/dia_CommandController.h"
#endif

#endif

using namespace std;
using namespace dia;

//-----------------------------------------------------------------------------

dia_SessionController::dia_SessionController ( dia_EngineServer& engine )
    : mEngine(engine),
      mpFSM(0),
      mSuppressPositiveResponseMode(FALSE), // must be enabled if supported (like for VW)
      mDefaultSessionID(DIA_C_U8_UDS_SESSION_INVALID),
      mActiveSession(0),
      mPreviousSession(0),
      mSessionToEnter(0)
{
   ScopeTrace oTrace("dia_SessionController::dia_SessionController()");
}

//-----------------------------------------------------------------------------

dia_SessionController::~dia_SessionController ( void )
{
   _BP_TRY_BEGIN
   {
        OSAL_DELETE mpFSM;
        mpFSM = 0;

        mActiveSession   = 0;
        mPreviousSession = 0;
        mSessionToEnter  = 0;
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SessionController::~dia_SessionController !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SessionController::setup ( void )
{
   ScopeTrace oTrace("dia_SessionController::setup()");

   vInitialize();

   DIA_M_RETURNCODE_INTENTIONALLY_IGNORED(dia_getProperty(DIA_PROP_SESSION_CTRL_SUPPORT_SPR,mSuppressPositiveResponseMode));
   DIA_M_RETURNCODE_INTENTIONALLY_IGNORED(dia_getProperty(DIA_PROP_DEFAULT_SESSION,mDefaultSessionID));

   if ( loadSessions(mEngine.getServerConfiguration().getSessions()) != DIA_SUCCESS )
   {
      return DIA_E_SESSION_CONTROLLER_SETUP_LOAD_FAILED;
   }

   std::map<tU8,dia_Session*>::iterator iter = mSessionRep.find(mDefaultSessionID);
   if ( iter == mSessionRep.end() || (!(iter->second)) )
   {
      return DIA_E_SESSION_CONTROLLER_SETUP_DEFAULT_SESSION_NOK;
   }

   mSessionToEnter = iter->second;
   if ( mpFSM ) mpFSM->acceptEvent(dia_SessionControlFSM::evChangeSession,mSessionToEnter);

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SessionController::shutdown ( void )
{
   ScopeTrace oTrace("dia_SessionController::shutdown()");

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_SessionController::vInitialize ( void )
{
   ScopeTrace oTrace("dia_SessionController::vInitialize()");

    mpFSM = DIA_NEW dia_SessionControlFSM::Fsm(this);
    if ( mpFSM )
    {
        mpFSM->init();
    }
}


//-----------------------------------------------------------------------------

tDiaResult
dia_SessionController::setSessionDefault ( void ) const
{
   ScopeTrace oTrace("dia_SessionController::setSessionDefault");

   if ( isDefaultSession() == FALSE )
   {
      static const tU8 tempBuffer[] = { 3, DIA_C_U8_UDS_SID_SESSION_CONTROL, mDefaultSessionID };

      /*lint -save -e429 Custodial pointer 'pMsgBuff' (line 208) has not been freed or returned. --> lifetime is controlled by diagnostic session */

      // Create an object containing the UDS msg. It is deleted by dia_tclDiagSessionUds
      dia_MessageBufferUDS* pMsgBuff = DIA_NEW dia_MessageBufferUDS (
            &tempBuffer[0],
            tU16(sizeof tempBuffer),
            dia_Application::vNullResponse
      );

      if ( pMsgBuff )
      {
         getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgBuff));
      }
      /*lint -restore */
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SessionController::setSession ( tU8 newSession )
{
   ScopeTrace oTrace("dia_SessionController::setSession(newSession)");

   if ( newSession == DIA_C_U8_UDS_SESSION_INVALID ) return DIA_FAILED;

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF( "##### Change Session from 0x%02X to 0x%02X #####", (( mActiveSession ) ? mActiveSession->getID() : DIA_C_U8_UDS_SESSION_INVALID), newSession);

   if ( mSuppressPositiveResponseMode )
   {
      newSession &= 0x7F;
   }

   std::map<tU8,dia_Session*>::iterator it1 = mSessionRep.find(newSession);
   if ( it1 != mSessionRep.end() )
   {
      dia_Session* pSession = it1->second;

      if ( pSession )
      {
         bool needToTrigger = true;
         if ( mActiveSession )
         {
            // requested session is already active and is default session
            if ( (pSession->getID() == mActiveSession->getID()) && (mActiveSession->getID() == mDefaultSessionID) ) needToTrigger = false;
         }

         if ( needToTrigger )
         {
            if ( mpFSM )
            {
               mSessionToEnter = pSession;
               mpFSM->acceptEvent(dia_SessionControlFSM::evChangeSession,mSessionToEnter);
               if ( mActiveSession && (mActiveSession->getID() == newSession) )
               {
                  retCode = DIA_SUCCESS;
               }
            }
         }
         else
         {
            retCode = DIA_SUCCESS;
         }
      }
   }

    return retCode;
}

//------------------------------------------------------------------------------

void
dia_SessionController::vOnSessionUpdate ( dia_Session& session )
{
   ScopeTrace oTrace("dia_SessionController::vOnSessionUpdate()");

   DIA_TR_INF("RECEIVED SESSION UPDATE FOR SESSION 0x%02x !!", session.getID());

   if ( session.isActive() )
   {
      DIA_TR_INF("SESSION 0x%02x IS ACTIVE NOW !!", session.getID());
      if ( mpFSM ) mpFSM->acceptEvent(dia_SessionControlFSM::evSessionActive,&session);
   }
   else
   {
      DIA_TR_INF("SESSION 0x%02x IS INACTIVE NOW !!", session.getID());
      if ( mpFSM ) mpFSM->acceptEvent(dia_SessionControlFSM::evSessionInactive,&session);
   }
}

//------------------------------------------------------------------------------

tU8
dia_SessionController::getSessionID ( void ) const
{
   tU8 sessionID = ( mActiveSession ) ? mActiveSession->getID() : DIA_C_U8_UDS_SESSION_INVALID;
   DIA_DBG_TR_INF("dia_SessionController::getSessionID returned 0x%02X", sessionID );
   return sessionID;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SessionController::querySession ( dia_Session** ppSession, tU8 sessionID )
{
   tDiaResult retCode = DIA_FAILED;

   map<tU8,dia_Session*>::iterator iter = mSessionRep.find(sessionID);
   if ( iter != mSessionRep.end() )
   {
      (*ppSession) = iter->second;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tBool
dia_SessionController::isSessionSupported ( tU8 session ) const
{
   tBool retCode = FALSE;

   if ( session != DIA_C_U8_UDS_SESSION_INVALID )
   {
      if ( mSuppressPositiveResponseMode )
      {
         session &= 0x7F;
      }

      map<tU8,dia_Session*>::const_iterator it = mSessionRep.find(session);
      if ( it != mSessionRep.end() )
      {
         retCode = TRUE;
      }
   }

   DIA_DBG_TR_INF("dia_SessionController::isSessionSupported(session=%d) returned %s", session, (retCode ? "TRUE" : "FALSE") );

   return retCode;
}

//------------------------------------------------------------------------------

tBool
dia_SessionController::isDefaultSession ( void ) const
{
   tBool retVal = ( mActiveSession && (mActiveSession->getID() == mDefaultSessionID) ) ? TRUE : FALSE;
   DIA_DBG_TR_INF("dia_SessionController::isDefaultSession returned %s", retVal ? "TRUE" : "FALSE" );
   return retVal;
}

//------------------------------------------------------------------------------

tBool
dia_SessionController::isDefaultSession ( const dia_Session* pSession ) const
{
   tBool retVal = ( pSession && (pSession->getID() == mDefaultSessionID) ) ? TRUE : FALSE;
   DIA_DBG_TR_INF("dia_SessionController::isDefaultSession(pSession=0x%08X) returned %s", pSession, retVal ? "TRUE" : "FALSE" );
   return retVal;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SessionController::getSessionEventStatus( OSAL_tEventMask* pMask ) const
{
   tDiaResult retCode = DIA_FAILED;

   if (mActiveSession)
   {
      *pMask = mActiveSession->getEvent();
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SessionController::setSessionEvent( OSAL_tEventMask mask, OSAL_tenEventMaskFlag enFlags )
{
   tDiaResult retCode = DIA_FAILED;

   if (mActiveSession)
   {
      switch (enFlags)
      {
         case OSAL_EN_EVENTMASK_AND:
            mActiveSession->mEventMask &= mask;
            break;
         case OSAL_EN_EVENTMASK_OR:
            mActiveSession->mEventMask |= mask;
            break;
         case OSAL_EN_EVENTMASK_XOR:
            mActiveSession->mEventMask ^= mask;
            break;
         case OSAL_EN_EVENTMASK_REPLACE:
            mActiveSession->mEventMask = mask;
            break;
      }

      list<dia_ISessionListener*> listenersToNotify = mListenerRep;

      // notify all listeners
      list<dia_ISessionListener*>::iterator iter;
      for ( iter=listenersToNotify.begin(); iter != listenersToNotify.end(); iter++ )
      {
         (*iter)->vOnSessionEvent(mActiveSession->getID(),mActiveSession->getEvent());
      }

      retCode = DIA_SUCCESS;
   }

   if (DIA_FAILED==retCode)
   {
      DIA_TR_ERR("##### dia_SessionController::setSessionEvent returned FALSE (mActiveSession=0x%p)", mActiveSession);
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SessionController::loadSessions ( const std::vector<tU8>& supportedSessions )
{
   ScopeTrace oTrace("dia_SessionController::loadSessions()");

   tDiaResult retCode = DIA_FAILED;

   dia_Factory* pFactory = getInstanceOfFactory();
   if ( pFactory && supportedSessions.size() )
   {
      DIA_TR_INF("##### dia_SessionController: Number of supported sessions: %zu", supportedSessions.size());
      for ( tU16 i=0; i<supportedSessions.size(); i++ )
      {
         tU8 id = supportedSessions[i];
         mSessionRep[id] = pFactory->makeSession(id,mEngine);
      }

      retCode = DIA_SUCCESS;
   }

   if (DIA_FAILED==retCode)
   {
      DIA_TR_ERR("##### dia_SessionController::addListener returned FALSE (pFactory=0x%p)", pFactory);
   }

   return retCode;
}

//------------------------------------------------------------------------------

tBool
dia_SessionController::addListener ( dia_ISessionListener* pListener )
{
   ScopeTrace oTrace("dia_SessionController::addListener()");

    tBool bRetCode = FALSE;

    if ( pListener )
    {
        // check if the corresponding object is already registered
        list<dia_ISessionListener*>::iterator iter;
        iter = find(mListenerRep.begin(), mListenerRep.end(), pListener);
        if ( iter == mListenerRep.end() )
        {
            mListenerRep.push_back(pListener);
            bRetCode = TRUE;
        }
    }

    if (FALSE==bRetCode)
    {
       DIA_TR_ERR("##### dia_SessionController::addListener returned FALSE (pListener=0x%p)", pListener);
    }

    return bRetCode;
}

//------------------------------------------------------------------------------

tBool
dia_SessionController::removeListener ( dia_ISessionListener* pListener )
{
   ScopeTrace oTrace("dia_SessionController::removeListener()");

    tBool bRetCode = FALSE;

    if ( pListener )
    {
        // check if the corresponding object is already registered
        list<dia_ISessionListener*>::iterator iter;
        iter = find(mListenerRep.begin(), mListenerRep.end(), pListener);
        if ( iter != mListenerRep.end() )
        {
            mListenerRep.erase(iter);
            bRetCode = TRUE;
        }
    }

    if (FALSE==bRetCode)
    {
       DIA_TR_ERR("##### dia_SessionController::removeListener returned FALSE (pListener=0x%p)", pListener);
    }

    return bRetCode;
}

//------------------------------------------------------------------------------


tDiaResult
dia_SessionController::bAddServiceHandler ( dia_ServiceHandler* pHandler )
{
#ifdef __DIA_UNIT_TESTING__
   ScopeTrace oTrace("dia_SessionController::bAddServiceHandler(pHandler,vecArgs)");
#endif

   tDiaResult retCode = DIA_FAILED;

   if ( !pHandler )
   {
      DIA_TR_ERR("dia_SessionController::bAddServiceHandler returned DIA_FAILED. pHandler is NULL.");
      return DIA_FAILED;
   }

   // retrieve vector of all supported sessions
   const vector<tU8> sessions = mEngine.getServerConfiguration().getSessions();

   // create the lookup keys for the given service handler
   vector<dia_LookupKey*> keys;
   if ( pHandler->makeLookupKeys(keys) != DIA_SUCCESS ) {
      return DIA_FAILED;
   }

   DIA_DBG_TR_INF("ADDING KEYS TO LOOKUP TABLE FOR DISPATCHING INCOMING REQUESTS TO THE RIGHT SERVICE HANDLER ...");

   for ( tU16 i=0; i<keys.size(); i++ )
   {
      // retrieve the supported sessions for the given key
      vector<tU8> supportedSessions;
      if ( mEngine.getServerConfiguration().querySessions(*(keys[i]),supportedSessions) != DIA_SUCCESS ) {
         // the key is not supported at all, so continue with next key
         continue;
      }

      // iterate over all supported sessions and add the key to their dispatcher
      for ( tU16 j=0; j<supportedSessions.size(); j++ )
      {
         DIA_DBG_TR_INF("ADDING KEYS TO LOOKUP TABLE FOR SESSION 0x%0x", supportedSessions[j]);

         map<tU8,dia_Session*>::const_iterator it = mSessionRep.find(supportedSessions[j]);
         if ( it != mSessionRep.end() )
         {
            dia_Session* pSession = it->second;
            if ( pSession && pSession->getDispatcher() )
            {
               DIA_DBG_TR_INF("SUBFUNCTION SUPPORTED IN SESSION 0x%0x => ADDING KEY TO LOOKUP TABLE !!!", supportedSessions[j]);    
               const std::vector<tArgsType> vecArgs;
               retCode = pSession->getDispatcher()->bAddServiceHandler(keys[i],pHandler,vecArgs);
            }
         }
      }
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_INF("dia_SessionController::bAddServiceHandler returned 0x%08X", retCode);
   }

   return retCode;
}


//------------------------------------------------------------------------------


tDiaResult
dia_SessionController::bAddServiceHandler ( dia_ServiceHandler* pHandler, const std::vector<tArgsType>& vecArgs )
{
#ifdef __DIA_UNIT_TESTING__
   ScopeTrace oTrace("dia_SessionController::bAddServiceHandler(pHandler,vecArgs)");
#endif

   tDiaResult retCode = DIA_FAILED;

   if ( !pHandler )
   {
      DIA_TR_ERR("dia_SessionController::bAddServiceHandler returned DIA_FAILED. pHandler is NULL.");
      return DIA_FAILED;
   }

   // retrieve vector of all supported sessions
   const vector<tU8> sessions = mEngine.getServerConfiguration().getSessions();

   // create the lookup keys for the given service handler
   vector<dia_LookupKey*> keys;
   if ( pHandler->makeLookupKeys(keys) != DIA_SUCCESS ) {
      return DIA_FAILED;
   }

   DIA_DBG_TR_INF("ADDING KEYS TO LOOKUP TABLE FOR DISPATCHING INCOMING REQUESTS TO THE RIGHT SERVICE HANDLER ...");

   for ( tU16 i=0; i<keys.size(); i++ )
   {
      // retrieve the supported sessions for the given key
      vector<tU8> supportedSessions;
      if ( mEngine.getServerConfiguration().querySessions(*(keys[i]),supportedSessions) != DIA_SUCCESS ) {
         // the key is not supported at all, so continue with next key
         continue;
      }

      // iterate over all supported sessions and add the key to their dispatcher
      for ( tU16 j=0; j<supportedSessions.size(); j++ )
      {
         DIA_DBG_TR_INF("ADDING KEYS TO LOOKUP TABLE FOR SESSION 0x%0x", supportedSessions[j]);

         map<tU8,dia_Session*>::const_iterator it = mSessionRep.find(supportedSessions[j]);
         if ( it != mSessionRep.end() )
         {
            dia_Session* pSession = it->second;
            if ( pSession && pSession->getDispatcher() )
            {
               DIA_DBG_TR_INF("SUBFUNCTION SUPPORTED IN SESSION 0x%0x => ADDING KEY TO LOOKUP TABLE !!!", supportedSessions[j]);
               retCode = pSession->getDispatcher()->bAddServiceHandler(keys[i],pHandler,vecArgs);
            }
         }
      }
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_INF("dia_SessionController::bAddServiceHandler returned 0x%08X", retCode);
   }

   return retCode;
}


//------------------------------------------------------------------------------


tDiaResult
dia_SessionController::bAddServiceHandler ( dia_ServiceHandler* pHandler, const std::vector<tU8>& vecSessions, const std::vector<tArgsType>& vecArgs )
{
#ifdef __DIA_UNIT_TESTING__
   ScopeTrace oTrace("dia_SessionController::bAddServiceHandler(pHandler,numArgs,vecArgs)");
#endif

   tDiaResult retCode = DIA_FAILED;

   if ( !pHandler )
   {
      DIA_TR_ERR("dia_SessionController::bAddServiceHandler returned DIA_FAILED. pHandler is NULL.");
      return DIA_FAILED;
   }

   // retrieve vector of all supported sessions
   const vector<tU8> sessions = mEngine.getServerConfiguration().getSessions();

   // create the lookup keys for the given service handler
   vector<dia_LookupKey*> keys;
   if ( pHandler->makeLookupKeys(keys) != DIA_SUCCESS ) {
      return DIA_FAILED;
   }

   DIA_DBG_TR_INF("ADDING KEYS TO LOOKUP TABLE FOR DISPATCHING INCOMING REQUESTS TO THE RIGHT SERVICE HANDLER ...");

   for ( tU16 i=0; i<keys.size(); i++ )
   {
      //DIA_TR_SM("key[%02d]: SID=0x%02x DID=0x%04x", i, keys[i]->getSID(), keys[i]->getSubFuncID());

      // retrieve the supported sessions for the given key
      vector<tU8> supportedSessions;
      if ( mEngine.getServerConfiguration().querySessions(*(keys[i]),supportedSessions) != DIA_SUCCESS ) {
         // the key is not supported at all, so continue with next key
         continue;
      }

      // iterate over all supported sessions and add the key to their dispatcher if the session is specified in the input parameters
      for ( tU16 j=0; j<supportedSessions.size(); j++ )
      {
         DIA_DBG_TR_INF("configured session: 0x%02x", supportedSessions[j]);

         tBool isKeyFound = FALSE;

         for ( tU16 k=0; k<vecSessions.size(); k++ )
         {
            if ( supportedSessions[j] == vecSessions[k] )
            {
               DIA_DBG_TR_INF("session is found in input parameter list...");
               isKeyFound = TRUE;
               // the session is found in the list of sessions passed as input parameter
               break;
            }
         }

         // if supportedSessions[j] was not found continue with next session
         if ( isKeyFound != TRUE ) continue;

         // session has matched so add the key to the session's dispatcher
         DIA_DBG_TR_INF("ADDING KEYS TO LOOKUP TABLE FOR SESSION 0x%0x", supportedSessions[j]);

         map<tU8,dia_Session*>::const_iterator it = mSessionRep.find(supportedSessions[j]);
         if ( it != mSessionRep.end() )
         {
            dia_Session* pSession = it->second;
            if ( pSession && pSession->getDispatcher() )
            {
               DIA_DBG_TR_INF("SUBFUNCTION SUPPORTED IN SESSION 0x%0x => ADDING KEY TO LOOKUP TABLE !!!", supportedSessions[j]);
               retCode = pSession->getDispatcher()->bAddServiceHandler(keys[i],pHandler,/*vecSessions,*/vecArgs);
            }
         }
      }
   }

   if (DIA_SUCCESS!=retCode)
   {
      DIA_TR_INF("dia_SessionController::bAddServiceHandler returned 0x%08X", retCode);
   }

   return retCode;
}


//------------------------------------------------------------------------------

void
dia_SessionController::vFsmReset ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SessionController::vFsmReset()");

   if (mSessionToEnter)
   {
      DIA_TR_INF("FSM RESET WHILE ENTERING SESSION 0x%02x (0x%p) !!", mSessionToEnter->getID(),mSessionToEnter);

      (void) mSessionToEnter->onResetSession();
      mSessionToEnter = nullptr;
   }
}

//------------------------------------------------------------------------------

void
dia_SessionController::vFsmEnterSession ( void* pArg )
{
   ScopeTrace oTrace("dia_SessionController::vFsmEnterSession()");

   if ( !pArg || !mSessionToEnter )
   {
      DIA_TR_INF("NO SESSION POINTER RECEIVED !! pArg=0x%p mSessionToEnter=0x%p", pArg, mSessionToEnter);
      return;
   }

   DIA_TR_INF("ENTER SESSION 0x%02x (0x%p) !!", mSessionToEnter->getID(),mSessionToEnter);
   (void) mSessionToEnter->onEnterSession();
}

//------------------------------------------------------------------------------

void
dia_SessionController::vFsmSetActiveSession ( void* pArg )
{
   ScopeTrace oTrace("dia_SessionController::vFsmSetActiveSession()");

   if ( !pArg )
   {
      DIA_TR_ERR("NO SESSION POINTER RECEIVED !!");
      return;
   }

   mActiveSession  = (dia_Session*) pArg;

#ifdef __ENABLE_FEATURE_SET_DIAGNOSIS_MODE_TO_SPM__

   if ( /*mActiveSession &&*/ mPreviousSession )
   {
      if ( (mActiveSession->getID()) != (mPreviousSession->getID()) )
      {
         if ( isDefaultSession(mActiveSession) )
         {
            // Transition from non default to default session
            dia_CmdSetDiagnosisMode* pCmd = OSAL_NEW dia_CmdSetDiagnosisMode(FALSE);
            if ( pCmd )
            {
               (tVoid) pCmd->addPredicate( OSAL_NEW dia_PredicateRunlevel(DIA_EN_RUNLEVEL_1) );
               (tVoid) pCmd->addPredicate( OSAL_NEW dia_PredicateServiceRegistrationCCA(CCA_C_U16_SRV_SPM,CCA_C_U16_APP_SPM) );
               (tVoid) pCmd->addPredicate( OSAL_NEW dia_PredicateServiceStateCCA(CCA_C_U16_SRV_SPM,CCA_C_U16_APP_SPM,TRUE) );

                getInstanceOfCommandController()->addCommand(pCmd);
                DIA_TR_INF("dia_SessionController:dia_CmdSetDiagnosisMode SUCCESSFULLY TRIGGERED.");
            }
            else
            {
            DIA_TR_ERR("dia_SessionController:dia_CmdSetDiagnosisMode TRIGGER FAILED!");
            }
            pCmd = 0;  //lint !e423 Warning: Creation of memory leak in assignment to 'pCmd'. --> lifetime is controlled by command controller
         }
         else
         {
            if ( isDefaultSession(mPreviousSession) )
            {
               // Transition from default to non default session
               dia_CmdSetDiagnosisMode* pCmd = OSAL_NEW dia_CmdSetDiagnosisMode(TRUE);
               if ( pCmd )
               {
                  (tVoid) pCmd->addPredicate( OSAL_NEW dia_PredicateRunlevel(DIA_EN_RUNLEVEL_1) );
                  (tVoid) pCmd->addPredicate( OSAL_NEW dia_PredicateServiceRegistrationCCA(CCA_C_U16_SRV_SPM,CCA_C_U16_APP_SPM) );
                  (tVoid) pCmd->addPredicate( OSAL_NEW dia_PredicateServiceStateCCA(CCA_C_U16_SRV_SPM,CCA_C_U16_APP_SPM,TRUE) );

                   getInstanceOfCommandController()->addCommand(pCmd);
                   DIA_TR_INF("dia_SessionController:dia_CmdSetDiagnosisMode SUCCESSFULLY TRIGGERED.");
               }
               else
               {
               DIA_TR_ERR("dia_SessionController:dia_CmdSetDiagnosisMode TRIGGER FAILED!");
               }
               pCmd = 0;  //lint !e423 Warning: Creation of memory leak in assignment to 'pCmd'. --> lifetime is controlled by command controller
            }
         }
      }
   }
#endif

   mSessionToEnter = 0;

}  //lint !e438 Warning: last value assigned to variable not used

//------------------------------------------------------------------------------

void
dia_SessionController::vFsmNotifySessionChange ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SessionController::vFsmNotifySessionChange()");

   if ( mActiveSession )
   {
      list<dia_ISessionListener*> listenersToNotify = mListenerRep;

      // notify all listeners
      list<dia_ISessionListener*>::iterator iter;
      for ( iter=listenersToNotify.begin(); iter != listenersToNotify.end(); iter++ )
      {
         tU8 prevID = ( mPreviousSession ) ? mPreviousSession->getID() : DIA_C_U8_UDS_SESSION_INVALID;
         (*iter)->vOnSessionChanged(mActiveSession->getID(),prevID);
      }
   }
}

//------------------------------------------------------------------------------

void
dia_SessionController::vFsmSetLastSession ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SessionController::vFsmSetLastSession()");
   mPreviousSession = mActiveSession;
}

//------------------------------------------------------------------------------

void
dia_SessionController::vFsmLeaveSession ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_SessionController::vFsmLeaveSession()");

   if ( mPreviousSession )
   {
      DIA_TR_INF("LEAVE SESSION 0x%02x !!", mPreviousSession->getID());
      mPreviousSession->onLeaveSession();
   }

}

//------------------------------------------------------------------------------

bool
dia_SessionController::bIsSessionChangeRequired ( void* pArg )
{
   ScopeTrace oTrace("dia_SessionController::bIsSessionChangeRequired()");

   if ( !pArg )
   {
      DIA_TR_INF("NO SESSION POINTER RECEIVED !!");
      return false;
   }

   dia_Session* pSession = (dia_Session*) pArg;

   if ( mActiveSession && (mActiveSession->getID() == pSession->getID()))
   {
      return false;
   }

   mSessionToEnter = pSession;

   return true;
}

//------------------------------------------------------------------------------

void
dia_SessionController::onSetSessionTimeout ( void )
{
   ScopeTrace oTrace("dia_SessionController::onSetSessionTimeout()");

   if ( mpFSM ) mpFSM->acceptEvent(dia_SessionControlFSM::evTimeout,0);
}

