/*
 * \file        dia_SelftestController.cpp
 *
 * \brief       This code controls selftest procedure for all external components
 *
 * \details     {insert file description here}
 *
 * \author      kaa1hi
 * \date        Nov 27, 2014
 *
 * \copyright   Robert Bosch Car Multimedia 2014
 */

#ifndef __INCLUDED_DIA_COMMON__
#include "common/depricated/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SYSTEM_ADAPTERS__
#include "common/framework/sysadapters/dia_common_system_adapters.h"
#endif

#ifndef DIA_SELFTESTCONTROLLER_H_
#include "common/framework/application/dia_SelftestController.h"
#endif

#ifndef DIA_SELFTESTCONTROLLERPLUGIN_H_
#include "common/framework/application/dia_SelftestControllerPlugin.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SELFTESTCONTROLLER_LISTENER__
#include "common/interfaces/dia_ISelftestControllerListener.h"
#endif

const char* ResultTypeToTxt[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COUNT] = {
   "RESULT_UNKNOWN",       // DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN = 0,
   "SELFTEST_REQUESTED",   // DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED,
   "RESULT_IN_PROGRESS",   // DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS,
   "RESULT_COMPLETED_OK",  // DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK,
   "RESULT_COMPLETED_NOK", // DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK,
   "RESULT_ABORTED",       // DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED,
   "RESULT_TIME_OUT"       // DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_TIMED_OUT
};

//-----------------------------------------------------------------------------

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_SelftestController);

//-----------------------------------------------------------------------------

dia_SelftestController::dia_SelftestController ( tVoid )
   : mSelftestResult(DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN),
     mpFSM(0)
{
   dia_tclFnctTrace oTrace("dia_SelftestController::dia_SelftestController()");
}

//------------------------------------------------------------------------------

dia_SelftestController::~dia_SelftestController ( tVoid )
{
   _BP_TRY_BEGIN
   {
      (tVoid) tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SelftestController::~dia_SelftestController !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//------------------------------------------------------------------------------

tDiaResult
dia_SelftestController::setup ( tVoid )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::setup()");

   tDiaResult retCode = DIA_FAILED;

   // create the state machine object
   mpFSM = OSAL_NEW dia_SelftestControllerFSM::Fsm(this);

   if ( mpFSM )
   {
       DIA_TR_INF( "### Initializing State Machine ###");
       mpFSM->init();
       DIA_TR_INF( "### Done ###");
       retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SelftestController::tearDown ( tVoid )
{
   if ( mpFSM )
   {
      OSAL_DELETE mpFSM;
      mpFSM = 0;
   }

   return DIA_SUCCESS;
}


//------------------------------------------------------------------------------

tDiaResult
dia_SelftestController::addSelftestControllerPlugin ( dia_SelftestControllerPlugin* pPlugin )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::addSelftestControllerPlugin()");

   tDiaResult retCode = DIA_FAILED;

   if ( pPlugin )
   {
      std::map<dia_eSelftestControllerPluginType,dia_SelftestControllerPlugin*>::iterator iter = mPluginRep.find(pPlugin->getUID());
      if ( iter == mPluginRep.end() )
      {
         DIA_TR_INF("#######################################################");
         DIA_TR_INF("#");
         DIA_TR_INF("# ADDING SELFTEST CONTROL PLUGIN \"%s\"", pPlugin->getName());
         DIA_TR_INF("#");
         DIA_TR_INF("#######################################################");
         mPluginRep[pPlugin->getUID()] = pPlugin;
         retCode = DIA_SUCCESS;
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SelftestController::removeSelfTestControllerPlugin ( dia_eSelftestControllerPluginType uniqueID )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::removeSelfTestControllerPlugin()");

   tDiaResult retCode = DIA_FAILED;

   std::map<dia_eSelftestControllerPluginType,dia_SelftestControllerPlugin*>::iterator iter = mPluginRep.find(uniqueID);
   if ( iter != mPluginRep.end() )
   {
      dia_SelftestControllerPlugin* pPlugin = iter->second;
      mPluginRep.erase(uniqueID);
      OSAL_DELETE pPlugin;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SelftestController::startSelfTests ( void )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::startSelfTests()");

   tDiaResult retCode = DIA_SUCCESS;
   size_t vectSize = mPluginRep.size();

   DIA_TR_INF("dia_SelftestController::startSelfTests number of plugins: %zu", vectSize);

   if (0!=vectSize)
   {
      if ( mpFSM )
      {
         DIA_TR_INF("CURRENT STATE: %s", mpFSM->getStateName());
      }
      DIA_TR_INF("SEND EVENT: dia_SelftestControllerFSM::evRequest");
      if ( mpFSM ) mpFSM->acceptEvent(dia_SelftestControllerFSM::evRequest,0);

      // if one of all components failed at start, send event 'evStartFailed'
      if (DIA_SUCCESS!=getSelfTestResult(&mSelftestResult))
      {
         mSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN;
         DIA_TR_ERR("ERROR mSelftestResult set to %s", ResultTypeToTxt[mSelftestResult]);
      }

      switch (mSelftestResult)
      {
         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK:
         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS:
         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED:
            /* do nothing, success */
         break;

         default:
            retCode = DIA_FAILED;

            DIA_TR_INF("SEND EVENT: dia_SelftestControllerFSM::evStartFailed");
            if ( mpFSM ) mpFSM->acceptEvent(dia_SelftestControllerFSM::evStartFailed,0);

         break;
      }
   }
   else
   {
      retCode = DIA_FAILED;
   }

   DIA_TR_INF("dia_SelftestController::startSelfTests return %s",  (DIA_FAILED==retCode? "FAILED": "SUCCESS"));

   return retCode;
}

//-----------------------------------------------------------------------------

bool
dia_SelftestController::bAllResponsesReceived(void) const
{
   dia_tclFnctTrace oTrace("dia_SelftestController::bAllResponsesReceived()");
   bool retVal;

   size_t vectSize = mPluginRep.size();

   if (0==vectSize)
   {
      DIA_TR_INF("dia_SelftestController::bAllResponsesReceived return TRUE");
      retVal = true;
   }
   else
   {
      tU32 numSelftestRequested = 0;
      tU32 numUnknown = 0;

      std::map<dia_eSelftestControllerPluginType,dia_SelftestControllerPlugin*>::const_iterator it = mPluginRep.begin();

      DIA_TR_INF("dia_SelftestController::bAllResponsesReceived plugin number: %zu", vectSize);

      while ( it!=mPluginRep.end() )
      {
         dia_SelftestControllerPlugin* pPlugin = it->second;

         if (NULL==pPlugin)
         {
            DIA_TR_ERR("dia_SelftestController::bAllResponsesReceived NULL PTR");
            numUnknown++;
         }
         else
         {
            switch (pPlugin->getSelftestResult())
            {
               case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN:
                  DIA_TR_ERR("Status of %s: UNKNOWN", pPlugin->getName());
                  numUnknown++;
               break;

               case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED:
                  DIA_TR_INF("Status of %s: SELFTEST_REQUESTED", pPlugin->getName());
                  numSelftestRequested++;
               break;

               default:
                  /* do nothing */
               break;
            }
         }
         ++it;
      }

      retVal = ( ((0==numSelftestRequested) && (0==numUnknown))? true: false);
   }

   DIA_TR_INF("dia_SelftestController::bAllResponsesReceived return %s", (retVal ? "TRUE": "FALSE") );

   return retVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SelftestController::getSelfTestResult ( dia_eSelftestControllerResultType* pResult )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::getSelfTestResult()");

   if (NULL==pResult)
   {
      return DIA_FAILED;
   }

   size_t vectSize = mPluginRep.size();
   const tU32 maxNumberOfPlugins = 255;

   // happy engineering, sunny day ;)
   dia_eSelftestControllerResultType retVal = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK;

   if (vectSize>maxNumberOfPlugins)
   {
      DIA_TR_INF("TOO MANY PLUGINS. INTERNAL ERROR. NUM OF PLUGINS IS %zu. MAX IS %u", vectSize, maxNumberOfPlugins);
      retVal = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED;
   }
   else
   {
      if (0!=vectSize)
      {
         tU8 resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COUNT] = {0};

         std::map<dia_eSelftestControllerPluginType,dia_SelftestControllerPlugin*>::iterator it = mPluginRep.begin();

         while ( it!=mPluginRep.end() )
         {
            dia_SelftestControllerPlugin* pPlugin = it->second;

            if (NULL!=pPlugin)
            {
               dia_eSelftestControllerResultType selftestResult = pPlugin->getSelftestResult();

               DIA_TR_INF("Get selftest result for plugin %s: %s", pPlugin->getName(), ResultTypeToTxt[selftestResult]);

               if (selftestResult<DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COUNT)
               {
                  resultArray[selftestResult]++;
               }
               else
               {
                  DIA_TR_INF("ERROR! DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN for plugin %s", pPlugin->getName());
                  resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN]++;
               }
            }
            else
            {
               DIA_TR_INF("ERROR! NULL PTR for plugin");
               resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN]++;
            }
            ++it;
         }

         DIA_TR_INF("########################################################################################");
         DIA_TR_INF("###### ResultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN]            = %d ######", resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN]);
         DIA_TR_INF("###### ResultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED] = %d ######", resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED]);
         DIA_TR_INF("###### ResultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS]        = %d ######", resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS]);
         DIA_TR_INF("###### ResultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK]       = %d ######", resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK]);
         DIA_TR_INF("###### ResultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK]      = %d ######", resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK]);
         DIA_TR_INF("###### ResultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED]            = %d ######", resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED]);
         DIA_TR_INF("###### ResultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_TIMED_OUT]          = %d ######", resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_TIMED_OUT]);
         DIA_TR_INF("########################################################################################");

         //check the array, set appropriate priorities for results
         if (resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN]>0)
         {
            DIA_TR_ERR("ERROR: DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED");
            retVal = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED;
         }
         else if (resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED]>0)
         {
            DIA_TR_INF("DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED");
            retVal = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED;
         }
         else if (resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_TIMED_OUT]>0)
         {
            DIA_TR_INF("DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_TIMED_OUT");
            retVal = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_TIMED_OUT;
         }
         else if (resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED]>0)
         {
            DIA_TR_INF("DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED");
            retVal = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED;
         }
         else if (resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS]>0)
         {
            DIA_TR_INF("DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS");
            retVal = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS;
         }
         else if (resultArray[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK]>0)
         {
            DIA_TR_INF("DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK");
            retVal = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK;
         }

         DIA_TR_INF("###################################################################################");
         DIA_TR_INF("############ DECISION FOR STATUS: %s ##########", ResultTypeToTxt[retVal]);
         DIA_TR_INF("###################################################################################");
      }
   }

   *pResult = retVal;

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SelftestController::requestSelfTestStatusIfInProgress ( void )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::requestSelfTestStatusIfInProgress()");

   tDiaResult retVal = DIA_SUCCESS;

   std::map<dia_eSelftestControllerPluginType,dia_SelftestControllerPlugin*>::iterator it = mPluginRep.begin();
   tU32 i = 0;

   while ( it!=mPluginRep.end() )
   {
      dia_SelftestControllerPlugin* pPlugin = it->second;

      if (NULL!=pPlugin)
      {
         dia_eSelftestControllerResultType selftestResult = pPlugin->getSelftestResult();

         DIA_TR_INF("Current selftest result for plugin[%d] %s: %s", i, pPlugin->getName(), ResultTypeToTxt[selftestResult]);

         if (DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS==selftestResult)
         {
            DIA_TR_INF("Send request to get selftest status for %s", pPlugin->getName());

            if (DIA_SUCCESS!=pPlugin->forceSelftestResult())
            {
               retVal = DIA_FAILED;
               DIA_TR_ERR("ERROR! forceSelftestResult finished UNSUCCESSFULLY for plugin %s", pPlugin->getName());
            }
         }
      }
      else
      {
         DIA_TR_INF("ERROR! NULL PTR for plugin");
         retVal = DIA_FAILED;
      }
      ++it;
      ++i;
   }

   DIA_TR_INF("dia_SelftestController::requestSelfTestStatusIfInProgress return %s", (DIA_SUCCESS==retVal) ? "DIA_SUCCESS": "DIA_FAILED" );

   return retVal;
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmHandleError ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmHandleError()");
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmRequestResultFailed ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmRequestResultFailed()");

   dia_ISelftestControllerListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_ISelftestControllerListener>(&pListener);
   if ( pListener )
   {
      DIA_TR_INF("SEND RESULT TO SERVICE HANDLERS: %s", ResultTypeToTxt[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED]);
      pListener->vOnSelfTestResult(DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED);
   }
   else
   {
      DIA_TR_ERR("NULL PTR. vFsmHandleError");
   }
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmStopFailed ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmStopFailed()");

   /* Impossible to stop selftest */
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmRequestSelftest ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmRequestSelftest()");

   std::map<dia_eSelftestControllerPluginType,dia_SelftestControllerPlugin*>::iterator it = mPluginRep.begin();
   size_t vectSize = mPluginRep.size();

   if (0!=vectSize)
   {
      while ( it != mPluginRep.end() )
      {
         dia_SelftestControllerPlugin* pPlugin = it->second;

         if (NULL!=pPlugin)
         {
            DIA_TR_INF("Start selftest for plugin %s", pPlugin->getName());

            if ( DIA_SUCCESS==(pPlugin->startSelftest()) )
            {
               DIA_TR_INF("Selftest started for Selftest Controller Plugin %s", pPlugin->getName());

               pPlugin->setSelftestResult(DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED);
            }
            else
            {
               DIA_TR_INF("##### Selftest Plugin Not Started %s !! #####", pPlugin->getName());

               pPlugin->setSelftestResult(DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED);
            }
         }
         else
         {
            DIA_TR_ERR("##### PTR NULL. ERROR!  #########");
         }
         ++it;
      }
   }
   else
   {
      DIA_TR_ERR("#################################################################");
      DIA_TR_ERR("#################################################################");
      DIA_TR_ERR("##### THERE ARE NO PLUGING. PLEASE CHECK CONFIGURATION  #########");
      DIA_TR_ERR("#################################################################");
      DIA_TR_ERR("#################################################################");
   }
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmRequestSuccess ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmRequestSuccess()");

   dia_ISelftestControllerListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_ISelftestControllerListener>(&pListener);
   if ( pListener )
   {
      DIA_TR_INF("SEND RESULT TO SERVICE HANDLERS: %s", ResultTypeToTxt[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS]);
      pListener->vOnSelfTestResult(DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS);
   }
   else
   {
      DIA_TR_ERR("NULL PTR. vFsmRequestSuccess");
   }
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmRequestFailure ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmRequestFailure()");

   dia_ISelftestControllerListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_ISelftestControllerListener>(&pListener);
   if ( pListener )
   {
      DIA_TR_INF("SEND RESULT TO SERVICE HANDLERS: %s", ResultTypeToTxt[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED]);
      pListener->vOnSelfTestResult(DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED);
   }
   else
   {
      DIA_TR_ERR("NULL PTR. vFsmRequestFailure");
   }
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmSelftestInProgress ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmSelftestInProgress()");

   /* Do nothing, just ignore this command. */
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmStartFailed ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmStartFailed()");

   dia_ISelftestControllerListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_ISelftestControllerListener>(&pListener);
   if ( pListener )
   {
      DIA_TR_INF("SEND RESULT TO SERVICE HANDLERS: %s", ResultTypeToTxt[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED]);
      pListener->vOnSelfTestResult(DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED);
   }
   else
   {
      DIA_TR_ERR("NULL PTR. vFsmStartFailed");
   }
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmRequestResultInProgress ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmRequestResultInProgress()");

   dia_ISelftestControllerListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_ISelftestControllerListener>(&pListener);
   if ( pListener )
   {
      DIA_TR_INF("SEND RESULT TO SERVICE HANDLERS: %s", ResultTypeToTxt[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS]);
      pListener->vOnSelfTestResult(DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS);
   }
   else
   {
      DIA_TR_ERR("NULL PTR. vFsmRequestResultInProgress");
   }
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmSelftestFailed ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmSelftestFailed()");

   /* This function should be called after one selftest lasts too long (timeout) */

   dia_ISelftestControllerListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_ISelftestControllerListener>(&pListener);
   if ( pListener )
   {
      DIA_TR_INF("SEND RESULT TO SERVICE HANDLERS: %s", ResultTypeToTxt[DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK]);
      pListener->vOnSelfTestResult(DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK);
   }
   else
   {
      DIA_TR_ERR("NULL PTR. vFsmSelftestFailed");
   }
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmSelftestDone ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmSelftestDone()");

   dia_ISelftestControllerListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_ISelftestControllerListener>(&pListener);
   if ( pListener )
   {
      // The variable 'mSelftestResult' is ready to be used in this place.
      DIA_TR_INF("SEND RESULT TO SERVICE HANDLERS: %s", ResultTypeToTxt[mSelftestResult]);
      pListener->vOnSelfTestResult(mSelftestResult);
   }
   else
   {
      DIA_TR_ERR("NULL PTR. vFsmSelftestDone");
   }
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmStopSelftest ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmStopSelftest()");

   /* Impossible to stop selftest */
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmStopDone ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmStopDone()");

   /* Impossible to stop selftest */
}

//------------------------------------------------------------------------------

void
dia_SelftestController::vFsmRequestResult ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vFsmRequestResult()");

   if (DIA_SUCCESS!=getSelfTestResult(&mSelftestResult))
   {
      mSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN;
      DIA_TR_ERR("ERROR mSelftestResult set to %s", ResultTypeToTxt[mSelftestResult]);
   }

   dia_ISelftestControllerListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_ISelftestControllerListener>(&pListener);
   if ( pListener )
   {
      DIA_TR_INF("SEND RESULT TO SERVICE HANDLERS: %s", ResultTypeToTxt[mSelftestResult]);
      pListener->vOnSelfTestResult(mSelftestResult);
   }
   else
   {
      DIA_TR_ERR("NULL PTR. vFsmRequestResult");
   }
}

//------------------------------------------------------------------------------

tVoid
dia_SelftestController::vOnSelftestControllerPluginUpdate ( dia_SelftestControllerPlugin& plugin )
{
   dia_tclFnctTrace oTrace("dia_SelftestController::vOnSelftestControllerPluginUpdate(plugin)");

   DIA_TR_INF("vOnSelftestControllerPluginUpdate -> NAME = %s , ID   = %d", plugin.getName(), plugin.getUID());

   if (DIA_SUCCESS!=getSelfTestResult(&mSelftestResult))
   {
      mSelftestResult = DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN;
      DIA_TR_ERR("ERROR mSelftestResult set to %s", ResultTypeToTxt[mSelftestResult]);
   }

   DIA_TR_INF("dia_SelftestController::vOnSelftestControllerPluginUpdate Got result: %s", ResultTypeToTxt[mSelftestResult]);

   if ( ! mpFSM ) {
      DIA_TR_ERR("ERROR: mpFSM not initialized. Process will fail!");
      return;
   }

   //if all components are ready, send event 'evResultReady'
   if (bAllResponsesReceived())
   {
      std::string CurrState(mpFSM->getStateName());
      DIA_TR_INF("dia_SelftestController::vOnSelftestControllerPluginUpdate curr state: %s", CurrState.c_str());

      switch(mSelftestResult)
      {
         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_IN_PROGRESS:
         {
            if ("Requested"==CurrState)
            {
               DIA_TR_INF("All component has replied. SEND EVENT: dia_SelftestControllerFSM::evConfirmedSuccess");
               mpFSM->acceptEvent(dia_SelftestControllerFSM::evConfirmedSuccess, 0);
            }
            else if ("Started"==CurrState)
            {
               DIA_TR_INF("All component has replied. Selftest still in progress for all components.");

               DIA_TR_INF("SEND EVENT: dia_SelftestControllerFSM::evRequestResult");
               mpFSM->acceptEvent(dia_SelftestControllerFSM::evRequestResult, 0);
            }
            else
            {
               DIA_TR_INF("dia_SelftestController::vOnSelftestControllerPluginUpdate ERROR. NOT EXPECTED BEHAVIOUR.");
            }
         }
         break;

         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_OK:
         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COMPLETED_NOK:
         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_ABORTED:
            DIA_TR_INF("All component has replied. SEND EVENT: dia_SelftestControllerFSM::evResultReady");
            mpFSM->acceptEvent(dia_SelftestControllerFSM::evResultReady, 0);
         break;

         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_UNKNOWN:
         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_SELFTEST_REQUESTED:
         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_TIMED_OUT:
         case DIA_EN_SELFTESTCONTROLLER_RESULT_TYPE_COUNT:
            DIA_TR_INF("dia_SelftestController::vOnSelftestControllerPluginUpdate: Do nothing");
            break;
      }

   }
}

