/*!
 * \file       dia_SelfTimer.h
 *
 * \brief      This file contains generic declaration and definition of a timer.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2019 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SELF_TIMER__
#define __INCLUDED_DIA_SELF_TIMER__

#ifndef __INCLUDED_DIA_INTERFACE_TIMER_LISTENER__
#include "common/interfaces/dia_ITimerListener.h"
#endif



#ifndef __INCLUDED_DIA_SELF_MSG_H__
#include "common/framework/application/dia_SelfMsg.h"
#endif


namespace dia {

template<class CLSTARGET, class MSG>
class SelfTimer: public  dia_ITimerListener {
public:
   struct MsgTimeout {
      MsgTimeout(uint32_t index):
         mIndex(index)
      {}
      uint32_t mIndex;
   };

   SelfTimer(CLSTARGET *target, RefCounter &refCounter):
      mRefCounter(refCounter),
      mRefCounterVal(nullptr),
      mTarget(target),
      mExpectedIndex(0),
      mPeriodMsec(0)
   {
      ScopeTrace oTrace("SelfTimer::SelfTimer()");
      /* 
         don't use reference here, it is not sure that refCounter is already fully constructed.
       */
      mTimer.s32Create();
      mTimer.addTimerListener(this);
   }

   ~SelfTimer() {
      ScopeTrace oTrace("SelfTimer::~SelfTimer()");
      mTimer.removeTimerListener(this);
      if (mRefCounterVal) {
         // time has been used. decrement refCounterVal ...
         mRefCounterVal->decrement();
         // .. and check if there are users left
         if (mRefCounterVal->isOrphan()) {
            // if not, deleted refCounterVal
            delete mRefCounterVal;
         }
      }
   }
   

   void stop() {
      mExpectedIndex++;
      mTimer.s32SetTime(0,0);
   }

   void start(uint32_t msec, uint32_t periodMsec=0 ) {
      DIA_TR_INF("SelfTimer::start() msec=%u", msec);
      if (!mRefCounterVal) {
         // first start of timer. increment refCounterVal. 
         // like this we make sure, that refCounterVal does not get deleted
         mRefCounterVal=mRefCounter.getVal();
         mRefCounter.increment();
      }
      mExpectedIndex++;
      mPeriodMsec=periodMsec;
      mTimer.s32SetTime(0,0);
      mTimer.s32SetTime(msec,mPeriodMsec);
   }

   // callback for system-timer:
   // is called in context of timer-thread, send event to self to handle is in correct thread context
   void vOnTimerElapsed(dia_TimerID /* id */) {
      ScopeTrace oTrace("SelfTimer::onTimeout()");
      MsgTimeout msg(mExpectedIndex);
      SelfMsg< SelfTimer<CLSTARGET, MSG >, SelfTimer<CLSTARGET, MSG >::MsgTimeout >::createAndPost(this, mRefCounterVal, &dia::SelfTimer<CLSTARGET, MSG>::onTimeoutEvent, msg);
   }

   // callback for timeout-event we send to ourself
   tDiaResult onTimeoutEvent(MsgTimeout msg) {
      ScopeTrace oTrace("SelfTimer::onTimeoutEvent()");
      DIA_TR_INF("SelfTimer::onTimeoutEvent() mExpectedIndex=%u msg->mIndex=%u", mExpectedIndex, msg.mIndex);
      if (mExpectedIndex!=msg.mIndex) {
         // if mExpectedIndex ischanged  timer has been stopped of restarted while event was in queue 
         return DIA_SUCCESS;
      }
      MSG targetMsg;
      uint32_t expectedIndexPre=mExpectedIndex;
      mTarget->onTimeout(&targetMsg);
      if (expectedIndexPre==mExpectedIndex && mPeriodMsec) {
         // timer has not been stopped or restarted and we have a periodical timer:
         mExpectedIndex++;
      }
      return DIA_SUCCESS;
   }

private:
   MSG mMsg;
   dia_Timer mTimer;
   RefCounter &mRefCounter;
   RefCounter::RefCounterVal *mRefCounterVal;
   CLSTARGET *mTarget;
   uint32_t mExpectedIndex;
   uint32_t mPeriodMsec;
};

}
#endif
