/*
 * dia_RemoteControl.cpp
 *
 *  Created on: 18.07.2012
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SYSTEM_ADAPTERS__
#include "common/framework/sysadapters/dia_common_system_adapters.h"
#endif

#ifndef __INCLUDED_DIA_REMOTECONTROL__
#include "common/framework/application/dia_RemoteControl.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_REMOTECONTROL_LISTENER__
#include "common/interfaces/dia_IRemoteControlListener.h"
#endif

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_RemoteControl)

#ifndef __DIA_UNIT_TESTING__

dia_RemoteControl*
getInstanceOfRemoteControl ( void )
{
   return dia_RemoteControl::getInstance();
}

void
releaseInstanceOfRemoteControl ( void )
{
   return dia_RemoteControl::deleteInstance();
}

#endif

//-----------------------------------------------------------------------------

dia_RemoteControl::dia_RemoteControl ( void )
   : mRemoteControlMode(DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE),
     mPrevRemoteControlMode(DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE),
     mErrorCode(DIA_E_NOERROR),
     mIsResultReady(FALSE),
     mpFSM(0),
     mRemoteControlSettings(DIA_C_U32_REMOTE_CTRL_SETTINGS_DEFAULT)
{

   dia_tclFnctTrace oTrace("dia_RemoteControl::dia_RemoteControl()");
}

//------------------------------------------------------------------------------

dia_RemoteControl::~dia_RemoteControl ( void )
{
   _BP_TRY_BEGIN
   {
      (void) tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_RemoteControl::~dia_RemoteControl !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RemoteControl::setup ( void )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::setup()");

   tDiaResult retCode = DIA_FAILED;

   // create the state machine object
   if ( dia_RemoteControlFSM::Fsm::createFSM (&mpFSM,this) == true )
   {
      DIA_TR_INF( "### dia_RemoteControl state machine instantiated and initialized ###");
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RemoteControl::tearDown ( void )
{
   if ( mpFSM )
   {
      OSAL_DELETE mpFSM;
      mpFSM = 0;
   }

   return DIA_SUCCESS;
}


//------------------------------------------------------------------------------

tDiaResult
dia_RemoteControl::addRemoteControlPlugin ( dia_RemoteControlPlugin* pPlugin )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::addRemoteControlPlugin()");

   if ( !pPlugin ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   std::map<dia_UID,dia_RemoteControlPlugin*>::iterator iter = mPluginRep.find(pPlugin->getUID());
   if ( iter == mPluginRep.end() )
   {
      DIA_TR_INF("#######################################################");
      DIA_TR_INF("#");
      DIA_TR_INF("# ADDING REMOTE CONTROL PLUGIN \"%s\"", pPlugin->getName());
      DIA_TR_INF("#");
      DIA_TR_INF("#######################################################");
      mPluginRep[pPlugin->getUID()] = pPlugin;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RemoteControl::removeRemoteControlPlugin ( dia_UID uid )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::removeRemoteControlPlugin()");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   std::map<dia_UID,dia_RemoteControlPlugin*>::iterator iter = mPluginRep.find(uid);
   if ( iter != mPluginRep.end() )
   {
      dia_RemoteControlPlugin* pPlugin = mPluginRep[uid];
      mPluginRep.erase(uid);
      OSAL_DELETE pPlugin;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tU16
dia_RemoteControl::numberOfRemoteControlPlugins ( void ) const
{
   return (tU16) mPluginRep.size();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RemoteControl::setRemoteControlMode ( dia_eRemoteCtrlMode mode, dia_RemoteControlSettings settings )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::setRemoteControlMode()");

   tDiaResult retCode = DIA_SUCCESS;

   mRemoteControlSettings = settings;

   DIA_TR_INF("setRemoteControlMode():mode=%d, settings=%d.", mode, settings);

   switch ( mode )
   {
   case DIA_EN_REMOTE_CONTROL_MODE_ACTIVE:
      if ( mpFSM ) mpFSM->acceptEvent(dia_RemoteControlFSM::evStart,0);
      break;

   case DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE:
      if ( mpFSM ) mpFSM->acceptEvent(dia_RemoteControlFSM::evStop,0);
      break;

   default:
      retCode = DIA_E_NOT_SUPPORTED;
      break;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

dia_eRemoteCtrlMode
dia_RemoteControl::getRemoteControlMode ( void )
{
   return mRemoteControlMode;
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::handleTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::handleTimeout()");

   if ( mpFSM ) mpFSM->acceptEvent(dia_RemoteControlFSM::evTimeout, 0);
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmReset ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmReset()");
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmPrepare ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmPrepare()");

   // initiate the iterator used to iterate over all available remote control plugins
   mPluginIter = mPluginRep.begin();
   mPluginIterActive = mPluginActivatedRep.begin();
   if ( !(mPluginIter->second) )
   {
      DIA_TR_INF("##### No RemoteControl Plugin Found in Repository !! #####");
   }
   else
   {
      DIA_TR_INF("##### Starting with RemoteControl Plugin %s !! #####", mPluginIter->second->getName());
   }
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmActivatePlugin ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmActivatePlugin()");

   if ( ( mPluginIter != mPluginRep.end() ) && mPluginIter->second )
   {
      DIA_TR_INF("##### Request RemoteControl Activation for Plugin %s !! #####", mPluginIter->second->getName());
      (void) mPluginIter->second->requestRemoteControlMode(DIA_EN_REMOTE_CONTROL_MODE_ACTIVE,mRemoteControlSettings);
   }
   else
   {
      DIA_TR_INF("##### No RemoteControl Plugin Found in Repository !! #####");
   }
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmSetNextPlugin ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmSetNextPlugin()");

   if ( mPluginIter != mPluginRep.end() )
   {
      DIA_TR_INF("##### Switch to next RemoteControl Plugin !! #####");

      mPluginIter++;

      if ( mPluginIter != mPluginRep.end() )
      {
         if ( mPluginIter->second )
         {
            DIA_TR_INF("##### Continuing with RemoteControl Plugin %s !! #####", mPluginIter->second->getName());
         }
      }
      else
      {
         DIA_TR_INF("##### No further RemoteControl plugins available !! #####");
      }
   }
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmSetNextDeactivationPlugin ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmSetNextDeactivationPlugin()");

   if ( mPluginIterActive != mPluginActivatedRep.end() )
   {
      DIA_TR_INF("##### Switch to next RemoteControl Plugin to Deactivate!! #####");

      mPluginIterActive++;

      if ( mPluginIterActive != mPluginActivatedRep.end() )
      {
         if ( mPluginIterActive->second )
         {
            DIA_TR_INF("##### Continuing with RemoteControl Plugin for Deactivation %s !! #####", mPluginIterActive->second->getName());
         }
      }
      else
      {
         DIA_TR_INF("##### No further RemoteControl plugins available to Deactivate!! #####");
      }
   }
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmSetActivatedPlugin ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmSetActivatedPlugin()");

   if ( mPluginIter != mPluginRep.end() )
   {
      DIA_TR_INF("##### Populate the activated RemoteControl Plugin !! #####");

      if ( mPluginIter->second )
      {
         dia_RemoteControlPlugin* pPlugin = mPluginIter->second;
         mPluginActivatedRep[pPlugin->getUID()] = pPlugin;
         mPluginIterActive++;
      }
      else
      {
         DIA_TR_INF("##### No further RemoteControl plugins available !! #####");
      }
   }

}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmPrepareDeactivation ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmPrepareDeactivation");
   // initiate the iterator used to iterate over all available remote control plugins
   mPluginIterActive = mPluginActivatedRep.begin();
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmEvaluateResponse ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmEvaluateResponse()");
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmDeactivatePlugin ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmDeactivatePlugin()");

   if ( ( mPluginIterActive != mPluginActivatedRep.end() ) && mPluginIterActive->second )
   {
      DIA_TR_INF("##### Request RemoteControl Deactivation for Plugin %s !! #####", mPluginIterActive->second->getName());
      (void) mPluginIterActive->second->requestRemoteControlMode(DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE,mRemoteControlSettings);
   }
   else
   {
      DIA_TR_INF("##### No RemoteControl Plugin Found in Repository !! #####");
   }
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmSetErrCode_OK ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmSetErrCode_OK()");
   mErrorCode = DIA_SUCCESS;
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmSetErrCode_Timeout ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmSetErrCode_Timeout()");
   mErrorCode = DIA_E_TIMEOUT;
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmNotify ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmNotify()");

   dia_IRemoteControlListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_IRemoteControlListener>(&pListener);
   if ( pListener )
   {
      pListener->vOnRemoteControlModeChanged(mRemoteControlMode,mPrevRemoteControlMode);
   }
   else
   {
      DIA_TR_INF("NO REMOTECONTROL LISTENER FOUND !!!");
   }
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmResultReady ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmResultReady()");
   mIsResultReady = TRUE;
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmSetModeON ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmSetModeON()");

   DIA_TR_INF("###########################################");
   DIA_TR_INF("#");
   DIA_TR_INF("# Activating RemoteControl !!");
   DIA_TR_INF("#");
   DIA_TR_INF("###########################################");

   mPrevRemoteControlMode = mRemoteControlMode;
   mRemoteControlMode = DIA_EN_REMOTE_CONTROL_MODE_ACTIVE;
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vFsmSetModeOFF ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vFsmSetModeOFF()");

   DIA_TR_INF("###########################################");
   DIA_TR_INF("#");
   DIA_TR_INF("# Deactivating RemoteControl !!");
   DIA_TR_INF("#");
   DIA_TR_INF("###########################################");

   mPrevRemoteControlMode = mRemoteControlMode;
   mRemoteControlMode = DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE;
}

//------------------------------------------------------------------------------

bool
dia_RemoteControl::bIsActivationComplete ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::bIsActivationComplete()");

   tU16 pluginCount = 0;

   std::map<dia_UID,dia_RemoteControlPlugin*>::iterator iter = mPluginRep.begin();

   for ( ; iter != mPluginRep.end(); iter++ )
   {
      dia_RemoteControlPlugin* pPlugin = iter->second;

      // increment counter if plugin is in remote control mode
      if ( pPlugin && (pPlugin->eGetMode() == DIA_EN_REMOTE_CONTROL_MODE_ACTIVE) ) pluginCount++;
   }

   if ( pluginCount == mPluginRep.size() )
   {
      DIA_TR_INF("##### ALL REMOTE CONTROL PLUGINS ACTIVATED #####");
      return true;
   }

   return false;
}

//------------------------------------------------------------------------------

bool
dia_RemoteControl::bIsActivationFailed ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::bIsActivationFailed()");

   bool result = true;

   if ( ( mPluginIter != mPluginRep.end() ) && mPluginIter->second )
   {
      if ((mPluginIter->second->eGetMode() == DIA_EN_REMOTE_CONTROL_MODE_ACTIVE))
      {
         result = false;
      }
   }

   return result;
}

//------------------------------------------------------------------------------


bool
dia_RemoteControl::bIsDeactivationComplete ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::bIsDeactivationComplete()");

   tU16 pluginCount = 0;

   std::map<dia_UID,dia_RemoteControlPlugin*>::iterator iter = mPluginActivatedRep.begin();

   for ( ; iter != mPluginActivatedRep.end(); iter++ )
   {
      dia_RemoteControlPlugin* pPlugin = iter->second;

      // increment counter if plugin is in remote control mode
      if ( pPlugin && (pPlugin->eGetMode() == DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE) ) pluginCount++;
   }

   if ( pluginCount == mPluginActivatedRep.size() )
   {
      DIA_TR_INF("##### ALL REMOTE CONTROL PLUGINS DEACTIVATED #####");
      return true;
   }

   return false;
}

//------------------------------------------------------------------------------

void
dia_RemoteControl::vOnRemoteControlPluginUpdate ( dia_RemoteControlPlugin& plugin )
{
   dia_tclFnctTrace oTrace("dia_RemoteControl::vOnRemoteControlPluginUpdate(plugin)");

   DIA_TR_INF("###########################################");
   DIA_TR_INF("#");
   DIA_TR_INF("# vOnRemoteControlPluginUpdate !!");
   DIA_TR_INF("#");
   DIA_TR_INF("# NAME = %s", plugin.getName());
   DIA_TR_INF("# UID  = 0x%08x", plugin.getUID());
   DIA_TR_INF("# MODE = %d", plugin.eGetMode());
   DIA_TR_INF("#");
   DIA_TR_INF("###########################################");

   if ( mpFSM ) mpFSM->acceptEvent(dia_RemoteControlFSM::evResponseReceived,(void*) &plugin);
}

